
/**************************************************************************
                        VCNVT.C

 Note : This file is for the voice file conversion between the ZyXEL 2, 3-bits
        ADPCM format and the Sound Blaster .VOC file format. (9.6k sampling
        uncompressed waveform)

 Usage : vcnvt  0  sfile dfile
                ^    ^     ^
                |    |     +--- Destination file path and name
                |    |
                |    +--- Source file path and name
                |
                +----- 0 : convert from .ZyXEL ADPCM to .VOC format
                +----- 1 : convert from .VOC to ZyXEL 2 bits ADPCM format
                +----- 2 : convert from .VOC to ZyXEL 3 bits ADPCM format

 Modified by Lin Chinru, Apr. 10, 1994. " For the Firmware > 6.12 "
 Created by Lin Chinru, Jan. 5, 1993.  R&D Dep., ZyXEL Comm. Corp.
 Copyright 1993, ZyXEL Communications Corporation
 **************************************************************************/
#include <stdio.h>

void ToVoc(int) ;
void ToAdpcm(int) ;
void Quit(int,char *) ;
void Adaptive(char) ;


char    Rcnt, CompBit ;
FILE    *SourFh, *DestFh ;

int     Rdata, EstMax, Delta, BlkCnt, Data, Pack ;
int     MaxTbl[] = { 0x399A, 0x3A9F, 0x4D14, 0x6607 } ;

/* ------------------------------------------------- */
/* Reserved buffers for the ZyXEL ADPCM file header  */
/* ------------------------------------------------- */
char    ZheadBuf0[16] ;
char    ZheadBuf1[16] = { 'Z','y','X','E','L',2,0,0,0,0,0,0,0,0,0,0 } ;

/* ------------------------------------------ */
/* Reserved buffers for the .VOC file header  */
/* ------------------------------------------ */
char    VheadBuf0[32] ;
char    VheadBuf1[32] = { 'C','r','e','a','t','i','v','e',' ',\
                          'V','o','i','c','e',' ','F','i','l','e',\
                          0x1a,0x1a,0,0x0a,1,0x29,0x11,\
                          1,0x82,0x70,0,0x98,0 } ;
char    VheadBuf2[4]  = { 2, 0x80, 0x70, 0 } ;

void main(int argc, char *argv[])
{
        /* ---------- Open file and check for legality -------------- */
        if ( argc!=4 || (*argv[1]!='0' && *argv[1]!='1' && *argv[1]!='2') )
                Quit(0,"") ;
        if ( (SourFh=fopen(argv[2],"rb")) == NULL )
                Quit(1,argv[2]) ;
        if ( (DestFh=fopen(argv[3],"wb")) == NULL )
                Quit(1,argv[3]) ;

        /* ------------------ */
        /* Initial parameters */
        /* ------------------ */
        Rdata   = 0 ;
        Rcnt    = 8 ;
        Delta   = 5 ;
        EstMax  = 0 ;
        BlkCnt  = 0x7080 ;

        /* -------------------------------- */
        /*  ZyXEL ADPCM -> .VOC conversion  */
        /* -------------------------------- */
        if ( *argv[1]=='0' )       {
                fread( ZheadBuf0, sizeof(char), 16, SourFh) ;
                if ( strcmp( ZheadBuf0, ZheadBuf1, 6) || ZheadBuf0[20]==0 )
                        Quit(2,argv[2]) ;
                fwrite( VheadBuf1, sizeof(char), 32, DestFh) ;
                CompBit = ZheadBuf0[10] ;
                if ( CompBit == 1 ) {
                        MaxTbl[0] = 0x3800 ;
                        MaxTbl[1] = 0x5600 ;
                }
                while ( (Data=getc(SourFh)) != EOF )    {
                        if ( CompBit == 1 )     {
                                ToVoc((Data&0xc0)>>6) ; /* XX-- ---- */
                                ToVoc((Data&0x30)>>4) ; /* --XX ---- */
                                ToVoc((Data&0x0c)>>2) ; /* ---- XX-- */
                                ToVoc(Data&0x03) ;      /* ---- --XX */
                        }
                        else    {
                                ToVoc((Data&0xe0)>>5) ; /* XXX- ---- */
                                ToVoc((Data&0x1c)>>2) ; /* ---X XX-- */
                                Pack = (Data&0x03)<<1 ;
                                Data = getc(SourFh) ;
                                ToVoc(Pack|((Data&0x80)>>7)) ;
                                ToVoc((Data&0x70)>>4) ; /* -XXX ---- */
                                ToVoc((Data&0x0e)>>1) ; /* ---- XXX- */
                                Pack = (Data&0x01)<<2 ;
                                Data = getc(SourFh) ;
                                ToVoc(Pack|((Data&0xc0)>>6)) ;
                                ToVoc((Data&0x38)>>3) ; /* --XX X--- */
                                ToVoc(Data&0x07) ;      /* ---- -XXX */
                        }
                }
                while ( BlkCnt-- )
                        putc( 0x7f, DestFh) ;
                putc( 0, DestFh) ;
        }

        /* -------------------------------- */
        /*  .VOC -> ZyXEL ADPCM conversion  */
        /* -------------------------------- */
        else    {
                fread( VheadBuf0, sizeof(char), 26, SourFh) ;
                if ( strcmp( VheadBuf0, VheadBuf1, 22) )
                        Quit(2,argv[2]) ;
                CompBit = *argv[1]-'0' ;
                if ( CompBit == 1 ) {
                        MaxTbl[0] = 0x3800 ;
                        MaxTbl[1] = 0x5600 ;
                }
                ZheadBuf1[10] = CompBit ;
                fwrite( ZheadBuf1, sizeof(char), 16, DestFh) ;

                /* Pack = 0->not accepted, 1->read sampling rate
                          2->read data, others->bypass          */
                while ( (Pack=getc(SourFh)) && (Pack!=EOF) ) {
                        /* Read the block length */
                        fread( &BlkCnt, sizeof(int), 1, SourFh) ;
                        /* Bypass this byte */
                        getc(SourFh) ;
                        /* Read the sampling rate, 9600 is required */
                        if ( Pack == 1 )        {
                                if ( (Data=getc(SourFh))<147 ||
                                      Data>157 || getc(SourFh) )
                                        Quit(2,argv[2]) ;
                                BlkCnt -= 2 ;
                        }
                        while ( BlkCnt-- )      {
                                Data = getc(SourFh) ;
                                if ( Pack <= 2 )
                                        ToAdpcm(Data) ;
                        }
                }
        }
        fclose(SourFh) ;
        fclose(DestFh) ;
}


/**************************************************************************
        ToVoc()



 Copyright 1992, ZyXEL Communications Corporation
 ************************************************************************/
void ToVoc(int Vdata)
{
int     Wdata ;

        Adaptive((char)Vdata) ;
        if ( EstMax > 8191 )
                Wdata = 8191 ;
        else if ( EstMax < -8192 )
                Wdata = -8192 ;
        else
                Wdata = EstMax ;
        putc( ((Wdata>>6)+128)&0xff, DestFh ) ;
        if ( !(--BlkCnt) )      {
                fwrite( VheadBuf2, sizeof(char), 4, DestFh) ;
                BlkCnt  = 0x7080 ;
        }
}

/**************************************************************************
        ToAdpcm()



 Copyright 1992, ZyXEL Communications Corporation
 ************************************************************************/
void ToAdpcm(int Edata)
{
char    TmpCompBit, DataBit ;


        DataBit = 0 ;
        Rdata  &= 0xff00 ;

        Edata -= 128 ;
        Edata <<= 6 ;
        Edata += 32 ;
        /* Check for the waveform data and quantize this data */
        if ( Edata -= EstMax )  {
                TmpCompBit = 2*CompBit ;
                /* ----------------------------------------------------- */
                /* If the data is negative, set flag and change the sign */
                /* ----------------------------------------------------- */
                if ( Edata < 0 ) {
                        Edata = -Edata ;
                        DataBit = TmpCompBit ;
                }
                /* --------------------------------------------------- */
                /* Quantize the waveform data, delta value is adaptive */
                /* --------------------------------------------------- */
                while ( ((Edata-=Delta)>0) && --TmpCompBit )
                        DataBit += 1 ;
                /* ---------------------------- */
                /* Rdata is the compressed data */
                /* ---------------------------- */
                Rdata |= ( DataBit << (7-CompBit) ) ;
        }
        /* ------------------------------------------------------ */
        /* check if the compressed data can be pack into one byte */
        /* ------------------------------------------------------ */
        TmpCompBit = CompBit+1 ;
        while ( TmpCompBit-- )     {
                Rdata <<= 1 ;
                if ( !(--Rcnt) )    {
                        putc(Rdata>>8,DestFh) ;
                        Rcnt = 8 ;
                }
        }
        /* -------------------------------------- */
        /* Adaptive the Delta and Estimat Maximum */
        /* -------------------------------------- */
        Adaptive(DataBit) ;
}

/**************************************************************************
        Adaptive(DataBit)



 Copyright 1992, ZyXEL Communications Corporation
 ************************************************************************/
void Adaptive(char DataBit)
{
int     TmpMax ;
char    TmpData, SignBit ;
long    TmpDelta ;
long    t0 ;

/******* This part is modified for the Firmware > 6.12 *************/
    t0 = (long)EstMax ;
    t0 *= 4093 ;
    t0 += 2048 ;
    t0 /= 4096 ;
    EstMax = (int)t0 ;
/*******************************************************************/


        SignBit = DataBit & (2*CompBit) ;
        DataBit &= ~(2*CompBit) ;
        if ( (Delta&1) && !SignBit )
                ++EstMax ;

        /* ------------------- */
        /* Calculate the Delta */
        /* ------------------- */
        TmpDelta = Delta ;
        TmpDelta *= MaxTbl[DataBit] ;
        TmpDelta += 8192 ;
        TmpDelta >>= 14 ;

        /* -------------------- */
        /* Calculate the EstMax */
        /* -------------------- */
        TmpMax  = (Delta>>1) ;
        while ( DataBit-- )
                TmpMax += Delta ;
        if ( SignBit )
                EstMax -= TmpMax ;
        else
                EstMax += TmpMax ;

        Delta = (int)TmpDelta ;
}

/**************************************************************************
        Quit()



 Copyright 1992, ZyXEL Communications Corporation
 ************************************************************************/
void Quit(int Num, char *MsgStr)
{
        if ( Num == 0 ) {
                printf("Usage : vcnvt 0 sfile dfile.\n");
                printf("              ^   ^     ^\n") ;
                printf("                       Destination file\n");
                printf("                  Source file\n") ;
                printf("               Convert type : \n") ;
                printf("    0 : convert from .ZyXEL ADPCM to .VOC format.\n") ;
                printf("    1 : convert from .VOC to ZyXEL 2 bits ADPCM.\n") ;
                printf("    2 : convert from .VOC to ZyXEL 3 bits ADPCM.\n") ;
        }
        else if ( Num == 1 )
                printf("Can't open %s\n", MsgStr) ;
        else if ( Num == 2 )
                printf("File format error on %s\n", MsgStr) ;
        exit(1) ;
}




