/*
 * File......: TBACHO.PRG
 * Author....: Steve Kolterman
 * CIS ID....: 76320,37
 * Date......: November 5, 1993
 * 
 * See the README.1ST file in TBACHO.ZIP for restrictions on redistribution.
 *
*/

/*  $DOC$
 *  $FUNCNAME$
 *     TBACHOICE()
 *  $ONELINER$
 *     Replace ACHOICE() with a Tbrowse object and multiple features.
 *  $SYNTAX$
 *     TBACHOICE( <nToprow>,<nTopcol>[,<nBotrow>][,<nBotcol>],<aArray>,     ;
 *           [<cBoxtype>],[<cBoxcolor>],[<cBoxtitle>],[<nTitlePos>],        ;
 *           [<cUselcolor>],[<cTitlecolor>],[<cBarcolor>],[<cHkcolor>],     ;
 *           [<cShadow>],[<lExecute>],[<nMsgrow>],[<nMsgcol>],              ;
 *           [<cMsg.color>],[cElevbar],[cEbarcolor],[<cEbarside>],          ;
 *           [<cNoSelcolor>],[<cTagch>],[<nStartelem>],[<lRscreen>],        ;
 *           [<nTimeout>],[<nTimeoutval>],[<lSaveTags>],[<cUserfunc>] )
 *      --> nOption
 *
 *  $ARGUMENTS$
 *
 *  <nToprow>   is the top row of the box to be drawn.  Required.
 *
 *  <nTopcol>   is the top column of the box to be drawn.  Required.
 *
 *  <nBotrow>   is the bottom row of the box to be drawn.  The default is
 *     <nToprow>+Len(<aArray>)+1 or maxrow()-2, whichever is less.
 *
 *  <nBotcol>   is the bottom column of the box to be drawn.  The default
 *     is <nTopcol>+width of the widest element in <aArray> +2.
 *
 *     It's been pointed out that the number of commas required to default
 *     the <nBotrow> and <nBotcol> params. is at least slightly confusing.
 *     So, some documentation on the requirements:
 *     Default both:                           7,9,,   ARRAY:
 *     Default <nBotrow>, specify <nBotcol>:   7,9,,20 ARRAY:
 *     Default <nBotcol>, specify <nBotrow>:   7,9,15, ARRAY:
 *
 *  Note:  the above four parameters have NO corresponding A_CHOICE()
 *     keywords.  A_CHOICE() is the pseudo-function defined in TBACHO.CH.
 *
 *  <aArray>    is the array of options to present to the user.  Each
 *     element can hold as many as five subelements, or as few as one.
 *     Required.  Additional documentation below.  A_CHOICE() keyword: ARRAY.
 *
 *  <cnBoxtype> is the type of box to draw.  Uses DispBox().  The
 *     default is a double-line box.  A_CHOICE() keyword: BOXTYPE.
 *
 *  <cBoxcolor> is the color with which to draw the box.  The default is
 *     Setcolor().  A_CHOICE() keyword: BOXCOLOR.
 *
 *  <cBoxtitle> is title of the box drawn on <nToprow>.  The default is
 *     no title.  A_CHOICE() keyword: BOXTITLE.
 *
 *  <nTitlepos>  is the starting column position (to the right of
 *     <nTopcol>) at which to draw <cBoxtitle>.  The default is 1.
 *     A_CHOICE() keyword: TITLEPOS.
 *
 *  <cUselcolor> is the color with which to draw unselected options.
 *     The default is Setcolor().  A_CHOICE() keyword: USELCOLOR.
 *
 *  <cTitlecolor> is the color with which to draw the box title.  The
 *     default is yellow on red.  A_CHOICE() keyword: TITLECOLOR.
 *
 *  <cBarcolor>  is the color with which to draw the selection bar.
 *     The default is yellow on black.  A_CHOICE() keyword: BAR_COLOR.
 *
 *  <cHkcolor>  is the default color with which to draw the hotkeys for
 *     for each option.  This is used when no hotkey color is supplied
 *     in <aArray>.  The default is hiwhite on the current background
 *     color.  A_CHOICE() keyword: HOTKEYCOLOR.
 *
 *  <cShadow>   is a character string, either "L" or "R" (for left or
 *     right) to designate the side of the box where a shadow will appear.
 *     Leave this NIL to avoid drawing a shadow.  You might also leave
 *     this NIL if you choose to use a .C or .ASM shadow function, which
 *     is a good idea.  Shadoww(), included below, is flat-out SLOW.
 *     A_CHOICE() keyword: SHADOW.
 *
 *  <lExecute>  turn on/off execution of option when first letter is
 *     pressed.  Rule:  setting in element 5 of each <aArray> subarray
 *     overrides <lexecute>.  If that element is left NIL, the <lexecute>
 *     setting is used.  If <lexecute> is not passed and element 5 is NIL,
 *     auto execution is turned ON.  A_CHOICE() keyword: AUTOEXEC.
 *
 *  <nMsgrow>  is the row on which to draw a message for each option.
 *     The default is two rows below the bottom of the box.  A_CHOICE()
 *     keyword:  MES_ROW.
 *
 *  <nMsgcol>  is the column at which to draw a message for each option.
 *     The default is <nBotcol> +2.  To CENTER the message on the screen,
 *     pass "C", to center within the box, pass "CB".  A_CHOICE()
 *     keyword: MES_COL.
 *
 *  <cMsgcolor>  is the default color with which to draw messages.  This
 *     color is used when element 4 of each <aArray> subarray is left NIL.
 *     The default is Setcolor().  A_CHOICE() keyword: MES_COLOR.
 *
 *  <cElevbar>  is the ASCII character to use as the elevator bar drawn
 *     on the box.  Leave this NIL to draw no elevator bar.  A_CHOICE()
 *     keyword:  ELEVBAR.
 *
 *  <cEbarcolor>  is the color with which to draw the elevator bar.
 *     This is ignored if <cElevbar> is NIL.  A_CHOICE() keyword:
 *     ELEVBAR_COLOR.
 *
 *  <cEbarside>  is a character string, either "L" or "R" (for left or
 *     right) to designate the side of the box on which to draw the
 *     elevator bar.  This is ignored if <cElevbar> is NIL.  A_CHOICE()
 *     keyword: ELEVBAR_SIDE.
 *
 *  <cNoselcolor>  is the color with which to draw unselectable options.
 *     The default is white on black.  A_CHOICE() keyword: NOSELCOLOR.
 *
 *  <cTagchar> is the ASCII character to use to draw tags that would
 *     appear to the right of each option.  The default is *DIS*abled
 *     tagging.  The default tag is "" (chr(251)).  A_CHOICE() keyword:
 *     TAGCHAR.
 *
 *  <nStartelem>  is the number of the option where the selection bar
 *     will first be placed.  Leave this NIL to begin at option 1.
 *     A_CHOICE() keyword: START_ELEM.
 *
 *  <lRestscrn>  is a logical to designate whether or not the screen
 *     coordinates occupied by the box and/or shadow should be restored
 *     before TBACHOICE() returns.  The default is .T.  A_CHOICE() keyword:
 *     REST_SCREEN.
 *
 *  <nTimeout>  is the number of seconds after which TBACHOICE() will
 *     timeout and return to the function/proced. which called it.  The
 *     default is 0.  A_CHOICE() keyword: TIME_OUT.
 *
 *  <nTimeoutVal> is an optional alternative numeric value TBACHOICE() will
 *     RETURN when/if it times out.  The default is the current element
 *     number.  A_CHOICE() keyword: TIME_OUT VALUE.
 *
 *  <lSaveTags> is an optional logical to indicate that you want any tags
 *     to remain with their respective options, thereby available for
 *     display during the next call of TBACHOICE().  Default .F.  A_CHOICE()
 *     keyword:  SAVE_TAGS.
 *
 *  <bUserfunc>  is a code block containing a function call to be
 *     executed after each key press.  You need to write just two formal
 *     parameters to allow the run-time passing of the key pressed and the
 *     current element number, e.g.:
 *               { |key,num| Myfunc( key,num [,other params.] ) }
 *     Unlimited extra parameters may be passed.  Of course, make certain
 *     to also write 'receptors' for them in 'Myfunc()' itself...as in the
 *     above example.  The default is NO user function.  A_CHOICE() keyword:
 *     USERFUNC.
 *
 *  $RETURNS$
 *     the number of the selected option, or 0 if [Esc] is pressed.
 *
 *  $DESCRIPTION$
 *     TBACHOICE() is a greatly enhanced, fully featured, and now mouse-
 *     supported replacement for Achoice(), based on a Tbrowse object. 
 *     Each element of <aArray> (the array you pass to it) is itself an
 *     array.  Each element can solely composed of "Option" (below), but
 *     may be composed as follows to take full advantage of the function's
 *     features:
 *
 *         Option   ,     Message      ,HotKeyPos,HotKeyColor,Selectable
 *     { "Utilities","System Utilities", 3       ,"+gr/b"    ,.T. }
 *
 *     All elements except for the first, the option itself, are optional.
 *     IF 'Message' is NIL, no message is displayed.  'HotKeyPos' is the
 *     position within 'Option' of the hotkey.  In the example above, the
 *     hotkey for 'Utilities' is the first 'i', i.e., at position 3.  The
 *     default is 1.  'HotKeyColor' is the color to use for the hotkey
 *     display.  The default is hiwhite  on the current background color.
 *     'Selectable' is a logical indicating whether or not that option can
 *     be selected.  The default is .T.
 *
 *     The A_CHOICE() UDC in TBACHO.CH makes using TBACHOICE() a breeze.
 *     The myriad of parameters can be written in any order.  Only <nToprow>,
 *     <nTopcol>, and <aArray> are required.  See the example below.
 *
 *     There may be some confusion over 'unselected' and 'unselectable'
 *     options.  The former refers to any option not currently covered
 *     by the selection bar.  The latter refers to options you have
 *     designated unselectable in subelement 5 of <aArray>, i.e., by
 *     writing .F.
 *
 *     To enable tagging, pass any ASCII character as <cTagchar>.  To
 *     tag/untag all options, press [SPACE].  To tag/untag individual
 *     options, press [+] and [-] respectively.  On press of [+], browse
 *     moves to the next element in the display.  To test for the tagged
 *     status of an option, use the WAS_TAGGED() UDC in TBACHO.CH. To
 *     check the entire array for tags, use Aeval() in conjunction with
 *     Was_Tagged() as in the example below.  When tagging is enabled, the
 *     string "Tags" will be written across the bottom row of the box in
 *     hiwhite on the current background color.
 *
 *     Because TBACHOICE() takes over the [SPACE],[+], and [-] keys, it saves
 *     any SET KEY procedures you might have set them to, and restores same
 *     on returning.  Any other procedures you might have SET KEYs to will
 *     fly when TBACHOICE() is called...thanks to the Inkey() replacement,
 *     SKINkey().
 *
 *     The pice de resistance of TBACHOICE() is its ability to execute
 *     a user function designed entirely by you.  It is called after each
 *     keypress, and because it is completely open-ended, extends the
 *     the reach of TBACHOICE() to the limits of Clipper.  See the docu-
 *     mentation under <bUserfunc> above.
 *
 *
 *     Test compile:  CLIPPER tbacho /n/w/m/dSK_TEST
 *     Test link   :  BLINKER FI tbacho LIB nanfor
 *
 *
 *     Mouse support
 *     =============
 *     Mouse support is provided via the Nanforum Toolkit FT_M* functions.
 *     Most actions are tied to the left button.  The equivalent of pressing
 *     [Esc] comes via the right button.  These left button clicks will
 *     produce the designated actions:
 *
 *     Mouse cursor outside box                :  K_ENTER (select option)
 *     Mouse cursor at box top left corner     :  browse:goTop()
 *                         bottom left corner  :  browse:goBottom()
 *                         top right corner    :  browse:pageUp()
 *                         bottom right corner :  browse:pageDown()
 *
 *     Mouse cursor at option, tagging ENabled
 *     --------------------------------------------
 *     Selection bar moves to option, subsequent press to tag or untag.  Do
 *     this for as many options as you want to tag/untag, then move mouse
 *     cursor outside the box.  Press again to select.  Tagging overrides
 *     <lExecute>, so with tagging on and <lExecute> .F., subsequent presses
 *     while inside the box coordinates simply tag/untag.
 *
 *     Mouse cursor at option, tagging DISabled
 *     ---------------------------------------------
 *     IF <lExecute> is turned on, the option is immediately selected.  If
 *     turned off, selection bar moves to option.  Press again to select.
 *
 *     To Eliminate Mouse Support
 *     ------------------------
 *     Compile with /n/w/m/dNOMOUSE
 *
 *     To Enable/Disable Mouse at Runtime
 *     ----------------------------------
 *     You must have compiled with /n/w/m.  In your code, write:
 *     SET KEY <yourKeyChoice> TO MouseOnOff()
 *
 *     Then, at runtime of course, press <yourKeyChoice> to toggle the
 *     mouse.  In the test routine, K_ALT_M is the toggle.  It is YOUR 
 *     responsibility to make sure the mouse is properly initialized 
 *     PRIOR to calling TBACHOICE():
 *     FT_MInit()       // initialize.
 *     nRet:= A_CHOICE( 7,9,, ARRAY:t_array )
 *
 *     IF you choose to begin with an initialized, but non-visible mouse
 *     cursor, as long as you've properly initialized and have SET a KEY
 *     to MouseOnOff(), you'll be able to toggle the mouse on at runtime.
 *
 *
 *  $EXAMPLES$
 *  nOpt := A_CHOICE( 7,9,, ARRAY:t_array )   // the minimum
 *
 *  nOpt := A_CHOICE( 7,9,, ;
 *            ARRAY:t_arrey ;
 *            USERFUNC:{|a,b| UserFunc(a,b,any1)};
 *            BOXTYPE:B_SINGLE  ;
 *            BOXTITLE:title  ;
 *            SHADOW:"FT" ;
 *            TAGCHAR:chr(17) ;
 *            REST_SCREEN:.F. ;
 *            AUTOEXEC:.F. ;
 *            MES_COLOR:MSG_COLOR ;
 *            ELEVBAR:"" ;
 *            NOSELCOLOR:"bg/n" ;
 *            MES_COL:"C" )
 *
 *  Check only the RETURNed element for whether tagged:
 *  IIF( Was_Tagged(chr(17),t_arrey,nOpt), MoreProcessing(), )
 *
 *  Check entire 't_arrey':
 *  Aeval( t_arrey,{|e,n| IIF( Was_Tagged(chr(17),t_arrey,n ), ;
 *                            MoreProcessing(t_arrey),NIL ) } )
 *
 *
 *  $END$
 */

#include "inkey.ch"
#include "box.ch"
#include "setcurs.ch"
#include "tbacho.ch"

#define KEY_ELEM         1
#define BLK_ELEM         2
#define AOPT             1
#define AMSG             2
#define AHOT             3
#define ACLR             4
#define ASEL             5
#define OUTTA_TIME       999
#define TIMED_OUT        (nKey==OUTTA_TIME)
#define HOTKEY_PRESS     (nElem > 0)
#define METHOD_PRESS     (nMeth_num > 0 .and. nMeth_num <= 10)
#define TAGS             (valtype(cTagchar)=="C")
#define TAG_PRESS        (TAGS .and. (nMeth_num > 11))
#define CONTINUING       (nKey <> K_ESC)
#define OUTTA_HERE       EXIT
#define ATTOP            (nAindex==1)
#define ATBOTT           (nAindex==Len(aArrey))
#define USEL_COLOR       FG(Setcolor())+"/"+BG(Setcolor())
#define BARCOLOR         IIF(Iscolor(),"+gr/n","n/w")
#define TITLECOLOR       IIF(Iscolor(),"+gr/r","n/w")
#define DEMOCOLOR        IIF(Iscolor(),"+gr/b","+w/n")
#define HK_COLOR         IIF(Iscolor(),"w+/"+ BG(Setcolor()),"w+/n")
#define SELECTABLE       (IIF(Len(aArrey[nAindex])==5 .and. aArrey[nAindex][5]<> NIL,;
                         aArrey[nAindex][5],lAexec))
#define NOSELECT         (Len(aArrey[nAindex])==5 .and. !(aArrey[nAindex][5]))
#define DEFAULT_TAG      ""
#define UP_ARROW_POS     t+2,nCol4bar
#define DN_ARROW_POS     b-2,nCol4bar
#define UP_ARROW         IIF(nTop_elem > 1,chr(24),chr(25))
#define DN_ARROW         IIF(nBot_elem < nNumelems,chr(25),chr(24))
#define GOING_UP         (nMeth_num==10 .or. Ltrim(Str(nMeth_num)) $ "2468")
#define GOING_DOWN       (Ltrim(Str(nMeth_num)) $ "13579")
#define MESG_LEN         aMsgdata[1]
#define MESG_COL         aMsgdata[2]
#xtrans MRow()     =>    FT_MGetX()

#xtranslate DISPMSG(<r>,<c>,<msg>[,<color>])           => ;
            SetPos(<r>,<c>); DispOut(<msg>[,<color>])
#translate  Clear([<t>,<l>,<b>,<r>])                   => ;
            SCROLL([<t>,<l>,<b>,<r>])
#command    DEFAULT <p> TO <val> [,<pn> TO <valn>]     => ;
            IIF( <p> == NIL, <p> := <val>, ) ;
            [; IIF( <pn> == NIL, <pn> := <valn>,) ]
#command    STABILIZE <o> => WHILE !<o>:stabilize() ; ENDDO

#ifndef K_SPACEBAR
#define K_SPACEBAR 32
#endif
#ifndef K_PLUS
#define K_PLUS  43
#define K_MINUS 45
#endif

STATIC aMsgData[2],aArrSave:= {},oSaveObj

#ifndef NOMOUSE
 STATIC lMouseOn:= .T.
#endif

#ifdef SK_TEST

Function Test( passes )

//                 Item       Msg         HotKeyPos/HotkeyColor/Selectable
LOCAL t_arrey:= { {"Larry"   ,"larry"    ,   ,"w+/b"          },;
                  {"Dick"    ,"dick"     ,   ,"b/r"           },;
                  {"Harry"   ,           ,  3,       ,.F.     },;
                  {"Steve"   ,"steve"    ,  4,"g/bg"          },;
                  {"Michelle","michelle"                      },;
                  {"Barnabas",           ,  6,"gr+/w"         },;
                  {"Fred"    ,"fred"                          },;
                  {"Lisa"    ,"lisa"     ,  3,"n/r"           },;
                  {"Eleanor" ,"eleanor"  ,  4,"bg/r"          },;
                  {"Anthony" ,"anthony"  ,  3                 },;
                  {"Charles" ,"charles"  ,   ,       ,.F.     },;
                  {"Ollie"   ,"ollie"    ,  4,"r/w"           },;
                  {"George"  ,           ,  5                 },;
                  {"Paula"   ,"paula"                         },;
                  {"Jack"    ,"jack"     ,  4                 },;
                  {"Quinten" ,"quinten"                       },;
                  {"Nancy"   ,"nancy"    ,  5,"w/n"           },;
                  {"Warren"  ,"warren"   ,  1,"n/gr*"         } }
LOCAL t_arrey2:= {{"Warren"  ,"warren"   ,   ,"w+/b"          },;
                  {"Charles" ,"charles"                       },;
                  {"Ollie"   ,"ollie"    ,  4,"r/w"           },;
                  {"George"  ,           ,  5                 },;
                  {"Paula"   ,"paula"    ,  3,"gr+/bg"        },;
                  {"Harry"   ,           ,  3,       ,.F.     },;
                  {"Michelle","michelle" ,   ,"gr+/gr"        },;
                  {"Anthony" ,"anthony"  ,  2                 } }

LOCAL title:= " SK Demo ",ret1,xx,nColor:= Setcolor( DEMOCOLOR ),o_blink
LOCAL any1:= "User function called!",ret2
LOCAL any2:= "User function2 called!"

DEFAULT passes to 3; passes:= IIF(valtype(passes)=="C",val(passes),passes)

Clear()

o_blink:= SetBlink(.F.)
#ifndef NOMOUSE
 FT_MInit() ; FT_MCursor(.T.)
 SET KEY K_ALT_M TO MouseOnOff()
#endif

FOR xx:= 1 to passes

  ret1:= A_CHOICE( 7,9,, ARRAY:t_arrey TITLEPOS:2 START_ELEM:ret1 ;
           USERFUNC:{|a,b| UserFunc(a,b,any1,.F.,1,.T.)} ;
           BOXTYPE:B_SINGLE  BOXTITLE:title  SHADOW:"FT" TAGCHAR:chr(17);
           REST_SCREEN:.F. AUTOEXEC:.F. MES_COLOR:"+w/b" ELEVBAR:"" ;
           SAVE_TAGS:.T. )

  @ 1,0 say "Returned element "+Ltrim(Str(ret1))+" "
  IF ret1 > 0
     @ 2,0 say "That was "+IIF( Was_Tagged(chr(17),t_arrey,ret1) ,;
               "a Tagged","an UNtagged")+"  element  "
  ENDIF
  @ 3,0 say "Press, Please "; inkey(0)

  Clear()
  ret2:= A_CHOICE( 5,9,20,40 ARRAY:t_arrey2  BOXTYPE:B_DOUBLE ELEVBAR:"" ;
            BOXTITLE:" SK Test2 " AUTOEXEC:.T. ELEVBAR_COLOR:"+w/r" ;
            MES_COLOR:"+w/gr" USERFUNC:{|a,b| UserFunc(a,b,any2,.T.,3,.F.,4)} ;
            REST_SCREEN:.F. ELEVBAR_SIDE:"R" TIME_OUT:4 MES_COL:"CB" ;
            START_ELEM:3 SHADOW:"L" BAR_COLOR:"+r/gr*" TIME_OUT VALUE:-999 )

  @ 1,0 say "Returned element "+Ltrim(Str(ret2))+" "
  IF ret2 > 0
    @ 2,0 say "That was "+IIF( Was_Tagged(DEFAULT_TAG,t_arrey2,ret2) ,;
               "a Tagged","an UNtagged")+"  element  "
  ENDIF
  @ 3,0 say "Press, Please "; inkey(0)
  Clear()
  
NEXT

SetBlink(o_blink)
QUIT
RETURN NIL

#endif

FUNCTION TBACHOICE( t,l,b,r,aArrey,xBoxtp,cBoxcolor,cBoxttl,nTtlpos,;
         cUselcolor,cTtlcolor,cBarcolor,cHkcolor,lShad,lAexec,nMsgrow,nMsgcol,;
         nMsgcolor,cEbar,cEbarcolor,cEbarside,cNoselcolor,cTagchar,nStartelem,;
         lRscreen,nTimeout,nTimeout_val,lSaveTag,bUfunc )

LOCAL nCurs,nKey:= 0,nMeth_num:= 0,nNumelems:= Len(aArrey),cAchscrn,;
      nColor,nElem:= 0,lEx_req:= .F.,lUfcont:= .T.,nTop_elem,nBot_elem,;
      lFirstEntry:= .T.,nCol4bar,lDidtag:=.F.,nAindex,nRow:= Row(),;
      nCol:= Col(),aHotkeys[3],aKeys,oBr,lDecr:= .F.,lCansel:= .T.,;
      cDir:= "D",cScrn

#ifndef NOMOUSE
 LOCAL lOldmouse
#endif

DEFAULT xBoxtp TO B_DOUBLE,       cTtlcolor TO TITLECOLOR,;
        cBarcolor TO BARCOLOR,    lRscreen TO .T. ,;
        nMsgcol TO l+2,           cNoselcolor TO "w/n" ,;
        nMsgcolor TO USEL_COLOR,  cBoxcolor TO Setcolor(),;
        cUselcolor TO USEL_COLOR, lAexec TO .T. ,;
        cEbarcolor TO Setcolor(), lSaveTag TO .F. ,;
        cEbarside TO "L",         nStartelem TO 1 ,;
        nTimeout TO 0,            nTtlpos TO 1

#ifndef NOMOUSE
 lMouseOn:= FT_MCursor() ; lOldmouse:= FT_MCursor()
#endif

nCurs := SetCursor(SC_NONE)
SR_keys( "S",aHotkeys )

IF b==NIL
   b:= IIF( (t+Len(aArrey)+1) >= maxrow()-2,maxrow()-2,(t+Len(aArrey)+1) )
ENDIF

DEFAULT nMsgrow TO b+2
cScrn:= SaveScreen( nMsgrow,nMsgcol,nMsgrow,maxcol() )

r:= PrepArray( aArrey,l,r,TAGS,cTagchar,lSaveTag )
cAchscrn := SaveScreen( t,l-2,b+2,r+2 )

IF aArrey==aArrSave
   nAindex:= oSaveObj:cargo[1]
   oBr:= oSaveObj ; oBr:inValidate()
ELSE
   nAindex:= 1
   oBr:= TBrowsenew( t+1,l+1,b-1,r-1 )
   oBr:addColumn( TBcolumnnew("",{|| aArrey[nAindex][AOPT]} ) )
   oBr:getColumn(1):width   := (r-1 -l)
   oBr:gotopblock           := {|| nAindex := 1}
   oBr:gobottomblock        := {|| nAindex := nNumelems}
   oBr:skipblock            := {|n| ASkip( n,@nAindex,aArrey )}
   oBr:colorspec            += ","+cUselcolor+","+cBarcolor+","+cNoselcolor
   oBr:getColumn(1):colorblock:= { || ;
                   IIF(NOSELECT,{8},{6}) ,;
                   oBr:getColumn(1):defcolor:= IIF(NOSELECT,{8,8},{6,7}) }
   oBr:cargo:= Array(1)
ENDIF

aArrSave:= AClone(aArrey)
lCanSel:= ( Ascan(aArrey,{|e| Len(e)==ASEL .and. (e[ASEL]==NIL ;
                              .or. e[ASEL]) }) ) == 0
aKeys:= CursKeys()

PaintBox( t,l,b,r,xBoxtp,cBoxcolor,cBoxttl,cTtlcolor,nTtlpos,lShad,TAGS )
nCol4bar  := IIF(Upper(cEbarside)=="L",l,r)
IIF( cEbar <> NIL,ElevBar( t+1,nCol4bar,b,cEbar,cEbarcolor,cEbarside ), )

oBr:autolite(.F.)

WHILE CONTINUING

   DispBegin()

   STABILIZE oBr

   IF lCanSel         // at least one option is selectable.
      WHILE NOSELECT
         IIF( cDir=="U",IIF(ATTOP,oBr:goBottom(),oBr:up()) ,;
                        IIF(ATBOTT,oBr:goTop(),oBr:down() ) )
         STABILIZE oBr
      ENDDO
   ENDIF

   nTop_elem:= 1+nAindex-oBr:rowPos
   nBot_elem:= nTop_elem+oBr:rowcount-1

   IF lFirstEntry .and. nStartelem > 1
      cDir:= HotKeyPress( oBr,aArrey,nStartelem,nAindex,nTop_elem,nBot_elem )
      nAindex:= nStartelem
      nTop_elem:= 1+nAindex-oBr:rowPos; nBot_elem:= nTop_elem+oBr:rowcount-1
   ENDIF

   HotKeyColor( t,l,nTop_elem,aArrey,oBr:rowcount,cHkcolor )

   DispMsgg( nMsgrow,nMsgcol,aArrey,nAindex,nMsgcolor,l,r )

   IF cEbar <> NIL
      DispMsg( UP_ARROW_POS,UP_ARROW,cEbarcolor )
      DispMsg( DN_ARROW_POS,DN_ARROW,cEbarcolor )
   ENDIF

   oBr:hilite()

   DispEnd()

#ifndef NOMOUSE
   IF lMouseOn
      IIF( !FT_MCursor(), FT_MCursor(.T.), )
   ENDIF
#endif

   // idle mode...of sorts.
   IF valtype(bUfunc)=="B"
      lUfcont:= Eval( bUfunc,nKey,IIF(lDidtag .and. lDecr,nAindex-1,nAindex) )
   ENDIF

   IF lEx_req .or. !lUfcont; OUTTA_HERE; ELSE; nKey:= 0; ENDIF

   **************************************************************************
   nKey     := SKInkey( nTimeout,oBr,aArrey,nAindex,t,l,b,r,TAGS,cTagchar,lAexec )
   **************************************************************************

   nMeth_num := Ascan( aKeys, {|e| nKey == e})
   nElem     := Ascan( aArrey,{|e| IIF(Len(e) >= AHOT .and. e[AHOT]<>NIL,;
                       Upper(Chr(nKey)) == Upper(Subs(Ltrim(e[AOPT]),e[AHOT],1)) ,;
                       Upper(Chr(nKey)) == Upper(Left(Ltrim(e[AOPT]),1)) ) } )

#ifndef NOMOUSE
   IF lMouseOn
      IIF( FT_MCursor(),FT_MCursor(.F.),)
   ENDIF
#endif

   DO CASE
   CASE TIMED_OUT
      lEx_req:= .T. ; nTimeout_val:= IIF(nTimeout_val==NIL,nAindex,nTimeout_val)
   CASE HOTKEY_PRESS
      cDir   := HotKeyPress(oBr,aArrey,nElem,nAindex,nTop_elem,nBot_elem)
      lEx_req:= SELECTABLE; nAindex:= nElem
   CASE METHOD_PRESS .or. nKey==K_ENTER
      ExecKey( nKey,oBr,ATTOP,ATBOTT )
      lEx_req:= ((nKey==K_ENTER) .and. !NOSELECT)
   CASE TAG_PRESS
      lDidtag:= TagPress( oBr,aArrey,nAindex,nKey,cTagchar,@lDecr )
   ENDCASE


   cDir   := IIF(GOING_UP,"U",IIF(GOING_DOWN,"D",cDir) )
   lEx_req:= IIF( nKey==0,.T.,lEx_req ) ; lFirstEntry:= .F.

ENDDO

oBr:cargo[1]:= nAindex ; oSaveObj:= oBr
Aeval( aArrey,{|e| e[AOPT]:= Ltrim(e[AOPT]) } )

IIF( lRscreen,RestScreen( t,l-2,b+2,r+2,cAchscrn ), )
RestScreen( nMsgrow,nMsgcol,nMsgrow,maxcol(),cScrn )
SetPos(nRow,nCol); SetCursor(nCurs)
SR_keys( "R",aHotkeys ) ; aMsgData:= Array(2)

#ifndef NOMOUSE
 FT_MCursor(lOldmouse)
#endif

RETURN IIF( nKey==K_ESC, 0, IIF(TIMED_OUT,nTimeout_val,nAindex) )
************************************************************************
STATIC FUNCTION ASKip( nNum_elems, nAindex, aArrey)
LOCAL nSave_aindex := nAindex
nAindex:= IIF( nAindex+nNum_elems > Len(aArrey), Len(aArrey),;
          IIF( nAindex+nNum_elems < 1, 1, nAindex+nNum_elems ))
RETURN (nAindex - nSave_aindex)
*************************************************************************
STATIC FUNCTION HotKeyPress( oBr,aArrey,nElem,nAindex,nTopElem,nBot_elem )
LOCAL nCur_elem:= nAindex,nNew_elem:= nElem,nDest,cDir:= "D"

WHILE nCur_elem < nNew_elem            // descending
   nDest:= MIN( nNew_elem,nBot_elem ) ; cDir:= "D"
   WHILE nCur_elem < nDest; oBr:down(); nCur_elem++; ENDDO    // speeding
   STABILIZE oBr
   WHILE nCur_elem < nNew_elem 
      oBr:down() ; STABILIZE oBr
      nCur_elem++
   ENDDO
ENDDO
WHILE nCur_elem > nNew_elem            // ascending
   nDest:= MAX( nNew_elem,nTopElem ) ; cDir:= "U"
   WHILE nCur_elem > nDest; oBr:up(); nCur_elem--; ENDDO      // speeding
   STABILIZE oBr
   WHILE nCur_elem > nNew_elem
      oBr:up() ; STABILIZE oBr
      nCur_elem--
   ENDDO
ENDDO

RETURN cDir
*************************************************************************
STATIC FUNCTION DispMsgg( r,c,aArrey,nPos,cMsgcolor,nLcol,nRcol )
LOCAL cColor2use
IF MESG_LEN==NIL ; MESG_LEN:= 0 ; MESG_COL:= 999; ENDIF
IIF( MESG_LEN > 0,Clear( r,MESG_COL,r,MESG_COL+MESG_LEN -1 ), )
IF Len(aArrey[nPos]) >= AMSG .and. aArrey[nPos][AMSG] <> NIL  // if msg. to disp.
   cColor2use:= IIF(Len(aArrey[nPos]) >= ACLR .and. aArrey[nPos][ACLR]<>NIL,;
                aArrey[nPos][ACLR],cMsgcolor)
   IF valtype(c)== "C" // indicating Centering
      c:= IIF(c=="C",Int( ((maxcol()+1)/2) -(Len(aArrey[nPos][AMSG])/2) ),;
                     Int( (nLcol+ (nRCol-nLcol)/2) -(Len(aArrey[nPos][AMSG])/2)) )
   ENDIF
   DispMsg( r,c,aArrey[nPos][AMSG],cColor2use )
   MESG_LEN:= Len(aArrey[nPos][AMSG]) ; MESG_COL:= c
ENDIF
RETURN NIL
*************************************************************************
/*
this is here for test purposes.  the default is NO user func.
*/
#ifdef SK_TEST

FUNCTION UserFunc( key,elem_num,anything,lAexec,st_elem,tag,tmout )
LOCAL ret_val:= .T.
DEFAULT tmout TO 0
@ 09,45 say "            LASTKEY: "+Ltrim(Str(key))+"  "
@ 10,45 say "CURRENT ELEMENT NUM: "+Ltrim(Str(elem_num))+"  "
@ 11,45 say "  AUTO-EXECUTION IS: "+IIF(lAexec,"ON ","OFF")
@ 12,45 say "STARTING AT ELEMENT: "+Ltrim(Str(st_elem))
@ 13,45 say "         TAGGING IS: "+IIF(tag,"ENABLED ","DISABLED")
@ 14,45 say "            TIMEOUT: "+IIF(tmout >0,Ltrim(Str(tmout))+" secs.  ",;
                                    "INACTIVE      ")
IF anything <> NIL; @ 16,45 say anything; ENDIF

/*
return .F. if you want to leave TBACHOICE() after whatever
processing you do here.
*/

RETURN (ret_val)
#endif

**************************************************************************
#define ELEM2USE    aArrey[nTopElem+nI]
#define CANT_SELECT (Len(ELEM2USE)==5 .and. !ELEM2USE[ASEL])

STATIC FUNCTION HotKeyColor( t,l,nTopElem,aArrey,nRowcount,cHkcolor )
LOCAL nI:= 0,cColor2use,nCol2use,nCharpos,nX, ;
      nUpper:= MIN(Len(aArrey),nRowcount)
FOR nX:= 1 TO nUpper
    cColor2use:= IIF( Len(ELEM2USE) >=ACLR .and. ELEM2USE[ACLR]<>NIL,;
                 ELEM2USE[ACLR], IIF(cHkcolor<>NIL,cHkcolor,HK_COLOR) )
    nCol2use:=   IIF( Len(ELEM2USE) >=AHOT .and. ELEM2USE[AHOT]<>NIL,;
                 l+1+ELEM2USE[AHOT],l+2)
    nCharpos:=   IIF( Len(ELEM2USE) >=AHOT .and. ELEM2USE[AHOT]<>NIL,;
                 ELEM2USE[AHOT],1 )
    IF !CANT_SELECT
       SetPos( t+nX,nCol2use )
       DispOut( SUBS(Ltrim(ELEM2USE[AOPT]),nCharpos,1),cColor2use )
    ENDIF
    nI++
NEXT
RETURN NIL
****************************************************************************
STATIC FUNCTION CursKeys()
RETURN (  { ;
          K_DOWN,     ;
          K_UP,       ;
          K_PGDN,     ;
          K_PGUP,     ;
          K_CTRL_PGDN,;
          K_CTRL_PGUP,;
          K_CTRL_END ,;
          K_CTRL_HOME,;
          K_END,      ;
          K_HOME,     ;
          K_ENTER,    ;
          K_SPACEBAR, ;
          K_PLUS,     ;
          K_MINUS } )
****************************************************************************
STATIC FUNCTION ExecKey( nKey,oBr,lTop,lBot )
DO CASE
   CASE nKey==K_DOWN       ; IIF(lBot,oBr:goTop(),oBr:down())
   CASE nKey==K_UP         ; IIF(lTop,oBr:goBottom(),oBr:up())
   CASE nKey==K_PGDN       ; IIF(lBot,oBr:goTop(),oBr:pagedown())
   CASE nKey==K_PGUP       ; IIF(lTop,oBr:goBottom(),oBr:pageup())
   CASE nKey==K_CTRL_PGDN .or. nKey==K_CTRL_END .or. nKey==K_END
                             IIF(lBot,oBr:goTop(),oBr:gobottom())
   CASE nKey==K_CTRL_PGUP .or. nKey==K_CTRL_HOME .or. nKey==K_HOME
                             IIF(lTop,oBr:goBottom(),oBr:goTop())
ENDCASE
RETURN NIL
****************************************************************************
STATIC FUNCTION ElevBar( t,nCol4bar,b,cBar,cColor )
LOCAL nC:= 0
Aeval( Array(b-t),{|e,n| SetPos(t+nC,nCol4bar),DispOut(cBar,cColor),nC++ })
RETURN NIL
****************************************************************************
#define TARGET   aArrey[nPos][AOPT]
#define TAGGED   (cTagchar $TARGET)
#define AEV_TARG aArrey[n][AOPT]
#define AEV_TAGD (cTagchar $AEV_TARG)

STATIC FUNCTION TagPress( oBr,aArrey,nPos,nKey,cTagchar,lDecr )
LOCAL lDidtag:= .F.

IF (nKey==K_PLUS .and. !TAGGED) .or. (nKey==K_MINUS .and. TAGGED)
   TARGET:= IIF( (nKey==K_PLUS .and. !TAGGED) ,;
                Left(TARGET,Len(TARGET)-1)+cTagchar ,;
            IIF( (nKey==K_MINUS .and. TAGGED) ,;
                Strtran(TARGET,cTagchar," ")  ,;
                TARGET ))
   oBr:refreshcurrent(); lDidtag:= .T.
   IF nKey==K_PLUS .and. TAGGED
      oBr:down() ; lDecr:= (nPos < Len(aArrey))
   ENDIF
ENDIF

IF nKey==K_SPACEBAR
   IF !(Ascan(aArrey,{|e| cTagchar $ e[AOPT] }) > 0)
      Aeval(aArrey,{|e,n| IIF( !(Len(e)==ASEL .and. !e[ASEL]) ,;
                         AEV_TARG:= Left(AEV_TARG,Len(AEV_TARG)-1)+cTagchar,) })
   ELSE; Aeval(aArrey,{|e,n| AEV_TARG:= Strtran(AEV_TARG,cTagchar," ") })
   ENDIF
   oBr:refreshall() ; lDidtag:= .T.
ENDIF
RETURN (lDidtag)
****************************************************************************
#translate CenterB( <b>,<l>,<r>,<msg>[,<color>] ) => ;
           SetPos(<b>,(<l>+Int((<r>-<l> -Len(<msg>))/2) ) ) ;;
           DispOut(<msg>[,<color>])

STATIC FUNCTION PaintBox( t,l,b,r,xBoxtp,cBoxcolor,cBoxttl,cTtlcolor,nTtlpos,cShad,lTags )
 IF cShad <> NIL
    IIF( cShad=="FT",FT_Shadow( t,l+1,b,r ), )
    IIF( cShad $"LR",Shadoww( t,l,b,r,Upper(cShad) ), )
 ENDIF
 DispBox( t,l,b,r,xBoxtp,cBoxcolor )
 IF cBoxttl <> NIL
    DispMsg( t,l+nTtlpos,cBoxttl,cTtlcolor )
 ENDIF
 IF lTags .and. (r-l) >= 4  // 4 == Len("tags")
    CenterB( b,l,r,"Tags","+w/"+BG(cBoxcolor) )
 ENDIF
RETURN NIL
****************************************************************************
STATIC FUNCTION PrepArray( aArrey,l,r,lTags,cTagchar,lSaveTag )
LOCAL nRet,nElem
DEFAULT cTagchar TO ""
nElem:= Ascan(aArrey,{|e| cTagchar $ e[AOPT]} )
IF lTags .and. lSaveTag .and. (nElem > 0)
   r:= 0
   Aeval( aArrey,{|e| e[AOPT]:= " "+e[AOPT],r:= MAX(r,Len(e[AOPT])) } )
   nRet:= (l+r+1)
ELSE
   Aeval( aArrey,{|e| e[AOPT]:= " " +AllTrim( ;
                  IIF( lTags,StrTran(e[AOPT],cTagchar),e[AOPT])) })
   IF valtype(r)=="N"
      Aeval( aArrey,{|e| e[AOPT]:= Padr(e[AOPT],r-l-1) })
   ELSEIF (r==NIL); r:= 0
      Aeval( aArrey,{|e| r:= MAX( r,Len(e[AOPT]) ) })
      r+= IIF( !lTags,(l+2),(l+3) )
      IIF( lTags,Aeval( aArrey,{|e| e[AOPT]:= Padr(e[AOPT],r-l-1) }), )
   ENDIF
   nRet:= r
ENDIF
RETURN nRet
*****************************************************************************
STATIC FUNCTION BG( cColor )
LOCAL nStartpos:= AT("/",cColor)+1
LOCAL nEndpos:= IIF( "," $ cColor,AT(",",cColor),Len(cColor)+1 )
RETURN Upper(Subs( cColor,nStartpos,(nEndpos-nStartpos) ))
*****************************************************************************
STATIC FUNCTION FG( cColor )
RETURN Upper(Subs( cColor,1,AT("/",cColor)-1 ))
*****************************************************************************
STATIC FUNCTION SKInkey( nSecs,oBr,aArrey,nAindex,t,l,b,r,lTags,cTagchar,lAexec )
LOCAL bBlock,nKey:= 0,lLooping:= .T.

WHILE lLooping

#ifndef NOMOUSE
    nKey:= MouseHandler( nSecs,oBr,aArrey,nAindex,t,l,b,r,lTags,cTagchar,lAexec )
    lLooping:= .F.
    IF nKey==0 .and. Nextkey() <> 0
#endif
       nKey:= Inkey( nSecs )
       IF ( bBlock := Setkey(nKey) ) <> NIL
          Eval( bBlock, Procname(1), Procline(1), Readvar(),Getactive() )
       ELSE; lLooping:= .F.
       ENDIF
#ifndef NOMOUSE
    ENDIF
#endif

ENDDO

RETURN (nKey)
****************************************************************************
#ifndef NOMOUSE

STATIC FUNCTION MouseHandler( nSecs,oBr,aArrey,nAindex,t,l,b,r,lTags,cTagchar,lAexec )
LOCAL nR:= 0,nCurrow,nNumpos:= 0,nKey:= 0,aCur_elem,lDown:= .F.,;
      nTime:= Seconds()

   // while no button or key pressed.
   WHILE Nextkey()==0 .and. nKey==0
      IF nSecs > 0 .and. (Seconds() >= nTime+nSecs)
         nKey:= OUTTA_TIME
      ELSEIF FT_Mbutprs(1)==2               // right button == ESC
         nKey:= K_ESC
      ELSEIF FT_Mbutprs(0)==1           // left button pressed
         DO CASE
           CASE FT_Minregion( t,l,t,l ) .or. FT_Minregion( b,l,b,l )
                nKey:= IIF( FT_Minregion( t,l,t,l ),K_CTRL_PGUP,;  // upper left
                       IIF( FT_Minregion( b,l,b,l ),K_CTRL_PGDN,nKey) ) // lower left
           CASE FT_Minregion( t,r,t,r ) .or. FT_Minregion( b,r,b,r )
                nKey:= IIF( FT_Minregion( t,r,t,r ),K_PGUP, ;      //upper right corner
                       IIF( FT_Minregion( b,r,b,r ),K_PGDN,nKey )) //lower left corner
           CASE !(FT_Minregion(t,l,b,r))
                nKey:= K_ENTER
           OTHERWISE
                IF FT_MInregion(t+1,l+1,t+Len(aArrey),r-1)
                   // mouse row.
                   nR:= MRow()
                   // what row does current elem occupy?
                   nCurrow:= oBr:nTop+oBr:rowPos-1
                   // difference between this and nR is number of positions to move.
                   nNumpos:= IIF( nR==nCurrow,0,ABS(nR-nCurrow))
                   aCur_elem:= aArrey[ nAindex+ IIF( nR > nCurrow,nNumpos,-nNumpos ) ]
                   IF nNumpos==0
                      nKey:= IIF(lTags, ;
                             IIF(cTagchar $ aCur_elem[1],K_MINUS,K_PLUS),;
                                K_ENTER)
                   ELSEIF (nNumpos > 0)
                      // if no hotkeys, we'll move ourselves and return -1.
                      // -1 indicating no handling in the main loop.
                      lDown:= (nR > nCurrow)
                      WHILE nR > nCurrow ; oBr:down(); nCurrow++ ; ENDDO
                      WHILE nR < nCurrow ; oBr:up()  ; nCurrow-- ; ENDDO
                      // if element is unselectable, move one above or below.
                      IF Len(aCur_elem)==ASEL .and. aCur_elem[ASEL]<>NIL .and. ;
                         !aCur_elem[ASEL]
                         IIF(lDown,oBr:down(),oBr:up())
                      ENDIF
                      nKey:= IIF( lTags .or. !lAexec, -1, K_ENTER)
                   ENDIF
                ENDIF
         ENDCASE
         // clear left mouse press.         
         WHILE (FT_MButprs(0)==1) ; ENDDO
      ENDIF
   ENDDO

RETURN (nKey)

#endif

****************************************************************************
STATIC FUNCTION SR_Keys( cAction,aHotkeys )
IF cAction=="S"
   Aeval( {K_SPACEBAR,K_PLUS,K_MINUS},;
          {|e,n| aHotkeys[n]:= Setkey( e ),Setkey( e,NIL )} )
ELSEIF cAction=="R"
   Aeval( {K_SPACEBAR,K_PLUS,K_MINUS},;
          {|e,n| Setkey( e,aHotkeys[n] )} )
ENDIF
RETURN NIL
****************************************************************************
STATIC FUNCTION Shadoww( t,l,b,r,cSide )
LOCAL cBx
DEFAULT cSide TO "R"
l+= IIF(cSide=="R",2,-2); r+= IIF(cSide=="R",2,-2)
cBx:= SaveScreen( ++t,l,++b,r )
RestScreen( t,l,b,r,Transf( cBx,Replic("x"+chr(8),Len(cBx)/2) ) )
RETURN NIL
****************************************************************************

#ifndef NOMOUSE

FUNCTION MouseOnOff()
IIF( lMouseOn,FT_MCursor( !FT_MCursor() ), )
lMouseOn:= !lMouseOn
RETURN NIL

#endif

