//----------------------------------------------------------------------------
//           Name: BackupMenu()
//        Purpose: Prompt user for source path to backup
//
//         Author: Wilson H. Yuen
//   Date created: 03-26-93
//   Time created: 11:19:55pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: BackupMenu()   --> NIL
//
//   Return Value: NIL
//
//    Description: Display backup menu option.
//
//           Note: Based on my current verion of Funcky & Exospace.
//                 If the current directory is "C:\BACKUP", Funcky's
//                 CurDir() will return:
//
//                 "C:\BACKUP"    //-- non-Exospace
//
//                 "BACKUP"       //-- with Exospace
//
//----------------------------------------------------------------------------
Function BackupMenu()
  #include "FUNCky.ch"
  #include "Inkey.ch"
  #include "SetCurs.ch"
  #include "Backup.ch"
  #define  pBACKUP  1
  #define  pRESTORE 2
  memvar getlist
  STATIC cSource                                      //-- source path
  STATIC cTarget                                      //-- target path

  local cCurColor   := SetColor("W+/G,W+/R,,,B/G")    //-- get/set current color settings
  local cCurCuror   := SetCursor(SC_NORMAL)           //-- get/set current cursor setting
  local cCurDir     := CurDir() + "\"                 //-- get current directory
  local cSaveScreen                                   //-- screen variable
  local nOption                                       //-- backup/restore option
  local cMsg                                          //-- misc message variable


  set scoreboard off
  cls(pREAD)
  cSaveScreen := SaveScreen(0,0,24,80)

  do while .t.
    cMsg := NIL

    RestScreen(0,0,24,80,cSaveScreen)
    nOption := ALERT("Backup or Restore Data?",;
                     { 'Backup', 'Restore', 'Exit' },;
                     {"W+/BR", "GR+/BR", "BG+/BR, W+/B"})

    If nOption == 0 .or. nOption == 3
       Exit
    Else

        If nOption == pBACKUP
           if cSource == NIL
              cSource := cCurDir + "*.d*"      + space(70)
              cTarget := cCurDir + "BKDBF.BK1" + space(30)
           endif
        Else
           if cSource == NIL
              cSource := cCurDir + "BKDBF.BK1" + space(30)
              cTarget := cCurDir + space(70)
           endif
        Endif

        //-- Get source and target path
        Box(15,15,18,65,pSINGLE_LINE,pALERT,1)
        @ 16,17 say "Enter source path:" get cSource PICTURE "@!S28"
        @ 17,17 say "Enter target path:" get cTarget PICTURE "@!S28"
        read

        do case
           case LastKey() == K_ESC
           exit
           case nOption == pBACKUP
                *----------------------------------
                *  Backup                         |
                *----------------------------------
                if FILECOUNT( cSource, A_NORMAL) == 0
                   //--  Are there files to be backed up
                   cMsg := "No files found to be backed up"

                elseIf empty(cTarget)
                   cMsg := {"Backup path and file name must be entered"}

                elseif IsDir(cTarget)
                   cMsg := {"Backup path and file name you entered is a sub-directory",;
                            "Please enter a valid path and file name"}

                //--  Target file exist
                elseif File(cTarget) .and.;
                       ALERT({"The file "+alltrim(cTarget)+" already exist.",;
                              "Do you want to overwrite this file?"},;
                              { 'Yes', 'No' },;
                              { "W+/BR", "GR+/BR", "BG+/BR, W+/B" } ) == 2
                      loop
                else

                   //-- no errors found

                   Zip( cSource, cTarget )

                endif


            case nOption == pRESTORE
                *----------------------------------
                *  Restore                        |
                *----------------------------------
                do case
                case ! file(cSource) .or. fopen(cSource) == -1
                   //--  Dose backup source file exist
                   cMsg := {"Backup file "+alltrim(cSource),;
                            "cannot be found or opened."}
                case ! empty(cTarget) .and. ! DirExist(cTarget)
                   //--  Dose target subdirectory exist
                   cMsg := {"Target path "+alltrim(cTarget)+" does not exist",;
                            "or the system was unable to create it.",;
                            "Please enter a valid target path."}
                case cMsg == NIL
                   //--  Restore from backup file
                   Unzip( cSource, cTarget )

                endcase

        endcase
    Endif

    If ! cMsg == NIL
       ErrMsg(cMsg)
    Endif
    RestScreen(0,0,24,80,cSaveScreen)
  enddo
  SetCursor(cCurCuror)
  SetColor(cCurColor)
  RestScreen(0,0,24,80,cSaveScreen)

Return(NIL)
* eof BackupMenu()
*
*
