//----------------------------------------------------------------------------
//           Name: Alert()
//        Purpose: Display a simple modal dialog box
//
//         Author: Wilson H. Yuen
//   Date created: 03-21-93
//   Time created: 05:58:52pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Alert( cMsg_, cOpt_, cColor_, lBeep ) --> nRetVal
//
//       Argument: cMsg_   - message to display in array or character format
//                 cOpt_   - list of possible responses
//                 cColor_ - [1] color to display the box in
//                           [2] color to display the message in
//                           [3] color to display the options in
//                 lBeep   - TRUE sound a beep, default TRUE
//
//   Return Value: nRetVal - Returns a numeric value indicating which option
//                           was chosen.  If the Esc key is pressed, the
//                           value returned is zero.
//
//    Description: Replaces Clipper's Alert()
//
//        Example: LOCAL aOptions := { 'Yes', 'No' }
//
//                 If ALERT("Shell out to DOS ?",;
//                          aOptions,;
//                           { "W+/BR", "GR+/BR", "BG+/BR, W+/B" } ) == 1
//                    DOSShell()
//                 Endif
//
//----------------------------------------------------------------------------
#include "setcurs.ch"
#include "Backup.ch"

//-- Alert()
#define  pALERT_BOXCOLOR  1
#define  pALERT_MSGCOLOR  2
#define  pALERT_OPTCOLOR  3


FUNCTION Alert( cMsg_, cOpt_, cColor_, lBeep )
    local nOptRow                               //-- starting row for option
    local nOptCol                               //-- starting column for option
    local nCounter                              //-- used in for/next loops
    local nArrayLen                             //-- length of the array of messages
    local nArrayLen2                            //-- length of the array of messages
    local cCurrScrn                             //-- save old screen so it can be restored later
    local nStartCol                             //-- starting column position of the message
    local nCurCursor := SetCursor( SC_NONE )    //-- cursor status
    local cMsg       := ""                      //-- the option message
    local cMsg2_     := {}                      //-- the array of messages to display
    local nMsgRow    := 12                      //-- the starting row to display the message on
    local nWidth     := 0                       //-- stores the width of the longest message
    local nRetVal    := 0                       //-- return value
    local cCurColor  := SetColor()              //-- current color setting.


    DEFAULT cOpt_   TO {"OK"},;
            cColor_ TO { "W+/R", "GR+/R", "BG+/R, W+/B" },;
            lBeep   TO pFALSE

    *-----------------------------------------------------------------------
    *  start by making sure all values have been passed.  If not, assign   |
    *  defaults.  message array.  If the message passed is character type, |
    *  then convert it to an array.                                        |
    *-----------------------------------------------------------------------
    do case
    case ValType(cMsg_) == "C"                   //  character type message was passed
        aadd(cMsg2_, cMsg_)
    case ValType(cMsg_) == "A"                   //  array type message was passed
        cMsg2_ := cMsg_
    endcase

    *-------------------------
    *  create option line.   |
    *-------------------------
    nArrayLen := len( cOpt_ )             //  number of options
    for nCounter := 1 to nArrayLen
        cMsg := cMsg + cOpt_[ nCounter ] + "   "
    next
    cMsg := Trim( cMsg )
    aadd( cMsg2_, cMsg )                  //  add the option line to the list


    *----------------------------------------
    *  determine length of longest message  |
    *----------------------------------------
    nArrayLen := len( cMsg2_ )
    for nCounter := 1 to nArrayLen
        nWidth := max( nWidth, len( cMsg2_[nCounter] ) )
    next

    *----------------------------------------------------------
    *  calculate the left column of the start of the message  |
    *----------------------------------------------------------
    nStartCol := round( (lastcol()-nWidth) /2,0 )


    *----------------------------------------------------------
    *  save the screen where the message will be displayed    |
    *----------------------------------------------------------
    nArrayLen2 := nArrayLen + 2
    cCurrScrn  := SaveScreen( nMsgRow-1, nStartCol-6, nMsgRow+nArrayLen2, nStartCol+nWidth+3 )


    *--------------------
    *  draw the box     |
    *--------------------
    Box( nMsgRow-1, nStartCol-4, nMsgRow+nArrayLen+1 , nStartCol+nWidth+3, , atoattr(cColor_[ pALERT_BOXCOLOR ]), 1 )


    *---------------------------
    *  display the message     |
    *---------------------------
    nArrayLen := nArrayLen - 1
    for nCounter := 1 to nArrayLen
        CENTER( nMsgRow-1+nCounter, cMsg2_[ nCounter ], cColor_[ pALERT_MSGCOLOR ] )
    next


    CENTER( nMsgRow+nCounter, cMsg2_[ nCounter ], cColor_[ pALERT_MSGCOLOR ] )

    *---------------------------------------
    *  calculate position of first option  |
    *---------------------------------------
    nOptRow := nMsgRow + nCounter
    nOptCol := 40 - len( cMsg2_[ nCounter ]) / 2


    SetColor( cColor_[ pALERT_OPTCOLOR ])
    nArrayLen := len( cOpt_ )
    for nCounter := 1 to nArrayLen
        @ nOptRow, nOptCol PROMPT cOpt_[ nCounter ]
        nOptCol := nOptCol + 3 + len( cOpt_[ nCounter ] )
    next
    if lBeep
       beep()
    endif

    Menu to nRetVal

    RestScreen( nMsgRow-1, nStartCol-6, nMsgRow+nArrayLen2, nStartCol+nWidth+3, cCurrScrn )

    SetCursor( nCurCursor )                         // reset the old cursor shape
    SetColor( cCurColor )

return( nRetVal )
* end Alert()
*
*
//----------------------------------------------------------------------------
//           Name: Center()
//        Purpose: Center a string
//
//         Author: Wilson H. Yuen
//   Date created: 04-02-93
//   Time created: 10:26:42pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Center( nRow, cString, cColor ) --> NIL
//
//       Argument: nRow    - row to center string on
//                 cString - string to center
//                 cColor  - color to display string with
//
//   Return Value: NIL
//
//    Description: Center a string.
//
//        Example: CENTER( 2, "Hello world", "W+/B" )
//
//----------------------------------------------------------------------------
Function Center(nRow, cString, cColor)
  local start_pos := round( ( 79 - len(cString) ) /2,0 )
  cColor := if( cColor == NIL, SetColor(), cColor)
  @ nRow, start_pos say cString COLOR cColor
return( NIL )
* eof center()
*
*
