* FileName: GBROWSE.PRG
*
* This file contains the functions required to allow a browse object within
* a READ command. The browse functions as a scrolling multi-field get.
*
* (c) Copyright 1992, John D. Lay
* ALL RIGHTS RESERVED
*
***************************************************************************
* This is the beginning of a Get:Browse                                   *
***************************************************************************
*
* Create Get Object for the following command:
*
* @ <Top>, <Left> TO <Right>, <Bottom> BROWSE
*                                  [ALIAS <(alias)>]
*                                  [PICTURE <pic>]
*                                  [COLOR <clr>]
*                                  [FIELDS <fields,...>]
*                                  [DEFAULT  <(defld1)> TO <defval1>
*                                          [,<(defldn)> TO <defvaln>]]
*                                  [VALID <vald>]
*                                  [FOR <for>] 
* => GetNewBrowse( <Top>, <Left>, <Right>, <Bottom>, <(alias)>,
*                  <Picture>, { <(fields)> }, 
*                  { {<(defld1)>,<(defval1)>}[, {<(defldn)>,<(defvaln)>] },
*                  <{valid}>, <{for}>,<clr> )
*
#include "Set.ch"
#include "Inkey.ch"
#include "Getexit.ch"
#include "SetCurs.ch"
#include "Fresh.ch"
*
* Defines for Get Cargo := Browse object
#define GETTYPE      1  // "BROWSE"
#define GETWINDOW    2  // Array: { nTop, nLeft, nBottom, nRight }
#define GETCARGO     3  // Cargo
#define GBALIAS      1
#define GBFOR        2
#define GBVALID      3
#define GBREADONLY   4 
#define GBPICTURE    5
#define GBDEFAULT    6
#define GBAPPEND     7
#define GBFIELDS     8 
#define GBCARGO      8 

#define K_UNDO  K_CTRL_U

**************************************************************************
*
* Function: GetNewBrowse( nTop, nLeft, nBottom, nRight, cAlias, cPicture,;
*                         aFields, aDefaults, bValid, bFor, cColors,     ;
*                         lReadOnly )
*
* This function is the constructor method for the getbrowse object.
* Input: nTop, nLeft, nBottom, nRight: Screen position
*        cAlias:    file alias
*        cPicture:  Picture to use for get
*        aFields:   array of fields in browse
*        aDefaults: array of default values for fields (input or not)
*        bValid:    valid block for get
*        bFor:      valid for browse
*        cColors:   screen colors
*        lReadOnly: if true then do not allow editing
*
STATIC bInsToggle := { || SETCURSOR( IF( READINSERT( !READINSERT() ), ;
                                         SC_NORMAL, SC_INSERT ) ) }

FUNCTION GetNewBrowse( nTop, nLeft, nBottom, nRight, cAlias, cPicture,    ;
                       aFields, aDefaults, bValid, bFor, cColors, lReadOnly )

   LOCAL oGet    := GetNew( nTop, nLeft ),;
         oBrowse := TBrowseNew( nTop, nLeft, nBottom, nRight )
   LOCAL x, nNumFields, cFieldName, nSel
      
   bFor      := IF( bFor   == NIL, {|| .T. }, bFor   )
   bValid    := IF( bValid == NIL, {|| .T. }, bValid )
   cAlias    := IF( EMPTY( cAlias ), ALIAS(), cAlias )
   lReadOnly := IF( lReadOnly == NIL, .F., lReadOnly )

   nSel := SELECT( cAlias )
   FOR x := 1 TO LEN( aDefaults )
      AADD( aDefaults[ x ], ; 
            FIELDWBLOCK( aDefaults[ x, 1 ], nSel ) )
   NEXT x

   oGet:name      := cAlias
   oGet:picture   := cPicture
   oGet:colorSpec := cColors
   oGet:Reader := {| get, aGets, nGetPos | gbReader( get, aGets, @nGetPos ) }

   IF EMPTY( aFields )
      aFields := (cAlias)->(DBSTRUCT())
   ENDIF

   nNumFields := LEN( aFields )

   oBrowse:colorSpec := cColors

   oBrowse:cargo := ARRAY( GBCARGO )
   
   oBrowse:cargo[ GBALIAS ]    := cAlias
   oBrowse:cargo[ GBFOR ]      := bFor
   oBrowse:cargo[ GBVALID]     := bValid
   oBrowse:cargo[ GBPICTURE ]  := cPicture
   oBrowse:cargo[ GBDEFAULT ]  := aDefaults
   oBrowse:cargo[ GBREADONLY ] := lReadOnly
   oBrowse:cargo[ GBAPPEND ]   := .F.

   oBrowse:skipBlock      := {| n | gbSkipper( n, oBrowse ) }
   oBrowse:goTopBlock     := {|   | gbTop( oBrowse )        }
   oBrowse:goBottomBlock  := {|   | gbBottom( oBrowse )     }
   oBrowse:autoLite := .F.

   oBrowse:cargo[ GBFIELDS ] := ARRAY( nNumFields )
   FOR x := 1 TO nNumFields
      IF VALTYPE( aFields[ x ] ) == "A"
         cFieldName := aFields[ x, 1 ]
      ELSE
         cFieldName := aFields[ x ]
      ENDIF
      oBrowse:cargo[ GBFIELDS, x ] := cFieldName
      oBrowse:addColumn(TBColumnNew(,FIELDWBLOCK(cFieldName,SELECT(cAlias))))
   NEXT x

   gbTop( oBrowse )

   oBrowse:refreshall()

   DO WHILE !oBrowse:stabilize()
   ENDDO

   oGet:cargo := { "BROWSE", { nTop, nLeft, nBottom, nRight }, oBrowse }

   oBrowse:autoLite := .T.

   RETURN oGet
**************************************************************************
*
* Function: gbReader( oGet )
*
* This function is the Reader method for the getbrowse object.
* It 1) activates the browse (get), 
*    2) PreValidates the get, 
*    3) calls gbApplyKey for each keyboard message,
*    4) Cleans up the browse prior to exit (stabilize),
*    5) de-activates the browse (get)
* 
*
FUNCTION gbReader( oGet, aGetList, nPos )
   LOCAL oBrowse, nMsButton, nMsX, nMsY, nNumGets, n, nKey 
   LOCAL nMsGetPos

*  Retrieve browse object
   oBrowse := oGet:cargo[ 3 ]

*  Activate "Get"
   oBrowse:hiLite()

*  Check WHEN clause
	IF GetPreValidate( oGet )

*  Process keyboard messages
      WHILE oGet:exitState == GE_NOEXIT
         nMsButton := 0
         WHILE ( ( nKey := inkey() ) == 0 .AND. ( nMsButton == 0 ) )
            MsStatus( @nMsButton, @nMsX, @nMsY )
         ENDDO
         IF ( nMsButton == MB_LEFT )
            nMsGetPos := nPos
            nNumGets  := LEN( aGetList )
            FOR n := 1 to nNumGets
               // See if mouse press is in a GET
               IF IsMouseInGet( aGetList[n], nMsX, nMsY )
                  nMsGetPos := n            // Assign it to mget position
                  EXIT
               ENDIF
            NEXT n
            IF ( nPos == nMsGetPos )
               // Click within browse
               gbApplyKey( oGet, oBrowse, 0, nMsButton, nMsX, nMsY )
            ELSE
               oGet:exitState := GE_WRITE
               nPos := IF( nMsGetPos > 0, nMsGetPos, nPos )
            ENDIF
         ELSEIF ( nMsButton == MB_RIGHT )
            __keyboard( CHR( K_ESC ) )
         ELSE
            gbApplyKey( oGet, oBrowse, nKey, 0, nMsX, nMsY )
         ENDIF
      ENDDO

   ENDIF

*  Clean up display (ensure browse is stable)
   DO WHILE !oBrowse:stabilize()
   ENDDO

*  De-Activate "Get"
   oBrowse:deHiLite()

*  Done...
   RETURN NIL
**************************************************************************
*
* Function: gbApplyKey( oGet, oBrowse, nKey )
*
* This function will process keystrokes for the getbrowse.
* It handles dispatching keyboard messages to the get and browse
* objects.
*
STATIC FUNCTION gbApplyKey( oGet, oBrowse, nKey, nMsPress, nMsX, nMsY )
   LOCAL lUpdateSave, nBrowseX 
   LOCAL cAlias := oBrowse:cargo[ GBALIAS ]

   MsHideAt( oBrowse:nTop, oBrowse:nLeft, oBrowse:nBottom, oBrowse:nRight )
   DO CASE
      CASE nMsPress == MB_LEFT
         nBrowseX := (oBrowse:nTop + oBrowse:rowpos) - 1
         DO WHILE nBrowseX != nMsX
            DO CASE
               CASE nBrowseX > nMsX
                  --nBrowseX
                  oBrowse:up()
               CASE nBrowseX < nMsX
                  ++nBrowseX
                  oBrowse:down()
            ENDCASE
         ENDDO

      CASE nKey == K_INS
         IF gbAppend( oBrowse )
            EVAL( bInsToggle )
         ENDIF

      CASE nKey == K_UP
         IF oBrowse:hitTop
            oGet:exitState := GE_UP
         ELSE
            gbAppend( oBrowse, .F. ) // Turn Off Append 
            oBrowse:up()
         ENDIF

      CASE nKey == K_SH_TAB
         oGet:exitState := GE_UP

      CASE nKey == K_DOWN
         IF oBrowse:hitBottom
            IF oBrowse:cargo[ GBREADONLY ]
               oGet:exitState := GE_DOWN
            ELSE
               IF !gbAppend( oBrowse )
                  gbAppend( oBrowse,.T. ) // Turn Append On
                  oBrowse:down()  // Show blank line
                  WHILE (!oBrowse:stabilize()) ; END
               ELSE
                  oGet:exitState := GE_DOWN
               ENDIF
            ENDIF
         ELSE
            oBrowse:down()
         ENDIF

      CASE nKey == K_TAB
         oGet:exitState := GE_DOWN

      CASE nKey == K_ENTER
         IF oBrowse:hitBottom 
            IF oBrowse:cargo[ GBREADONLY ]
               oGet:exitState := GE_ENTER
            ELSE
               IF !gbAppend( oBrowse )
                  gbAppend( oBrowse, .T. ) // Turn Append On
                  oBrowse:down()  // Show blank line
                  WHILE (!oBrowse:stabilize()) ; END
               ELSE
                  oGet:exitState := GE_ENTER
               ENDIF
            ENDIF
         ELSE
            IF oBrowse:colPos == oBrowse:colCount
               oBrowse:down()
               oBrowse:home()
            ELSE
               oBrowse:right()
            ENDIF
         ENDIF

      CASE nKey == K_ESC
         oGet:exitState := GE_ESCAPE

      CASE nKey == K_PGUP
         IF oBrowse:hitTop
            oGet:exitState := GE_WRITE
         ELSE
            gbAppend( oBrowse, .F. ) // Turn Off Append 
            oBrowse:pageDown()
         ENDIF

      CASE nKey == K_CTRL_PGUP
         IF oBrowse:hitTop
            oGet:exitState := GE_WRITE
         ELSE
            gbAppend( oBrowse, .F. ) // Turn Off Append 
            oBrowse:goTop()
         ENDIF

      CASE nKey == K_PGDN
         IF oBrowse:hitBottom
            oGet:exitState := GE_WRITE
         ELSE
            gbAppend( oBrowse, .F. ) // Turn Off Append 
            oBrowse:pageDown()
         ENDIF

      CASE nKey == K_CTRL_PGDN
         IF oBrowse:hitBottom
            oGet:exitState := GE_WRITE
         ELSE
            gbAppend( oBrowse, .F. ) // Turn Off Append 
            oBrowse:goBottom()
         ENDIF

      CASE nKey == K_CTRL_HOME
         oGet:exitState := GE_TOP

      CASE nKey == K_CTRL_W
         oGet:exitState := GE_WRITE

      CASE nKey == K_HOME
         oBrowse:home()

      CASE nKey == K_END
         oBrowse:end()

      CASE nKey == K_RIGHT
         oBrowse:Right()

      CASE nKey == K_LEFT
         oBrowse:Left()

      CASE nKey == K_DEL
         IF !oBrowse:cargo[ GBREADONLY ] .AND. !gbAppend( oBrowse )
            ReadUpdated( TRUE )
            IF (cAlias)->(DELETED()) 
               (cAlias)->(DBRECALL())
            ELSE
               (cAlias)->(DBDELETE())
            ENDIF
         ENDIF

   OTHERWISE
      IF ( nKey >= 32 .AND. nKey <= 255 )
         KEYBOARD CHR( nKey )
         lUpdateSave := UPDATED()
         gbGetIt( oBrowse )
         ReadUpdated( UPDATED() .OR. lUpdateSave )
      ENDIF

   ENDCASE

   DO WHILE !oBrowse:Stabilize()
   ENDDO

   IF oBrowse:hitTop .AND. (nKey != K_CTRL_PGUP)
      oGet:exitState := GE_UP
   ELSEIF oBrowse:hitBottom .AND. (nKey != K_CTRL_PGDN)
      IF oBrowse:cargo[ GBREADONLY ]
         oGet:exitState := GE_ENTER
      ELSE
         IF gbAppend( oBrowse )
            oGet:exitState := GE_ENTER
         ELSE
            gbAppend( oBrowse, .T. ) // Turn Append on
         ENDIF
      ENDIF
   ENDIF

   MsShow()

   RETURN NIL
**************************************************************************
*
* Function: gbGetIt( oBrowse )
STATIC FUNCTION gbGetIt( oBrowse )
   LOCAL cKeyCmd, ;
         cOldKeyValue := "",;
         cNewKeyValue := ""
   LOCAL oCol, xGetVar, cFieldName, x, nLen, oGet, nRec
   LOCAL cAlias := oBrowse:cargo[ GBALIAS ]

   DO WHILE !oBrowse:stabilize() ; END

   cKeyCmd := (cAlias)->(INDEXKEY( 0 ))
   IF !EMPTY( cKeyCmd )
      cOldKeyValue := &cKeyCmd
   ENDIF

   oCol := oBrowse:getColumn( oBrowse:colPos ) // Get current Column object

   cFieldName := oBrowse:cargo[ GBFIELDS, oBrowse:colPos ]

   xGetVar := EVAL( oCol:block )  // Setup get variable

   nLen := LEN( xGetVar )

   IF gbAppend( oBrowse )
      FOR x := 1 TO LEN( oBrowse:cargo[ GBDEFAULT ] )
         IF cFieldName == oBrowse:cargo[ GBDEFAULT, x, 1 ] 
            xGetVar := oBrowse:cargo[ GBDEFAULT, x, 2 ]
            IF VALTYPE( xGetVar ) $ "CM"
               xGetVar := PAD( xGetVar, nLen )
            ENDIF
            EXIT
         ENDIF
      NEXT x
   ENDIF

   oGet := GETNEW( ROW(), COL(), ;
                   {|x|IF(x==NIL, xGetVar, xGetVar := x) },;
                   cFieldname,, oBrowse:colorSpec )

   oGet:picture   := oBrowse:cargo[ GBPICTURE ]
   oGet:postBlock := oBrowse:cargo[ GBVALID   ]

   FWReadModal( { oGet } )  // Just Do It.

   IF LASTKEY() != K_ESC
      IF gbAppend( oBrowse )
         (cAlias)->(DBAPPEND())  // Add Record
         FOR x := 1 TO LEN( oBrowse:cargo[ GBDEFAULT ] ) // Set defaults
            EVAL( oBrowse:cargo[ GBDEFAULT, x, 3 ], ;
                  oBrowse:cargo[ GBDEFAULT, x, 2 ]  )
         NEXT x
         gbAppend( oBrowse, .F. )
      ENDIF
      IF (cAlias)->(RLOCK())
         EVAL( oCol:block, xGetVar )  // Save new Value
      ELSE
         TONE( 220, 4 )
      ENDIF
   ENDIF

   KEYBOARD CHR( ExitKey() ) // Stuff movement key

   IF !EMPTY( cKeyCmd )
      cNewKeyValue := &cKeyCmd
   ENDIF

   IF cOldKeyValue == cNewKeyValue
      oBrowse:refreshCurrent()
   ELSE
      nRec := (cAlias)->(RECNO())
      oBrowse:refreshAll()
      WHILE !oBrowse:stabilize() ; END
      WHILE ( (cAlias)->(RECNO()) != nRec ) .AND. !oBrowse:hitTop
         oBrowse:up()
         WHILE !oBrowse:stabilize() ; END
      END
   ENDIF

   IF !EVAL( oBrowse:cargo[ GBFOR ] )
      oBrowse:refreshAll()
      WHILE !oBrowse:stabilize() ; END
   ENDIF

   RETURN NIL
**************************************************************************
*
* Function: ExitKey( )
*
* This function will return a movement key based upon the exit key of 
* the last get.
*
STATIC FUNCTION ExitKey()

   LOCAL nKey

   nKey := LASTKEY()

   IF (nKey >= 32) .AND. (nKey <= 255) // Typeout
      nKey := K_RETURN
   ENDIF

   RETURN (nKey)
**************************************************************************
*
* Function: GetPreValidate( oGet )
*
* This function handles processing of the Get VALID clause.
*
STATIC FUNCTION GetPreValidate( oGet )
   LOCAL lWhen := .T.

   oGet:exitState := GE_NOEXIT

   IF oGet:preBlock <> NIL
      lWhen := EVAL( oGet:preBlock, oGet )
   ENDIF

   IF !(lWhen)
      oGet:exitState := GE_WHEN
   ENDIF

   RETURN (lWhen)
**************************************************************************
*
* Function: gbSkipper( n, oBrowse )
* 
* This function is the SKIP method for the BROWSE object.  It handles
* movement within the file being browsed.
*
STATIC FUNCTION gbSkipper( nNum2Skip, oBrowse )
   LOCAL nNumSkipped := 0,;
         nSaveRec    := 0
   LOCAL cAlias := oBrowse:cargo[ GBALIAS ],;
         bFor   := oBrowse:cargo[ GBFOR ]

   IF EVAL( bFor )
      nSaveRec := (cAlias)->(RECNO())
   ENDIF

   IF nNum2Skip == 0 .OR. (cAlias)->(LASTREC()) == 0
      (cAlias)->(DBSKIP( 0 ))
   ELSEIF nNum2Skip > 0 .AND. !((cAlias)->(EOF()))
      DO WHILE nNumSkipped < nNum2Skip
         (cAlias)->(DBSKIP( 1 ))
         IF EVAL( bFor ) .OR. (cAlias)->(EOF())
            ++nNumSkipped
            nSaveRec := (cAlias)->(RECNO())
         ENDIF
         IF (cAlias)->(EOF())
            oBrowse:hitBottom := .T.
            EXIT
         ENDIF
      ENDDO
   ELSEIF nNum2Skip < 0
      DO WHILE nNumSkipped > nNum2Skip
         (cAlias)->(DBSKIP( -1 ))
         IF (cAlias)->(BOF())
            oBrowse:hitTop := .T.
            EXIT
         ENDIF
         IF EVAL( bFor )
            --nNumSkipped
            nSaveRec := (cAlias)->(RECNO())
         ENDIF
      ENDDO
   ENDIF

   IF nSaveRec <> 0 
      (cAlias)->(DBGOTO( nSaveRec ))
   ENDIF

   RETURN (nNumSkipped)
**************************************************************************
*
* Function: gbTop( oBrowse )
*
* This function is the goTop method for the browse object.
* It handles moving to the beginning of the browsed file.
*
STATIC FUNCTION gbTop( oBrowse )
   LOCAL cAlias := oBrowse:cargo[ GBALIAS ],;
         bFor   := oBrowse:cargo[ GBFOR ]

   
   (cAlias)->(DBGOTOP())
   DO WHILE !((cAlias)->(EVAL( bFor ))) .AND. !((cAlias)->(EOF()))   
      (cAlias)->(DBSKIP( 1 ))
   ENDDO

   IF (cAlias)->(EOF())
      oBrowse:hitBottom := .T.
   ENDIF

   RETURN NIL
**************************************************************************
*
* Function: gbBottom( oBrowse )
*
* This function is the goBottom method for the browse object.
* It handles moving to the End of the browsed file.
*
STATIC FUNCTION gbBottom( oBrowse )
   LOCAL cAlias := oBrowse:cargo[ GBALIAS ],;
         bFor   := oBrowse:cargo[ GBFOR ]

   (cAlias)->(DBGOBOTTOM())
   DO WHILE !((cAlias)->(EVAL( bFor ))) .AND. !((cAlias)->(BOF()))
      (cAlias)->(DBSKIP( -1 ))
   ENDDO

   IF (cAlias)->(BOF())
      oBrowse:hitTop := .T.
   ENDIF

   RETURN NIL
**************************************************************************
*
* Function: gbAppend( oBrowse, lInsert )
*
* This function is a GET/SET function for the Append mode, adding new 
* records.
*
STATIC FUNCTION gbAppend( oBrowse, lInsert )

   LOCAL  lCurrentMode

   lCurrentMode := oBrowse:cargo[ GBAPPEND ]

   IF VALTYPE( lInsert ) == "L"
      oBrowse:cargo[ GBAPPEND ] := lInsert 
   ENDIF

   RETURN (lCurrentMode)
***************************************************************************
