UNIT ExitHandler ;
{$O-}
(****************************************************************************)
(* EXITHANDLER.PAS (TURBO PASCAL 6.0 unit)                                  *)
(* Standard error checking and exit handling routines                       *)
(****************************************************************************)
(* Revision history:                                                        *)
(*  04-Sep-1991 created by hs                                               *)
(*  19-Mar-1992 revised: F (file) argument to CheckRes changed into FN      *)
(*              (filename), and some other changes forced by this.          *)
(*              Installed typed constant renamed into StdExitInstalled      *)
(*              and moved to INTERFACE. Null variable removed. {$O-}        *)
(*              inserted.                                                   *)
(*  08-Jun-1992 add stack trace routines; whole unit restructured; some     *)
(*              incompatible changes, but no features removed.              *)
(*  07-Jul-1992 further changes concerning installation of handlers.        *)
(****************************************************************************)
(* This unit was written by Hans Schleichert at Institute of Ophthalmology, *)
(* Department of Visual Science, Judd Street, London WC1H 9QS               *)
(****************************************************************************)

(****************************************************************************)
(****************************************) INTERFACE (***********************)
(****************************************************************************)

    PROCEDURE CheckRes (
        Res, Code : Integer ; FN : String ; Addr : Pointer
    ) ; (****************************************************************)
    (* check a condition code against ERR_Ok                            *)
    (* Input:                                                           *)
    (*  Res     primary code to check                                   *)
    (*  Code    secondary code                                          *)
    (*  FN      pointer to name of file causing the error               *)
    (*  Addr    code address where the error should be generated        *)
    (* Notes:                                                           *)
    (*  1 - If Res=errOk, nothing happens.                              *)
    (*  2 - If Res<>errOk, the value of Code and a copy of FN^ are      *)
    (*      stored in global variables accessible to this unit's exit   *)
    (*      handling routines. Then a run-time error is generated at the*)
    (*      address supplied with Addr with the code supplied as Res.   *)
    (*  3 - Set Addr to NIL if the run-time error should be generated   *)
    (*      at the address of the CheckRes call.                        *)
    (*  4 - If the condition code to be checked doesn't result from a   *)
    (*      file operation, set FN to NIL. If no secondary code should  *)
    (*      be generated, supply the value 0 (or errOk) for Code.       *)
    (*  5 - Cf. StdExitHandler description below.                       *)
    (********************************************************************)

    PROCEDURE InstallStackTrace ; (**************************************)
    (* Install the stack trace exit handler                             *)
    (********************************************************************)

    PROCEDURE InstallStdExit (Report : Boolean) ; (**********************)
    (* Install the standard exit handler                                *)
    (* Input:                                                           *)
    (*  Report      If True, the error address will not be cleared when *)
    (*              a run-time error occurs (useful for debugging)      *)
    (* Note:                                                            *)
    (*  Also use this procedure for changing the Report flag.           *)
    (********************************************************************)

    PROCEDURE RemoveStackTrace ; (***************************************)
    (* Remove the stack trace exit handler                              *)
    (********************************************************************)

    PROCEDURE RemoveStdExit ; (******************************************)
    (* Remove the standard exit handler                                 *)
    (********************************************************************)

    PROCEDURE ReportErr (Res, Code : Integer ; FN : String) ; (**********)
    (* report an error condition                                        *)
    (* Input:                                                           *)
    (*  Res     primary error code                                      *)
    (*  Code    secondary code                                          *)
    (*  FN      pointer to name of file causing the error               *)
    (* Notes:                                                           *)
    (*  1 - As opposed to CheckRes, even a success code generates a     *)
    (*      report.                                                     *)
    (*  2 - The generated report consists of the error message assoc-   *)
    (*      iated with Code if Code<>errOk, the error message associat- *)
    (*      ed with Res, and the string FN preceded by 'File: ' if FN   *)
    (*      is not empty.                                               *)
    (********************************************************************)

    PROCEDURE SetErrorInfo (Code : Integer ; FN : String) ; (************)
    (* set the secondary error code and related filename                *)
    (* Input:                                                           *)
    (*  Code    secondary error code                                    *)
    (*  FN      name of related file                                    *)
    (* Note:                                                            *)
    (*  Use this procedure to supply secondary info for RTL-generated   *)
    (*  errors.                                                         *)
    (********************************************************************)

    PROCEDURE StackTrace (Action : Pointer ; Normalize : Boolean) ; (****)
    (* Trace the Turbo Pascal stack by stack frames                     *)
    (* Input:                                                           *)
    (*  Action      Address of a routine to be called with every stack  *)
    (*              frame                                               *)
    (*  Normalize   If True, all addresses will be normalized before    *)
    (*              being passed to the action routine                  *)
    (* Notes:                                                           *)
    (*  1 - A stack frame is a TStackFrame structure, although the CS   *)
    (*      field may actually not belong to the structure and thus be  *)
    (*      invalid. This is the case when the frame was built by a     *)
    (*      NEAR call.                                                  *)
    (*  2 - The first stack frame is located at SS:BP. Every stack      *)
    (*      frame's BP field contains the offset of the next stack      *)
    (*      frame, the segment always being SS. The last stack frame    *)
    (*      has a BP field of 0, and the data in the IP and CS fields   *)
    (*      should be regarded as invalid.                              *)
    (*  3 - Normalization changes CS values from physical to logical    *)
    (*      segment addresses. Logical addresses are helpful if you     *)
    (*      want to locate the error in the source code by means of the *)
    (*      Turbo Pascal compiler. The normalization process takes two  *)
    (*      steps:                                                      *)
    (*      (1) If CS points to an overlaid unit, CS is replaced by the *)
    (*          segment address of the overlay header record.           *)
    (*      (2) Then, PrefixSeg+$10 is subtracted from CS.              *)
    (*      Normalization takes place on a copy if the stack frame and  *)
    (*      never on the frame itself.                                  *)
    (*  4 - StackTrace tries to distinguish between "valid" and         *)
    (*      "invalid" CS values. A valid CS must be greater than or     *)
    (*      equal to PrefixSeg+$10, and less than DSeg. An invalid CS   *)
    (*      will cause StackTrace to assume a NEAR call while a valid   *)
    (*      CS will result in assuming a FAR call. A NEAR call is also  *)
    (*      assumed, regardless of CS, when the BP field contains the   *)
    (*      current frame's offset plus 4.                              *)
    (*  5 - No FAR return address can be assumed to be NEAR while a     *)
    (*      NEAR address may under some circumstances be considered as  *)
    (*      FAR.                                                        *)
    (*  6 - StackTrace terminates when one of these conditions is met   *)
    (*      in the current stack frame:                                 *)
    (*      (1) The number in the BP field is 0.                        *)
    (*      (2) The number in the BP field is odd.                      *)
    (*      (3) The number in the BP field is less than the current     *)
    (*          stack frame's offset plus 4.                            *)
    (*  7 - The Action parameter points to a routine which is called    *)
    (*      once for each stack frame, except when one of the           *)
    (*      conditions from note 6 is met. This routine must be a       *)
    (*      nested PROCEDURE using the FAR call model which takes a     *)
    (*      Word-type, a Pointer-type, and a Boolean-type argument:     *)
    (*          PROCEDURE ActionRtn (Frame : Word ; CSIP : Pointer ;    *)
    (*              FarCall : Boolean) ; FAR ;                          *)
    (*      The parameters have the following meanings:                 *)
    (*          Frame   Offset part of the address of the current stack *)
    (*                  frame; see also below.                          *)
    (*          CSIP    Return address stored in the stack frame. The   *)
    (*                  Segment part should be considered invalid if    *)
    (*                  FarCall=False, and is normalized if             *)
    (*                  FarCall=True and normalization was requested.   *)
    (*          FarCall True if CS is "valid" as defined in note 4.     *)
    (*      When the final stack frame has been found, i. e. one of the *)
    (*      conditions from note 6 is true, the action routine gets     *)
    (*      called once more. If this was because BP=0, the Frame para- *)
    (*      meter is 0; otherwise, it is 1. CSIP is NIL, and FarCall is *)
    (*      False.                                                      *)
    (*  8 - When called from within an exit handler, the action routine *)
    (*      has to ignore the first and second stack frame.             *)
    (*  9 - The action routine can access local variables, the data     *)
    (*      segment, and I/O services.                                  *)
    (********************************************************************)

(****************************************************************************)
(****************************************) IMPLEMENTATION (******************)
(****************************************************************************)

    USES
        Dos,
        Error, FormatOutput, OvrService ;

    PROCEDURE NilProc ; FAR ; ASSEMBLER ; ASM END ;

    TYPE
        TActionProc = PROCEDURE ;
        TActions = RECORD
            ExitProc, TraceProc : TActionProc ;
        END ;

    CONST   (************************** initialized local work area *********)
        SecCode : Integer = errOk ;     (* secondary error code             *)
        RelFile : STring = '' ;         (* error-related filename           *)
        SaveExit : Pointer = Ptr ($FFFF, $FFFF) ;
                                        (* saved ExitProc                   *)
        Actions : TActions = (ExitProc: NilProc; TraceProc: NilProc) ;
                                        (* exit actions                     *)
        DoReport : Boolean = True ;     (* error report flag                *)

    PROCEDURE DoExit ; FAR ; FORWARD ;
    PROCEDURE DoTrace ; FAR ; FORWARD ;
    PROCEDURE InstallHandler ; NEAR ; FORWARD ;

    PROCEDURE CheckRes (
        Res, Code : Integer ; FN : String ; Addr : Pointer
    ) ; ASSEMBLER ;

        ASM     (********************** PROCEDURE CheckRes ******************)
            MOV     AX,[Res]
            OR      AX,AX
            JZ      @Done
            PUSH    AX
            MOV     AX,[Code]
            MOV     [SecCode],AX
            MOV     DX,DS
            LDS     SI,[FN]
            MOV     ES,DX
            MOV     DI,OFFSET RelFile
            CLD
            LODSB
            STOSB
            MOV     CL,AL
            MOV     CH,0
            REP     MOVSB
            MOV     DS,DX
            LES     DI,[Addr]
            PUSH    ES
            PUSH    DI
            CALL    ErrorAt
@Done:  END ;   (********************** PROCEDURE CheckRes ******************)

    PROCEDURE InstallStackTrace ;

        BEGIN   (********************** PROCEDURE InstallStackTrace *********)
            InstallHandler ;
            Actions.TraceProc := DoTrace
        END ;   (********************** PROCEDURE InstallStackTrace *********)

    PROCEDURE InstallStdExit (Report : Boolean) ;

        BEGIN   (********************** PROCEDURE InstallStdExit ************)
            InstallHandler ;
            Actions.ExitProc := DoExit ;
            DoReport := Report
        END ;   (********************** PROCEDURE InstallStdExit ************)

    PROCEDURE RemoveStackTrace ;

        BEGIN   (********************** PROCEDURE RemoveStackTrace **********)
            Actions.TraceProc := NilProc
        END ;   (********************** PROCEDURE RemoveStackTrace **********)

    PROCEDURE RemoveStdExit ;

        BEGIN   (********************** PROCEDURE RemoveStdExit *************)
            Actions.ExitProc := NilProc
        END ;   (********************** PROCEDURE RemoveStdExit *************)

    PROCEDURE ReportErr (Res, Code : Integer ; FN : String) ; ASSEMBLER ;

        PROCEDURE DoReport (VAR F : String) ;

            BEGIN   (****************** PROCEDURE DoReport ******************)
                IF Code <> errOk THEN WriteLn (DosErrorMsg (Code)) ;
                WriteLn (DosErrorMsg (Res)) ;
                IF F <> '' THEN WriteLn ('File: ', F)
            END ;   (****************** PROCEDURE DoReport ******************)

        ASM     (********************** PROCEDURE ReportErr *****************)
            LES     DI,[FN]
            PUSH    ES
            PUSH    DI
            PUSH    BP
            CALL    DoReport
        END ;   (********************** PROCEDURE ReportErr *****************)

    PROCEDURE StackTrace (Action : Pointer ; Normalize : Boolean) ;
        ASSEMBLER ;

        ASM     (********************** PROCEDURE StackTrace ****************)
            MOV     BX,BP
@@1:        MOV     AX,[SS:BX]
            OR      AX,AX
            JE      @@9
            TEST    AL,01h
            JNZ     @@10
            ADD     BX,4
            XOR     DX,DX
            CMP     AX,BX
            JB      @@10
            PUSH    AX
            PUSH    AX
            MOV     CX,[WORD PTR SS:BX-2]
            JE      @@6
@@2:        MOV     BX,[WORD PTR SS:BX]
            MOV     AX,OvrLoadList
@@3:        OR      AX,AX
            JE      @@5
            MOV     ES,AX
            CMP     BX,[ES:TOvrHeader.Segment]
            JE      @@4
            MOV     AX,[ES:TOvrHeader.Next]
            JMP     @@2
@@4:        MOV     BX,ES
@@5:        CMP     BX,SEG @Data
            JAE     @@6
            MOV     AX,[PrefixSeg]
            ADD     AX,10h
            SUB     BX,AX
            JB      @@6
            MOV     DL,1
            JMP     @@7
@@6:        XOR     BX,BX
@@7:        PUSH    BX
            PUSH    CX
            PUSH    DX
            PUSH    [Word PTR SS:BP]
            CALL    [Action]
            POP     BX
            JMP     @@1
@@8:        MOV     AX,1
            PUSH    AX
            DEC     AL
            JMP     @@10
@@9:        XOR     AX,AX
            PUSH    AX
@@10:       PUSH    AX
            PUSH    AX
            PUSH    AX
            PUSH    [Word PTR SS:BP]
            CALL    [Action]
        END ;   (********************** PROCEDURE StackTrace ****************)

    PROCEDURE SetErrorInfo (Code : Integer ; FN : String) ; ASSEMBLER ;

        ASM     (********************** PROCEDURE SetErrorInfo **************)
            MOV     AX,[Code]
            MOV     [SecCode],AX
            MOV     DX,DS
            LDS     SI,[FN]
            MOV     ES,DX
            MOV     DI,OFFSET RelFile
            CLD
            LODSB
            STOSB
            MOV     CL,AL
            MOV     CH,0
            REP     MOVSB
            MOV     DS,DX
        END ;   (********************** PROCEDURE SetErrorInfo **************)

    PROCEDURE DoExit ;

        BEGIN   (********************** PROCEDURE DoExit ********************)
            IF ErrorAddr <> NIL THEN
            BEGIN
                IF SecCode <> errOk THEN WriteLn (DosErrorMsg (SecCode)) ;
                WriteLn (DosErrorMsg (ExitCode)) ;
                IF RelFile <> '' THEN
                WriteLn ('File: "', RelFile, '"') ;
                IF NOT DoReport THEN ErrorAddr := NIL
            END ;
        END ;   (********************** PROCEDURE DoExit ********************)

    PROCEDURE DoTrace ;

        VAR
            Count : Word ;

        PROCEDURE DoShow (
            Frame : Word ; CSIP : Pointer ; FarCall : Boolean
        ) ; FAR ;

            BEGIN   (****************** PROCEDURE DoShow ********************)
                IF Frame = 0 THEN Exit ;
                IF Frame = 1 THEN
                BEGIN
                    WriteLn ('Stack is corrupted - Illegal stack frame') ;
                    Exit
                END ;
                Inc (Count) ;
                IF Count <= 2 THEN Exit ;
                Write ('   Stack frame at ', foHexWord (Frame), ': ') ;
                IF FarCall THEN WriteLn (foHexAddress (CSIP), ' FAR')
                ELSE WriteLn ('     ', foHexWord (Word (CSIP)), ' NEAR') ;
            END ;   (****************** PROCEDURE DoShow ********************)

        BEGIN   (********************** PROCEDURE DoTrace *******************)
            Count := 0 ;
            WriteLn ('Stacked return address trace:') ;
            StackTrace (@DoShow, True)
        END ;   (********************** PROCEDURE DoTrace *******************)

    PROCEDURE Handler ; FAR ;

        BEGIN   (********************** PROCEDURE Handler *******************)
            Actions.ExitProc ;
            Actions.TraceProc ;
            IF LongInt (SaveExit) <> -1 THEN ExitProc := SaveExit ;
            LongInt (SaveExit) := -1
        END ;   (********************** PROCEDURE Handler *******************)

    PROCEDURE InstallHandler ;

        BEGIN   (********************** PROCEDURE InstallHandler ************)
            IF LongInt (SaveExit) = -1 THEN
            BEGIN
                SaveExit := ExitProc ; ExitProc := @Handler
            END
        END ;   (********************** PROCEDURE InstallHandler ************)

END .