//
// strsearch.cpp: str class member functions for searching and replacing
// Author       : Roy S. Woll
//
// Copyright (c) 1993 by Roy S. Woll
// You may distribute this source freely as long as you leave all files
// in their original form, including the copyright notice as is.
//
//
// Version 2.2      5/5/93
//    Fix bug in index relating to case sensitivity (backwards)
//
// Version 2.00     11/31/92
//    Support searching/replacing, regular expressions, case sensitivity
//

#include <string.h>
#include <limits.h>

#include "str.h"
#include "regx.h"


// 
// index member functions
//
int str::index(const char * s, int start) const{
   if (!caseSensitive()){
      str mylowstr(::lowercase(*this));
      const char * charptr = strstr(mylowstr(start), ::lowercase(s));
      if (charptr) return (charptr - mylowstr());
      else return (-1);
   }
   else {
      const char * charptr = strstr((*this)(start), s);
      if (charptr) return (charptr - data->buf);
      else return (-1);
   }
};

int str::index(const regX& reg, int start) const{
   int matchlen;
   return reg.index(*this, &matchlen, start, caseSensitive());
};


int str::index(const regX& reg, int * matchLen, int start) const
{
   return reg.index(*this, matchLen, start, caseSensitive());
};


//
// search operator functions
//

int str::search(const char * s, int* startPtr) const{
   *startPtr = index(s, *startPtr);
   return *startPtr>=0;
};

int str::search(const char * s, int start) const{
   return index(s, start)>=0;
};

int str::search(const regX& reg, int * startPtr) const{
   return search(reg, 0, startPtr);
};

int str::search(const regX& reg, int start) const{
   return search(reg, 0, &start);
};

int str::search(const regX& reg, str * matchPtr, int start) const{
   return search(reg, matchPtr, &start);
};

int str::search(const regX& reg, str * matchPtr, int * startPtr) const{

   int matchLen;
   int start = reg.index(*this, &matchLen, *startPtr, caseSensitive());
   if (start>=0) {
      if (matchPtr) *matchPtr = operator()(start, matchLen);
      if (startPtr) *startPtr = start;
      return 1;
   }
   else {
      if (matchPtr) *matchPtr = "";
      if (startPtr) *startPtr = -1;
      return 0;
   };

};


int str::replace(const regX& reg, const char* replaceStr,
                 int start, int numReplacements)
{
  return replace(reg, replaceStr, &start, numReplacements);
};

int str::replace(const regX& reg, const char* replaceStr,
                 int* startPtr, int numReplacements)
{
   int& start = *startPtr;
   if (numReplacements==0) return 0;

   int countReplacements=0, matchLen;
   int replaceLen = strlen(replaceStr);

   do {
      start = index(reg, &matchLen, start);
      if (start<0) break;
      countReplacements++;
      operator()(start, matchLen) = replaceStr;
      start+= replaceLen;           //skip past newly added data

      if (!--numReplacements) break;
   } while (1);

   return countReplacements;

};



int str::replace(const char * searchStr, const char* replaceStr,
                 int start, int numReplacements)
{
  return replace(searchStr, replaceStr, &start, numReplacements);
};

int str::replace(const char * searchStr, const char* replaceStr,
                 int* startPtr, int numReplacements)
{

   if (numReplacements==0) return 0;

   int& start = *startPtr;
   int countReplacements=0;
   int searchLen = strlen(searchStr), replaceLen = strlen(replaceStr);

   do {

      start = index(searchStr, start);
      if (start<0) break;
      countReplacements++;
      operator()(start, searchLen) = replaceStr;
      start+= replaceLen;           //skip past newly added data

      if (!--numReplacements) break;

   } while (1);

   return countReplacements;

};


int str::replaceAll(const regX& reg, const char* replaceStr, int start)
{
  return replace(reg, replaceStr, &start, INT_MAX);
};

int str::replaceAll(const char * searchStr, const char* replaceStr, int start)
{
  return replace(searchStr, replaceStr, &start, INT_MAX);
};

