/* kbos.c -- operating system and BIOS keyboard access
 * Copyright (C) 1995, 1996 Markus F.X.J. Oberhumer
 * For conditions of distribution and use, see copyright notice in kb.h 
 */

#if defined(__EMX__)
#  include <sys/emx.h>		/* must be first include file */
#endif
#include <stdlib.h>
#include <kb.h>
#include "_kb.h"
#if defined(__GO32__)
#  include <gppconio.h>
#  undef kbhit  /* want to be able to call kbhit from libc */
#endif


/* this file contains lots of #if ... */


/***********************************************************************
// BIOS wrapper
************************************************************************/

#if defined(__EMX__) || defined(__GO32__)

static __inline__ unsigned _my_bios_keybrd(unsigned cmd)
{
#if defined(__DJGPP__)
	__dpmi_regs regs;
#else
	union REGS regs;
#endif

#if defined(__EMX__)
	/* ((_emx_env & 0x0200) != 0) || "emx: not in DOS_MODE" */
	/* ((_emx_env & 0x0800) == 0) || "emx: -ac not enabled for _int86()" */
	if (!KB_USE_INT86())
		return 0;
#endif /* __EMX__ */

	regs.h.al = 0;
	regs.h.ah = cmd;
	KB_INT86(0x16,&regs);

	if (cmd == KB_KEYBRD_READY || cmd == KB_NKEYBRD_READY)
	{
		if (regs.x.flags & 0x40)	/* zero flag set -> no key is waiting */
			return 0;
		else if (regs.x.ax == 0)
			return 0x0000ffff;		/* Control-Break */
		else
			return regs.x.ax;
	}
	else
		return regs.x.ax;
}

#define _KB_BIOSKEY(x)	_my_bios_keybrd(x)

#endif /* __EMX__ || __GO32__ */


#if defined(__KB_MSDOS)
#  if defined(__BORLANDC__) || defined(__TURBOC__) || defined(__GO32__)
#    ifndef _KB_BIOSKEY
#      define _KB_BIOSKEY(x)	bioskey(x)
#    endif
#  endif
#  if defined(_MSC_VER) || defined(__WATCOMC__)
#    ifndef _KB_BIOSKEY
#      define _KB_BIOSKEY(x)	_bios_keybrd(x)
#    endif
#  endif
#endif


/***********************************************************************
// BIOS level - lowest and (hopefully) fastest access level
// If not applicable, OS level is used
************************************************************************/

unsigned kb_bioskey(unsigned cmd)
{
#if defined(_KB_BIOSKEY)
	return _KB_BIOSKEY(cmd);
#else
	return 0;
#endif
}


int kb_bios_kbhit(void)
{
	if (_kb_mode)
		return kb_kbhit();

#if !defined(_KB_BIOSKEY)
	return kb_os_kbhit();
#elif defined(__GO32__)
	return kbhit();				/* this uses BIOS and is damned fast */
#else
# if defined(__EMX__)
	if (!KB_USE_INT86())
		return kb_os_kbhit();	/* _int86() not allowed */
# endif
	return (_KB_BIOSKEY(KB_NKEYBRD_READY) != 0);
#endif
}


unsigned kb_bios_getkey(void)
{
	if (_kb_mode)
		return kb_getkey();

	if (!kb_bios_kbhit())
		return 0;

#if !defined(_KB_BIOSKEY)
	return kb_os_getkey();
#elif defined(__GO32__)
	return getxkey();			/* this uses BIOS */
#else
# if defined(__EMX__)
	if (!KB_USE_INT86())
		return kb_os_getkey();	/* _int86() not allowed */
# endif
	return kb_bios_keycode(_KB_BIOSKEY(KB_NKEYBRD_READ));
#endif
}



/***********************************************************************
// emx getch() wrapper
// returns 0 if no key is available
************************************************************************/

#if defined(__EMX__)

static int _last_key = -1;

static __inline__ int _my_read_kbd(int eat)
{
	if (_last_key == -1)
		_last_key = _read_kbd(0,0,0);		/* echo, wait, sig */
	if (_last_key == -1)
		return 0;							/* no key available */
	if (eat)
	{
		int k = _last_key;
		_last_key = -1;
		return k;
	}
	else
		return 1;
}

#define _my_kbhit()		_my_read_kbd(0)
#define _my_getch()		_my_read_kbd(1)
		
#endif /* __EMX__ */


/***********************************************************************
// Linux getch() wrapper
// returns 0 if no key is available
//
// adapted from code written by Tommy Frandsen and Harm Hanemaayer
// see svgalib 1.2.9: src/vgamisc.c
//
// This code is slow and doesn't work very well because stdin
// is only temporary changed for each attempt to read a key
************************************************************************/

#if defined(__KB_LINUX)

static int _last_key = 0;

#include <stdio.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#if 0
static __inline__ int _my_read_a_key(void)
{
    struct timeval tv;
    fd_set fds;
    char c;
    int fd = fileno(stdin);

    tv.tv_sec = tv.tv_usec = 0;
    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    if (select(fd + 1, &fds, 0, 0, &tv) <= 0)
		return 0;
	if (read(fileno(stdin), &c, 1) != 1)
	    return 0;
	return (unsigned char) c;				/* Return key. */
}
#else
static __inline__ int _my_read_a_key(void)
{
	struct termio zap, original;
	char c;
	int e;
	int fd = fileno(stdin);

	if (ioctl(fd, TCGETA, &original) != 0)	/* Get termio */
		return 0;
	zap = original;
	zap.c_cc[VMIN] = 0;						/* Modify termio  */
	zap.c_cc[VTIME] = 0;
	zap.c_lflag = 0;
	if (ioctl(fd, TCSETA, &zap) != 0)		/* Set new termio */
		return 0;
	e = read(fileno(stdin), &c, 1);			/* Read one char */
	ioctl(fd, TCSETA, &original);			/* Restore termio */
	if (e != 1)
		return 0;							/* No key pressed. */
	return (unsigned char) c;				/* Return key. */
}
#endif


static __inline__ int _my_read_kbd(int eat)
{
	if (_last_key == 0)
		_last_key = _my_read_a_key();
	if (_last_key == 0)
		return 0;							/* no key available */
	if (eat)
	{
		int k = _last_key;
		_last_key = 0;
		return k;
	}
	else
		return 1;
}

#define _my_kbhit()		_my_read_kbd(0)
#define _my_getch()		_my_read_kbd(1)
		

#endif /* __KB_LINUX */



/***********************************************************************
// OS level
************************************************************************/

int kb_os_kbhit(void)
{
	if (_kb_mode)
		return kb_kbhit();

#if defined(__EMX__)
	/* use BIOS if possible, cause it's much faster */
	if (KB_USE_INT86())
		return kb_bios_kbhit();
	return _my_kbhit() != 0;
#elif defined(__KB_LINUX)
	return _my_kbhit() != 0;
#else
	return kbhit() != 0;
#endif
}


/* do NOT use BIOS, but be compatible */

#if !defined(_my_getch)
#  define _my_getch()	getch()
#endif

unsigned kb_os_getkey(void)
{
	unsigned k;

	if (_kb_mode)
		return kb_getkey();

	if (!kb_os_kbhit())
		return 0;

	k = _my_getch();
#if defined(__KB_MSDOS)
	if (k != 0)
		return kb_os_keycode(k,0);
	k = _my_getch();				/* read again */
	return kb_os_keycode(0,k);
#elif defined(__KB_LINUX)
	return k;
#endif
}


/***********************************************************************
// emx+gcc lacks a real kbhit() and getch()
************************************************************************/

#if defined(__EMX__)

int kbhit(void)
{
	return kb_os_kbhit();
}

int getch(void)
{
	if (_kb_mode)
	{
		int k = kb_getkey();
		return k == 0 ? -1 : k;
	}
	return _my_getch();
}

#endif /* __EMX__ */


/*
vi:ts=4
*/
