/**********************************************************************/
/*                                                                    */
/*  REXX Drive Waste Scanner                                          */
/*                                                                    */
/*  Written by Mike Ruskai <mruskai@microfone.net>                    */
/*                                                                    */
/*  Distribute and modify freely, but it'd be nice if you give me     */
/*  credit if you distribute a modified form.                         */
/*                                                                    */
/*  Usage:  DRIVECHK.CMD [<drive letter>: [/S]]                       */
/*                                                                    */
/*  Parameters are optional.  <drive letter>: is self explanatory.    */
/*  The /S switch tells the program skip extended attributes, and     */
/*  can only be used when the drive letter is passed.  If no options  */
/*  are chosen, the program will prompt for the input of the drive    */
/*  to scan, and whether or not to check extended attributes.         */
/*                                                                    */
/*  This REXX script will scan an entire drive to determine how much  */
/*  space would be wasted for your specific files with the HPFS and   */
/*  FAT file systems.  The display of this information depends on     */
/*  which file system is actually being used.                         */
/*                                                                    */
/*  I've tried to include as much information as I know how.  In      */
/*  addition to the normal file system usage for a file and a         */
/*  directory, the extended attributes are also added up and used     */
/*  in space and slack space determinations.  For files that do not   */
/*  have any extended attributes, the size of the Fnode (512 bytes)   */
/*  for the HPFS scenario is added to the slack space for the file,   */
/*  since it would not exist in the FAT scenario.  That also goes     */
/*  for zero-byte files, which take up 512 bytes on HPFS drives, but  */
/*  zero bytes on FAT drives.                                         */
/*                                                                    */
/*  What's *not* included is the space consumed by the system areas   */
/*  specific to the file system.  The reason for this exclusion is    */
/*  that I don't know how to measure this size for HPFS volumes       */
/*  without capturing the output of CHKDSK (time-consuming), since I  */
/*  see no correlation between system area size and total disk size.  */
/*  My own HPFS drives have system areas ranging from 2MB to 6.5MB,   */
/*  with smaller drives sometimes having bigger system areas than     */
/*  some larger drives.                                               */
/*                                                                    */
/*  As stated, extended attributes are included, but the process by   */
/*  which they are is quite slow.  The only way via REXX to find the  */
/*  size of a file's extended attributes was to use the output of     */
/*  CMD.EXE or 4OS2.EXE's DIR command (using the /N parameter on FAT  */
/*  volumes).  Piping the output to RXQUEUE was very slow, and        */
/*  redirecting the output to a file, then reading from the file was  */
/*  much faster, but still slow.  4OS2.EXE tends to be about 60%      */
/*  slower than CMD.EXE in loading itself into memory (more features  */
/*  to load).  This would normally prompt you to type CMD before      */
/*  running this, given the frequency with which the interpreter is   */
/*  called (if you knew this about CMD and 4OS2, that is), but here,  */
/*  you needn't bother.  Because of some extra work necessitated by   */
/*  the output of CMD, the speed balances out - both CMD and 4OS2     */
/*  are equally snail-like in this process.  Because of the horrible  */
/*  speed, you might want to skip EA checking altogether.  The        */
/*  significance of EA's in space determinations vary from drive to   */
/*  drive, though.  If you know that your files have quite a lot of   */
/*  extended attributes, you should just suffer to get results which  */
/*  more accurately represent reality.                                */
/*                                                                    */
/**********************************************************************/

call RxFuncAdd 'SysLoadFuncs','RexxUtil','SysLoadFuncs'
call SysLoadFuncs

/* Checks to see if a drive was passed on the command line, and       */
/* whether or not to skip extended attributes.                        */

CurrDir=directory()
arg TestDrive SkipSwitch

if TestDrive\='' then do
    if directory(TestDrive||'\')\=TestDrive||'\' then do
        say 'Invalid drive, '||'"'TestDrive'"'||'.'
        exit
    end
    if translate(SkipSwitch)='/S' then SkipEA=1
        else SkipEA=0
end

/* If no drive was passed to the program, it prompts the user to      */
/* input one, repeating if an invalid drive is given.                 */

else do
    do until ans=1
        call SysCls
        if ans=0 then do
            call SysCurPos 8,1
            say '"'TestDrive'"'||', is an invalid drive.'
        end
        call SysCurPos 10,1
        say 'Which drive do you wish to check (e.g. C)?'
        call charout ,': '
        TDrive=translate(SysGetKey('echo'))
        TestDrive=TDrive||':'        
        if directory(TestDrive||'\')\=TestDrive||'\' then ans=0
            else ans=1
    end
    ans='2'
    do until ans=1
        call SysCls
        if ans=0 then do
            call SysCurPos 8,1
            say "Please answer only with 'Y' or 'N'."
        end
        call SysCurPos 10,1
        say 'Do you wish to skip EA checking? (Y/N)'
        call charout ,': '
        res=translate(SysGetKey('echo'))
        if res\='Y' & res\='N' then ans=0
            else do
                ans=1
            if res='Y' then SkipEA=1
                else SkipEA=0
            end
    end
end

call directory TestDrive||'\'

/* Uses the total drive space as reported by SysDriveInfo to deter-   */
/* mine the cluster size that FAT would use on the drive.  It then    */
/* attempts to write a file with a long name to determine if the      */
/* drive being scanned is HPFS or FAT, and orders the allocation unit */
/* variables accordingly (affects report at program conclusion).      */

DriveSize=word(SysDriveInfo(TestDrive),3)
if DriveSize<16777216 then ClustSize=4096
else do
    if DriveSize>=16777216 & DriveSize<134217728 then ClustSize=2048
    else do
        RawClusterSize=format((DriveSize/65536),,0)
        BinClust=x2b(d2x(RawClusterSize))
        ValidPartStart=pos('1',BinClust)
        DigitValue=length(substr(BinClust,ValidPartStart))
        ClustSize=2
        do i=1 to DigitValue-1
            ClustSize=ClustSize*2
        end
    end
end
tfile='Testing For HPFS File System'
rc=lineout(tfile,'test')
if rc=1 then do
    FileSystem.1='FAT'
    FileSystem.2='HPFS'
    AUnit.1=ClustSize
    AUnit.2=512
    Fnode.1=0
    Fnode.2=512
end
else do
    FileSystem.1='HPFS'
    FileSystem.2='FAT'
    call stream tfile,'c','close'
    call SysFileDelete tfile
    AUnit.1=512
    AUnit.2=ClustSize
    Fnode.1=512
    Fnode.2=0
end

/* Quick check to see if CMD.EXE or 4OS2.EXE is being used.  It has   */
/* an effect on how EA size checking is done, since the display for   */
/* each command processor is slightly different.                      */

p1=right(d2x(random(65535)),4,'0')
p2=right(d2x(random(65535)),4,'0')
TShChk='\'||p1||p2||'.TMP'
'@ver>'||TShChk
call SysFileSearch '4OS2',TShChk,'ShCheck.'
call stream TShChk,'c','close'
call SysFileDelete TShChk
if ShCheck.0=0 then UShell='CMD'
    else UShell='4OS2'

/* Initializes the variables used for the main procedure.             */

TotFiles=0
TotSize=0
TotDirs=0
TotWaste.1=0
TotWaste.2=0
TotDirSize.1=0
TotDirSize.2=0

/* Uses CMD.EXE or 4OS2.EXE to get a quick list of all subdirectories */
/* on the drive for the computation of their space consumption, and   */
/* that of the files they contain.                                    */

call SysCls
ClrStr="                                        "
ClrStr=ClrStr||ClrStr
Data=ProcDir(TestDrive||'\')
TotSize=TotSize+word(Data,3)
do i=1 to 2
    TotWaste.i=TotWaste.i+word(Data,i)
    TotDirSize.i=TotDirSize.i+AUnit.i
end
TotFiles=TotFiles+word(Data,4)
p1=right(d2x(random(65535)),4,'0')
p2=right(d2x(random(65535)),4,'0')
TDList='\'||p1||p2||'.TMP'
call SysCurPos 12,0
call SysCurState 'off'
say "Getting list of directories..."
say "(could take a small while, and your HD LED may tell you nothing)"
if UShell='4OS2' then 
    '@dir/b/ad/s > '||TDList
else '@dir/f/ad/s >'||TDList
check=lines(TDList)
call SysCls
do while check=1
    WDir=linein(TDList)
    call SysCurPos 12,0
    say ClrStr
    call SysCurPos 12,0
    if length(WDir)>68 then 
        DDir=left(WDir,68)||'>'
    else
        DDir=WDir
    say "Scanning : "||DDir
    TotDirs=TotDirs+1
    Data=ProcDir(WDir)
    TotSize=TotSize+word(Data,3)
    do i=1 to 2
        TotWaste.i=TotWaste.i+word(Data,i)
        TotDirSize.i=TotDirSize.i+AUnit.i
    end
    TotFiles=TotFiles+word(Data,4)
    DEAData=TallyDirEA(WDir)
    do i=1 to 2
        TotDirSize.i=TotDirSize.i+word(DEAData,i)
    end
    check=lines(TDList)
end
call stream TDList,'c','close'
call SysFileDelete TDList

AllWaste=TotWaste.1-TotWaste.2
AllDir=TotDirSize.1-TotDirSize.2
AllTot=AllWaste+AllDir
if AllTot>=0 then 
    if AllTot=0 then
        TotalDiff=0
else do
    Modifier='*LOST*'
    TotalDiff=abs(AllTot)
end
else do
    Modifier='*SAVED*'
    TotalDiff=abs(AllTot)
end

/* Reports the data in a formatted manner.  Each of the numerical     */
/* variables is deliminated with commas for ease of reading.  First   */
/* you reverse the string.  Then you take the first three characters  */
/* of the reversed string (i.e. last three characters of the original */
/* string), writing the remaining characters back to the variable.    */
/* The three characters are appended to the final string with a comma */
/* at the end, and the process is repeated until there are three or   */
/* less characters left in the reversed string variable.  The final   */
/* write of the formatted string appends the remaining digits.  The   */
/* string is then reversed again (putting it back to normal), which   */
/* results in a readable number:                                      */
/*                                                                    */
/*  2484693813       - Original string                                */
/*  3183964842       - Reversed                                       */
/*  318,             - Three chars and comma appended to final string */
/*  318,396,         - Next three and a comma                         */
/*  318,396,484,     - Next three and a comma                         */
/*  318,396,484,2    - Last char(s), three or less, no comma          */
/*  2,484,693,813    - Reversed back to normal                        */
/*                                                                    */
/* The screen clears, the data is presented, and it says how much     */
/* space is saved or lost by using the current file system.           */


TDriveSize=reverse(DriveSize)
FormDriveSize=''
do while length(TDriveSize)>3
    parse var TDriveSize TriDig +3 TDriveSize
    FormDriveSize=FormDriveSize||TriDig||','
end
FormDriveSize=reverse(FormDriveSize||TDriveSize)

TDiff=reverse(TotalDiff)
FormTotalDiff=''
do while length(TDiff)>3
    parse var TDiff TriDig +3 TDiff
    FormTotalDiff=FormTotalDiff||TriDig||','
end
FormTotalDiff=reverse(FormTotalDiff||TDiff)

do i=1 to 2
    TDirSpace.i=reverse(TotDirSize.i)
    FormDirSpace.i=''
    do while length(TDirSpace.i)>3
        parse var TDirSpace.i TriDig +3 TDirSpace.i
        FormDirSpace.i=FormDirSpace.i||TriDig||','
    end

    FormDirSpace.i=reverse(FormDirSpace.i||TDirSpace.i)
    TWasteSpace.i=reverse(TotWaste.i)
    FormWasteSpace.i=''
    do while length(TWasteSpace.i)>3
        parse var TWasteSpace.i TriDig +3 TWasteSpace.i
        FormWasteSpace.i=FormWasteSpace.i||TriDig||','
    end
    FormWasteSpace.i=reverse(FormWasteSpace.i||TWasteSpace.i)

    TUnit.i=reverse(AUnit.i)
    FormUnitSize.i=''
    do while length(TUnit.i)>3
        parse var TUnit.i TriDig +3 TUnit.i
        FormUnitSize.i=FormUnitSize.i||TriDig||','
    end
    FormUnitSize.i=reverse(FormUnitSize.i||TUnit.i)
end

TSize=reverse(TotSize)
FormTotSize=''
do while length(TSize)>3
    parse var TSize TriDig +3 TSize
    FormTotSize=FormTotSize||TriDig||','
end
FormTotSize=reverse(FormTotSize||TSize)

TFiles=reverse(TotFiles)
FormTotFiles=''
do while length(TFiles)>3
    parse var TFiles TriDig +3 TFiles
    FormTotFiles=FormTotFiles||TriDig||','
end
FormTotFiles=reverse(FormTotFiles||TFiles)

TDirs=reverse(TotDirs)
FormTotDirs=''
do while length(TDirs)>3
    parse var TDirs TriDig +3 TDirs
    FormTotDirs=FormTotDirs||TriDig||','
end
FormTotDirs=reverse(FormTotDirs||TDirs)

call SysCls
call SysCurPos 5,0
say "Drive "||TestDrive||" "||FormDriveSize||" bytes"
say ""
say "Total files                           : "||FormTotFiles
if SkipEA=1 then 
say "Total size of files                   : "||FormTotSize||" bytes"
else
say "Total size of files and EA's          : "||FormTotSize||" bytes"
say ""
say "Current statistics using "||FileSystem.1||" :"
say ""
say "Allocation unit size                  : "||FormUnitSize.1||" bytes"
say "Total space consumed by directories   : "||FormDirSpace.1||" bytes"
if SkipEA=1 then
say "Total wasted space for files          : "||FormWasteSpace.1||" bytes"
else                                         
say "Total wasted space for files and EA's : "||FormWasteSpace.1||" bytes"
say ""
say "If it was being used with "||FileSystem.2||" :"
say ""
say "Allocation unit size                  : "||FormUnitSize.2||" bytes"
say "Total space consumed by directories   : "||FormDirSpace.2||" bytes"
if SkipEA=1 then
say "Total wasted space for files          : "||FormWasteSpace.2||" bytes"
else
say "Total wasted space for files and EA's : "||FormWasteSpace.2||" bytes"
say ""
if TotalDiff=0 then do
say "Oddly enough, by using "||FileSystem.1||" instead of "||FileSystem.2||","
say "you've neither saved nor lost any disk space on drive "||TestDrive||"."
end
else do
say "By using "||FileSystem.1||" instead of "||FileSystem.2||", you've"
say Modifier||" "||FormTotalDiff||" bytes on drive "||TestDrive||"."
end
call directory CurrDir
exit

/******************** Begin TallyFiles Procedure **********************/
/* Procedure to determine total size and waste of all files in a      */
/* given directory, for both FAT and HPFS file systems.  The list of  */
/* files is obtained using SysFileTree.  For each file, it's total    */
/* size and the size of its EA's (unless skipped) are tallied for the */
/* file size total, and the waste of the file and it's EA's is added  */
/* to the waste total.  Calculations are performed by the SlackSpace  */
/* procedure, for both FAT and HPFS file systems.                     */

TallyFiles: Procedure expose Fnode.1 Fnode.2 SkipEA UShell AUnit.1 AUnit.2 

TotSize=0
TotWaste.1=0
TotWaste.2=0
arg Dir
call directory Dir
call SysFileTree '*','files.','F'
do i=1 to files.0
    if SkipEA=1 then do
        FSize=word(files.i,3)
        TotSize=TotSize+FSize
        Waste=SlackSpace(Fsize)
        TotWaste.1=TotWaste.1+word(Waste,1)+Fnode.1
        TotWaste.2=TotWaste.2+word(Waste,2)+Fnode.2
    end
    else do
        file=substr(files.i,38)
        p1=right(d2x(random(65535)),4,'0')
        p2=right(d2x(random(65535)),4,'0')
        tfile='\'||p1||p2||'.TMP'
        if UShell='4OS2' then do
            '@dir/a/knm '||'"'file'"'||'>'||tfile
            filestring=linein(tfile)
            UFEASize=word(filestring,4)
            UFFSize=word(filestring,3)
            EACommaPos=pos(',',UFEASize)
            if EACommaPos\=0 then
                EASize=delstr(UFEASize,EACommaPos,1)
            else EASize=UFEASize
            FCommaPos=pos(',',UFFSize)
            if FCommaPos\=0 then do until FCommaPos=0
                UFFSize=delstr(UFFSize,FCommaPos,1)
                FCommaPos=pos(',',UFFSize)
            end 
            FSize=UFFSize
        end
        else do
            '@dir/a/n '||'"'file'"'||'>'||tfile
            do 5
                scrap=linein(tfile)
            end
            filestring=linein(tfile)
        EASize=word(filestring,4)
        FSize=word(filestring,3)
        end
        TotSize=TotSize+FSize+EASize
        FWaste=SlackSpace(FSize)
        FWaste.1=word(FWaste,1)
        FWaste.2=word(FWaste,2)
        if EASize=0 then do
            TotWaste.1=TotWaste.1+FWaste.1+Fnode.1
            TotWaste.2=TotWaste.2+FWaste.2+Fnode.2
        end
        else do
            EAWaste=SlackSpace(EASize)
            EAWaste.1=word(EAWaste,1)
            EAWaste.2=word(EAWaste,2)
            TotWaste.1=TotWaste.1+FWaste.1+EAWaste.1
            TotWaste.2=TotWaste.2+FWaste.2+EAWaste.2
        end
        call stream tfile,'c','close'
        call SysFileDelete tfile
    end
end
return TotSize TotWaste.1 TotWaste.2 files.0
/******************** End TallyFiles Procedure ************************/

/******************** Begin SlackSpace Procddure **********************/
/* Procedure to determine the wasted space of the given file size.    */
/* This procedure receives the file size and allocation unit size     */
/* from the main program.  It divides the file size by the size of    */
/* the allocation unit, then multiplies the integer portion the       */
/* result to obtain the space occupied only by completely used        */
/* allocation units.  The difference between this and the size of the */
/* file is then subtracted from the size of the allocation unit which */
/* results in the amount of wasted space, in bytes.  Of course, if    */
/* there is no partial allocation unit usage, there is no wasted      */
/* space, and 0 is returned in such a case.                           */

SlackSpace: Procedure expose AUnit.1 AUnit.2

arg FileSize
do i=1 to 2
    IntUnit.i=FileSize/AUnit.i
    StrEnd.i=pos('.',IntUnit.i)
    if StrEnd.i=0 then Waste.i=0
    else do
        AUMult.i=substr(IntUnit.i,1,(StrEnd.i-1))
        FullUnits.i=AUMult.i*AUnit.i
        Diff.i=FileSize-FullUnits.i
        Waste.i=AUnit.i-Diff.i
    end
end
return Waste.1 Waste.2
/******************** End SlackSpace Procedure ************************/

/******************** Begin ProcDir Procedure *************************/
/* Procedure to get all information about a specific directory, which */
/* includes total file and EA size and wasted space (for both FAT and */
/* HPFS).                                                             */

ProcDir: Procedure expose FNode.1 FNode.2 AUnit.1 AUnit.2 UShell SkipEA

arg DirName
TSize=TallyFiles(DirName)
FSize=word(TSize,1)
Waste.1=word(TSize,2)
Waste.2=word(TSize,3)
NumFiles=word(TSize,4)
return Waste.1 Waste.2 FSize NumFiles
/******************** End ProcDir Procedure ***************************/

/******************** Begin TallyDirEA Procedure **********************/
/* Procedure to tally the extended attributes of a given directory,   */
/* which is done regardless of the SkipEA status, since it doesn't    */
/* really take that long.  The parent of the given directory is       */
/* changed to, so we can do a DIR listing of all directories, and     */
/* parse the information about the specific one we want from the list */
/* returned.  There seems to be no way to list only a specific        */
/* directory in either CMD.EXE or 4OS2.EXE.  Once the EA size for the */
/* directory is found, the total actual space taken up for each       */
/* allocation unit is calculated, and returned.                       */

TallyDirEA: Procedure expose AUnit.1 AUnit.2 UShell

arg DirName
tDirName=reverse(DirName)
BSPos=pos('\',tDirName)
BDName=translate(reverse(substr(tDirName,1,(BSPos-1))))
call directory DirName||'\..'
p1=right(d2x(random(65535)),4,'0')
p2=right(d2x(random(65535)),4,'0')
tfile='\'||p1||p2||'.TMP'
if UShell='4OS2' then do
    '@dir/n/m/k/h/ad > '||tfile
    do until found=1
        dirstring=linein(tfile)
        if translate(substr(dirstring,46))=BDName then do
            UFEASize=word(dirstring,4)           
            EACommaPos=pos(',',UFEASize)
            if EACommaPos\=0 then
                EASize=delstr(UFEASize,EACommaPos,1)
            else EASize=UFEASize
            found=1
        end
        else found=0
    end
end
else do
    '@dir/n/ad > '||tfile
    do 5
        scrap=linein(tfile)
    end
    do until found=1
        dirstring=linein(tfile)
        if translate(substr(dirstring,41))=BDName then do
            EASize=word(dirstring,4)
            found=1
        end
        else found=0
    end
end
call stream tfile,'c','close'
call SysFileDelete tfile
DSData=SlackSpace(EASize)
DirEATot.1=EASize+word(DSData,1)
DirEATot.2=EASize+word(DSData,2)
return DirEATot.1 DirEATot.2
/******************** End TallyDirEA Procedure ************************/
