/*****************************************************************
*
* PROJECT:        Swiss Perfect 3 Round-Robin Pairing Tester
*
*                 Copyright (C) 1996 Robert Rozycki
*                 SwissPerfect General Public Licence
*
* FILE:           sprr.cpp
*
* DESCRIPTION:    Main module.
*
* AUTHOR:         Robert Rozycki   rozycki@perth.dialix.oz.au
*
* PORTABILITY:    ANSI C++
*
* $Revision: 1.1 $
*
* $Date: 1996/02/19 11:23:47 $
*
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "sp3rrpar.h"

#define SPRR_VERSION "SPRR 1.00.0, Copyright (C) 1996 Robert Rozycki"


// Local function prototypes

static void DoPairing( int players_count, int multi_rounds, int pair_system,
                       int disp_start, int disp_end );
static void PrintUsage();
static void DisplayRound( int *round_table, int tables, int round,
                          int players_count );

////////////////////////////////////////////////////////////////
//
//  function main()
//
//  Command line must tell us how many players we want to pair (-p),
//  how many times we play the same tournament (-m),
//  what pairing system to use (-r if Rutsch rather than standard)
//  and which rounds to print out (-d).
//

int main( int argc, char *argv[] )
{
   int i;
   int pair_system = 0;
   int players_count = 10, multi_rounds = 1;
   int start_round = 0, end_round = 0;
   char *p;

// Read parameters and configure execution
   if( argc < 2 )
   {
      PrintUsage();
      return 0;
   }
   for( i = 1; i < argc; i++ )
   {
      if( argv[i][0] != '-' )
         continue;
      switch( argv[i][1] )
      {
         case 'p':
            players_count = atoi( argv[i+1] );
            i++;
            break;
         case 'm':
            multi_rounds = atoi( argv[i+1] );
            i++;
            break;
         case 'r':
            pair_system = 1;
            break;
         case 'd':
            start_round = atoi( argv[i+1] );
            p = argv[i+1];
            while( *p != '-' && *p != 0 )
               p++;
            if( *p == '-' )
               p++;
            end_round = atoi( p );
            i++;
            break;
         default :
            continue;
      }
   }
   DoPairing( players_count, multi_rounds, pair_system,
              start_round, end_round );
   return 0;
}

///////////////////////////////////////////////////////////////////////
//
//  Checks and process parameters.
//  Creates, uses and destroys pairing object.
//
static void DoPairing( int players_count, int multi_rounds, int pair_system,
                int disp_start, int disp_end )
{
   int i, j, res, rounds, tot_rounds, tables, curr_round;
   int *round_table;
   RoundRobinPair_c *RRPairMngr;

// check parameters
   if( players_count <= 0 )
   {
      printf( "Number of players must be > 0 \n" );
      return;
   }
   if( multi_rounds <= 0 )
      multi_rounds = 1;

   if( pair_system == 1 )
      RRPairMngr = new Rutsch_c( );
   else
      RRPairMngr = new StandardRR_c( );

// calculate number of rounds
   rounds = ( players_count - 1 ) + ( players_count & 0x01 );
   tot_rounds = rounds * multi_rounds;
   if( disp_start == 0 && disp_end == 0 )
   {
      disp_start = 1;
      disp_end = tot_rounds;
   }
   if( disp_start <= 0 )
      disp_start = 1;
   if( disp_end < disp_start )
      disp_end = disp_start;

// show data
   printf("Running parameters:\n");
   printf("   -number of players = %d\n", players_count );
   printf("   -multirounds       = %d\n", multi_rounds );
   printf("   -pairing system    = " );
   if( pair_system == 1 )
      printf( "Rutsch\n" );
   else
      printf( "standard\n" );
   printf("   -display rounds    = %d-%d\n\n", disp_start, disp_end );

   tables = ( players_count + 1 ) >> 1;
   round_table = new int[ 2 * ( tables+1 ) ];
   if( round_table == NULL )
   {
      printf("DoPairing(): cannot allocate pairing table\n");
      return;
   }
   memset( round_table, 0, 2 * ( tables+1 ) * sizeof( int ) );

   for( i = 1; i <= multi_rounds; i++ )
   {
      for( j = 1; j <= rounds; j++ )
      {
         curr_round = (i-1) * rounds + j;
         if( curr_round < disp_start || curr_round > disp_end )
            continue;
         res = RRPairMngr->GetPairing( round_table, players_count, j, i );
         if( res != SUCCESS )
         {
            printf("GetPairing(): Out of memory\n");
            break;
         }
         DisplayRound( round_table, tables, curr_round, players_count );
      }
   }
   delete round_table;
   delete RRPairMngr;
}

///////////////////////////////////////////////////////////////////////
//
//  Displays pairing on the screen
//

static void DisplayRound( int *round_table, int tables, int round,
                          int players_count )
{
   int i, left_player, right_player;
   char buf[100];
   int  n = 1, tens = 1;

   buf[ 0 ] = 0;
   while( ( tens *= 10 ) <= players_count )
      n++;

   printf( "round %d\n", round );
   for( i = 0; i < tables; i++ )
   {
      left_player = round_table[ i << 1 ];
      right_player = round_table[ ( i << 1 ) + 1];
      if( right_player != 0 )
      {
         sprintf( buf+strlen(buf), "%*d-%-*d ",
                  n, left_player, n, right_player );
      }
      else
      {
         sprintf( buf+strlen(buf), "%*d", n, left_player );
      }
      if( strlen( buf ) > 60 )
      {
         printf( "%s\n", buf );
         buf[0] = 0;
      }
   }
   printf( "%s\n", buf );
}

//////////////////////////////////////////////////////////////////////
//
//   Prints usage information 
//
static void PrintUsage()
{
   printf( SPRR_VERSION );
   printf("\nUsage: sprr [options]");
   printf("\nOptions:");
   printf("\n   -p N      number of players (default 10)" );
   printf("\n   -m        number of multiple rounds (default 1)" );
   printf("\n   -d N1-N2  rounds to be displayed (from N1 to N2)");
   printf("\n   -r        Rutsch system" );
   printf("\n\n");
}

