program ErrFlags;

{$M 16384,16384,16384}

(*  ErrFlags; version 2.1  //  Checks nodelist segments for flag errors    *)
(*  Copyright 1995, jonny bergdahl data AB. Freeware. All rights deserved  *)

(*  Send comments to jonny bergdahl at 2:200/150@fidonet                   *)

(*  Revision history                                                       *)
(*  Date   Version  Change                                                 *)
(*  960127 2.1      Increased limit of inbound files to 80                 *)
(*                  Implementation of a 'NoTouch' mode                     *)

Uses DOS;


Type
  TSegmentFile = record
                   FileName : String[13];
                   RptFile  : String[13];
                   Notifier : String[24];
                 end;
  TFlag        = string[10];
  TConvFlag    = record
                   First    : TFlag;
                   Last     : TFlag;
                 end;

Var
  RptFile  : Text;
  CMTFileName,
  CTLFileName,
  TABFileName : String[13];
  DefaultZone,
  ThisZone    : Word;
  DefaultNet,
  ThisNet     : Word;
  SegmentFile : Array[1..80] of TSegmentFile;       (* !! 2.1 960127 *)
  SegmentNum  : Byte;
  OldDir,
  InboundPath,
  NotifyPath,
  NotifyCmd   : String;
  ExecutePath,
  ExecuteCmd  : String;
  ApprFlags   : Array[1..80] of TFlag;
  UserFlags   : Array[1..80] of TFlag;
  ConvFlags   : Array[1..80] of TConvFlag;
  ReduntFlags : Array[1..80] of TConvFlag;
  ApprNum,
  UserNum,
  ConvNum,
  ReduntNum   : Byte;
  FlagErr,
  ReduntErr   : Byte;
  TotFlagErr,
  TotUserErr,
  TotReduErr,
  ThisFlagErr,
  ThisUserErr,
  ThisReduErr : Byte;

  LastErr     : Word;
  Touch,                                         (* !! 2.1 960127 *)
  AnyProcessed: Boolean;


  (* Generic functions                                                 *)

function LastError : Word;   (* Saves the last I/O error code *)

  begin
    LastErr:=IOresult;
    LastError:=LastErr;
  end;

function Upper(ins:String):String;   (* Uppercase string conversion *)
  Var
    L : Integer;
  Begin
    For L:=1 to Length(Ins) do
      ins[L]:=UpCase(Ins[L]);
    Upper:=Ins;
  end;

function StrToWord(var Inp : String):Word;   (* String to Word *)

  Var
    Value : Word;
    Error : Integer;

  begin
    Val(Inp,Value,Error);
    StrToWord:=Value;
  end;

function WordToStr(Inp : Word):String;      (* Word to String *)

  Var
    Temp : String[12];

  begin
    Str(Inp,Temp);
    WordToStr:=Temp;
  end;

function FirstWord(var Instr:String):String;
                             (* This function returns the first word in a *)
                             (* string, and removes it from the original  *)
                             (* string. Used to read the setup files      *)
  Var
    Position : Byte;

  begin
    Position :=1;
    while ((Instr[Position]<>' ') and (Instr[Position]<>#9) and
           (Position <= Length(InStr))) do Inc(Position);
    FirstWord:=Copy(InStr,1,Pred(Position));
    while (((Instr[Position]=' ') or (Instr[Position]=#9)) and
           (Position<=Length(InStr))) do Inc(Position);
    InStr:=Copy(InStr,Position,255);
  end;

function Extract(Var Line:String):String;
                             (* This function works as the above, but for *)
                             (* comma separated files like the nodelist   *)

  begin
    If pos(',',Line)<>0 then
      begin
        Extract:=Copy(Line,1,Pred(Pos(',',Line)));
        Delete(Line,1,Pos(',',Line));
      end
    else
      begin
        Extract:=Line;
        Line:='';
      end;
  end;

  (* Program specific functions and procedures                           *)

procedure SignOn;     (* Initialises the program and extracts parameters *)

  Var
    StdOut : File;
    L      : Integer;
    Temp   : String;

  begin
    CTLFileName:='ERRFLAGS.CTL';
    Assign(StdOut,'');
    Rewrite(StdOut);
    WriteLn('ErrFlags; version 2.1  //  Checks nodelist segments for flag errors');
    WriteLn('Copyright 1995, 1996 jonny bergdahl data AB. Freeware. All rights deserved');
    WriteLn;
    Temp:=Upper(ParamStr(1));
    If (Temp[2]='?') then
      begin
        WriteLn('Syntax:');
        WriteLn;
        WriteLn('ERRFLAGS [control file]');
        WriteLn;
        WriteLn('    control file  Optional control file name. Can be used when processing');
        WriteLn('                  segments in different zones.');
        WriteLn('                  Default file name is ERRFLAGS.CTL');
        Halt(0);
      end;
    If ParamStr(1)<>'' then
      CTLFileName:=ParamStr(1);
    GetDir(0,OldDir);
  end;

procedure ParseCTLfile;    (* Parses the configuration file  *)

  Var
    CTLFile : Text;
    Temp1   : String;
    Temp2   : String;

  begin
    SegmentNum:=0;
    TABFileName:='ERRFLAGS.TAB';
    CMTFileName:='ERRFLAGS.CMT';
    ExecutePath:='';
    ExecuteCmd:='';
    NotifyPath:='';
    NotifyCmd:='';
    Touch:=True;                          (* !! 960127 *)
    Assign(CTLFile,CTLFileName);
    {$I-}
    Reset(CTLFile);
    {I+}
    If LastError<>0 then
      begin
        WriteLn('! Unable to open ',CTLFileName,' - exiting...');
        Halt(1);
      end;
    While not EOF(CTLFile) do
      begin
        {$I-}
        ReadLn(CTLFile,Temp1);
        {$I+}
        If LastError<>0 then
          begin
            WriteLn('! Error reading ',CTLFileName,' - exiting...');
            Halt(1);
          end;
        If Temp1[1]<>';' then
          begin
            Temp2:=Firstword(Temp1);
            If Upper(Temp2)='ZONE' then
              DefaultZone:=StrToWord(Temp1);
            If Upper(Temp2)='NET' then
              DefaultNet:=StrToWord(Temp1);
            If Upper(Temp2)='NOTOUCH' then          (* !! 960127 *)
              Touch:=False;
            If Upper(Temp2)='FILE' then
              begin
                Inc(SegmentNum);
                With SegmentFile[SegmentNum] do
                  begin
                    FileName := FirstWord(Temp1);
                    Notifier := FirstWord(Temp1);
                    RptFile  := FirstWord(Temp1);
                    If RptFile='' then
                      RptFile:=Copy(FileName,1,Pos('.',FileName))+'RPT';
                  end;
              end;
            If Upper(Temp2)='INBOUND' then
              InboundPath:=Temp1;
            If Upper(Temp2)='NOTIFY' then
              NotifyCmd:=Temp1;
            If Upper(Temp2)='NOTIFYPATH' then
              NotifyPath:=Temp1;
            If Upper(Temp2)='EXECUTE' then
              ExecuteCmd:=Temp1;
            If Upper(Temp2)='EXECUTEPATH' then
              ExecutePath:=Temp1;
            If Upper(Temp2)='TABFILE' then
              TABFileName:=Temp1;
            If Upper(Temp2)='CMTFILE' then
              CMTFileName:=Temp1;
          end;
      end;
   If (InboundPath<>'') and (InboundPath[Length(InboundPath)]<>'\') then
     InboundPath:=InboundPath+'\';
   close(CTLFile);
 end;

procedure ParseTabFile;    (* Parses the approved flag file *)

  Var
    TabFile : Text;
    Temp1,
    Temp2   : String;

  begin
    ApprNum:=0;
    UserNum:=0;
    ConvNum:=0;
    ReduntNum:=0;
    Assign(TabFile,TabFileName);
    {$I-}
    Reset(TabFile);
    {I+}
    If LastError<>0 then
      begin
        WriteLn('! Unable to open ',TabFileName,' - exiting...');
        Halt(1);
      end;
    While not EOF(TabFile) do
      begin
        {$I-}
        ReadLn(TabFile,Temp1);
        {$I+}
        If LastError<>0 then
          begin
            WriteLn('! Error reading ',TabFileName,' - exiting...');
            Halt(1);
          end;
        If Temp1[1]<>';' then
          begin
            Temp2:=Firstword(Temp1);
            If Upper(Temp2)='FLAGS' then
              while Temp1<>'' do
                begin
                  Inc(ApprNum);
                  ApprFlags[ApprNum]:=FirstWord(Temp1);
                end;
            If Upper(Temp2)='USER' then
              while Temp1<>'' do
                begin
                  Inc(UserNum);
                  UserFlags[UserNum]:=FirstWord(Temp1);
                end;
            If Upper(Temp2)='CONVERT' then
              while Temp1<>'' do
                begin
                  Inc(ConvNum);
                  ConvFlags[ConvNum].First:=FirstWord(Temp1);
                  ConvFlags[ConvNum].Last:=FirstWord(Temp1);
                end;
            If Upper(Temp2)='REDUNDANT' then
              begin
                Temp2:=FirstWord(Temp1);
                while Temp1<>'' do
                  begin
                    Inc(ReduntNum);
                    ReduntFlags[ReduntNum].First:=Temp2;
                    ReduntFlags[ReduntNum].Last:=FirstWord(Temp1);
                  end;
              end;
          end;
      end;
    close(TabFile);
  end;

function Equal(Test, Against : String):Boolean;
                              (* Check flags for equality, supporting *)
                              (* a few macros in the process          *)
  Var
    Passed  : Boolean;
    L       : Integer;


  begin
    Passed:=(Test[0]=Against[0]);
    L:=1;
    while (passed and (L<=Byte(Test[0]))) do
      begin
        Case Against[L] of
          '$' : Passed:=Test[L] in ['A'..'Z','a'..'z'];
          '@' : Passed:=Test[L] in ['0'..'9'];
          '*' : Passed:=True;
        else
          Passed:=Test[L]=Against[L];
        end;
        Inc(L);
      end;
    Equal:=Passed;
  end;


function Fixflags(Inp : String; RepNode : String):String;
                                   (* Checks a segment line for errors, *)
                                   (* and fix the problems              *)
  Var
    L1,
    L2,
    L3,
    L4,
    UserStart    : Byte;
    NrmFlags,
    UsrFlags     : String;
    nNFlags      : Byte;
    nUFlags      : Byte;
    NFlag        : Array[1..20] of TFlag;
    UFlag        : Array[1..20] of TFlag;
    Passed       : Boolean;

  begin
                             (* First of all, make any conversions *)
                             (* as stated in the CTL file          *)
    For L1:=1 to ConvNum do
      begin
        If Pos(ConvFlags[L1].First,Inp) <>0 then
          Inp:=Copy(Inp,1,Pred(Pos(ConvFlags[L1].First,Inp)))+
               ConvFlags[L1].Last+
               Copy(Inp,Pos(ConvFlags[L1].First,Inp)+
                    Length(ConvFlags[L1].First),255);
      end;
                             (* Then we will divide the flags into normal *)
                             (* and user flags *)
    NrmFlags:=Inp;
    UsrFlags:='';
    UserStart:=Pos(',U',Inp);
    If UserStart<>0 then
      begin
        NrmFlags:=Copy(Inp,1,Pred(UserStart));
        UsrFlags:=Copy(Inp,UserStart,255);
        If UsrFlags[3]=',' then
          Delete(UsrFlags,1,3)
        else
          Delete(UsrFlags,1,2);
        While Pos(',U',UsrFlags)<>0 do
          Delete(UsrFlags,Pos(',U',UsrFlags)+1,1);
      end;
                    (* Now lets check for illegal flags *)
    nNFlags:=0;
    While Length(NrmFlags)>0 Do
      begin
        Inc(nNFlags);
        NFlag[nNFlags]:=extract(NrmFlags);
      end;
    L1:=1;
    For L1:=1 to nNFlags do
      begin
        Passed:=False;
        For L2:=1 to ApprNum do
          If not passed then Passed:=Equal(NFlag[L1],ApprFlags[L2]);
        If Passed=False then
          begin
            WriteLn(' Illegal flag '+nFlag[L1]+' for ',RepNode);
            WriteLn(RptFile,'Illegal flag '+nFlag[L1]+' for ',RepNode);
            Inc(ThisFlagErr);
            nFlag[L1]:='';
          end;
      end;
                   (* And for illegal user flags *)
    nUFlags:=0;
    While Length(UsrFlags)>0 Do
      begin
        Inc(nUFlags);
        UFlag[nUFlags]:=extract(UsrFlags);
      end;
    For L1:=1 to nUFlags do
      begin
        Passed:=False;
        For L2:=1 to UserNum do
          if not passed then Passed:=Equal(UFlag[L1],UserFlags[L2]);
        If Passed=False then
          begin
            Inc(ThisUserErr);
            WriteLn(' Illegal user flag '+UFlag[L1]+' for ',RepNode);
            WriteLn(RptFile,'Illegal user flag '+UFlag[L1]+' for ',RepNode);
            nFlag[L1]:='';
          end;
      end;
              (* Last but not least, let's remove redundant flags *)

    For L1:=1 to nNFlags do
      begin
        For L2:=1 to ReduntNum do
          begin
            if NFlag[L1]=ReduntFlags[L2].First then (* We got a match *)
              For L3:=1 to (nNFLags) do           (* Now check redundancy *)
                If NFlag[L3]=ReduntFlags[L2].Last then (* Got it *)
                  begin
                    WriteLn(' Redundant flag ',NFlag[L3],' due to ', NFlag[L1],' in 2:',RepNode);
                    WriteLn(RptFile,'Redundant flag ',NFlag[L3],' due to ', NFlag[L1],' in 2:',RepNode);
                    NFLag[L3]:='';
                    Inc(ThisReduErr);
                  end;
          end;
      end;
    Inp:='';
    For L1:=1 to NNFlags do
      If nFlag[L1]<>'' then Inp:=Inp+','+nFlag[L1];
    If nUFlags>0 then
      Inp:=Inp+',U';
    For L1:=1 to nUFlags do
      If uFlag[L1]<>'' then Inp:=Inp+','+uFlag[L1];
    FixFlags:=Inp;
  end;

function ParseNodeLine(Inp : String):String;
                               (* initial parser of segment line      *)
                               (* Since MakeNl makes checks on the    *)
                               (* normal fields, we do not check them *)
                               (* but chacks should go here           *)
  Var
    CurrentNode : String[25];
    Coord       : String[10];
    Node        : String[5];
    Line        : String;

begin
  Coord:=Extract(Inp);
  Node:=Extract(Inp);
  CurrentNode:=WordToStr(ThisZone)+':'+WordToStr(ThisNet)+'/'+Node;
  If Coord='Zone' then
    begin
      ThisZone:=StrToWord(Node);
      CurrentNode:=Node+':'+Node+'/0';
    end;
  If (Coord='Region') or (Coord='Host') then
    begin
      ThisNet:=StrToWord(Node);
      CurrentNode:=WordToStr(ThisZone)+':'+Node+'/0';
    end;
  ParseNodeLine:=Coord+','+Node+','+Extract(Inp)+','+Extract(Inp)+','+
                 Extract(Inp)+','+Extract(Inp)+','+Extract(Inp)+
                 FixFlags(Inp,CurrentNode);
end;


procedure CheckSegment(InF, RptF, Notif : String);
                                (* Checks complete segment files, creating *)
                                (* a fixed segment file together with a    *)
                                (* report file for notification            *)
  Var
    TmpFile,
    InFile,
    OutFile  : Text;
    Temp     : String;

  begin
    ThisZone:=DefaultZone;
    ThisNet:=DefaultNet;
    ThisFlagErr:=0;
    ThisUserErr:=0;
    ThisReduErr:=0;
    If Touch then         (* !! 960127 No need to erase in NoTouch mode *)
      begin
        Assign(InFile,'NET$TEMP.$$$');
        {$I-}
        Erase(InFile);
        {$I+}
        If LastError=0 then;  (* This would be the normal *)
      end;
    Assign(InFile,InF);
    If Touch then             (* !! 960127 Don't rename in NoTouch mode *)
      begin
        {$I-}
        Rename(InFile,'NET$TEMP.$$$');
        {$I+}
        If LastError<>0 then
          begin
            If LastErr<>2 then  (* file not found is normal *)
              WriteLn('! Unable to rename ',InF,' - error ',WordToStr(LastErr));
            Exit;
          end;
        Assign(OutFile,InF);   (* !! 960127 Moved outfile creation code *)
        {$I-}
        Rewrite(OutFile);
        {$I+}
        If LastError<>0 then
          begin
            WriteLn('! Unable to create ',InF,' - error ',WordToStr(LastErr));
            Close(InFile);
            Exit;
          end;
      end;
    {$I-}
    Reset(InFile);
    {$I+}
    If LastError<>0 then
      begin
        if LastErr<>2 then       (* !! 960127 *)
          WriteLn('! Unable to open ',InF,' - error ',WordToStr(LastErr));
        Exit;
      end;
    Assign(RptFile,RptF);
    {$I-}
    Rewrite(RptFile);
    {$I+}
    If LastError<>0 then
      begin
        WriteLn('! Unable to create ',InF,' - error ',WordToStr(LastErr));
        Exit;
      end;
    Assign(TmpFile,CMTFileName);
    {$I-}
    Reset(TmpFile);
    {$I+}
    If LastError=0 then
      begin
        while not EOF(TmpFile) do
          begin
            Readln(TmpFile,Temp);
            WriteLn(RptFile,Temp);
          end;
        writeLn(RptFile);
        close(TmpFile);
      end;
    WriteLn(' Processing file ',InF);
    WriteLn(RptFIle,'Processing file ',Inf,#13#10);
    While not Eof(InFile) Do
      begin
        ReadLn(InFile,Temp);
        If Temp[1]<>';' then             (* !! 960127 *)
          Temp:=ParseNodeLine(Temp);
        If Touch  then                   (* !! 960127 *)
          WriteLn(OutFile,Temp);
      end;
    close(InFile);
    If Touch then    (* !! 960127 *)
      begin
        {$I-}
        Erase(InFile);
        {$I+}
        If LastError<>0 then
          begin
            WriteLn('! Unable to delete temporary file - error ',WordToStr(LastErr));
            Close(OutFile);
            Close(RptFile);
            exit;
          end;
        Close(OutFile);
      end;
    WriteLn(' Segment has ',ThisFlagErr+ThisUserErr,' erronous flags , and ',ThisReduErr,' redundant flags');
    WriteLn(RptFile);
    WriteLn(RptFile,'List has ',ThisFlagErr+ThisUserErr,' erronous flags , and ',ThisReduErr,' redundant flags');
    WriteLn(RptFile);
    WriteLn(RptFile,'--- ErrFlags 2.1 by jonny bergdahl');    (* !! 960127 *)
    Close(RptFile);
    Inc(TotFlagErr,ThisFlagErr);
    Inc(TotUserErr,ThisUserErr);
    Inc(TotReduErr,ThisReduErr);
    AnyProcessed:=True;
    If NotifyCmd<>'' then   (* Only run notification if defined *)
      begin
        {$I-}
        ChDir(NotifyPath);
        {$I+}
        If LastError<>0 then
          writeLn('! Unable to CD to path ',NotifyPath, ' - error ',LastErr);
        Temp:=Upper(NotifyCmd);
        If Pos('%NODE%',Temp)<>0 then
          NotifyCmd:=Copy(NotifyCmd,1,Pred(Pos('%NODE%',Temp)))+Notif+
                Copy(NotifyCmd,Pos('%NODE%',Temp)+6,255);
        Temp:=Upper(NotifyCmd);
        If Pos('%FILE%',Temp)<>0 then
          NotifyCmd:=Copy(NotifyCmd,1,Pred(Pos('%FILE%',Temp)))+RptF+
                Copy(NotifyCmd,Pos('%FILE%',Temp)+6,255);
        WriteLn(' Notify:');
        WriteLn('  ',NotifyCmd);
        Exec(GetEnv('COMSPEC'),'/C '+NotifyCmd);
        ChDir(OldDir);
      end;
  end;

Var
  Loop : Integer;

begin                          (* Main program  *)
  AnyProcessed:=False;
  SignOn;
  ParseCTLFile;
  ParseTabFile;
  TotFlagErr:=0;
  TotUserErr:=0;
  TotReduErr:=0;
  For Loop:=1 to SegmentNum do
    with SegmentFile[Loop] do
      CheckSegment(InboundPath+FileName,RptFile,Notifier);
  If AnyProcessed and (NotifyCmd<>'') then
    begin
      ChDir(ExecutePath);
      {$I+}
      If LastError<>0 then
        writeLn('! Unable to CD to path ',ExecutePath, ' - error ',LastErr);
      WriteLn(' Execute:');
      WriteLn('  ',ExecuteCmd);
      Exec(GetEnv('COMSPEC'),'/C '+ExecuteCmd);
      ChDir(OldDir);
    end;
  WriteLn(' Total flag errors:      ',TotFlagErr);
  WriteLn(' Total user flag errors: ',TotUserErr);
  WriteLn(' Total redundant flags:  ',TotReduErr);
end.