/*******************************************************************/
/* GraphZ.h - header file to support the GraphZ graphical library. */
/* Copyright  1995, ZSoft. All rights reserved.                   */
/*******************************************************************/


/***************************************************/
/* Let's start with some defines for the colors... */
/***************************************************/
#define COLOR_BLUE		RGB(0,0,255)
#define COLOR_GREEN 	RGB(0,255,0)
#define COLOR_RED		RGB(255,0,0)
#define COLOR_YELLOW	RGB(255,255,0)
#define COLOR_PURPLE	RGB(255,0,255)
#define COLOR_CYAN		RGB(0,255,255)
#define COLOR_GRAY		RGB(128,128,128)
#define COLOR_NAVY		RGB(0,0,128)
#define COLOR_BOTTLE	RGB(0,128,0)
#define COLOR_CARMEN	RGB(128,0,0)
#define COLOR_BROWN		RGB(128,128,0)
#define COLOR_MAGENTA	RGB(128,0,128)
#define COLOR_TEMPEST	RGB(0,128,128)
#define COLOR_WHITE		RGB(255,255,255)
#define COLOR_STEAL		RGB(192,192,192)
#define COLOR_BLACK		RGB(0,0,0)





/***************************/
/* Real coordinates points */
/***************************/
typedef struct
	{
	double x;
	double y;
	int curve;
	} GZPOINT, NEAR *PGZPOINT, huge *LPGZPOINT;



#define MAXFONTNAME			41

/*******************/
/* TITLE structure */
/*******************/
typedef struct
	{
	char	szTitle[81];                /* the graph's tile */
	char	szFontName[MAXFONTNAME];	/* title font family name */
	int		nFontSize;                  /* title font size in pts */
	BOOL	bBold;                      /* bold or regular */
	BOOL	bItalic;                    /* italic or regular */
	BOOL	bUnderline;                 /* underline or not */
	BOOL	bGraved;                    /* graved or not (display only) */
	BOOL	bEmbossed;                  /* embossed or not (display only) */
	BOOL	bBorder;                    /* border or not (display only) */
	COLORREF color;                     /* RGB text color (ignored for sculpted styles )*/
	} TITLE, NEAR *PTITLE, FAR *LPTITLE;


/*******************/
/* ALARM structure */
/*******************/
typedef struct
	{
	double	limit;					/* Alarm limit */
	BYTE	nSymbol;				/* Symbol to draw if point overgoes the limit */
	BOOL	bHorizontalLineAtLimit;	/* Draw an horizontal line to show the limit */
	LOGPEN	pen;					/* Pen to use for the horizontal line */
	BYTE	reserved[2];			/* reserved for internal use. Do not modify ! */
	} ALARM, NEAR *PALARM, FAR *LPALARM;


/*******************/
/* CURVE structure */
/*******************/
typedef struct
	{
	char	szLegend[41];       	/* name for this curve */
	LOGPEN	pen;                	/* logical color pen */
	BYTE	nSymbol;            	/* symbol (see the symbol codes definitions below) */
	BOOL	bNoLine;            	/* plot no line, only symbols */
	BOOL	bAlarms;            	/* Check for alarms at run-time ? */
	ALARM	alarmLow, alarmHigh;	/* Alarm settigns... */
	BYTE    reserved[10];       	/* reserved for internal use. Do not modify ! */
	} CURVE, NEAR *PCURVE, FAR *LPCURVE;


/************************************************************/
/* Symbol codes definitions for the CURVE.nSymbol attribute */
/************************************************************/
#define	SYM_NONE		0
#define SYM_CIRCLE		1
#define SYM_SQUARE		2
#define SYM_PLUS		3
#define SYM_CROSS		4
#define SYM_DIAMOND		5
#define SYM_DOT			6
#define SYM_STAR		7
#define SYM_TRIANGLE	8


/*******************/
/* GRAPH structure */
/*******************/
#define NUM_CURVES			16
typedef struct
	{
	char	szIniFile[81];              /* the path and filename of the settings file */
	TITLE	title;                      /* the graph's title */
	char	szXLabel[41];               /* label for the X axis (horizontal) */
	char	szYLabel[41];               /* label for the Y axis (vertical) */
	double	xRange;                     /* x-value range that has to be displayed */
	double	yMin, yMax;                 /* y-scale boundaries when not in autoscale mode */
	double	logBase;                    /* base for logarithm scaling (eg. 10 for decimal log) */
	BOOL	bAutoscale;                 /* autoscale the Y axis */
	BOOL	bLogarithmic;               /* logarithmic Y-scale */
	BOOL	bGrid;                      /* display a grid ? */
	BOOL	bTics;                      /* display tics and tic labels */
	BOOL	bShowAllObjects;            /* show/hide all the objects underneath the graph box */
	WORD	wTimeFormat;                /* date/time format code for the x axis (see the codes below) */
	LONG	bufferLen;                  /* number of buffered values for each curve */
	WORD	numCurves;                  /* number of curves */
	char	szFontName[MAXFONTNAME];    /* font family name for all but the title */
	int		nFontSize;				    /* font size in pts */
	COLORREF background;			    /* curves-box background RGB-color */
	LOGPEN	penGrid;				    /* the pen to use for the grid */
	LOGPEN	penTics;				    /* the pen to use for the tics */
	CURVE	curves[NUM_CURVES];         /* the curves definition */
	BYTE	reserved[530];
	} GRAPH, NEAR *PGRAPH, FAR *LPGRAPH;



/********************************************************/
/* date/time codes for the GRAPH.wTimeFormat attribute. */
/* Combine with the bitwise or operator (|).            */
/********************************************************/
#define DATE_NO			0x0000
#define DATE_DMY		0x0001
#define DATE_YMD		0x0002
#define DATE_MDY		0x0003
#define DATE_DmY		0x0004
#define DATE_YmD		0x0005
#define DATE_mDY		0x0006

#define TIME_NO			0x0000
#define TIME_HM			0x0100
#define TIME_MS			0x0200
#define TIME_HMS		0x0300
#define TIME_S99		0x0400
#define TIME_MS99		0x0500
#define TIME_HMS99		0x0600




/************************************************************************/
/* A parent of a GraphZ window receives the WM_COMMAND message with		*/
/* wParam = GraphZ window's id, and (LPGZNOTIFY)lParam = far pointer	*/
/* to a GZNOTIFY structure, when the users wants to perform some.		*/
/* This application should then validate (return TRUE) or invalidate	*/
/* (return FALSE) the action.											*/
/************************************************************************/
typedef struct
	{
	WORD wNotify;		/* Notification code (see below) */
	HWND hWnd;			/* Handle of the GraphZ window that is notifying its parent */
	LPGRAPH lpGraph;	/* far pointer to a GRAPH structure */
	LPSTR lpszFileName;	/* settings file name */
	} GZNOTIFY, NEAR *PGZNOTIFY, FAR *LPGZNOTIFY;
#define GZUA_SETSETTINGS 1	/* The user clicked OK in the Settings dialog box. */
							/* The settings are about to be changed. */
							/* lpGraph = the new settings as wanted by the user. */
							/* You may modify the values in lpGraph. */
							/* lpszFileName = settings file name */
#define GZUA_LOADSETTINGS 2	/* The user clicked LOAD in the Settings dialog box. */
							/* The settings are about to be loaded. */
							/* lpGraph = NULL. */
							/* lpszFileName = settings file name to load from. */
#define GZUA_SAVESETTINGS 3	/* The user clicked SAVE in the Settings dialog box. */
							/* The currently active settings are about to be saved. */
							/* lpGraph = NULL. */
							/* lpszFileName = settings file name to save to. */





/***********************/
/* Function prototypes */
/***********************/
BOOL FAR PASCAL GZRegister(LPSTR szName, LPSTR szCode);
BOOL FAR PASCAL GZGetSettings(HWND hWnd, LPGRAPH lpGraph);
BOOL FAR PASCAL GZSetSettings(HWND hWnd, LPGRAPH lpGraph);
BOOL FAR PASCAL GZAddPoints(HWND hWnd, LPGZPOINT lpPoints, WORD nNumCurves);
BOOL FAR PASCAL GZAddPointsSeries(HWND hWnd, GZPOINT huge **lpPointsSeries, WORD nNumCurves, WORD nArrayLength);
BOOL FAR PASCAL GZAddCurveSegment(HWND hWnd, double huge *lpX, double huge *lpY, LONG lNumPoints, WORD nCurve);
BOOL FAR PASCAL GZLoadIni(HWND hWnd);
BOOL FAR PASCAL GZWriteIni(HWND hWnd);
BOOL FAR PASCAL GZSetIni(HWND hWnd, LPSTR lpszIniFile);
void FAR PASCAL GZExtractDate(double datetime, LPINT lpnDay, LPINT lpnMonth, LPINT lpnYear);
void FAR PASCAL GZExtractTime(double datetime, LPINT lpnHour, LPINT lpnMinute, LPINT lpnSecond, LPINT lpnHundredth);
void FAR PASCAL GZMakeDuration(double FAR *lpDuration, WORD nDays, WORD nHour, WORD nMinute, WORD nSecond, WORD nHundredth);
void FAR PASCAL GZMakeDate(double FAR *lpDate, WORD nDay, WORD nMonth, WORD nYear, WORD nHour, WORD nMinute, WORD nSecond, WORD nHundredth);
BOOL FAR PASCAL GZShowSettingsDialog(HWND hWnd);
BOOL FAR PASCAL GZPrint(HWND hWnd, BOOL bUseDefaultPrinter);
BOOL FAR PASCAL GZShowAllObjects(HWND hWnd, BOOL bShow);
BOOL FAR PASCAL GZClearBuffers(HWND hWnd);
BOOL FAR PASCAL GZClearCurveBuffer(HWND hWnd, WORD nCurve);
BOOL FAR PASCAL GZRefresh(HWND hWnd);
BOOL FAR PASCAL GZSetWMFLogo(LPSTR szFileName);
void FAR PASCAL GZGetNow(double FAR *now);
BOOL FAR PASCAL GZGetCurveSettings(HWND hWnd, LPCURVE lpCurve, WORD nCurve);
BOOL FAR PASCAL GZSetCurveSettings(HWND hWnd, LPCURVE lpCurve, WORD nCurve);
BOOL FAR PASCAL GZGetTitle(HWND hWnd, LPTITLE lpTitle);
BOOL FAR PASCAL GZSetTitle(HWND hWnd, LPTITLE lpTitle);
BOOL FAR PASCAL GZSetBufferLength(HWND hWnd, LONG lBufferLength);
BOOL FAR PASCAL GZSetNumCurves(HWND hWnd, WORD nNumCurves);
BOOL FAR PASCAL GZStartDisplayAt(HWND hWnd, double x);
BOOL FAR PASCAL GZZoom(HWND hWnd, double xMin, double xMax, double yMin, double yMax);
BOOL FAR PASCAL GZUnzoom(HWND hWnd);


/*
 *
 *
 * Include the following in your project's .DEF file:
 *
 * IMPORTS
 *			GZRegister=GraphZ.1
 *			GZAddPoints=GraphZ.2
 *			GZGetSettings=GraphZ.3
 *			GZSetSettings=GraphZ.4
 *			GZWriteIni=GraphZ.5
 *			GZLoadIni=GraphZ.6
 *			GZSetIni=GraphZ.7
 *			GZMakeDate=GraphZ.8
 *			GZExtractTime=GraphZ.9
 *			GZExtractDate=GraphZ.10
 *			GZMakeDuration=GraphZ.11
 *			GZAddPointsSeries=GraphZ.12
 *			GZShowSettingsDialog=GraphZ.13
 *			GZPrint=GraphZ.14
 *			GZShowAllObjects=GraphZ.15
 *			GZClearBuffers=GraphZ.16
 *			GZRefresh=GraphZ.17
 *			GZSetWMFLogo=GraphZ.18
 * 			GZGetNow=GraphZ.19
 *			GZClearCurveBuffer=GraphZ.20
 *			GZAddCurveSegment=GraphZ.21
 *			GZGetCurveSettings=GraphZ.22
 *			GZSetCurveSettings=GraphZ.23
 *			GZGetTitle=GraphZ.24
 *			GZSetTitle=GraphZ.25
 *			GZSetBufferLength=GraphZ.26
 *			GZSetNumCurves=GraphZ.27
 *			GZStartDisplayAt=GraphZ.28
 *			GZZoom=GraphZ.29
 *			GZUnzoom=GraphZ.30
 *
 * And make a case insensitive link and import.
 *
 * Also: Compile with BYTE alignement for the structures to be correct.
 *
 */

