PROGRAM QHead; {v1.15 - Free DOS utility: Get message headers from QWK files.}
{$M 5120,0,0}  {5k stack, no heap needed}
{$N-,E- no math support needed}
{$X- function calls may not be discarded}
{$I- disable I/O checking (trap errors by checking IOResult)}
{$S- no stack checking code}

{===========================================================================}
                       (** Global declarations ... **)
{===========================================================================}

USES
  DOS, ArcType;

CONST
  cursorState : BYTE = 1;  {0..3}
  cursorData : ARRAY [0..3] OF CHAR = (#179, #47, #196, #92);

VAR
  qheader, qline : string[128];

  unqwk, unarc, unarj, unhap, unlha, unpak,
  unrar, unuc2, unzip, unzoo : PATHSTR;

VAR
  confnumb : WORD;
  ExtractAll : Boolean;

{===========================================================================}
                (** Custom help & exit procedure ... **)
{===========================================================================}

VAR
  SavedExitProc: POINTER;

PROCEDURE cursorOn; FORWARD;

PROCEDURE CustomExit; FAR;
{---- Always exit through here ----}
VAR
  message: STRING [79];
BEGIN
  ExitProc := SavedExitProc;
  cursorOn;
  IF (ExitCode > 0) THEN BEGIN
    Writeln('QHead v1.15 - Free DOS utility: Extract message headers from QWK packets.');
    Writeln('Copyright (c) August 24, 1995, by David Daniel Anderson - Reign Ware.');
    Writeln;
    Writeln('Usage:    QHead <QWKpacket(s)> [conference]');
    Writeln;
    Writeln('Where:    "[conference]" is any valid DOS filename, with an embedded conference');
    Writeln('          number.  If no number is found embedded within the filename, then');
    Writeln('          *all* the conferences'' headers in the QWK packet will be extracted.');
    Writeln;
    Writeln('Examples: QHead c:\qwks\*.qwk cnf100.hdr');
    Writeln('          QHead c:\qwk\channel1.qwk ch-all.hdr');
    Writeln;
    Writeln('Note:     DOS wildcards may be used when specifying the QWKpackets.');
  END;
  IF ErrorAddr <> NIL THEN
  BEGIN
    WriteLn ('An unanticipated error occurred, please contact DDA with the following data:');
    WriteLn ('Address = ', Seg (ErrorAddr^), ':', Ofs (ErrorAddr^));
    WriteLn ('Code    = ', ExitCode);
    ErrorAddr := NIL;
  END
  ELSE
    IF (ExitCode > 0) AND (ExitCode < 255) THEN BEGIN
      CASE ExitCode OF
        1 : message := 'No files found.  First parameter must be a valid file specification.';
        2 : message := 'The second parameter must contain a conference number.';
        5 : message := 'Not enough memory to extract MESSAGES.DAT - aborting!';
        6 : message := 'You cannot just specify a path, add "*.*" or "\*.*" for all files.';
        7 : message := 'File handling error.  Text file is most likely incomplete - or nonexistent.';
        ELSE  message := 'Unknown error.';
      END;
      WriteLn (#7, 'Error encountered (number ', ExitCode, '):'); WriteLn (message);
    END;
END;

{===========================================================================}
                      (** Supporting subroutines ... **)
{===========================================================================}

PROCEDURE CheckIO;
BEGIN
  IF IOResult <> 0 THEN Halt (7);
END;

PROCEDURE cursorOn; ASSEMBLER; ASM
  mov AH, 3; mov BH, 0; Int $10; AND CH, NOT $20; mov AH, 1; Int $10;
END;

PROCEDURE cursorOff; ASSEMBLER; ASM
  mov AH, 3; mov BH, 0; Int $10; OR CH, $20; mov AH, 1; Int $10;
END;

PROCEDURE updateCursor;
BEGIN
  cursorState := Succ (cursorState) AND 3;
  Write (cursorData [cursorState], ^H);
END;

FUNCTION WhereX: BYTE; ASSEMBLER;
(* Routine from SWAG *)
ASM
  MOV AH, 3     {Ask For current cursor position}
  MOV BH, 0     { On page 0 }
  Int 10h       { Return inFormation in DX }
  Inc DL        { Bios Assumes Zero-based. Crt.WhereX Uses 1 based }
  MOV AL, DL    { Return X position in AL For use in Byte Result }
END;

FUNCTION WhereY: BYTE; ASSEMBLER;
(* Routine from SWAG *)
ASM
  MOV AH, 3    {Ask For current cursor position}
  MOV BH, 0    { On page 0 }
  Int 10h      { Return inFormation in DX }
  Inc DH       { Bios Assumes Zero-based. Crt.WhereY Uses 1 based }
  MOV AL, DH   { Return Y position in AL For use in Byte Result }
END;

PROCEDURE GotoXY (X, Y: BYTE); ASSEMBLER;
(* Routine from SWAG *)
ASM
  MOV DH, Y    { DH = Row (Y) }
  MOV DL, X    { DL = Column (X) }
  Dec DH       { Adjust For Zero-based Bios routines }
  Dec DL       { Turbo Crt.GotoXY is 1-based }
  MOV BH, 0    { Display page 0 }
  MOV AH, 2    { Call For SET CURSOR POSITION }
  Int 10h
END;

procedure WriteCharAtCursor(x: char);
(* Routine from SWAG *)
var
  reg: registers;
begin
  reg.AH := $0A;
  reg.AL := ord(x);
  reg.BH := $00;    {* Display Page Number. * for Graphics Modes! *}
  reg.CX := 1;      {* Word for number of characters to write *}
  intr($10, reg);
end;

Procedure ClrEol;
(* Routine by DDA *)
VAR
  NumCol: WORD ABSOLUTE $0040: $004A; { Number of CRT columns (1-based) }
  x, y, DistanceToRight: byte;
BEGIN
  x := WhereX;
  y := WhereY;
  DistanceToRight := NumCol-x;
  Write('':DistanceToRight);
  WriteCharAtCursor(#32);
  GotoXY(x,y);
END;

FUNCTION leadingzero (CONST w: WORD): STRING;
VAR
  s : STRING;
BEGIN
  Str (w : 0, s);
  IF (Length (s) = 1) THEN
    s := '0' + s;
  leadingzero := s;
END;

FUNCTION Upper (w: STRING): STRING;
VAR
  cp : INTEGER;        {The position of the character to change.}
BEGIN
  FOR cp := 1 TO Length (w) DO
    w [cp] := UpCase (w [cp]);
  Upper := w;
END;

FUNCTION RTrim (InStr: STRING): STRING;
BEGIN
  WHILE (Length (InStr) > 0) AND (InStr [Length (InStr) ] IN [#0, #9, #32]) DO
    Dec (InStr [0]);
  RTrim := InStr;
END;

FUNCTION LTrim (InStr: STRING): STRING;
BEGIN
  WHILE (Length (InStr) > 0) AND (InStr [1] IN [#0, #9, #32]) DO
    Delete (InStr, 1, 1);
  LTrim := InStr;
END;

FUNCTION Squeeze (ss: STRING): STRING;
VAR
  controlCHAR: CHAR;
BEGIN
  FOR controlCHAR := #0 TO #31 DO
    WHILE (Ord (ss [0]) > 0) AND (Pos (controlCHAR, ss) > 0) DO
      ss [Pos (controlCHAR, ss) ] := #32;
  Squeeze := RTrim (LTrim (ss));
END;

FUNCTION IsFile (CONST filename: PATHSTR): BOOLEAN;
VAR
  Attr : WORD;
  f    : FILE;
BEGIN
  Assign (f, filename);
  GetFAttr (f, Attr);
  IsFile := (DosError = 0)
END;

PROCEDURE EraseFile (CONST CurrentFile: STRING);
VAR
  df: FILE;
BEGIN
  IF IsFile (CurrentFile) THEN BEGIN
    Assign (df, CurrentFile);
    SetFAttr (df, 0);
    Erase (df); CheckIO;
  END;
END;

{===========================================================================}
                       (** Primary subroutines ... **)
{===========================================================================}

PROCEDURE InitConfig;
VAR
  epath : PATHSTR;
  edir  : DIRSTR;
  ename : NAMESTR;
  eext  : EXTSTR;
  CfgFile        : TEXT;
  CfgLine,
  CfgVar, CfgVal : STRING [128];
  equalPos      : BYTE;

BEGIN
  qheader := '';
  qline := '';

  unQWK := 'gus';

  unARC := 'pkxarc';
  unARJ := 'arj e -y';
  unHAP := 'pah e';
  unLHA := 'lha e';
  unPAK := 'pak e /wa';
  unRAR := 'rar e';
  unUC2 := 'uc e -f';
  unZIP := 'pkunzip -# -o';
  unZOO := 'zoo -extract';

  FSplit (FExpand (ParamStr(0)), edir, ename, eext); { break up path into components }
  epath := edir + ename + '.cfg';
  IF IsFile (epath) THEN
  BEGIN
    Assign (CfgFile, epath);
    Reset (CfgFile); CheckIO;
    REPEAT  { find vars }
      ReadLn (CfgFile, CfgLine);
      equalPos := Pos ('=', CfgLine);
      IF (Length (CfgLine) > 8) THEN BEGIN

        CfgVar := LTrim (RTrim (Upper (Copy (CfgLine, 1, equalPos - 1))));
        CfgVal := LTrim (RTrim (Copy (CfgLine, equalPos + 1, Length (CfgLine) - equalPos)));

        IF (CfgVar = 'UNQWK') THEN
          unQWK := CfgVal
        ELSE IF (CfgVar = 'UNARC') THEN
          unARC := CfgVal
        ELSE IF (CfgVar = 'UNARJ') THEN
          unARJ := CfgVal
        ELSE IF (CfgVar = 'UNHAP') THEN
          unHAP := CfgVal
        ELSE IF (CfgVar = 'UNLHA') THEN
          unLHA := CfgVal
        ELSE IF (CfgVar = 'UNPAK') THEN
          unPAK := CfgVal
        ELSE IF (CfgVar = 'UNRAR') THEN
          unRAR := CfgVal
        ELSE IF (CfgVar = 'UNUC2') THEN
          unUC2 := CfgVal
        ELSE IF (CfgVar = 'UNZIP') THEN
          unZIP := CfgVal
        ELSE IF (CfgVar = 'UNZOO') THEN
          unZOO := CfgVal

        ELSE IF (CfgVar = 'QHEADER') THEN
          qheader := CfgVal
        ELSE IF (CfgVar = 'QLINE') THEN
          qline := CfgVal

      END;
    UNTIL EoF (CfgFile); { loop back to read another line }
    Close (CfgFile);
  END;
END;
{===========================================================================}

FUNCTION IsDir (CONST FileName: PATHSTR): BOOLEAN;
VAR
  Attr  : WORD;
  cFile : FILE;
BEGIN
  Assign (cFile, FileName);
  GetFAttr (cFile, Attr);
  IF (DosError = 0) AND ((Attr AND Directory) = Directory)
    THEN IsDir := TRUE
    ELSE IsDir := FALSE;
END;

FUNCTION GetFilePath (CONST PSTR: STRING; VAR sDir: DIRSTR): PATHSTR;
VAR
  dirinfo   : SEARCHREC;
  jPath     : PATHSTR;  { file path,       }
  jDir      : DIRSTR;   {      directory,  }
  jName     : NAMESTR;  {      name,       }
  jExt      : EXTSTR;   {      extension.  }
BEGIN
  jPath := PStr;
  IF jPath = '' THEN jPath := '*.*';
  IF (NOT (jPath[Length(jPath)] in [':','\'])) AND IsDir (jPath) THEN
    jPath:=jPath+'\';
  IF (jPath[Length(jPath)] in [':','\']) THEN
    jPath:=jPath+'*.*';

  FSplit (FExpand (jPath), jDir, jName, jExt);
  jPath := jDir+jName+jExt;

  sDir := jDir;
  GetFilePath := jPath;
END;

FUNCTION ExtractDAT (CONST QWKfile, DATfileName : STRING): BOOLEAN;
VAR
  X, Y, newX: BYTE;
  extract : STRING [80];
BEGIN
  extract := unqwk;

  X := WhereX;
  Y := WhereY;
  CASE GetArcType (QWKfile) OF
    ARC     : BEGIN extract := unARC; Write ('> assuming ARC '); END;
    ARJ     : BEGIN extract := unARJ; Write ('> assuming ARJ '); END;
    HAP     : BEGIN extract := unHAP; Write ('> assuming HAP '); END;
    LZH     : BEGIN extract := unLHA; Write ('> assuming LHA '); END;
    PAK     : BEGIN extract := unPAK; Write ('> assuming PAK '); END;
    RAR     : BEGIN extract := unRAR; Write ('> assuming RAR '); END;
    UC2     : BEGIN extract := unUC2; Write ('> assuming UC2 '); END;
    ZIP     : BEGIN extract := unZIP; Write ('> assuming ZIP '); END;
    ZOO     : BEGIN extract := unZOO; Write ('> assuming ZOO '); END;
    UNKNOWN : Write ('Trying user type,')
    ELSE
      Write ('File not found!');
  END;
  newX := WhereX;

  SwapVectors;
    Exec (GetEnv ('COMSPEC'), ' /c ' + extract + ' ' + QWKfile+ ' ' + DATfileName + ' >nul');
  SwapVectors;
  IF DosError = 8 THEN Halt (5);

  IF (Y = WhereY) and (WhereX >= newX) THEN
  BEGIN  {If we haven't moved to a new line... }
    GotoXY (X, Y);  {return to where we were at start of procedure}
    ClrEol;
  END;

  cursorOff;
  ExtractDAT := IsFile (DATfileName)
END;
{===========================================================================}

PROCEDURE ProcessHeader (VAR MSGFile: FILE; VAR TXTfile: TEXT; VAR NumChunks: INTEGER);

(* Note: the meaning of the status flag in the header of the QWK
  format specification is interpreted differently by different products.

  According to Patrick Y. Lee's "QWK Mail Packet File Layout" v1.0
  and Robomail v1.30, an asterisk ('*') means private and received,
  and the plus sign ('+') means private and NOT received.

  SLMR 2.1a, SPEED and OLX v1.53 seem to agree that the meaning of the
  two symbols is reversed.

  I believe the second meaning is accurate, so the private and
  received flags will be translated into the following symbols:

  public, unread   =  ' '  (#32)
  public, read     =  '-'  (#45)
  private, unread  =  '*'  (#42)
  private, read    =  '+'  (#43)
*)

TYPE
  MSGDATHdr = RECORD
                Status   : CHAR;
                MSGNum   : ARRAY [1..7] OF CHAR;
                Date     : ARRAY [1..8] OF CHAR;
                Time     : ARRAY [1..5] OF CHAR;
                WhoTo    : ARRAY [1..25] OF CHAR;
                WhoFrom  : ARRAY [1..25] OF CHAR;
                Subject  : ARRAY [1..25] OF CHAR;
                PassWord : ARRAY [1..12] OF CHAR;
                ReferNum : ARRAY [1..8] OF CHAR;
                NumChunk : ARRAY [1..6] OF CHAR;
                Alive    : BYTE;
                ConfNum  : WORD;
                Reserved : ARRAY [1..3] OF CHAR;
              END;
VAR
  VErr          : INTEGER;
  MessageHeader : MSGDATHdr;
  PRIVATE       : CHAR;
BEGIN
  updateCursor;
  BlockRead (MSGFile, MessageHeader, 1);
  Val (Squeeze (MessageHeader. NumChunk), NumChunks, VErr);
  IF (VErr <> 0) THEN NumChunks := 0;
  IF NumChunks <> 0 THEN
    WITH MessageHeader DO BEGIN
      IF ExtractAll OR (confnum = ConfNumb) THEN BEGIN
        IF (Pos (status, '+*~`!#') > 0)
           THEN PRIVATE := #158
           ELSE PRIVATE := #32;
        WriteLn (TXTfile,
        '{',ConfNum:4, ',',
        (RTrim (MSGNum)): 7, '}',
        PRIVATE:2, #32,
        Copy (Date, 1, 5), #32,
        Copy (WhoFrom, 1, 21), #32,
        Copy (WhoTo, 1, 21), #32,
        Copy (Subject, 1, 25), #32
        );
      END;
    END;
END;
{===========================================================================}

PROCEDURE ProcessFiles (VAR MSGFile: FILE; VAR TXTfile: TEXT);
VAR
  QWKrecs,
  Chunks    : INTEGER;
BEGIN
  QWKrecs := 2;                         { start at RECORD #2 }
  WHILE QWKrecs < FileSize (MSGFile) DO BEGIN
    Seek (MSGFile, QWKrecs - 1);
    ProcessHeader (MSGFile, TXTfile, Chunks);
    IF Chunks = 0 THEN
      Chunks := 1;
    Inc (QWKrecs, Chunks);
  END;
END;

FUNCTION GetConfNumb (CONST PSTR: STRING): PATHSTR;
VAR
  ConfNumbpath   : PATHSTR;    { ConfNumb file path,     }
  ConfNumbdir    : DIRSTR;     {             directory,  }
  ConfNumbname   : NAMESTR;    {             name,       }
  ConfNumbext    : EXTSTR;     {             extension.  }

  sTemp : NAMESTR;
  index : BYTE;
  VErr  : INTEGER;
BEGIN
  ExtractAll := FALSE;

  ConfNumbpath := PSTR;
  IF ConfNumbpath = '' THEN
     ConfNumbpath := 'QHEAD.OUT';

  IF ConfNumbpath [1] IN ['/', '-'] THEN Halt (255);
  FSplit (FExpand (ConfNumbpath), ConfNumbdir, ConfNumbname, ConfNumbext);
  IF (ConfNumbname = '') THEN Halt (6);

  sTemp := '';

  FOR index := 1 TO Length (ConfNumbname) DO
    IF ConfNumbname [index] IN ['0'..'9'] THEN
      sTemp := sTemp + ConfNumbname [index];

  IF sTemp = '' THEN BEGIN
    ExtractAll := TRUE;
    ConfNumb := 0;
  END
  ELSE BEGIN
    Val (sTemp, confnumb, VErr);  { confnumb is a GLOBAL var }
    IF VErr <> 0 THEN Halt (2);
  END;

  GetConfNumb := ConfNumbdir + ConfNumbname + ConfNumbext;
END;

{===========================================================================}
                           (** Main program ... **)
{===========================================================================}

CONST
  MSGFileName = 'MESSAGES.DAT';

VAR
  MSGFile : FILE;
  TXTfile : TEXT;

  QWKpath    : PATHSTR;    { QWK file path. }
  QWKdir     : DIRSTR;     { QWK file dir.  }
  TXTpath    : PATHSTR;    { TXT file path. }
  fileinfo   : SEARCHREC;

BEGIN
  SavedExitProc := ExitProc;
  ExitProc := @CustomExit;
  IF NOT (ParamCount in [1,2]) THEN Halt (255);

  cursorOff;

  TXTpath := GetConfNumb (ParamStr (2));
  InitConfig;
  QWKpath := GetFilePath (ParamStr (1), QWKdir);

  FindFirst (QWKpath, Archive, fileinfo); IF DosError <> 0 THEN Halt (1);
  WriteLn ('QHead v1.15 - Free QWK header extractor is now working.');
  WHILE DosError = 0 DO
  BEGIN
    QWKpath := QWKdir + fileinfo. Name;

    EraseFile (MSGFileName);
    Write ('Extracting MESSAGES.DAT from ', QWKpath, ' ...');
    IF ExtractDAT (QWKpath, MSGFileName) THEN BEGIN

      WriteLn (' done!');
      Assign (MSGFile, MSGFileName);
      Reset (MSGFile, 128); CheckIO;
      Assign (TXTfile, TXTpath);
      IF IsFile (TXTpath) THEN BEGIN
        Append (TXTfile); CheckIO;
        Write ('Appending ', TXTpath);
      END
      ELSE BEGIN
        Rewrite (TXTfile); CheckIO;
        IF qheader <> '' THEN
          WriteLn (TXTfile, qheader);
        IF qline <> '' THEN
          WriteLn (TXTfile, qline);
        Write ('Creating ', TXTpath);
      END;
      Write (', and now extracting headers', #32);
      ProcessFiles (MSGFile, TXTfile);
      WriteLn (#8, ', done!');
      Close (MSGFile); CheckIO;
      Close (TXTfile); CheckIO;
      EraseFile (MSGFileName);

    END
    ELSE
      WriteLn ('- bad QWK - skipping.');
    FindNext (fileinfo);
  END;
  WriteLn ('Mission accomplished!');
END.
