/*
 * Copyright (c) 1995 Eugene W. Stark
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Eugene W. Stark.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 5. No copying or redistribution in any form for commercial purposes is
 *    permitted without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY EUGENE W. STARK (THE AUTHOR) ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * GEDCOM lineage-linked database structure definitions
 */

/*
 * Counts of various kinds of top-level records
 */

extern int total_individuals;
extern int total_families;
extern int total_events;
extern int total_sources;
extern int total_notes;
extern int total_repositories;
extern int total_submitters;

/*
 * Flag controlling capitalization of surnames
 */

extern int capitalization;

/*
 * Arrays for each access to top-level records
 */

struct individual_record **all_individuals;
struct family_record **all_familes;

/*
 * Access to the hierarchical index tree
 */

extern struct index_node *index_root;
extern struct index_node *surname_head;

/*
 * Parameters controlling production of index
 */

extern int index_depth;
extern int index_width;

/*
 * Database structure definitions
 */

struct individual_record {
  int serial;
  char *xref;
  struct name_structure *personal_name;
  char *title;
  char sex;
  char *refn;
  char *rfn;
  char *afn;
  struct xref *fams, *lastfams;
  struct xref *famc, *lastfamc;
  struct xref *sources, *lastsource;
  struct note_structure *notes, *lastnote;
  struct event_structure *events, *lastevent;
  struct individual_record *next;
};

struct family_record {
  char *xref;
  char *refn;
  struct xref *husband;
  struct xref *wife;
  struct xref *children, *lastchild;
  struct xref *sources, *lastsource;
  struct note_structure *notes, *lastnote;
  struct event_structure *events, *lastevent;
  struct family_record *next;
};

struct source_record {
  char *xref;
  char *text;
  struct continuation *cont;
};

struct name_structure {
  char *name;
  int surname_start;
  int surname_end;
  char *surname;
};

struct place_structure {
  char *name;
  struct note_structure *notes, *lastnote;
};

struct note_structure {
   char *xref;
   char *text;
   struct continuation *cont;
   struct note_structure *next;
};

struct event_structure {
  struct tag *tag;
  char *date;
  struct place_structure *place;
  struct event_structure *next;
};

struct xref {
  char *id;
  union {
    struct individual_record *individual;
    struct family_record *family;
    struct source_record *source;
  } pointer;
  struct xref *next;
};

struct continuation {
  char *text;
  struct continuation *next;
};

struct index_node {
    int id;				/* UID for constructing file names */
    int level;				/* Level number for output */
    struct individual_record *first;	/* First individual under this index */
    struct individual_record *last;	/* Last individual under this index */
    struct index_node *parent;		/* Parent index node (if any) */
    struct index_node *children;	/* Sub-indexes (if any) */
    struct index_node *next;		/* Next index at same level */
    struct index_node *prev;		/* Previous index at same level */
};

/*
 * Function prototypes
 */

void process_records(struct node *np);
void process_individual_record(struct node *np);
void process_family_record(struct node *np);
void process_event_record(struct node *np);
void process_note_record(struct node *np);
void process_submitter_record(struct node *np);
void process_repository_record(struct node *np);
void process_source_record(struct node *np);
struct event_structure *process_event(struct node *np);
struct note_structure *process_note(struct node *np);
struct xref *process_xref(struct node *np);
struct name_structure *process_name(struct node *np);
void link_records(struct node *np);
void link_individual_record(struct node *np);
void link_family_record(struct node *np);
void index_individuals();
void index_surnames();
int compare_name(struct individual_record **ipp1,
		 struct individual_record **ipp2);

