/*******************************************************************************
*
*       Copyright (c) 1988-95 OS/tools Incorporated -- All rights reserved
*
*       This source code is supplied AS-IS.  No garantees, no promises.
*       Use as you see fit.
*
*   NOTE: The device driver changes some of the DCB fields during normal
*            processing.  The only way to be sure that you change only those
*            fields you intend to change is to use the following sequence to
*            modify a device's DCB:
*
*            Read the DCB (GetDCB()), make any required changes to the 
*            variables just read, then write the DCB (SendDCB()) when
*            making changes to a device's Device Control Block.
*
*     This is NOT the source used to build the IOCTL.DLL supplied on the 
*     distribution diskette.  This sample code is intended for you to compile
*     and link with your communications application.
*
*     You may want to link with the IOCTL.DLL dynamic library included
*     instead of using this code.  Read "IOCTL.TXT" for details on using the
*     IOCTL.DLL.
*
*    The IOctlErrorMessageBox function will only work in a Presentation Manager
*    program.
*
*******************************************************************************/
#define INCL_DEV
#define INCL_DOS
#include <os2.h>
#include <stdio.h>

typedef unsigned char BYTE;
typedef unsigned short WORD;

/*
**  Atom constants
*/
#define IC_ATM_LINEWRT      3001
#define IC_ATM_LINERD       3002
#define IC_ATM_DCBWRT       3003
#define IC_ATM_DCBRD        3004
#define IC_ATM_BAUDWRT      3005
#define IC_ATM_BAUDRD       3006
#define IC_ATM_FXOFF        3007
#define IC_ATM_FXON         3008
#define IC_ATM_MDMGET       3009
#define IC_ATM_MDMSET       3010
#define IC_ATM_FLUSH        3011
#define IC_ATM_COMEVENTS    3012
#define IC_ATM_COMEVENT     3013
#define IC_ATM_COMERROR     3014
#define IC_ATM_XMITSTAT     3015
#define IC_ATM_COMSTAT      3016
#define IC_ATM_BRKON        3017
#define IC_ATM_BRKOFF       3018
#define IC_ATM_RCVQLEN      3019
#define IC_ATM_TXQLEN       3020
#define IC_ATM_IMMBYTE      3021

#define EIC_ATM_ERRORCD     5001
#define EIC_ATM_FUNCTION    5002
#define EIC_ATM_IOCTL       5003

/*
** constants used by FlushBuffers function
**
**  INPUT causes the receive buffer to be flushed
**  OUTPUT causes the transmit buffer to be flushed
*/
#define INPUT         1
#define OUTPUT        2

/*
**      Used by the "SetModemSignals" and "GetOutputModemSignals" functions 
**      (46H).
**
**  Turn bits ON (one) in the "ModemSigOn" field to activate a modem signal
**   all other bits must be off (zero).
**
**  Turn bits OFF in the "ModemSigOff" field to de-activate a modem signal
**   all other bits must be on (one)
**
**   Bit        Signal
**    0           DTR
**    1           RTS
**    2           OUT1
**    4           LOOP
**
**    Marking any other bit will cause the function to be rejected by
**    device driver.  Bits turning on a signal will take precedence over
**    bits turning off a signal.
**
**    Example One:
**
**      stLineChar.ModemSigOn = 0x02;
**      stLineChar.ModemSigOff = 0xfe;
**
**      Sending these values will cause the RTS signal to be activated and
**      will cause the DTR Signal to be de-activated.
**
**    Example Two:
**
**      stLineChar.ModemSigOn = '\x03'
**      stLineChar.ModemSigOff = '\xff'
**
**      Sending these values will cause the RTS and DTR signals to be 
**      activated.
**
**  NOTE:  Bits two (OUT1) and four (LOOP) will be available only when the
**            device driver being used to access a device is COMi and the
**            extension "Enable Modem Extensions" is enabled for that device.
*/
typedef struct
  {
  BYTE byModemSigOn;
  BYTE byModemSigOff;
  }MDMSSIG;

/*
**      Used by "SetLineCharacteristics" and "GetLineCharacteristics" functions
**      (42H and 62H)
**
**      DataBits (word length) field bit definitions:
**
**      0x00-0x04       Reserved
**      0x05            5 data bits
**      0x06            6 data bits
**      0x07            7 data bits
**      0x08            8 data bits
**      0x09-0xff        Reserved
**
**      StopBits (stop bits) field bit definitions:
**
**      0x00            1 stop bit
**      0x01            1.5 stop bits (valid only with 5-bit word length)
**      0x02            2 stop bits (not valid with 5-bit word length)
**
**      Parity field bit definitions:
**
**      0x00            no parity
**      0x01            odd parity
**      0x02            even parity
**      0x03            mark parity (parity bit always 1)
**      0x04            space parity (parity bit always 0)
**      0x05-0xff       Reserved
**
**      The "bTransmittingBreak" field is used only for the 
**      "GetLineCharacteristice" function and returns TRUE (one)
**      if the hardware is currently transmitting a break, and returns
**      FALSE (zero) if the hardware is not currently transmitting
**      a break;
*/
typedef struct
  {
  BYTE byDataBits;
  BYTE byParity;
  BYTE byStopBits;
  BOOL bTransmittingBreak;
  }LINECHAR;

/*
** used by the "SetBaudRate function (43H)
**
**  Set "lBaudRate" to the desired baud rate.  Set "byFraction" to zero
**  under normal conditions.  See the "Physical Device Driver" technical
**  reference under Category One, Function 43H for more detail.
*/
typedef struct
  {
  LONG lBaudRate;
  BYTE byFraction;
  }BAUDRT;

/*
**      Used by the "GetBaudRate" function (63H)
*/
typedef struct
  {
  BAUDRT stCurrentBaud;
  BAUDRT stHighestBaud;
  BAUDRT stLowestBaud;
  }BAUDST;

/*
**      Used by the "SetDCB" and GetDCB" functions (53H and 73H)
**
**      DCB flag bit definitions:
**
**      Flags1:
**
**      bit 7   = Unused
**      bit 6   = enable DSR input Handshaking
**      bit 5   = enable DCD output Handshaking
**      bit 4   = enable DSR output Handshaking
**      bit 3   = enable CTS output Handshaking
**      bit 2   = Unused
**      bit 0-1 = DTR Control Mode
**        BR1 BR0
**           0   0      Disable
**           0   1      Enable
**           1   0      Input Handshaking
**           1   1      Output Handshaking
**
**      Flags2:
**
**      bit 6-7 = RTS Control Mode
**        BR7 BR6
**           0   0      Disable
**           0   1      Enable
**           1   0      Input Handshaking
**           1   1      Toggling on Transmit
**
**      bit 5 = Automatic Receive Flow Control
**           0 = Normal
**           1 = Full_Duplex
**
**      bit 4   = enable break replacement
**      bit 3   = enable NULL striping
**      bit 2   = enable error Replacement
**      bit 1   = enable automatic receive flow control (Xon/Xoff)
**      bit 0   = enable automatic transmit flow control (Xon/Xoff)
**
**      Flags3:
**
**      bit 7 = Transmit Buffer Load Count
**          0 = no transmit buffering
**          1 = 16 byte transmit buffering
**
**      bits 5-6 = receive trigger level
**        BR6 BR5
**           0   0      one character
**           0   1      four characters
**           1   0      eight characters
**           1   1      fourteen characters
**
**      bits 3-4 = Extended Hardware Buffering
**       BR4 BR3
**          0   0      not supported - ignored - no change to fifo mode
**          0   1      extended hardware buffering disabled
**          1   0      extended hardware buffering enabled
**          1   1      automatic protocol override
**
**      bits 1-2 = Read Timeout Processing
**        BR2 BR1
**          0   0       invalid
**          0   1       normal read timeout processing
**          1   0       wait for something, read timeout processing
**          1   1       no wait, read timeout processing
**
**      bit 0    = Enable infinite write timeout processing
**
*/
typedef struct
  {
  WORD wWrtTimeout;
  WORD wReadTimeout;
  BYTE byFlags1;
  BYTE byFlags2;
  BYTE byFlags3;
  BYTE byErrChar;
  BYTE byBrkChar;
  BYTE byXonChar;
  BYTE byXoffChar;
 }DCB;

VOID IOctlErrorMessageBox(ULONG idErrMsg,WORD wFunction,WORD wError);
void SendXonXoff(HFILE hCom,WORD wSignal);
void FlushComBuffer(HFILE hCom,WORD wDirection);
APIRET ForceXoff(HFILE hCom);
APIRET ForceXon(HFILE hCom);
APIRET GetBaudRate(HFILE hCom,LONG *plBaud);
APIRET SetBaudRate(HFILE hCom,LONG lBaud);
APIRET GetLineCharacteristics(HFILE hCom,LINECHAR *pstLineChar);
APIRET SetLineCharacteristics(HFILE hCom,LINECHAR *pstLineChar);
APIRET SetModemSignals(HFILE hCom,BYTE byOnByte,BYTE byOffByte,WORD *pwCOMerror);
APIRET GetModemOutputSignals(HFILE hCom,BYTE *pbySignals);
APIRET GetModemInputSignals(HFILE hCom,BYTE *pbySignals);
APIRET GetTransmitQueueLen(HFILE hCom,WORD *pwQueueLen,WORD *pwByteCount);
APIRET GetReceiveQueueLen(HFILE hCom,WORD *pwQueueLen,WORD *pwByteCount);
APIRET BreakOff(HFILE hCom,WORD *pwCOMerror);
APIRET BreakOn(HFILE hCom,WORD *pwCOMerror);
APIRET GetCOMerror(HFILE hCom,WORD *pwCOMerror);
APIRET GetCOMevent(HFILE hCom,WORD *pwCOMevent);
APIRET GetCOMstatus(HFILE hCom,BYTE *pbyCOMstatus);
APIRET GetXmitStatus(HFILE hCom,BYTE *pbyCOMstatus);
APIRET SendDCB(HFILE hCom,DCB *pstComDCB);
APIRET GetDCB(HFILE hCom,DCB *pstComDCB);

extern HAB habAnchorBlock;
extern char szPortName[];
extern HWND hwndFrame;

VOID IOctlErrorMessageBox(ULONG idErrMsg,WORD wFunction,WORD wError)
  {
  CHAR szErrMsg[80];
  CHAR szMsgString[200];
  LONG lLength;

  lLength = WinLoadString(habAnchorBlock,NULLHANDLE,idErrMsg,sizeof(szMsgString),szMsgString);
  WinLoadString(habAnchorBlock,NULLHANDLE,EIC_ATM_FUNCTION,sizeof(szErrMsg),szErrMsg);
  lLength += sprintf(&szMsgString[lLength],"\n\n%s 0x%02X",szErrMsg,wFunction);
  WinLoadString(habAnchorBlock,NULLHANDLE,EIC_ATM_ERRORCD,sizeof(szErrMsg),szErrMsg);
  sprintf(&szMsgString[lLength],"\n%s 0x%04X",szErrMsg,wError);
  lLength = WinLoadString(habAnchorBlock,NULLHANDLE,EIC_ATM_IOCTL,sizeof(szErrMsg),szErrMsg);
  sprintf(&szErrMsg[lLength]," %s",szPortName);
  /*
  ** Display a message box with MB_OK and CUA warning style
  */
  WinMessageBox(HWND_DESKTOP,
                hwndFrame,
           (PSZ)szMsgString,
           (PSZ)szErrMsg,
                0,
               (MB_MOVEABLE | MB_OK | MB_CUAWARNING));
  }

APIRET GetDCB(HFILE hCom,DCB *pstComDCB)
  {
  ULONG ulDataLen = sizeof(DCB);
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x73,0L,0L,&ulParmLen,(PVOID)pstComDCB,sizeof(DCB),&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_DCBRD,0x73,rc);
  return(rc);
  }

APIRET SendDCB(HFILE hCom,DCB *pstComDCB)
  {
  ULONG ulDataLen = 0L;
  ULONG ulParmLen = sizeof(DCB);
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x53,(PVOID)pstComDCB,sizeof(DCB),&ulParmLen,0L,0L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_DCBWRT,0x53,rc);
  return(rc);
  }

APIRET SendByteImmediate(HFILE hCom,BYTE bySendByte)
  {
  ULONG ulDataLen = 0L;
  ULONG ulParmLen = 1L;
  USHORT rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x44,(PVOID)&bySendByte,1L,&ulParmLen,0L,0L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_IMMBYTE,0x44,rc);
  return(rc);
  }

APIRET GetXmitStatus(HFILE hCom,BYTE *pbyCOMstatus)
  {
  ULONG ulDataLen = 1l;
  ULONG ulParmLen = 0l;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x65,0L,0L,&ulParmLen,(PVOID)pbyCOMstatus,1L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_XMITSTAT,0x65,rc);
  return(rc);
  }

APIRET GetCOMstatus(HFILE hCom,BYTE *pbyCOMstatus)
  {
  ULONG ulDataLen = 1L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x64,0L,0L,&ulParmLen,pbyCOMstatus,1L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_COMSTAT,0x64,rc);
  return(rc);
  }

APIRET GetCOMevent(HFILE hCom,WORD *pwCOMevent)
  {
  ULONG ulDataLen = 2L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x72,0L,0L,&ulParmLen,pwCOMevent,2L,&ulDataLen)) != 0)
  	IOctlErrorMessageBox(IC_ATM_COMEVENT,0x72,rc);
  return(rc);
  }

APIRET GetCOMerror(HFILE hCom,WORD *pwCOMerror)
  {
  ULONG ulDataLen = 2L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x6d,0L,0L,&ulParmLen,pwCOMerror,2L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_COMERROR,0x6d,rc);
  return(rc);
  }

APIRET BreakOn(HFILE hCom,WORD *pwCOMerror)
  {
  ULONG ulDataLen = 2L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x4b,0L,0L,&ulParmLen,pwCOMerror,2L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_BRKON,0x4b,rc);
  return(rc);
  }

APIRET BreakOff(HFILE hCom,WORD *pwCOMerror)
  {
  ULONG ulDataLen = 2L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x45,0L,0L,&ulParmLen,pwCOMerror,2L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_BRKOFF,0x45,rc);
  return(rc);
  }

APIRET GetReceiveQueueLen(HFILE hCom,WORD *pwQueueLen,WORD *pwByteCount)
  {
  ULONG ulDataLen = 4L;
  ULONG ulParmLen = 0L;
  APIRET rc;
  struct
    {
    WORD wByteCount;
    WORD wQueueLen;
    }stWordPair;

  if ((rc = DosDevIOCtl(hCom,0x01,0x68,0L,0L,&ulParmLen,&stWordPair,4L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_RCVQLEN,0x68,rc);
  *pwQueueLen = stWordPair.wQueueLen;
  *pwByteCount = stWordPair.wByteCount;
  return(rc);
  }

APIRET GetTransmitQueueLen(HFILE hCom,WORD *pwQueueLen,WORD *pwByteCount)
  {
  ULONG ulDataLen = 4L;
  ULONG ulParmLen = 0L;
  APIRET rc;
  struct
    {
    WORD wByteCount;
    WORD wQueueLen;
    }stWordPair;

  if ((rc = DosDevIOCtl(hCom,0x01,0x69,0L,0L,&ulParmLen,&stWordPair,4L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_TXQLEN,0x69,rc);
  *pwQueueLen = stWordPair.wQueueLen;
  *pwByteCount = stWordPair.wByteCount;
  return(rc);
  }

APIRET GetModemInputSignals(HFILE hCom,BYTE *pbySignals)
  {
  ULONG ulDataLen = 1L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x67,0L,0L,&ulParmLen,pbySignals,1L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_MDMSET,0x67,rc);
  return(rc);
  }

APIRET GetModemOutputSignals(HFILE hCom,BYTE *pbySignals)
  {
  ULONG ulDataLen = 1L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x66,0L,0L,&ulParmLen,pbySignals,1L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_MDMGET,0x66,rc);
  return(rc);
  }

APIRET SetModemSignals(HFILE hCom,BYTE byOnByte,BYTE byOffByte,WORD *pwCOMerror)
  {
  ULONG ulDataLen = 2L;
  ULONG ulParmLen = 2L;
  APIRET rc;
  MDMSSIG stModemSetSignals;

  stModemSetSignals.byModemSigOn = byOnByte;
  stModemSetSignals.byModemSigOff = byOffByte;

  if ((rc = DosDevIOCtl(hCom,0x01,0x46,&stModemSetSignals,2L,&ulParmLen,pwCOMerror,2L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_MDMSET,0x46,rc);
  return(rc);
  }

APIRET SetLineCharacteristics(HFILE hCom,LINECHAR *pstLineChar)
  {
  ULONG ulDataLen = 0L;
  ULONG ulParmLen = sizeof(LINECHAR);
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x42,pstLineChar,sizeof(LINECHAR),&ulParmLen,0L,0L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_LINERD,0x42,rc);
  return(rc);
  }

APIRET GetLineCharacteristics(HFILE hCom,LINECHAR *pstLineChar)
  {
  ULONG ulDataLen = sizeof(LINECHAR);
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x62,0L,0L,&ulParmLen,pstLineChar,sizeof(LINECHAR),&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_LINERD,0x62,rc);
  return(rc);
  }

APIRET SetBaudRate(HFILE hCom,LONG lBaud)
  {
  ULONG ulDataLen = 0l;
  ULONG ulParmLen = sizeof(BAUDRT);
  APIRET rc;
  BAUDRT stBaudRate;

  stBaudRate.lBaudRate = lBaud;
  stBaudRate.byFraction = 0;
  if ((rc = DosDevIOCtl(hCom,0x01,0x43,&stBaudRate,sizeof(BAUDRT),&ulParmLen,0L,0L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_BAUDWRT,0x43,rc);
  return(rc);
  }

APIRET GetBaudRate(HFILE hCom,LONG *plBaud)
  {
  ULONG ulDataLen = sizeof(BAUDRT);
  ULONG ulParmLen = 0L;
  APIRET rc;
  BAUDST stBaudRate;

  *plBaud = 0;
  if ((rc = DosDevIOCtl(hCom,0x01,0x63,0L,0L,&ulParmLen,&stBaudRate,sizeof(BAUDRT),&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_BAUDRD,0x63,rc);
  *plBaud = stBaudRate.stCurrentBaud.lBaudRate;
  return(rc);
  }

APIRET ForceXon(HFILE hCom)
  {
  ULONG ulDataLen = 0L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x48,0L,0L,&ulParmLen,0L,0L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_FXON,0x48,rc);
  return(rc);
  }

APIRET ForceXoff(HFILE hCom)
  {
  ULONG ulDataLen = 0L;
  ULONG ulParmLen = 0L;
  APIRET rc;

  if ((rc = DosDevIOCtl(hCom,0x01,0x47,0L,0L,&ulParmLen,0L,0L,&ulDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_FXOFF,0x47,rc);
  return(rc);
  }

void FlushComBuffer(HFILE hCom,WORD wDirection)
  {
	ULONG pcbDataLen = 0;
	ULONG pcbParmLen = 1;
  WORD Error;
  BYTE byParam = 0;

	if ((Error = DosDevIOCtl(hCom,0x0b,(ULONG)wDirection,&byParam,1L,&pcbParmLen,0L,0L,&pcbDataLen)) != 0)
    IOctlErrorMessageBox(IC_ATM_FLUSH,0x0b,Error);
  }

void SendXonXoff(HFILE hCom,WORD wSignal)
  {
  DCB stComDCB;
  BYTE byTemp;

  GetDCB(hCom,&stComDCB);
  if (wSignal == SEND_XON)
    byTemp = stComDCB.byXonChar;
  else
    byTemp = stComDCB.byXoffChar;

  SendByteImmediate(hCom,byTemp);
  }

