(******************************************************************)
(*                                                                 *)
(*            SELFTEST.PAS                         April 95        *)
(*                                                                 *)
(*  SELFTEST requires two serial ports on the same computer. The   *)
(*  program transmits a test string on one port (FirstCOM) and     *)
(*  receives on a second port (SecondCOM), where the two ports are *)
(*  connected via a null modem adapter. The received string is     *)
(*  tested against the transmit string (they should be idenical).  *)
(*                                                                 *)
(*  Connect the two serial ports (on a single computer) together   *)
(*  using a null modem cable. Be sure to modify the configuration  *)
(*  section for non-standard PC ports or to setup your multiport   *)
(*  board. Note that many multiport boards are either Digiboard or *)
(*  BOCA board compatible.                                         *)
(*                                                                 *)
(*******************************************************************)


program selftest;
uses crt, PCL4P;

const
   PC = 1;
   DB = 2;
   BB = 3;

var
   BaudCode  : Integer;
   RetCode   : Integer;
   Version   : Integer;
   C         : Char;
   I         : Integer;
   Port      : Integer;
   Reset1st  : Boolean;
   Reset2nd  : Boolean;
   BufPtr    : Pointer;
   BufSeg    : Integer;
   TestString: String;
   TestLength: Integer;
   FirstCOM  : Integer;
   SecondCOM : Integer;
   TheSwitch : Integer;
   ComLimit  : Integer;

procedure SayError( Code : Integer );
var
   RetCode : Integer;
begin
   if Code < 0 then RetCode := SioError( Code )
   else if (Code and (FramingError or ParityError or OverrunError)) <> 0 then
      begin (* Port Error *)
         if (Code and FramingError) <> 0 then WriteLn('Framing Error');
         if (Code and ParityError)  <> 0 then WriteLn('Parity Error');
         if (Code and OverrunError) <> 0 then WriteLn('Overrun Error')
      end
end;

function ErrorCheck(Code : Integer) : Integer;
begin
  (* trap PCL error codes *)
  if Code < 0 then
     begin
       WriteLn;
       Write('ERROR: ');
       SayError( Code );
       if Reset1st then RetCode := SioDone(FirstCOM);
       if Reset2nd then RetCode := SioDone(SecondCOM);
       WriteLn('*** HALTING ***');
       Halt;
     end;
 ErrorCheck := Code;
end;

begin   (* main program *)
   Reset1st := FALSE;
   Reset2nd := FALSE;
   BaudCode := Baud9600;
   TheSwitch := 0;
   TestString := 'This is a test string';
   (* fetch PORT # from command line *)
   if ParamCount <> 3 then
      begin
         WriteLn('USAGE: "SELFTEST {PC|DB|BB} 1stCom 2ndCom"');
         halt;
      end;
   (* determine port type *)
   if (ParamStr(1)='pc') OR (ParamStr(1)='PC') then TheSwitch := PC;
   if (ParamStr(1)='db') OR (ParamStr(1)='DB') then TheSwitch := DB;
   if (ParamStr(1)='bb') OR (ParamStr(1)='BB') then TheSwitch := BB;
   (* check switch value *)
   if TheSwitch = 0 then
     begin
       WriteLn('Must specify "PC", "DB" or "BB" as 1st argument');
       WriteLn('EG:  SELFTEST PC 1 4');
       Halt
     end;
   (* set port limits *)
   if TheSwitch = PC then ComLimit := COM4;
   if TheSwitch = DB then ComLimit := COM8;
   if TheSwitch = BB then ComLimit := COM16;
   (* get FirstCom *)
   Val( ParamStr(2),FirstCom, RetCode );
   if RetCode <> 0 then
      begin
         WriteLn('1st COM port must be 1 to 20');
         Halt;
      end;
   FirstCom := FirstCom - 1;
   if (FirstCom<COM1) or (FirstCom>COM20) then
      begin
         WriteLn('1st COM port must be 1 to 20');
         Halt
      end;
   WriteLn('FirstCOM =',1+FirstCOM);
   (* get SecondCOM *)
   Val( ParamStr(3),SecondCom, RetCode );
   if RetCode <> 0 then
      begin
         WriteLn('2nd COM port must be 1 to 20');
         Halt;
      end;
   SecondCom := SecondCom - 1;
   if (SecondCom<COM1) or (SecondCom>COM20) then
      begin
         WriteLn('2nd COM port must be 1 to 20');
         Halt
      end;
   WriteLn('SecondCOM =',1+SecondCOM);
   (* check range limits *)
   if FirstCOM < COM1 then
     begin
       WriteLn('1stCom must be >= COM1');
       Halt;
     end;
   if SecondCOM > ComLimit then
     begin
       WriteLn('2ndCom must be <= COM',1+ComLimit);
       Halt;
     end;
   if FirstCOM >= SecondCOM then
     begin
       WriteLn('1stCom must be < 2ndCom');
       Halt;
     end;
   (* configure ports as necessary *)
   if TheSwitch = DB then
     begin
       (*** Custom Configuration: DigiBoard PC/8 ***)
       WriteLn('[ Configuring for DigiBoard PC/8 (IRQ5) ]');
       SioPorts(8,COM1,$140,DIGIBOARD);
       for Port := COM1 to COM8 do
         begin
            (* set DigiBoard UART addresses *)
            ErrorCheck( SioUART(Port,$100+8*Port) );
            (* set DigiBoard IRQ *)
            ErrorCheck( SioIRQ(Port,IRQ5) );
         end;
       end;
   if TheSwitch = BB then
     begin
        (*** Custom Configuration: BOCA BB2016 ***)
        WriteLn('[ Configuring for BOCA Board BB2016 (IRQ15) ]');
        SioPorts(16,COM1,$107,BOCABOARD);
        for Port := COM1 to COM16 do
          begin
            (* set BOCA Board UART addresses *)
            ErrorCheck( SioUART(Port,$100+8*Port) );
            (* set BOCA Board IRQ *)
            ErrorCheck( SioIRQ(Port,IRQ15) );
          end;
       end;
   if TheSwitch = PC then
     begin
       WriteLn('[ Configuring for standard PC ports]');
     end;
   (* setup 1K receive buffers *)
   GetMem(BufPtr,1024+16);
   BufSeg := Seg(BufPtr^) + ((Ofs(BufPtr^)+15) SHR 4);
   RetCode := ErrorCheck( SioRxBuf(FirstCOM, BufSeg, Size1024) );
   GetMem(BufPtr,1024+16);
   BufSeg := Seg(BufPtr^) + ((Ofs(BufPtr^)+15) SHR 4);
   RetCode := ErrorCheck( SioRxBuf(SecondCOM, BufSeg, Size1024) );
   (* using transmit interrupts ? *)
   if SioInfo('I') > 0 then
      begin
         (* setup 1K transmit buffers *)
         WriteLn('Setting up transmit buffers');
         GetMem(BufPtr,1024+16);
         BufSeg := Seg(BufPtr^) + ((Ofs(BufPtr^)+15) SHR 4);
         RetCode := ErrorCheck( SioTxBuf(FirstCOM, BufSeg, Size1024) );
         GetMem(BufPtr,1024+16);
         BufSeg := Seg(BufPtr^) + ((Ofs(BufPtr^)+15) SHR 4);
         RetCode := ErrorCheck( SioTxBuf(SecondCOM, BufSeg, Size1024) );
      end;
   (* reset FirstCOM *)
   RetCode := SioReset(FirstCOM,BaudCode);
   (* if error then try one more time *)
   if RetCode <> 0 then RetCode := ErrorCheck( SioReset(FirstCOM,BaudCode) );
   Reset1st := TRUE;
   (* Port successfully reset *)
   WriteLn('COM',1+FirstCOM,' reset @ 9600 Baud');
   (* reset SecondCOM *)
   RetCode := SioReset(SecondCOM,BaudCode);
   (* if error then try one more time *)
   if RetCode <> 0 then RetCode := ErrorCheck( SioReset(SecondCOM,BaudCode) );
   (* SecondCOM successfully reset *)
   WriteLn('COM',1+SecondCOM,' reset @ 9600 Baud');
   Reset2nd := TRUE;
   (* set port parmameters *)
   RetCode := ErrorCheck( SioParms(FirstCOM, NoParity, OneStopBit, WordLength8) );
   RetCode := ErrorCheck( SioParms(SecondCOM, NoParity, OneStopBit, WordLength8) );
   (* set FIFO level if have INS16550 *)
   RetCode := ErrorCheck( SioFIFO(FirstCOM, LEVEL_8) );
   RetCode := ErrorCheck( SioFIFO(SecondCOM, LEVEL_8) );
   (* flush ports *)
   RetCode := ErrorCheck( SioRxFlush(FirstCOM) );
   RetCode := ErrorCheck( SioRxFlush(SecondCOM) );
   WriteLn('***      SELFTEST: 04/02/95 ');
   Version := SioInfo('V');
   WriteLn('***       Library: ',Version SHR 4,'.',15 AND Version);
   WriteLn;
   TestLength := Length(TestString);
   (* send string *)
   Write('  Sending: ');
   for I := 1 to TestLength do
     begin
       C := TestString[I];
       RetCode := ErrorCheck( SioPutc(FirstCOM,C) );
       Write(C);
     end;
   WriteLn;
   (* receive string *)
   Write('Receiving: ');
   for I:= 1 to TestLength do
     begin
        RetCode := ErrorCheck( SioGetc(SecondCOM,18) );
        (* echo just received char *)
        Write(chr(RetCode));
        (* compare character *)
        if chr(RetCode) <> TestString[I] then
           begin
              WriteLn('ERROR: Expecting ',TestString[I],' received ',chr(RetCode));
              if Reset1st then SioDone(FirstCOM);
              if Reset2nd then SioDone(SecondCOM);
              Halt;
           end;
     end;
   WriteLn;
   WriteLn('SUCCESS: Test AOK !');
   RetCode := SioDone(FirstCOM);
   RetCode := SioDone(SecondCOM);
 end.

