/*****************************************************************************
   MODULE: rgett.c
  PURPOSE: recio character delimited time input functions
COPYRIGHT: (C) 1994-1995, William Pierpoint
 COMPILER: Borland C Version 3.1
       OS: MSDOS Version 6.2
  VERSION: 2.12
  RELEASE: January 29, 1995
*****************************************************************************/

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "recio.h"

extern int _risready(REC *rp, int mode);
extern char *_rfldstr(REC *rp, size_t len);
extern char *_rerrs(REC *rp, int errnum);

#define rtmfmt(rp) ((rp)->r_tmfmt)

int _rgett_c_warno;  /* warning number */

/****************************************************************************/
static int                   /* return number of digits read                */
    istrncpy(                /* copy up to n digits from src to dst         */
              char *dst,     /* destination string                          */
        const char *src,     /* source string                               */
                int n)       /* number of digits to copy                    */
/****************************************************************************/
{
    int i;
    
    /* copy up to n digits */
    for (i=0; i < n; i++) {
        if (isdigit(*src)) {
            *dst++ = *src++;
        } else {
            break;
        }
    }
    *dst = '\0';
    return i;
}

/****************************************************************************/
void                         /* returns nothing                             */
    _tminit(                 /* initialize a tm struct                      */
        struct tm *t)        /* tm struct                                   */
/****************************************************************************/
{
    t->tm_sec = 0;
    t->tm_min = 0;
    t->tm_hour = 0;
    t->tm_mday = 1;
    t->tm_mon = 0;
    t->tm_year = 70;
    t->tm_isdst = -1;
}

/****************************************************************************/
struct tm                    /* return tm                                   */
    timetotm(                /* convert time_t to struct tm                 */
        time_t time)         /* source time                                 */
/****************************************************************************/
{
    struct tm t;
    
    if (time == (time_t)-1) _tminit(&t);
    else memcpy(&t, localtime(&time), sizeof(struct tm));
    return t;
}

/****************************************************************************/
time_t                       /* return time (-1=error)                      */
    tmtotime(                /* convert struct tm to time_t                 */
        struct tm t)         /* broken-down time source                     */
/****************************************************************************/
{
    time_t time;             /* return time (-1 = error)    */
    
    /* note: see section 3.5 of design.txt for mktime range */
    time = mktime(&t);
    if (time == (time_t) -1) errno = ERANGE;
    return time;
}
 
/****************************************************************************/
time_t                       /* return time (-1=error)                      */
    sftotime(                /* convert formated string to time_t           */
        const char *s,       /* source string with time data                */
        const char *fmt)     /* time format                                 */
/****************************************************************************/
{
    time_t time=(time_t) -1; /* return time (-1=error) */
    struct tm t;             /* broken-down time */

    t = sftotm(s, fmt);
    if (!errno) time = tmtotime(t);
    return time;
}

/****************************************************************************/
struct tm                    /* return tm (if error, errno != 0)            */
    sftotm(                  /* convert formated string to tm               */
        const char *s,       /* source string with time data                */
        const char *fmt)     /* time format                                 */
/****************************************************************************/
{
    struct tm t;     /* tm struct for time */
    char buf[5];     /* string buffer */

    errno = 0;
    _rgett_c_warno = 0;
    _tminit(&t);
    for ( ; ; fmt++) {
            /* skip white space */
            while (isspace(*s)) s++;
            while (isspace(*fmt)) fmt++;
            if (!*s || !*fmt) break;

            if (*fmt == '%') {
            switch (*++fmt) {
            case 'm':  /* month (0 to 11) */
                s += istrncpy(buf, s, 2);
                if (*buf) {
                    t.tm_mon = atoi(buf) - 1;
                    if (t.tm_mon < 0 || t.tm_mon > 11) {
                        /* error */
                        errno = EINVDAT;
                        goto done;
                     }
                }
            	break;
            case 'd':  /* day of month (1 to 31) */
                s += istrncpy(buf, s, 2);
                if (*buf) {
                    t.tm_mday = atoi(buf);
                    if (t.tm_mday < 1 || t.tm_mday > 31) {
                        /* error */
                        errno = EINVDAT;
                        goto done;
                    }
                }
            	break;
            case 'y':  /* 2-digit year (1970 to 2069) */
                s += istrncpy(buf, s, 2);
                if (*buf) {
                    t.tm_year = atoi(buf);
                    if (t.tm_year < 70) t.tm_year += 100;
                }
            	break;
            case 'Y':  /* 4-digit year */
                s += istrncpy(buf, s, 4);
                if (*buf) {
                    t.tm_year = atoi(buf) - 1900;
                }
            	break;
            case 'H':  /* hour (0 to 23) */
                s += istrncpy(buf, s, 2);
                if (*buf) {
                    t.tm_hour = atoi(buf);
                    if (t.tm_hour > 23) {
                        /* error */
                        errno = EINVDAT;
                        goto done;
                    }
                }
            	break;
            case 'M':  /* minute (0 to 59) */
                s += istrncpy(buf, s, 2);
                if (*buf) {
                    t.tm_min = atoi(buf);
                    if (t.tm_min > 59) {
                        /* error */
                        errno = EINVDAT;
                        goto done;
                    }
                }
            	break;
            case 'S':  /* second (0 to 61) */
                       /* includes up to two leap seconds */
                s += istrncpy(buf, s, 2);
                if (*buf) {
                    t.tm_sec = atoi(buf);
                    if (t.tm_sec > 61) {
                        /* error */
                        errno = EINVDAT;
                        goto done;
                    }
                }
            	break;
            case '%':  /* literal % */
                if (*s=='%') {
                    s++;
                } else {
                    /* error */
                    errno = EINVDAT;
                    goto done;
                }
                break;
            default:
                /* error */
                errno = EINVAL;
                goto done;
            }
        } else {
            if (*s==*fmt) {
                s++;
                continue;
            } else if (isspace(*fmt)) {
                continue;
            } else {
                /* error */
                errno = EINVDAT;
                goto done;
            }
        }
    }
done:
    if (!errno) {
        /* warning if any non-whitespace part of format string left */
        while (isspace(*fmt)) fmt++;
        if (*fmt) _rgett_c_warno = R_WTMFMT;
    }
    return t;
}

/****************************************************************************/
struct tm                    /* return tm                                   */
    rgettm(                  /* get time from record stream                 */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
    struct tm t;             /* return tm */
    struct tm val;           /* conversion value         */
    char *fldptr;            /* pointer to field buffer  */

    _tminit(&t);
    if (_risready(rp, R_READ)) {
        fldptr = _rfldstr(rp, 0);
        if (fldptr) {
            strims(fldptr);
            for (;;) {
                if (*fldptr != '\0') {
                    val = sftotm(fldptr, rtmfmt(rp));
                    if (errno) {
                        switch (errno) {
                        case ERANGE:
                            fldptr = _rerrs(rp, R_ERANGE);
                            break;
                        case EINVAL:
                            fldptr = _rerrs(rp, R_EINVAL);
                            break;
                        case EINVDAT:
                            fldptr = _rerrs(rp, R_EINVDAT);
                            break;
                        }
                        if (fldptr) { continue; } else { goto done; } 
                    } else {
                        t = val;
                        if (_rgett_c_warno) rsetwarn(rp, _rgett_c_warno);
                        goto done;
                    }
                } /* missing data */ 
                fldptr = _rerrs(rp, R_EMISDAT); 
                if (fldptr) { continue; } else { goto done; }
            }
        } 
    }
done:
    return t;
}

/****************************************************************************/
time_t                       /* return time (-1=error)                      */
    rgett(                   /* get time from record stream                 */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
    time_t time=(time_t) -1; /* return time (-1 = error) */
    struct tm t;

    t = rgettm(rp);
    if (_risready(rp, R_READ)) {
        time = tmtotime(t);
        if (time == (time_t) -1) rseterr(rp, R_ERANGE);
    }
    return time;
}
