/* area:  Areacode Finder v1.4

  Program searches for telephone area codes/state information by
  area code, state name, city name, or postal codes.

  Author:  Jason Mathews <mathews@nssdc.gsfc.nasa.gov>
	   NASA/Goddard Space Flight Center - Code 633.2
	   Greenbelt Road
	   Greenbelt, MD 20771-0001  USA

  Please notify me at the address above if there are any missing area codes
  or errors found in this program.  I would appreciate any comments if you
  find this program useful or would like to see more features.

  Copyright (C) 1992-95 by Jason Mathews.  Permission is granted to any
  individual or institution to use, copy or redistribute this software so long
  as it is not sold for profit, provided this copyright notice is retained.

  Usage: area                   - help
	 area *                 - list all area codes
	 area XXX [XXX...]      - find match
	  where XXX is an area code, state, city, or postal code.

    States and cities are found by wildcard match if wildcard characters
    ('*' and/or '?') are specified, otherwise a substring comparison
    is made.  All comparisons are case insensitive; e.g., "NEW" and "New"
    are equivalent.

    - Two letter state/postal codes like TX for Texas, CA for California
      can be used, otherwise type in the state name.
    - Enter two word state names like "New*Jersey" or "New York".
    - Enter three digit area code to find a state such as 301 for Maryland.
    - Enter city names in between quotes like "Los Angeles"
      or substrings such as "cisco" for "San Francisco".
    - Enter "*" for a list of all Area Codes.

  Compilers: This program will compile and link with Turbo C 2.0, UNIX gcc,
	     VAX/VMS C, and other ANSI compilers.

    tcc -Z -O -d area.c    (MS-DOS)
    gcc -O area.c -o area   (unix)

  History:

  V1.0 26-Nov-92  - Original version for MS-DOS/UNIX.

  V1.1 20-Apr-94  - Optimized postal code search.
		  - Added 210, 407, 508, 516, 610, 708, 810, 903, 908,
		    909, 910, and 917 area codes (effective 1/94).
		  - Updated 706 and 905 area codes.
		  - Added Canada, Bahamas (BA), Bermuda (BM) postal codes.

  V1.2 24-May-94  - Fixed several database corrections.
		  - Added city search as suggested by Peter Mauzey.
		  - Added strstr function for some "non-ANSI" compilers.

  V1.3 1-Jun-94   - Combined city and state search for faster searches.
		  - Improved search with wildcard expression or substring
		  - search depending on the target string.
		  - Fixed wildcard evaluation with "*?" expression.
		  - Added JM, NT postal codes + more cities.

  V1.4 8-Feb-95   - Added 630 area code and updated Illinois info.

 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define N_ZONES  71
#define N_CODES 177

#ifdef __TURBOC__

# define ToLower(c) tolower(c)
# define ToUpper(c) toupper(c)

#else /* ?unix */

/* Avoid using const if compiler does not support it */
# ifndef __STDC__
#   define const
# endif

/*  some compilers don't check if a character is already upper or lower
 *  case in the toupper/tolower functions, so we define our own here.
 */

# ifndef _tolower
#   define _tolower(c) ((c) + 'a' - 'A')
# endif

# ifndef _toupper
#   define _toupper(c) ((c) + 'A' - 'a')
# endif

# define ToLower(c) (isupper(c) ? _tolower(c) : (c))
# define ToUpper(c) (islower(c) ? _toupper(c) : (c))

#endif /* ?__TURBOC__ */

#ifdef VMS
#  define EXIT exit
#else
#  define EXIT return
#endif /* VMS */


/* Global variables */

const char copyright[] =
	"   ** Area Code Finder v1.4 **    (C) Jason Mathews 1992-95\n";

int zone = -1;  /* zone code */

typedef struct zone_s {
    char *code;
    char *stateName;
} zone_t;

zone_t areazones[N_ZONES] = {
"AL","Alabama",
"AK","Alaska",
"AZ","Arizona",
"AR","Arkansas",
"BC","British Columbia",
"CA","California",
"CN","Canada",
"CO","Colorado",
"CT","Connecticut",
"DE","Delaware",
"DC","Washington, District of Columbia",
"FL","Florida",
"GA","Georgia",
"HI","Hawaii",
"ID","Idaho",
"IL","Illinois",
"IN","Indiana",
"IA","Iowa",
"KS","Kansas",
"KY","Kentucky",
"LA","Louisiana",
"ME","Maine",
"MD","Maryland",
"MA","Massachusetts",
"MI","Michigan",
"MN","Minnesota",
"MS","Mississippi",
"MO","Missouri",
"MT","Montana",
"NE","Nebraska",
"NV","Nevada",
"NH","New Hampshire",
"NJ","New Jersey",
"NM","New Mexico",
"NY","New York",
"NC","North Carolina",
"ND","North Dakota",
"OH","Ohio",
"OK","Oklahoma",
"OR","Oregon",
"PA","Pennsylvania",
"PR","Puerto Rico",
"RI","Rhode Island",
"SC","South Carolina",
"SD","South Dakota",
"TN","Tennessee",
"TX","Texas",
"UT","Utah",
"VT","Vermont",
"VI","Virgin Islands",
"VA","Virginia",
"WA","Washington",
"WV","West Virginia",
"WI","Wisconsin",
"WY","Wyoming",
"  ","Wide area (tollfree)",
"  ","International Access",
"BA","Bahamas",
"BM","Bermuda",
"JM","Jamaica",
"AB","Alberta",
"MB","Manitoba",
"NB","New Brunswick",
"NF","Newfoundland",
"NS","Nova Scotia",
"NT","N.W. Territory",
"ON","Ontario",
"PE","Prince Edward Island",
"PO","Quebec",
"SK","Saskatchewan",
"YT","Yukon Territory"
};

typedef struct code_s {
    short zcode;        /* index into zonelist (0..n) */

    short areaCode;     /* local area code.
			 * if areacode is -1 then the list continues from
			 * the previous entry.
			 */

    char *list;         /* list of cities, counties, ... with same areacode.
			 * if list is NULL (0) then this area code covers
			 * the entire area.
			 */
} code_t;

code_t areacodes[N_CODES+1] = {
0,205,0,
1,907,0,
2,602,0,
3,501,0,
4,604,0, /* British Columbia */

/* CALIFORNIA */
5,209,"Fresno and Stockton",
5,213,"Los Angeles",
5,310,"Beverly Hills, Long Beach, Whittier",
5,408,"San Jose, Sunnyvale",
5,415,"San Francisco area",
5,510,"Oakland, Berkeley, San Francisco",
5,619,"Palm Springs, San Diego and the Imperial Valley",
5,707,"Eureka, Napa, Santa Rosa",
5,714,"Anaheim, Orange County",
5,805,"Bakersfield, Ventura, Simi Valley",
5,818,"Burbank, Pasadena, Van Nuys",
5,909,"Riverside; Southwestern CA",
5,916,"Sacramento and South Lake Tahoe",

/* CANADA */
6,604,"British Columbia",
6,403,"Alberta and Yukon",
6,306,"Saskatchewan",
6,204,"Manitoba",
6,416,"Toronto, Ont.",
6,519,"London, Ont.",
6,613,"Ottawa, Ont.",
6,705,"East Ontario",
6,807,"West Ontario",
6,905,"Southern Ontario",
6,418,"Northern Quebec",
6,514,"Montreal, Quebec",
6,819,"NW Quebec",
6,709,"Newfoundland",
6,506,"New Brunswick",
6,902,"Nova Scotia, Prince Edward Island",

7,303,"Boulder, Denver, Grand Junction",
7,719,"Colorado Springs, Pueblo",
8,203,0,
9,302,0,
10,202,0,

/* Florida */
11,305,"Miami, Key West, Ft. Lauderdale",
11,407,"Orlando, West Palm Beach",
11,813,"Ft. Myers, Tampa, Winter Haven",
11,904,"Jacksonville, Tallahassee",

12,404,"Atlanta, Rome",
12,706,"Athens, Augusta, Columbus",
12,912,"Macon, Savannah, Waycross",
13,808,0,
14,208,0,

/* Illinois */
15,217,"Champaign-Urbana, Decatur, Springfield",
15,309,"Galesburg, Macomb, Peoria",
15,312,"Chicago, O'hare Airport",
15,618,"Alton, Carbondale, Centralia, Mount Vernon",
15,630, "Wireless Devices in Chicago (312) and suburbs (708)",
15,708,"Aurora, Elgin, Plano, Waukegan",
15,815,"De Kalb, Joliet, Moline, Ottawa, Rockford",

16,812,"Evansville",
16,219,"Gary, South Bend, Warsaw",
16,317,"Indianapolis, Kokomo",

17,712,"Council Bluffs, Sioux City",
17,515,"Des Moines",
17,319,"Dubuque",

18,316,"Dodge City, Emporia, Wichita",
18,913,"Hays, Kansas City, Overland Park, Topeka",

19,502,"Louisville, Frankfort, Paducah, Shelbyville",
19,606,"Ashland, Lexington, Middlesboro, Winchester",

20,504,"Baton Rouge, New Orleans",
20,318,"Lake Charles, Monroe, Shreveport",
21,207,0,

/* Maryland */
22,301,"Bethesda, Cumberland, Frederick, Greenbelt,",
22,-1,"Hagerstown, Rockville, Silver Spring",
22,410,"Annapolis, Baltimore, Ellicott City, Salisbury",

23,413,"Pittsfield, Springfield",
23,508,"Fall River, Framingham, New Bedford, Worchester",
23,617,"Boston, Cambridge, Plymouth",
24,313,"Detroit, Ann Arbor",
24,616,"Battle Creek, Grand Rapids, Kalamazoo",
24,517,"Jackson, Lansing",
24,810,"Flint",
24,906,"Escanaba, Sault Ste. Marie",
25,218,"Duluth",
25,612,"Minneapolis, St. Paul",
25,507,"Mankato, Rochester",
26,601,0,
27,816,"Belton, Independence, Kansas City, Marshall, St. Joseph, Sedalia",
27,314,"St. Louis, Cape Girardeau, Columbia, Fulton, Hannibal,",
27,-1,"Jefferson City, Mexico, Poplar Bluff, Rolla",
27,417,"Joplin, Springfield",
28,406,0,
29,402,"Omaha, Lincoln",
29,308,"North Platte, Scottsbluff",
30,702,0,
31,603,0,
32,201,"Newark, Hackensack, Jersey City, Patterson",
32,609,"Atlantic City, Camden, Trenton, Wildwood",
32,908,"Elizabeth, New Brunswick",
33,505,0,
34,518,"Albany, Schenectady",
34,607,"Binghamton, Corning",
34,716,"Buffalo, Niagara Falls, Rochester",
34,516,"Hempstead, Long Island area",
34,914,"Mount Vernon, White Plains",
34,212,"New York City (Manhattan and Bronx)",
34,718,"New York City (Queens, Brooklyn and Staten Island)",
34,917,"New York City (All Locations)",
34,315,"Syracuse, Watertown",
35,704,"Asheville, Charlotte, Salisbury",
35,910,"Greensboro, Winston-Salem",
35,919,"Raleigh",
36,701,0,
37,216,"Akron, Canton, Cleveland, Youngstown",
37,513,"Cincinnati, Dayton, Springfield",
37,614,"Columbus, Marion, Zanesville",
37,419,"Lima, Toledo",
38,918,"Bartlesville, McAlester, Muskogee, Tulsa",
38,405,"Enid, Oklahoma City, Norman, Ponca City, Stillwater",
39,503,0,
40,215,"Levittown, Philadelphia, Quakertown",
40,412,"Butler, Indiana, New Castle, Pittsburgh, Uniontown",
40,610,"Allentown, Bethlehem, Reading, Swarthmore, Valleyforge",
40,717,"Gettysburg, Harrisburg, Hershey, Lancaster, Scranton",
40,814,"Bradford, Clarion, Erie, Punxsutawney, Williamsburg",
41,809,"Anguilla, Antigua, Bahamas, Barbados, Bequia, Bermuda, Cayman Islands,",
41,-1,"Dominica, Dominican Republic, Grenada, Jamaica, Montserrat, Mustique,",
41,-1,"Nevis, Palm Island, Puerto Rico, St. Kitts, St. Lucia, St. Vincent,",
41,-1,"Trinidad and Tobago, Turks and Caicos, Union Island, Virgin Islands",
42,401,0,
43,803,0,
44,605,0,
45,615,"Chattanooga, Johnson City, Knoxville, Nashville",
45,901,"Jackson, Memphis",

/* Texas */
46,210,"San Antonio",
46,214,"Dallas, Ennis, Greenville, Jefferson, Longview, Sherman",
46,409,"Bay City, Beaumont, Bryan, College Station, Galveston, Huntsville",
46,512,"Austin, Brownsville, Corpus Christi, Del Rio, Eagle Pass,",
46,-1,"Laredo, McAllen, Victoria",
46,713,"Houston, Baytown, Pasadena",
46,806,"Amarillo, Dalhart, Lubbock",
46,817,"Fort Worth, Denton, Temple, Waco, Wichita Falls",
46,903,"Texarkana, Tyler",
46,915,"Abilene, Alpine, Big Spring, El Paso, Midland, Odessa",

47,801,0,
48,802,0,
49,809,0,
50,804,"Charlottesville, Lynchburg, Newport News,",
50,-1,"Norfolk, Portsmouth, Richmond, Williamsburg",
50,703,"Alexandria, Arlington, Fairfax, Fredricksburg,",
50,-1,"Roanoke, Staunton, Vienna, Winchester",
51,206,"Seattle, Olympia, Tacoma, Vancouver",
51,509,"Spokane, Walla Walla, Yakima",
52,304,0,
53,414,"Appleton, Milwaukee, Green Bay, Racine",
53,608,"Beloit, La Crosse, Madison",
53,715,"Eau Claire, Wausau",
54,307,0,
55,800,0,

56,11,0,  /* International */
57,809,0, /* Bahamas */
58,809,0, /* Bermuda */
59,809,0, /* Jamaica */

/* CANADA */

60,403,0, /* Alberta */
61,204,0, /* Manitoba */
62,506,0, /* New Brunswick */
63,709,0, /* Newfoundland */
64,902,0, /* Nova Scotia */
65,604,0, /* NW Territory */
66,416,"Toronto", /* Ontario */
66,519,"London",
66,613,"Ottawa",
66,705,"North Bay",
66,807,"Fort William, Thunder Bay",
66,905,"Southern Ontario",
67,902,0, /* Prince Edward */
68,418,"Quebec City",
68,514,"Montreal",
68,819,"Sherbrooke",
69,306,0, /* Saskatchewan */
70,403,0, /* Yukon Territory */

99,0,0    /* sentinel entry */
};

#ifndef __TURBOC__
/*---------------------------------------------------------------------*
 * strlwr - converts a string to lower-case
 *
 * Returns: pointer to the converted string
 *---------------------------------------------------------------------*/
char* strlwr( char* s )
{
  if (s != 0)
  {
      register char *ps = s;
      while (*ps)
      {
	  if (isupper((unsigned char)*ps))
	    *ps = _tolower((unsigned char)*ps);
	  ps++;
	}
    }
  return s;
}
#endif /* ?!__TURBOC__ */

/*-----------------------------------------------------------------------*
 * stristr - Scans a string for the occurrence of a given string
 *           ignoring case
 *
 * Returns:  1 if str1 contains str2, otherwise 0.
 *-----------------------------------------------------------------------*/
int stristr( const char *str1, const char *str2 )
{
  register const char *s1, *s2;
  char c1, c2;

  if (!*str2) return 1;           /* return 1 if str2 empty */

  do
  {
    s1 = str1; c1 = *s1;
    s2 = str2; c2 = *s2;
    while (c1 && c2)
    {
	if (isupper(c1)) c1 = _tolower(c1);
	if (isupper(c2)) c2 = _tolower(c2);
	if (c1==c2)    /* str1[i] == str2[i] ? */
	{
	    c1 = *(++s1);  c2 = *(++s2);
	}
	else break;
     }
     if (!*s2) return 1;    /* target parsed - match found */
     ++str1;    /* try next position of str1 */
  } while (*str1);
  return 0;
}

/*-----------------------------------------------------------------------*
 * hasWildCard: Checks if wildcard characters are present in string s
 *
 * Returns:     1 if '*' or '?' are found, 0 otherwise.
 *-----------------------------------------------------------------------*/
int hasWildCard( const char *s )
{
    while (*s)
    {
	if (*s == '*' || *s == '?') return 1;
	s++;
    }
    return 0;
}

/*-----------------------------------------------------------------------*
 * match:       match string s with wildcard pattern
 *
 * Returns:     1 if the pattern matches the string
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int match (const char *target, const char *pattern)
{
    register char symbol;
    register const char *s = target;
    const char *old_pat = 0;  /* address of last wildcard pattern */
    const char *old_target;   /* address of target string */

  /*  Parse each string character by character until
   *  a mismatch is found or the end of one is reached.
   */
 matchRetry:
    while (*pattern != 0)
    {
    switch (*pattern) {
	   case '?':
		goto matchNext; /* match any symbol, except NULL */

	   case '*':
		old_pat = pattern++;
		while (*pattern != 0)
		{
		  switch (*pattern) {
		   case '?':
			if (!*s) return 0; /* no match? */
			s++;  pattern++;
			continue;
		   case '*': /* handle multiple *'s */
			pattern++;
			continue;
		  } /* switch */
		  break;
		} /* while */
		/*  If the end of the expression was parsed
		 *  and the last character was a wildcard (*) then
		 *  the rest of the string is accepted.
		 */
		if (!*pattern) return 1;
		for (symbol = ToLower(*pattern);; s++)
		{
		    if (!*s) return 0; /* no match */
		    if (ToLower(*s) == symbol)
		    {
			old_target = s + 1;
			break;
		    }
		}
	goto matchNext2;

	   default:
		symbol = ToLower(*pattern);
	} /* switch */

	if (ToLower(*s) != symbol) break; /* mismatch? */

  matchNext:
	if (!*s) break;
  matchNext2:
	s++;  pattern++;   /* increment strings */
	symbol = ToLower(*pattern);
    } /* while */

    /* check if match failed */
    if (ToLower(*s) != symbol && old_pat && *old_target != 0)
    {
	/*  check if a wildcard was previously parsed, if so back up
	 *  to that pattern and try again.
	 */
	pattern = old_pat; /* reset pattern */
	s = old_target++;  /* reset target to last position */
	goto matchRetry;
    }

  /*    Check if the last two characters match:
   *    either strings must be NULL to end the loop,
   *    so they both must equal NULL for a match.
   */
    return (*pattern == *s);
}

/*-----------------------------------------------------------------------*
 * Printcode:   Print the entry for the specified areacode
 *
 *              Keep track of the previous zone and
 *              only print the state information when this changes.
 *-----------------------------------------------------------------------*/
void PrintCode (int code)
{
    int newzone = areacodes[code].zcode;

    /*  check if entry is continued from the previous one
     *  (identified by the -1 areacode) and already printed.
     */
    if (areacodes[code].areaCode < 0) return;
    if (newzone != zone)
    {
	zone = newzone;
	printf("\n%s  %s area code(s):\n",
		areazones[zone].code,
		areazones[zone].stateName);
    }
    printf("    %03d %s\n", areacodes[code].areaCode,
	areacodes[code].list ? areacodes[code].list : "All locations.");

    /* Check if areacode list continues on next entry */
    while (areacodes[++code].areaCode == -1)
	printf("        %s\n", areacodes[code].list);
}

/*-----------------------------------------------------------------------*
 * FindAbbrev:  Find postal code in list of states matching the string s
 *
 * returns:     1 if match found.
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int FindAbbrev (const char *s)
{
    int  i;
    char ch1, ch2;      /* target characters in uppercase */
    ch1 = ToUpper(*s);
    ch2 = ToUpper(*(s+1));
    for (i=0; i < N_ZONES; i++)
    {
	if (ch1==areazones[i].code[0] && ch2==areazones[i].code[1])
	{
		int code = 0;
	    /*  Locate areacode in list and
	     *  print all entries for that state.
	     */
		while (areacodes[code].zcode < i) code++;
		while (areacodes[code].zcode==i)
		{
			PrintCode(code++);
		}
		return 1;
	}
    }
    return 0;
}

/*-----------------------------------------------------------------------*
 * FindCode:    Find areacode in list
 *
 * returns:     1 if areacode is found.
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int FindCode (const char *s)
{
    int i;
    int code = atoi(s);
    if (code == 0) return 0; /* invalid code */

    for (i=0; i < N_CODES; i++)
    {
	if (areacodes[i].areaCode == code)
	{
		PrintCode(i);
		return 1;
	}
    }
    return 0;
}

/*-----------------------------------------------------------------------*
 * FindString:  Find all states and/or cities matching the pattern s
 *
 * Returns:     1 if string is found.
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int FindString (const char *s)
{
    int i;
    int rc = 0;  /* return code */
    int code, pos;
    char target[30];
    int wildcards = hasWildCard(s);
    int (*cmp)(const char*, const char*) = wildcards ? match : stristr;

    if (*s == '*' && !*(s+1))   /* show all codes and return */
    {
	for (code=0; code < N_CODES; code++)
		PrintCode(code);
	return 1;
    }

    strncpy(target, s, sizeof(target)); /* copy target string */
    strlwr(target);                     /* convert to lower case */

    for (code=0, i=0; i < N_ZONES; i++)
    {
	if (cmp(areazones[i].stateName, target))
	{
	    /* print all codes for found zone */
	    while (areacodes[code].zcode==i)
	    {
		PrintCode(code++);
	    }
	    rc = 1;
	}
	else /* try city search */
	{
	    while (areacodes[code].list != 0 && areacodes[code].zcode == i)
	    {
		if (cmp(areacodes[code].list, target))
		{
		    pos = code;
		    while (areacodes[pos].areaCode < 0) pos--;
		    PrintCode(pos);
		    rc = 1;
		}
		code++;
	    }
	    /* set zone position */
	    while (areacodes[code].zcode <= i) code++;
	}
    } /* for */
    return rc;
}


int main (int argc, char **argv)
{
    int i;

    printf(copyright);
    if (argc==1)
    {
	printf("\nProgram searches for telephone area codes,\n");
	printf("   as area xxx xxx xxx etc.\n");
	printf("   xxx is an Area Code, State name, or City.\n\n");
	printf("   Two letter state/postal codes like TX for Texas, CA for California\n");
	printf("      can be used, otherwise type in the state or city name.\n");
	printf("   Enter wildcard expressions such as: New* or *Green?b*\n");
	printf("   Enter substrings like: cisco or east\n");
	printf("   Enter multiple words in between quotes like: \"Los Angeles\" or \"New York\"\n");
	printf("   Enter area * for a list of all Area Codes.\n");
	EXIT(0);
    }

    for (i=1; i < argc; i++)
    {
	zone = -1;      /* reset zone number */
	if (hasWildCard(argv[i])) goto others;
	switch (strlen(argv[i])) {
	 case 2: /* find state zone: CA, MD, NY, ...
		  * if there's no match then try the
		  * general string search.
		  */
		if (!FindAbbrev(argv[i])) goto others;
		break;

	 case 3: /* find area code: 301, 718, ... */
		if (isdigit(*argv[i]) && FindCode(argv[i]))
			break;
		/* otherwise try general search */

	 default: /* state name: new*york, maryland, ... */
	 others:  /* or city name: Greenbelt, Fresno, ... */
		if (!FindString(argv[i]))
		    printf("\nNot found, %s.\n", argv[i]);
     } /* switch */
    } /* for each argument */

    EXIT(0);
}
