
/* ------------------------------------------------------------
 * Filename ............... Opinion.Cpp
 *
 * General Purpose ........ Opinion PEX
 * ------------------------------------------------------------
 * First date ............. 31/05/95 16:03
 *
 * First in version ....... 2.00
 *
 * Written by ............. Alain Schellinck
 *
 * Copyright .............. Nephilim software development
 * ------------------------------------------------------------
 * Revisions:
 * ----------
 *
 *   Date   | By |                  Purpose                   |
 * ---------+----+--------------------------------------------+
 * 10-06-95 | AS | Definable prompt in INI file (Kitt Brown)  |
 *          |    |                                            |
 *          |    |                                            |
 */

/*-------------------------------------------------------------------------*/

#include "opinion.hpp"
#pragma hdrstop

/*-------------------------------------------------------------------------*/

struct
{
   ChrP        name;
   ChrT        value;
}  colorList[16] =
{
   {  "Black"     , 0x00   }  ,  {  "+Black"    , 0x10   }  ,
   {  "Red"       , 0x01   }  ,  {  "+Red"      , 0x11   }  ,
   {  "Green"     , 0x02   }  ,  {  "+Green"    , 0x12   }  ,
   {  "Yellow"    , 0x03   }  ,  {  "+Yellow"   , 0x13   }  ,
   {  "Blue"      , 0x04   }  ,  {  "+Blue"     , 0x14   }  ,
   {  "Magenta"   , 0x05   }  ,  {  "+Magenta"  , 0x15   }  ,
   {  "Cyan"      , 0x06   }  ,  {  "+Cyan"     , 0x16   }  ,
   {  "White"     , 0x07   }  ,  {  "+White"    , 0x17   }  ,
};

/*-------------------------------------------------------------------------*/

struct
{
   ChrT     defUsrClr;
   ChrT     defSayClr;
   ChrT     lstUsrClr;
   ChrT     lstSayClr;
   FlgT     waitEnter;
   FlgT     clrScreen;
   FlgT     hideCopyright;
   ChrT     askText[80];
} setup;

/*-------------------------------------------------------------------------*/

#define MaxLines  128

/*-------------------------------------------------------------------------*/

LCL ChrT opinionPath [128     ];
LCL ChrP dataLine    [MaxLines];
LCL IntT linesUsed;

/*--] Code [---------------------------------------------------------------*/

/*
 * Routine   : centerLine()
 * Purpose   : center the specified line on the screen
 * ------------------------------------------------------------------------
 * Parameters: formatted string
 * Return    : None
 *
 */

LCL void centerLine(ChrP fs, ...)
{
   ChrT buffer[128];
   va_list v;

   va_start(v, fs);
   vsprintf(buffer, fs, v);
   va_end(v);

   IntT realLen = strlen(buffer);
   IntT textLen = 0;
   for(IntT cnt = 0; cnt < realLen; cnt++)
   {
      if(BytT(buffer[cnt]) >= 32)
         textLen++;
   }

   IntT leadingSpace = ((80 - textLen) / 2);
   printf("%*.*s%s", leadingSpace, leadingSpace, "", buffer);
}

/*
 * Routine   : getDate()
 * Purpose   : get the date
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : date in the format DDD DD MMM CCYY
 *
 */

LCL ChrP getDate()
{
   LCL ChrT buffer[32];

   ChrP days  [ 7] =
   {
      "Sunday"    ,  "Monday"    ,  "Tuesday"   ,
      "Wednesday" ,  "Thursday"  ,  "Friday"    ,
      "Saturday"

   };

   ChrP months[12] =
   {
      "January"   ,  "February"  ,  "March"     ,
      "April"     ,  "May"       ,  "June"      ,
      "July"      ,  "August"    ,  "September" ,
      "October"   ,  "November"  ,  "December"
   };

   time_t timer       = time(NULL);
   struct tm * tblock = localtime(&timer);

   sprintf(buffer,"%s %u %s %u",
                     days[tblock->tm_wday],
                     tblock->tm_mday,
                     months[tblock->tm_mon],
                     tblock->tm_year + 1900
          );

   return(buffer);
}

/*
 * Routine   : getColor()
 * Purpose   : scan the colorList array for the color
 * ------------------------------------------------------------------------
 * Parameters: color string
 * Return    : color value (-1 if not found)
 *
 */

LCL ChrT getColor(const ChrP colorStr, ChrT defAnswer)
{
   ChrT ret_val = -1;

   for(IntT cnt = 0; cnt < 16 && ret_val == -1; cnt++)
      if(stricmp(colorList[cnt].name, colorStr) == 0)
         ret_val = colorList[cnt].value;

   if(ret_val == -1)
      ret_val = defAnswer;

   return(ret_val);
}

/*
 * Routine   : getYesNo()
 * Purpose   : check if the ini variable is YES or NO
 * ------------------------------------------------------------------------
 * Parameters: INI variable, default answer
 * Return    : True if YES, False if NO, default answer if not yes or no
 *
 */

LCL FlgT getYesNo(const ChrP iniVar, FlgT defAnswer)
{
   FlgT ret_val = defAnswer;

   if(stricmp(iniVar, "YES") == 0)
      ret_val = True;
   else
      if(stricmp(iniVar, "NO") == 0)
         ret_val = False;

   return(ret_val);
}

/*
 * Routine   : init()
 * Purpose   : set the default values
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : None
 *
 */

LCL void init()
{
   setup.defUsrClr     = getColor("White"  , -1);
   setup.defSayClr     = getColor("Cyan"   , -1);
   setup.lstUsrClr     = getColor("Yellow" , -1);
   setup.lstSayClr     = getColor("Magenta", -1);
   setup.waitEnter     = True;
   setup.clrScreen     = True;
   setup.hideCopyright = False;
   strcpy(setup.askText, "Do you want to add you own opinion?");
}

/*
 * Routine   : readIni()
 * Purpose   : read the INI file
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : None
 *
 */

LCL void readIni()
{
   ChrT iniFile[128];
   sprintf(iniFile,"%s.%s", opinionPath, "INI");

   ChrT buffer[81];

   if(GetIniVar(iniFile, "WaitForEnter"    , buffer, 15) == True)
      setup.waitEnter      = getYesNo(buffer, setup.waitEnter );

   if(GetIniVar(iniFile, "ClearScreen"     , buffer, 15) == True)
      setup.clrScreen      = getYesNo(buffer, setup.clrScreen );

   if(GetIniVar(iniFile, "HideCopyright"   , buffer, 15) == True)
      setup.hideCopyright  = getYesNo(buffer, setup.clrScreen );

   if(GetIniVar(iniFile, "DefUserColor"    , buffer, 15) == True)
      setup.defUsrClr      = getColor(buffer, setup.defUsrClr );

   if(GetIniVar(iniFile, "DefCommentColor" , buffer, 15) == True)
      setup.defSayClr      = getColor(buffer, setup.defSayClr );

   if(GetIniVar(iniFile, "LastUserColor"   , buffer, 15) == True)
      setup.lstUsrClr      = getColor(buffer, setup.lstUsrClr );

   if(GetIniVar(iniFile, "LastCommentColor", buffer, 15) == True)
      setup.lstSayClr      = getColor(buffer, setup.lstSayClr );

   if(GetIniVar(iniFile, "AskText"         , buffer, 81) == True)
      strcpy(setup.askText, buffer);
}

/*
 * Routine   : readOpinions()
 * Purpose   : read the OPINIONS.DAT file
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : None
 *
 */

LCL void readOpinions()
{
   ChrT datFile[128];
   sprintf(datFile,"%s.%s", opinionPath, "DAT");

   linesUsed = 0;

   FILE *handle = fopen(datFile, "rt");
   if(handle)
   {
      while(!feof(handle) && linesUsed < MaxLines)
      {
         ChrT buffer[128];

         if(fgets(buffer, 127, handle) != NULL)
         {
            IntT lastPos = strlen(buffer) - 1;
            if(buffer[lastPos] == '\n')
               buffer[lastPos] = '\0';

            dataLine[linesUsed++] = strdup(buffer);
         }
      }

      fclose(handle);
   }
}

/*
 * Routine   : writeOpinions()
 * Purpose   : write the OPINIONS.DAT file
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : None
 *
 */

LCL void writeOpinions()
{
   ChrT datFile[128];
   sprintf(datFile,"%s.%s", opinionPath, "DAT");

   FILE *handle = fopen(datFile, "wt");
   if(handle)
   {
      for(IntT cnt = 0; cnt < linesUsed; cnt += 2)
         fprintf(handle, "%s\n%s\n",
                              dataLine[cnt    ],
                              dataLine[cnt + 1]
                );

      fclose(handle);
   }
}

/*
 * Routine   : addOpinion()
 * Purpose   : add an opinion line
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : True if an opinion is added, False if NOT
 *
 */

LCL FlgT addOpinion()
{
   FlgT ret_val = False;

   printf("\n\6%s ", setup.askText);
   FlgT status = Ask(False);
   printf("\n");
   if(status == True)
   {
      ChrT buffer[80];
      Input(buffer, 77, INPUT_ALL);
      if(strlen(buffer) > 3)
      {
         if(linesUsed + 2 >= MaxLines)
         {
            free(dataLine[0]);
            free(dataLine[1]);

            for(IntT cnt = 2; cnt < MaxLines; cnt)
               dataLine[cnt - 2] = dataLine[cnt];

            linesUsed -= 2;
         }

         dataLine[linesUsed++] = strdup(form("%s says on %s",
                                                CurUser->name,
                                                getDate()
                                            )
                                       );

         dataLine[linesUsed++] = strdup(buffer);

         ret_val = True;
      }
   }

   return(ret_val);
}

/*
 * Routine   : showOpinions()
 * Purpose   : display the opinions
 * ------------------------------------------------------------------------
 * Parameters: None
 * Return    : None
 *
 */

LCL void showOpinions()
{
   if(linesUsed >= 2)
   {
      if(setup.clrScreen)
         printf("\f");

      printf("\n");
      if(setup.hideCopyright == False)
      {
         centerLine("%c%s %cversion %c%s.%c (C) %s %s",
                        BLUE , projectName,
                        WHITE,
                        BLUE , versionText,
                        WHITE,
                        copyrightYear,
                        companyName
                   );

         printf("\n");

         centerLine("%c", WHITE);

         printf("\n\n");
      }

      IntT maxOnScreen = (CurUser->screenLength - 3);
      if(setup.hideCopyright == False)    // take 3 lines for the (C)
         maxOnScreen -= 3;

      IntT start = max(0, linesUsed - maxOnScreen);
      if((start % 2) != 0)
         start++;

      for(IntT cnt = start; cnt < linesUsed; cnt += 2)
      {
         if((cnt + 2) >= linesUsed)
         {
            printf("%c%s\n\1 %c%s\n",
                     setup.lstUsrClr, dataLine[cnt    ],
                     setup.lstSayClr, dataLine[cnt + 1]
                  );

         }
         else
         {
            printf("%c%s\n\1 %c%s\n",
                     setup.defUsrClr, dataLine[cnt    ],
                     setup.defSayClr, dataLine[cnt + 1]
                  );
         }
      }
   }
}

/*
 * Routine   : main()
 * Purpose   : the main function
 * ------------------------------------------------------------------------
 * Parameters: command line parameters
 * Return    : None
 *
 */

void main(IntT argc, ChrP argv[])
{
   // find out from where OPINION is started.
   // we can't use Config->pexpath, because ProBoard can now start pex'es
   // from another directory...
   strcpy(opinionPath, argv[0]);

   // strip the extension, so we keep <path>\opinion in this variable
   IntT length = strlen(opinionPath);
   IntT cnt;
   for(cnt = length - 1; opinionPath[cnt] != '.'; cnt--);
   opinionPath[cnt + 1] = '\0';

   init();
   readIni();

   for(cnt = 0; cnt < MaxLines; cnt++)
      dataLine[cnt] = NULL;

   readOpinions();

   for(cnt = 0; cnt < argc; cnt++)
   {
      if(stricmp(argv[cnt + 1], "SHOW") == 0)
      {
         showOpinions();
      }
      else
      {
         if(stricmp(argv[cnt + 1], "ADD") == 0)
         {
            if(addOpinion() == True)
               writeOpinions();

            setup.waitEnter = False;
         }
      }
   }

   if(setup.waitEnter)
   {
      printf("\n");
      centerLine("\7Press [\6Enter\7] to continue.");
      WaitKeys("\r");
   }

   for(cnt = 0; cnt < MaxLines; cnt++)
      if(dataLine[cnt] != NULL)
         free(dataLine[cnt]);
}

/*---------------------------] END OF THE CODE [---------------------------*/
