{***********************************************************************

FISHTANK.PAS -- This program demonstrates multi-object non-destructive
animation. The coral background is displayed on page 2 and copied to
page 0, the visual page. A packed pixel run file containing the 6 fish
is displayed on page 1, and then fg_getimage is used to load the fish
into the fish bitmaps.

To make the fish move, the background is copied to page 1 and the fish
are put over the background using fg_clpimage and fg_flpimage. The
fish are clipped at the edge of the screen. Page 1 is then copied to
page 0 using fg_copypage. This process is repeated in a loop.

To compile this program and link it with Fastgraph or Fastgraph/Light
4.0:

   TPC FISHTANK

For more examples of animation using Fastgraph, or for an evaluation
copy of Fastgraph/Light, call DDBBS at (702) 796-7134. For Fastgraph
voice support, call Ted Gruber Software at (702) 735-1980.

***********************************************************************}

program fishtank;
uses fgmain, fgbitmap, fgmisc, fgpr;

var

  old_mode : integer;
  status   : integer;

  { fish bitmaps }

  fish1 : array[0..699]  of byte;
  fish2 : array[0..1025] of byte;
  fish3 : array[0..883]  of byte;
  fish4 : array[0..839]  of byte;
  fish5 : array[0..681]  of byte;
  fish6 : array[0..1223] of byte;

const

  { palette values }

  colors : array[0..15] of integer = (0,1,2,3,4,5,6,7,16,0,18,19,20,21,22,23);

  { number of fish }

  NFISH = 11;

  { location of fish x & y }

  fish_x1 : array[0..5] of integer = (0,   64,128,200,  0, 80);
  fish_y1 : array[0..5] of integer = (199,199,199,199,150,150);

  { size of fish (width and height) }

  width   : array[0..5] of integer = ( 28, 27, 34, 28, 31, 34);
  height  : array[0..5] of integer = ( 25, 38, 26, 30, 22, 36);

  { for convenience, an array of pointers to fish bitmaps }

  fishes : array[0..5] of ^byte = (@fish1,@fish2,@fish3,@fish4,@fish5,@fish6);

{
************************************************************************
*                                                                      *
*            min -- determine the smaller of two integer values        *
*                                                                      *
************************************************************************
}

function min (value1, value2 : integer) : integer;

begin

  if (value1 <= value2) then
    min := value1
  else
    min := value2;

end;

{
************************************************************************
*                                                                      *
*            max -- determine the larger of two integer values         *
*                                                                      *
************************************************************************
}

function max (value1, value2 : integer) : integer;

begin

  if (value1 >= value2) then
    max := value1
  else
    max := value2;

end;

{
************************************************************************
*                                                                      *
*            get_fish -- fill up the fish bitmap arrays                *
*                                                                      *
************************************************************************
}

procedure get_fish;

var

  fish_num : integer;

begin

  { read the fish from a packed pixel run file and display them on page 1 }

  fg_setpage(1);
  fg_move(0,199);
  status := fg_showppr('FISH.PPR'+chr(0),320);

  { build the fish bitmaps }

  for fish_num := 0 to 5 do
  begin
    fg_move(fish_x1[fish_num],fish_y1[fish_num]);
    fg_getimage(fishes[fish_num]^,width[fish_num],height[fish_num]);
  end;

end;

{
************************************************************************
*                                                                      *
*      put_fish -- draw one of the six fish anywhere you want          *
*                                                                      *
************************************************************************
}

procedure put_fish (fish_num, x, y, fish_dir : integer);

begin

  { move to position where the fish will appear }

  fg_move(x,y);

  { draw a left- or right-facing fish, depending on fish_dir }

  if (fish_dir = 0) then
    fg_flpimage(fishes[fish_num]^,width[fish_num],height[fish_num])
  else
    fg_clpimage(fishes[fish_num]^,width[fish_num],height[fish_num]);

end;

{
************************************************************************
*                                                                      *
*             go_fish -- make the fish swim around                     *
*                                                                      *
************************************************************************
}

procedure go_fish;

type

  fish_array = array[0..NFISH-1] of integer;

const
{
*   There are 11 fish total, and 6 different kinds of fish. These
*   arrays keep track of what kind of fish each fish is, and how each
*   fish moves:
*
*   fish()   -- which fish bitmap applies to this fish?
*   x()      -- starting x coordinate
*   y()      -- starting y coordinate
*
*   xmin()   -- how far left (off screen) the fish can go
*   xmax()   -- how far right (off screen) the fish can go
*   xinc()   -- how fast the fish goes left and right
*   dir()    -- starting direction for each fish
*
*   ymin()   -- how far up this fish can go
*   ymax()   -- how far down this fish can go
*   yinc()   -- how fast the fish moves up or down
*   yturn()  -- how long fish can go in the vertical direction
*               before stopping or turning around
*   ycount() -- counter to compare to yturn
}

  fish   : fish_array =
                (   1,   1,   2,   3,   3,   0,   0,   5,   4,   2,   3);
  x      : fish_array =
                (-100,-150,-450,-140,-200, 520, 620,-800, 800, 800,-300);
  y      : fish_array =
                (  40,  60, 150,  80,  70, 190, 180, 100,  30, 130,  92);

  xmin   : fish_array =
                (-300,-300,-800,-200,-200,-200,-300,-900,-900,-900,-400);
  xmax   : fish_array =
                ( 600, 600,1100,1000,1000, 750, 800,1200,1400,1200, 900);
  xinc   : fish_array =
                (   2,   2,   8,   5,   5,  -3,  -3,   7,  -8,  -9,   6);
  dir    : fish_array =
                (   0,   0,   0,   0,   0,   1,   1,   0,   1,   1,   0);

  ymin   : fish_array =
                (  40,  60, 120,  70,  60, 160, 160,  80,  30, 110,  72);
  ymax   : fish_array =
                (  80, 100, 170, 110, 100, 199, 199, 120,  70, 150, 122);
  yturn  : fish_array =
                (  50,  30,  10,  30,  20,  10,  10,  10,  30,   20, 10);
  ycount : fish_array =
                (   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0);
  yinc   : fish_array =
                (   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0);

var

  i : integer;
  key, aux : byte;

begin

  { make the fish swim around }

  repeat
  begin

    { copy the background from page 2 to page 1 }

    fg_copypage(2,1);

    { put all the fish on the background }

    for i := 0 to NFISH-1 do
    begin
      fg_setpage(1);
      inc(ycount[i]);
      if (ycount[i] > yturn[i]) then
      begin
         ycount[i] := 0;
         yinc[i] := random(3) - 1;
      end;
      inc(y[i],yinc[i]);
      y[i] := min(ymax[i],max(y[i],ymin[i]));

      if (x[i] >= 0) and (x[i] < 320) then
        put_fish(fish[i],x[i],y[i],dir[i])
      else if (x[i] < 0) and (x[i] > -72) then
      begin
        fg_transfer(0,71,0,199,104,199,1,3);
        fg_setpage(3);
        put_fish(fish[i],x[i]+104,y[i],dir[i]);
        fg_transfer(104,175,0,199,0,199,3,1);
      end;
      inc(x[i],xinc[i]);
      if (x[i] <= xmin[i]) or (x[i] >= xmax[i]) then
      begin
        xinc[i] := -xinc[i];
        dir[i] := 1 - dir[i];
      end;
    end;

    { copy page 1 to page 0 }

    fg_setpage(0);
    fg_copypage(1,0);

    { intercept a keystroke, if it is escape exit the program }

    fg_intkey(key,aux);

  end;
  until (key = 27);

end;

{
************************************************************************
*                                                                      *
*                                 main                                 *
*                                                                      *
************************************************************************
}

begin

  { in case we're compiling for protected mode }

  fg_initpm;

  { make sure the system supports video mode 13 with 4 pages }

  if (fg_testmode(13,4) = 0) then
  begin
    writeln;
    writeln('This program requires an EGA or VGA card');
    writeln('with at least 128k. If an EGA card is');
    writeln('present, it must be the active adapter.');
    halt;
  end;

  { initialize the video environment }

  old_mode := fg_getmode;
  fg_setmode(13);
  fg_palettes(colors);
  randomize;

  { get the coral background from a file and put it on page 2 }

  fg_setpage(2);
  fg_move(0,199);
  status := fg_showppr('CORAL.PPR'+chr(0),320);

  { copy the background from page 2 to page 0, the visual page }

  fg_copypage(2,0);

  { get the fish }

  get_fish;

  { make the fish go }

  go_fish;

  { restore the original video state }

  fg_setmode(old_mode);
  fg_reset;

end.
