/*
 * Copyright (C) 1992, 1995 Stephen A. Wood.  All rights reserved.
 * 
 * This file is part of PAF C-Tools.
 * 
 * These programs are free software; you can redistribute them and/or modify
 * them under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * These programs are distributed in the hope that they will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA. The GNU General Public License can be
 * found in the file LICENSE.
 * 
 * The author of PAF C-Tools is Stephen A. Wood.  He may be reached on internet
 * at the address saw@cebaf.gov, or via mail at 328 Dominion Drive, Newport
 * News, VA 23602.
 * 
 * ------------
 * 
 * Update notes file with standard input.  The indiv2.dat and notes2.dat files
 * are modified by this program.  These files are backup to indiv2.bNN and
 * notes2.bNN.
 */

#include <stdio.h>
#include <string.h>
#include "pafsubs.h"

void            backup_file(enum paffiles filetype);
void            printnotes(RECORD_PTR rin);

main(int argc, char *argv[])
{
  PAF_FILE       *indiv_file, *notes_file;
  int             anum;
  short int       freelist[128];
  int             i;
  int verbose=FALSE;
  int bad_syntax=FALSE;
  int nobackup=FALSE;
  char *path="";
  int lineno;
  
  while(argc-- > 1 && !bad_syntax){
    argv++;
    if(argv[0][0] != '-'){
      path = *argv;
    } else
      switch(argv[0][1]){
      case 'v': case 'V':
	verbose = TRUE;
	break;
      case 'n': case 'N':   
	nobackup = TRUE;
	break;
      case 0:
      case 'h': case 'H':
      default:
	bad_syntax = TRUE;
	break;
      }
  }
  if(bad_syntax) {
    fputs("usage: nupdate [input_file] [-v]\n",stderr);
    fputs("  updates PAF notes from input_file or standard input\n",stderr);
    fputs("  -v verbose mode, displays progress\n",stderr);
    fputs("  -n don't make backups of note and indiv files\n",stderr);
    exit(1);
  }
      
  if(!nobackup) {
    backup_file(NOTE);
    backup_file(INDIV);
  }
  
  if(paf_open_indiv(path,'w') || paf_open_notes(path,'w')) {
    fprintf(stderr,"Failed to open PAF database in directory %s\n",path);
    exit(1);
  }
  notes_file = get_paf_filed(NOTE);
  indiv_file = get_paf_filed(INDIV);
  
  printf("\n");
  
  lineno = 0;
  {
    char            s[512];
    char           *not_eof;

    not_eof = gets(&s[1]);
    lineno++;
    while (not_eof) {
      int             newnote;
      int             len;
      int             c;
      int             stop, xstop;
      INDIV_REC       indiv;
      NOTE_REC        notes;
      int             padctr;
      RECORD_PTR      rin;
      RECORD_PTR      thispad, nextpad;

      /* Skip over preamble or junk between entries */
      while (not_eof && strncmp(&s[1], "%%", 2)!=0) {
	if(verbose && not_eof) fprintf(stderr,"Skipping line #%d\n",lineno);
	not_eof = gets(&s[1]);
	lineno++;
      }
      if(!not_eof) break;       /* Break if end reached */
      {
	char           *p;
	p = strchr(&s[1], '[');
	rin = atoi(++p);
      }
      if(verbose) fprintf(stderr,"Processing RIN %d\n",rin);
      not_eof = gets(&s[1]);
      lineno++;
      newnote = 1;
      stop = strncmp(&s[1], "%%--", 4);
      xstop = 1;
      len = strlen(&s[1]);
      newnote = 1;
      get_indiv_rec(rin, &indiv);
      if (stop) {
	thispad = indiv.notepad;
	nextpad = get_next(thispad, notes_file);
	/* printf("Pad=%d, Next=%d\n",thispad,nextpad); */
	if (thispad == 0) {
	  if(verbose) fprintf(stderr,"RIN %5d: Adding new notes\n", rin);
	  thispad = get_free_rec(notes_file);
	  nextpad = 0;
	  indiv.notepad = thispad;
	  write_indiv_rec(rin, &indiv);
	}
	padctr = 0;
	while (not_eof && xstop) {
	  if (len == 0 && stop) {
	    newnote = 1;
	  } else {
	    s[0] = newnote;
	    newnote = 0;
	    i = 0;
	    len += 2;
	    while (i < len) {
	      if (padctr < NOTEPADSIZE) {
		notes.notelines[padctr++] = s[i];
	      } else {
		padctr = 0;
		if (nextpad == 0)
		  nextpad = get_free_rec(notes_file);
		notes.next_pad = nextpad;
		/*
		 * printf("Writing %d with
		 * nextpad=%d\n",thispad,nextpad);
		 */
		write_note_rec(thispad, &notes);
		thispad = nextpad;
		nextpad = get_next(thispad, notes_file);
		notes.notelines[padctr++] = s[i];
	      }
	      /*
	       * if(s[i]>1) putchar(s[i]); else
	       * printf("%d",s[i]);
	       */
	      i++;
	    }
	    /* printf("\n"); */
	  }
	  xstop = stop;
	  not_eof = gets(&s[1]);
	  lineno++;
	  stop = strncmp(&s[1], "%%--", 4);
	  if (!stop) {
	    newnote = 0;
	    s[1] = 0;
	    len = 0;
	  } else {
	    len = strlen(&s[1]);
	  }
	}
	if (padctr > 0) {
	  if (nextpad != 0) {
	    if(verbose) fprintf(stderr,"RIN %5d: Removing some note pads\n", rin);
	    add_to_freelist(nextpad, notes_file);
	  }
	  notes.next_pad = 0;
	  /* printf("Writing pad %d\n",thispad); */
	  while(padctr < NOTEPADSIZE) { /* Fill rest of pad with nulls */
	    notes.notelines[padctr++] = '\0'; /* as some utilities depend on this */
	  }
	  write_note_rec(thispad, &notes);
	}
      } else {
	if (indiv.notepad == 0) {
	  if(verbose) fprintf(stderr,"RIN %5d: Notes remain empty\n", rin);
	} else {
	  add_to_freelist(indiv.notepad, notes_file);
	  if(verbose) fprintf(stderr,"RIN %5d: Removing notes\n", rin);
	  indiv.notepad = 0;
	  write_indiv_rec(rin, &indiv);
	}
	not_eof = gets(&s[1]);
	lineno++;
      }
    }
  }
  
  fclose(notes_file->ptr);
  fclose(indiv_file->ptr);
  exit(0);
}

void 
backup_file(enum paffiles filetype)
{
  char           *pos;
  FILE           *backup;
  char            command[100];
  char            backup_fname[30];
  PAF_FILE       *orig;
  int             i;

  orig = get_paf_filed(filetype);
  strcpy(backup_fname, orig->fname);
  pos = (char *) strrchr(backup_fname, '.');
  if (pos == NULL) {
    printf("nupdate: No . found in %s\n", backup_fname);
    exit(1);
  } else {
    pos++;
    strcpy(pos, "b00");
    pos++;
    i = 0;
    while ((backup = fopen(backup_fname, "rb")) != NULL) {
      fclose(backup);
      i++;
      sprintf(pos, "%2.2i", i);
    }
#ifdef unix
    strcpy(command, "cp ");
#else
    strcpy(command, "copy ");
#endif
    strcat(command, orig->fname);
    strcat(command, " ");
    strcat(command, backup_fname);
    printf("Backing up %s to %s\n", orig->fname, backup_fname);
    system(command);
  }
}
