/*
 * Copyright (C) 1992, 1995 Stephen A. Wood.  All rights reserved.
 * 
 * This file is part of PAF C-Tools.
 * 
 * These programs are free software; you can redistribute them and/or modify
 * them under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * The GNU General Public License can be found in the file LICENSE.
 * 
 * The author of PAF C-Tools is Stephen A. Wood.  He may be reached on internet
 * at the address saw@cegaf.gov, or via mail at 328 Dominion Drive,
 * Newport News, VA 23602.
 * 
 * ------------
 * 
 * This program finds all individuals that have no death date and were born
 * after 1900.  If the birthday is given, then a line is printed with the
 * persons birthday followed by their name.  After the name, the year of birth
 * is put in parenthesis.  If the person is female, the marriage record is
 * looked up.  If there is a husband found, his last name is used instead of
 * the womans last name.  Her last name is placed in brackets after the
 * husbands last name.  If the woman has several marriages in the database, the
 * marriage pointed to by her individual record is the one used.  This may or
 * may not be the latest marriage.
 */

#include <stdio.h>
#include <time.h>

#include "pafsubs.h"

void printrec(RECORD_PTR rin);

int sortflag = FALSE;
int sinceyear = 1900;
int bad_syntax = 0;

main(argc,argv)
int argc;
char *argv[];
{
  RECORD_PTR rin, rin_max;
  PAF_FILE *indiv_file;
  char *path="";

  while(argc-- > 1 && !bad_syntax){
    argv++;
    if(argv[0][0] != '-'){
      path = *argv;
    } else
      switch(argv[0][1]){
      case 's': case 'S':
	sortflag = TRUE;
	break;
      case 'y': case 'Y':
	sinceyear = atol(argv[0][2]);
	break;
      case 0:
      case 'h': case 'H':
      default:
	bad_syntax = TRUE;
	break;
      }
  }	

  if(bad_syntax) {
    fputs("usage: alive [path] [-yNNNN]\n",stderr);
    fputs("  Prints birthdays of every in the PAF database who is alive\n",stderr);
    fputs("\n  -yNNNN for people not dead, only lists thost born after NNNN\n",stderr);
    fputs("  [path] specifies what directory to find the PAF database\n",stderr);
    exit(1);
  }

  if(paf_open_name(path,'r') || paf_open_indiv(path,'r') || paf_open_marr(path,'r')) {
    fprintf(stderr,"Failed to open PAF database in directory %s\n",path);
    exit(1);
  }

  indiv_file = get_paf_filed(INDIV);
  rin_max = indiv_file->nrec;

  for(rin=1;rin<=rin_max;rin++){
    printrec(rin);
  }
  return(0);
}

void printrec(RECORD_PTR rin)
{
  INDIV_REC indiv, husband;
  NAME_REC namrec;
  MARR_REC marriage;
  int i;
  RECORD_PTR j;
  int year,month,day;
  int used_married_name;

  get_indiv_rec(rin,&indiv);

/* Check out what the other fields mean here */
/* Why do some live people have non-zero deathdate fields */
  if((indiv.deathdate&0xFCFFFF) == 0 && indiv.birthdate != 0
     && indiv.sex != 'D') {
    year = (indiv.birthdate & 0xFF) << 4;
    year += (indiv.birthdate >> 12) & 0xF;
    month = ((indiv.birthdate & 0xF00) >> 7);
    month += ((indiv.birthdate & 0x800000) >>23);
    day = ((indiv.birthdate & 0x7C0000) >> 18);
    if(year >= sinceyear && month != 0 && day != 0){
      if(indiv.sex == 'D') printf("Deleted: ");
      if(sortflag != 0){
	NAME_REC first,last;
	get_name_rec(indiv.names[0],&last);
	get_name_rec(indiv.names[1],&first);
	printf("%-10s%-10s|",last.name,first.name);
      }
      printf("%i/%i",month,day);
      for(i=1;i<4;i++){
	if(indiv.names[i] != 0){
	  get_name_rec(indiv.names[i],&namrec);
	  printf(" %s",namrec.name);
	}
      }
      used_married_name = 0;
      if(indiv.sex == 'F' && indiv.ownmarriage != 0) {
	get_marr_rec(indiv.ownmarriage,&marriage);
	if(marriage.husband != 0) {
	  get_indiv_rec(marriage.husband,&husband);
	  if(husband.names[0] != 0){
	    get_name_rec(husband.names[0],&namrec);
	    printf(" %s",namrec.name);
	    used_married_name = 1;
	  }
	}
      }
      if(indiv.names[0] != 0){
	if(used_married_name) printf(" [");
	else printf(" ");
	get_name_rec(indiv.names[0],&namrec);
	printf("%s",namrec.name);
	if(used_married_name) printf("]");
      }
      if(indiv.names[4] != 0){
	get_name_rec(indiv.names[4],&namrec);
	printf(", %s",namrec.name);
      }
      printf(" (%i)\n",year);
    }
  }
  return;
}
