// This may look like C code, but it is really -*- C++ -*-
//
// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//

#ifndef _TCSOCK_H
#define _TCSOCK_H

#ifndef _TCLOG_H
#include "tcLog.h"			// Ensure we have error logger definitions
#endif 

#ifndef _UTILITY_H
#include "utility.h"		// ensure we have the timeout definitions
#endif
/////////////////////////////////////////////////////////////////////////////
// Use these pre-processor to avoid overhead if we will never want to debug
// If the following symbols are not set appropriately, then the tcSocket
// classes will never log TC_DEBUG or TC_TRACE messages, which means that
// they cannot by logged to file etc even if the tcLogger is capable of
// that.  Normally, one would only want to do this if speed really
// mattered...
#ifdef _DEBUG
#define TCSOCK_TRACE	1		// 1 to generate socket trace messages
#define TCSOCK_DEBUG	1		// 1 to generate socket debug messages
#endif


/////////////////////////////////////////////////////////////////////////////
// Currently arbitrary Windows message number to use for socket notification.
// This should really be done via an atom or ReserveMessage or something...
#define WM_SOCKET			(WM_USER + 20)
#define WM_SOCKET_HOST		(WM_SOCKET + 1 )

#define IDT_SOCKET_TIMEOUT 1
#define SOCKET_TIMEOUT_GRANULARITY 5

/////////////////////////////////////////////////////////////////////////////
// The actual socket base class
//

class tcSocket {
	struct errDesc {						// Error message descriptor
		int		num;						// Error number
		char	*string;					// Error description
	};
  protected:
	static errDesc			errString[];	// Error strings
	long					evMask;			// Events we are interested in
	SOCKET					sock;			// The actual socket
	HWND					hWnd;			// Window handle
	int						lastErr;		// Last socket error
	tcLogger				*logger;		// Event logger
	char					*hostbuf;		// buffer for hostent struct   
	int						m_Port;			// storage for port during name resolution
	HANDLE					hosttask;		// The task handle during name resolutions
  	HTIMEOUT				m_hTimeout;
	SOCKET					winSock()	{	// Get socket ID
		return sock;
	};
	void					Init(SOCKET s);	// Standard init stuff
	void					updateMask();	// Set new event mask

  public:
	tcSocket(tcLogger *l, int type =SOCK_STREAM);
	tcSocket(tcLogger *l, SOCKET sock);
	virtual ~tcSocket();   // virtual to ensure called on derived cleanup
	BOOL	isValid() {
		return (sock != INVALID_SOCKET);
	}
	void	addMask(long m) {		// Add bit to event mask
		setMask(evMask | m);
	}
	void	delMask(long m) {		// Remove bit from event mask
		setMask(evMask & ~m);
    }
	void	setMask(long m) {		// Set new event mask
		if (evMask != m) {			// Don't change the mask unless we must
			evMask = m;
			updateMask();			// Actually call winsock to set the mask
		}
	}
	int				Connect(const char *host, int port); // Connect to server
	int				Connect(const char *host, const char *serv,
							const char *prot ="tcp");
	int				Listen(int port);				// Listen for client
	tcSocket		*Accept();						// Accept new client
	void			Close();						// Close down
	int				GetHost( const char * host , int port );
	
	virtual void	Connected(int error);
	virtual void	Accepted(int error);
	virtual void	Closed(int error);
	virtual void	Readable(int error);
	virtual void	Writable(int error);
	virtual void	OOBData(int error);
	virtual void	HostResolve(HANDLE task , int error);
	virtual void	Timeout();
		
	const char		*sockErrMessage(int err);
	int				lastError()			{ return lastErr;					};
	const char		*lastErrorMessage()	{ return sockErrMessage(lastErr);	};
	void			clearError()		{ lastErr = 0;						};

	// User accessible read/write functions
	virtual int		send(const char *data, int amt =-1) {
		if (amt == -1) amt = strlen(data);
    	return ::send(sock, data, amt, 0);
    }
	virtual int		recv(char *dest, int size) {
    	return ::recv(sock, dest, size, 0);
    }                          
	// User Accessable timeout functions
    void SetTimeout( int Secs , CTimeoutType Type )	{ ::TimeoutSet( m_hTimeout , Secs , Type ); }
    void ClrTimeout()								{ ::TimeoutClr( m_hTimeout ); }
    CTimeoutType	TypTimeout()					{ return ::TimeoutFindType( m_hTimeout ); }
  private:
  	int _Connect( sockaddr_in addr , int port );      
};

/////////////////////////////////////////////////////////////////////////////
// Derived sockets with some better buffer handling
//

class tcBufSocket : public tcSocket {
  protected:
	int		rbSize, wbSize;		// Buffer sizes
	int		nrBuf, nwBuf;		// Number of characters buffered
	char	*rBuf;				// Read buffer
	char	*wBuf;
	int		wrpending;			// Number of FD_WRITE msgs pending

	void	pullUp(char *buf, int &cnt, int amt) {	// Re-align a buffer
		cnt -= amt;									// Recalculate the counter
		if (cnt > 0) {								// Is there anything to do?
			memmove(buf, buf + amt, cnt);			// Copy remaining data
		}
	}
	void	rdPullUp(int amt)	{ pullUp(rBuf, nrBuf, amt);		};
	void	wrPullUp(int amt)	{ pullUp(wBuf, nwBuf, amt);		};
	void	Init(int rbw, int wbs);					// Set up the buffers
	int		sendBuffer();							// Flush pending data
	
  public:
	tcBufSocket(tcLogger *l, int rbs =1024, int wbs =1024);
	tcBufSocket(tcLogger *l, SOCKET s, int rbs =1024, int wbs =1024);
	virtual ~tcBufSocket();
	int				numRbuf()		{ return nrBuf;				};
	int				numWbuf()		{ return nwBuf;				};
	int				rdSpace()		{ return (rbSize - nrBuf);	};
	int				wrSpace()		{ return (wbSize - nwBuf);	};
	virtual int		send(const char *data, int amt =-1);
	int				sendi(const char *data, int amt =-1);
	virtual int		recv(char *dest, int size);

	void			Readable(int error);		// Callback from tcSocket
	void			Writable(int error);		// Callback from tcSocket

	virtual void	dataRead()		=0;			// Data available to read
	virtual void	dataWritten()	=0;			// Write buffer empty
	virtual void	readError(int error);		// Read error callback
	virtual void	writeError(int error);		// Write error callback
};

/////////////////////////////////////////////////////////////////////////////
// Trivial line-at-a-time layer on top of tcBufSocket
//

class tcLineBufSocket : public tcBufSocket {
  protected:
  public:
	tcLineBufSocket(tcLogger *l, int rbs =1024, int wbs =1024)
	: tcBufSocket(l, rbs, wbs) { };
	tcLineBufSocket(tcLogger *l, SOCKET s, int rbs =1024, int wbs =1024)
	: tcBufSocket(l, s, rbs, wbs) { };

	int				readLine(char *buffer, int len); // Read a line
	int				writeLine(const char *line) {	 // For consistent naming
		int rc = send(line);
		if (rc > 0) send("\n", 1);
        return rc;
	}

	void			dataRead();					// Callback from tcBuffSocket

	virtual void	lineRead()		=0;			// A whole line available
};
/////////////////////////////////////////////////////////////////////////////
// Trivial line-at-a-time layer on top of tcBufSocket
// which also includes CRLF sending
//

class tcCRLFLineBufSocket : public tcBufSocket {
  protected:
  public:
	tcCRLFLineBufSocket(tcLogger *l, int rbs =1024, int wbs =1024)
	: tcBufSocket(l, rbs, wbs) { };
	tcCRLFLineBufSocket(tcLogger *l, SOCKET s, int rbs =1024, int wbs =1024)
	: tcBufSocket(l, s, rbs, wbs) { };

	int				readLine(char *buffer, int len); // Read a line
	int				writeLine(const char *line) {	 // For consistent naming
#ifdef TCSOCK_TRACE
		logger->logf(TC_DEBUG, "tcCRLFLineBufSocket(%p): writeLine(%s)", this, line );
#endif
		int rc = send(line);
		if (rc > 0) send("\r\n", 2);
        return rc;
	}

	void			dataRead();					// Callback from tcBuffSocket

	virtual void	lineRead()		=0;			// A whole line available
};

/////////////////////////////////////////////////////////////////////////////
// Free function to initialise the library.  Needs an HWND to parent
// the hidden socket windows, an HINSTANCE for CreateWindow (although
// it can infer this from the HWND if omitted) and a class name to use
// when creating the hidden windows (with a default)

int tcSocketInit(HWND hwnd, const char *className ="tcSocketCL",
				 HINSTANCE hInst =NULL, BOOL DelayInit = TRUE );
int tcSocketExit();

#endif // _tcSock_h_
