/***********************************************************************
**
** nntpsock.CPP
**
** Copyright 1994 by Ewan Kirk <ewan@kirk.demon.co.uk>
**
** Permission to use, copy, modify, distribute, and sell this software and its
** documentation for any purpose is hereby granted without fee, provided that
** the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation, and that the name of Ewan Kirk not be used in
** advertising or publicity pertaining to distribution of the software without
** specific, written prior permission.  Ewan Kirk makes no representations
** about the suitability of this software for any purpose.  It is provided
** "as is" without express or implied warranty.
**
** EWAN KIRK DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
** INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
** EVENT SHALL EWAN KIRK BE LIABLE FOR ANY SPECIAL, INDIRECT OR
** CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
** DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
** TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
** PERFORMANCE OF THIS SOFTWARE.
**
** This file is part of WINDIS. 
**
** All the guts of the NNTPSocket are here.
** This is basically a general purpose NNTPClient.
** Does batching too!
**
** V1.0 - Initial Revision 01-Sep-94
** V1.1 - Fixed the time kick bug
*/
////////////////////////////////////////////////////////////////////
// nntpsock.cpp
// Implementation of the nntpclient socket
#include "stdafx.h"
#include "windis.h"
#include "utility.h"
#include "tcsock.h"
#include "csocket.h"
#include "nntpsock.h"          
#include "nntpdlg.h"
        
// The simple inline constructors for the time Socket
CNntpSocket::CNntpSocket( tcLogger * Log , CNntpDlg * pParent , int Rbs , int Wbs )
	: CLineSocket( Log , Rbs , Wbs )
{
	m_Parent = pParent;
	m_State = INIT;      
	m_BatchOutstanding = 0;
	m_Info = new CNntpInfo;
	m_LineBuf = new char[ NntpLineBufSize ];
	m_HistList = NULL;
	m_BatchIsOpen = FALSE;
	m_GotSomething = FALSE;
}

CNntpSocket::~CNntpSocket()
{
	delete m_Info;                              
	delete [] m_LineBuf;
}           



void CNntpSocket::lineRead()
{                                         
	BOOL rc = TRUE;  // Are we going to send update info?
	ClrTimeout();
	m_Info->TotalBytes += readLine( m_LineBuf , NntpLineBufSize - 1 );    
	switch( m_State ) 
	{
  	case INIT:				rc = DoInit( m_LineBuf ); 				break;
	case CONNECTED:			rc = DoConnected( m_LineBuf );			break;
 	case WAITGROUPSCONF:	rc = DoWaitGroupsConf( m_LineBuf );	break;
 	case WAITGROUPS:		rc = DoWaitGroups( m_LineBuf );		break;
 	case WAITNEWGROUPSCONF:	rc = DoWaitNewGroupsConf( m_LineBuf );	break;
 	case WAITNEWGROUPS:		rc = DoWaitNewGroups( m_LineBuf );		break;
 	case WAITARTIDSCONF:	rc = DoWaitArtIdsConf( m_LineBuf );	break;
 	case WAITARTIDS:		rc = DoWaitArtIds( m_LineBuf );		break;
 	case WAITARTSCONF:		rc = DoWaitArtsConf( m_LineBuf );		break;
 	case WAITARTS:			rc = DoWaitArts( m_LineBuf );			break;
 	case CLOSING:			rc = DoClosing( m_LineBuf );			break;
 	}
	if( rc )
		m_Parent->Info( m_Info );

	return;
}
// The StateTableFunctions
BOOL CNntpSocket::DoInit( char * /* Str */ )
{
	// should never get this.
	ASSERT( FALSE );
	return FALSE;
}

BOOL CNntpSocket::DoConnected( char * Str )
{
	// We expect a line of the form
	// 20x <newserver> news server ready
	// Read the line, parse it and if ok
	// then do send the request for newsgroups since time
	// and state = WAITARTIDS,
	// However, if we want to download the complete list
	// of newsgroups then send the command to do this
	// and then we should be in the state WAITFORGROUPS
	// We mark the start by getting the system time
	m_Info->SessionStart = time(NULL);
	int Code = ParseReply( Str);
	if ( Code == 200 || Code == 201 )
	{
		// we are ok
		// So kick into the download groups section		
		if( DownloadGroups() )
		{
			m_State = WAITGROUPSCONF;
			return TRUE;
		}
		// So we don't want to download the groups
		// Do we want to download new groups?
		if( DownloadNewGroups() )
		{
			m_State = WAITNEWGROUPSCONF;
			return TRUE;
		}
		// So we don't want to download the new groups
		// so now we just download the article ids
		if( DownloadArticleIds() )
		{
			m_State = WAITARTIDSCONF;
			return TRUE;
		}
		// There's no news groups in the file
		EndConversation(FALSE);
		return TRUE;
									
		// If we get here then there's been some fuck up
	}		
	return ProtocolError( Str , 200 );
}

BOOL CNntpSocket::DoWaitGroupsConf( char * Str )
{
	int Code = ParseReply( Str );
	if( Code == 215 )
	{
		m_State = WAITGROUPS;
		return TRUE;
	}
	return ProtocolError( Str , 215 );
}

// If we get this then we're trying to download the
// entire list of groups so read this group
// and stick it in the active file.
BOOL CNntpSocket::DoWaitGroups( char * Str )
{
	char * p = ParseBuffer( Str );
	if( p != NULL )
	{
		Str = strtok( p , " \n\t" ); // Just save the names for demon
		m_CurrentList.AddTail( CString( p ) );
		m_Parent->Trace( 1 , p );
		if( m_Info->GroupsGot++ % 10 != 9 )
			return FALSE;
		else
			return TRUE;
	}
	else
	{
		// We've got to the end of the groups list
		// Write it to a file
		CStdioFile GroupList;    
		CString KA9Q = GetNewsGroupListName();
		if( GroupList.Open( KA9Q , CFile::modeCreate | CFile::typeText ) )
		{
			while( !m_CurrentList.IsEmpty() )
			{
				GroupList.WriteString( m_CurrentList.RemoveHead() );
				GroupList.WriteString( "\n" );
			}
			GroupList.Close();
		}
		else
		{
			::MsgBox( MB_OK | MB_ICONHAND , IDS_MSG_COULDNT_WRITE_GROUP_LIST );
		}
		// Write out the information
		int LastTime = m_ThisTime.GetDay() + m_ThisTime.GetMonth() * 100;
		gConfig->SetDownloadGroupsLastTime( LastTime );
		gConfig->WriteCurrentSet();
		// We never download the new groups at this point
		// So we just ask for the articles
		if( DownloadArticleIds() )
		{
			m_State = WAITARTIDSCONF;
			return TRUE;
		}
		// If we get here, we've got a problem
		return ProtocolError( Str , -1 );
	}    
}

BOOL CNntpSocket::DoWaitNewGroupsConf( char * Str )
{    
	int Code = ParseReply( Str );
	if( Code == 231 )
	{
	 	m_State = WAITNEWGROUPS;
	 	return TRUE;
	}
	else            
		return	ProtocolError( Str ,231 );     
}

// Read the line
// if the line is not "." add to the new group file
// and wait some more
// else
// We should send the first article id and wait for
// an article like in the WAITARTIDS state.
BOOL CNntpSocket::DoWaitNewGroups( char * Str )
{
	char * p = ParseBuffer( Str );
	if( p != NULL )
	{
		m_CurrentList.AddTail( CString( p ) );
		m_Info->NewGroupsGot++;
		m_Parent->Trace( 0 , Str );
		return TRUE;
	}
	else
	{
		// We've got to the end of the new groups list  
		// So we should save it to a file
		CStdioFile GroupList;    
		CString GroupFile = GetNewsNewgroupName();
		if( GroupList.Open( GroupFile , CFile::modeWrite | CFile::typeText ) )
		{                                           
			while( !m_CurrentList.IsEmpty() )
			{
				CString Tmp = m_CurrentList.GetHead();
				GroupList.WriteString( Tmp );
				GroupList.WriteString( "\n" );
				Tmp = m_CurrentList.RemoveHead();
			};
			GroupList.Close();
		}
		if( DownloadArticleIds() )
		{
			m_State = WAITARTIDSCONF;
			return TRUE;
		}
		// And if we get here, it's a protocol error
		return ProtocolError( Str , -1 );
	}
}

BOOL CNntpSocket::DoWaitArtIdsConf( char * Str )
{
	int Code = ParseReply( Str );
	if( Code == 230 )
	{
	 	m_State = WAITARTIDS;
	 	return TRUE;
	}
	else            
		return	ProtocolError( Str , 230 );     
}

// Read the line 
// if line is not "." add it to the list
// and wait some more
// else,                           
// If we want the new groups then send the 
// command to get the new groups since time
// Send the first article ID and state = WAITART
BOOL CNntpSocket::DoWaitArtIds( char * Str )
{
	char * p = ParseBuffer( Str );
	if( p != NULL )
	{
		m_ArticlesToGet.AddTail( CString( p ) );
		if( m_Info->ArticlesToGet++ % 10 != 9 ) // 9 because then it's incremented!
			m_Parent->Trace( 1 , Str );
	}
	else
	{
		// We've got to the end of the new messages list   
		// Check to see if we still have more groups to get
		// the new news for
		if( DownloadArticleIds() )
		{
			m_State = WAITARTIDSCONF;
		}
		else
		{
			m_CurrentId = "";
			m_CurrentList.RemoveAll();
			DownloadArticle();     
			if( m_State != CLOSING )
				// State doesn't change really
				m_State = WAITARTSCONF;
		}
	}
	return TRUE;
}                  

BOOL CNntpSocket::DoWaitArtsConf( char * Str )
{
	int Code = ParseReply( Str );      
	m_Info->CurrentBytes = 0l;
	if( Code == 220 )
	{
	 	m_State = WAITARTS;
	 	return TRUE;
	}
	if( Code ==  430 )
	{
		// This article is unavailable
		// Trace it.
		// This is a very special case cludge
		m_Parent->Trace( 0 , Str );
		m_BatchOutstanding--;
		m_Info->Unavailable++;
		DownloadArticle();
		if( m_State != CLOSING )
				// State doesn't change really
				m_State = WAITARTSCONF;
		return TRUE;
	}               
	else            
		return	ProtocolError( Str , 220 );     
}

// if line is not ".", add it to the list and wait some more
// else
// stick the article in the batch, add the id to the news
// history file and then
// if there is another article to get, send the command
// to get it
// else we're finished. send quit message
// state = CLOSING
BOOL CNntpSocket::DoWaitArts( char * Str )
{
	char * p = ParseBuffer( Str );
	if( p != NULL )    
	{
		m_Info->ArticleBytes += ( strlen( p ) + 1 );
		m_Info->CurrentBytes += ( strlen( p ) + 1 );
		m_CurrentList.AddTail( CString( p ) );
		m_Parent->Trace( 4 , Str );
		return FALSE;
	}                             
	else
	{
		// Write the message to the batch file
		char Buffer[ 32 ];
		wsprintf( Buffer , "#! rnews %ld\n" , m_Info->CurrentBytes );
		if( !m_BatchIsOpen )
		{
			// open the batch file
			if( !OpenOrCreate( m_BatchFile , m_BatchName , CFile::modeWrite | CFile::typeText ) )
			{
				::MsgBox( MB_OK | MB_ICONHAND , IDS_MSG_COULDNT_CREATE_BATCH_FILE );
				return FALSE;
			}
			m_BatchFile.SeekToEnd();	          
			m_BatchIsOpen = TRUE;
		}
		m_BatchFile.WriteString( Buffer );
		while( !m_CurrentList.IsEmpty() )
		{
			m_BatchFile.WriteString( m_CurrentList.RemoveHead() );
			m_BatchFile.WriteString( "\n" );
			// remember the string contains the \n character
		}
		m_BatchFile.Flush();                  
		// Write the ID to the historyfile
		m_HistList->AddHistoryEntry( m_CurrentId );
		m_Parent->Trace( 0 , "Retrieved " + m_CurrentId );
		m_Info->ArticlesGot++;              
		// Done here I think
		m_BatchOutstanding--;
		DownloadArticle();
		if( m_State != CLOSING )
			// Download Article worked 
			m_State = WAITARTSCONF;
		return TRUE;
	}
}

// Close myself
// Socket will post a FD_CLOSE message which gets
// forwarded to the window. But of course, this
// doesn't happen because of the socket library
// clearing the event mask before closing.  Hmmmm
// Change this?
BOOL CNntpSocket::DoClosing( char * Str )
{					
	int Code = ParseReply( Str );
	if( Code != 205 )
	{         
		ProtocolError( Str , 205 );
	}                              
	if( m_BatchIsOpen )
		m_BatchFile.Close();
	m_BatchIsOpen = FALSE;
	m_Parent->Close( 0 , this );
	Close(); // Close myself
	return FALSE;
}                 


void CNntpSocket::dataWritten()
{                                               
	// For an nntp socket, if we've written some data,
	// we're going to read some almost right away
	SetTimeout( TIMEOUT_READ , TO_READING );
}

void CNntpSocket::Connected(int error)
{                               
	delMask( FD_CONNECT );
	m_State = CONNECTED;
	m_Parent->Connected( error , this );
}

void CNntpSocket::Closed(int error )
{
	// I thought that this would never get called but infact,
	// the server end may close the connection for some
	// reason
	m_Parent->Closed( error , this );
}

void CNntpSocket::Timeout()
{
	ASSERT_VALID( m_Parent );
	m_Parent->Timeout(this);
}


// This function will check the setup for a flag to
// tell you to periodically download new groups
// and will return send the required command if
// required (!).

BOOL CNntpSocket::DownloadGroups()
{                         
	if( gConfig->GetDownloadGroups() )
	{   
		// Check for time;
		CTime LastTime;
		CTimeSpan Span;
		int Last = gConfig->GetDownloadGroupsLastTime();
		if( Last != 0 )
		{
			// We just do this on the basis of days.  The string
			// is MMDD
			int Month = Last/100;
			int Day = Last - 100*Month;
			// This is pretty cludgy because we can only store limited
			// information in an int (unless we bugger about with bit
			// fields and basically, life's too short.  
			LastTime = CTime( m_ThisTime.GetYear() , Month , Day , 0 , 0 , 0 );
			Span = m_ThisTime - LastTime;
			// Catch year roll.
			if( Span < 0 )
			{
				CTimeSpan OneYear( 365 , 0 , 0 , 0 );
				Span += OneYear; 
			}
		}
		if( Span.GetDays() < gConfig->GetDownloadGroupsDays() )
			return FALSE; 
			                         
		char Buffer[ 64 ];
		if( Last == 0 )
			wsprintf( Buffer , "You have never " );
		else
			wsprintf( Buffer , "It has been %ld days since you last" , Span.GetDays() );
		if( ::MsgBox( MB_OKCANCEL , IDS_MSG_DOWNLOAD_COMPLETE_SET, (const char *)Buffer ) == IDCANCEL )
			return FALSE;

		// If we get here then we actually want to do the download						  
		CString Cmd = "LIST";
		writeLine( Cmd );
		m_Parent->Trace( 3 , Cmd );
		// Empty the current list
		while( !m_CurrentList.IsEmpty() )
			m_CurrentList.RemoveHead();
		// Set the timeout
		SetTimeout( TIMEOUT_WRITE , TO_WRITING );
		return TRUE;
	}
	else               
		return FALSE;
}

// This function asks to download the new groups since the time
BOOL CNntpSocket::DownloadNewGroups()
{   
	if( gConfig->GetDownloadNewGroups() )
	{
		CString Cmd = "NEWGROUPS " + m_LastTime.Format( "%y%m%d %H%M00 GMT" );
		writeLine( Cmd );       
		m_Parent->Trace( 3 , Cmd );
		// Empty the current list
		while( !m_CurrentList.IsEmpty() )
			m_CurrentList.RemoveHead();
		m_State = WAITNEWGROUPSCONF;
		SetTimeout( TIMEOUT_WRITE , TO_WRITING );
		return TRUE;
	}
	else
		return FALSE;
}
                

// This guy downloads the article ids for each group in turn
// What we do is remove the newsgroups from the newsgroups list
// until we're close to the 512 line length limit and send
// these lines
// EMK This is a cludge to be fixed.
BOOL CNntpSocket::DownloadArticleIds()
{                                   
	CString TimeStr = " ";
	TimeStr += m_LastTime.Format( "%y%m%d %H%M00 GMT" );
	if(	m_NewsGroups.IsEmpty() )
		return FALSE;
		
	// Construct the newnews line                      
	CString Cmd = "NEWNEWS ";               
	while( !m_NewsGroups.IsEmpty() )
	{
		CString Group = m_NewsGroups.GetHead();
		                                      
		// Set the limit at 400 for safety		                                      
		if( Cmd.GetLength()+Group.GetLength()+TimeStr.GetLength() > 400 )
		{
			break;
		}                    
		if( Cmd != "NEWNEWS " )
		{
			Cmd += ',';	// Not First case
		}
		Cmd += Group;                   
		// Remove the group we've already done
		Group = m_NewsGroups.RemoveHead();
	}
	writeLine( Cmd + TimeStr );
	m_Parent->Trace( 3 , Cmd );
	SetTimeout( TIMEOUT_WRITE , TO_WRITING );
	return TRUE;
}

// We'll have to check out batch download here.
BOOL CNntpSocket::DownloadArticle()
{   
   	// First check to see if we need to start the timer
   	// This might be better as a gettickcount?
   	if( m_Info->NewsStart == (time_t) 0 )
   		m_Info->NewsStart = time(NULL); 
   		
	if( m_ArticlesToGet.IsEmpty() )
	{
		EndConversation( TRUE );
		return TRUE;
	}
	// If we get here, we have something so we are going to
	// update the last date
	m_GotSomething = TRUE;
	while( m_BatchOutstanding < gConfig->GetNewsBatchDepth() )
	{                                                                
		if( m_ArticlesToGet.IsEmpty() )
		{
			EndConversation( TRUE );
			return TRUE;
		}		
    	m_CurrentId = m_ArticlesToGet.RemoveHead();
    	while( m_HistList->IsAlreadyGot( m_CurrentId ) )
    	{
       		// Do duplicate processing
	       	m_Info->Duplicates++;
   	    	m_Info->ArticlesToGet--;
	       	m_Parent->Trace( 0 , "Duplicate Article" + m_CurrentId );
			if( m_ArticlesToGet.IsEmpty() )
			{
				EndConversation(TRUE);
				return TRUE;
			}	
   	    	m_CurrentId = m_ArticlesToGet.RemoveHead();
    	}
		CString Id = "ARTICLE "+ m_CurrentId ;
		writeLine( Id );
		m_Parent->Trace( 3 , Id );
		m_BatchOutstanding++;
	}
	m_Parent->Info( m_Info );
	SetTimeout( TIMEOUT_WRITE , TO_WRITING );
	return TRUE;
}
                
void CNntpSocket::EndConversation( BOOL Save )
{
	if( m_BatchOutstanding == 0 )
	{
		if( Save ) 
			SaveNntpDat();
		writeLine( "QUIT" );
		m_Parent->Trace( 3 , "Quit" );
		m_State = CLOSING;
	}
}

void CNntpSocket::AbortConversation()
{
	if( m_Parent )
	{
		m_Parent->Trace( 3 , "Hard Close" );
		m_Parent->Close( 0 , this );
		// The parent calls the socket close function
		m_State = CLOSING;
	};
}

char * CNntpSocket::ParseBuffer( char * Buf)
{
	if( *Buf != '.' )
		// this is of no interest so just return the buffer
		return Buf;
	// More interesting.
	++Buf;
	if( *Buf == '\0' )
		// this is the last one
	    return NULL;
	else
		return Buf;
}

int CNntpSocket::ParseReply(const char * Line)
{       
	m_Parent->Trace( 2 , Line );                        
	return atoi( Line );	// This should be clever enough to
								// catch just the code
}

BOOL CNntpSocket::ProtocolError( const char * Str , int /* Expected */ )
{
	// There was some error so flag it
	m_Parent->Trace( 0 , Str );
	AbortConversation();
	return FALSE;
}

BOOL CNntpSocket::Init(CNNTPHistoryList * HistList, CTime ThisTime )
{
    // This sets the time up so that we can say what the last
    // time we connected was.  This used to be set in the
    // constructor wrongly
	char Buffer[ 256 ];                  
    
	m_ThisTime = ThisTime;
	m_HistList = HistList;
	m_NntpFileName = GetNewsDataName();
	m_BatchName = GetNewsBatchFileName();

	// Now open the nntp dat file
	CStdioFile NewsDat;
	if( !NewsDat.Open( m_NntpFileName , CFile::modeRead | CFile::typeText ) )
	{
			// Now we're in the shit
		::MsgBox( MB_OK | MB_ICONHAND , IDS_MSG_COULDNT_OPEN_NEWSDAT );
		return FALSE;                                        
	}
	m_Parent->Trace( 1 , IDS_TRACE_READING_NEWSDAT );	
	// First nonblank line is the news server and the time
	while( strlen( NewsDat.ReadString( Buffer , 255 ) ) == 0 )
		; // keep reading
	
	// parse the line
	char * p = strtok( Buffer , " \t\n" );
	// EMK need error checking here!!!	
	// Check this against the newsserver
	p = strtok( NULL , " \t\n" );
	// p is now the last date
	if( p == NULL )
		return FALSE;
	long Date = atol( p );
	int Year = int(Date / 10000L);
	int Month =  int( ( Date - Year * 10000L ) / 100L );
	int Day = int(  Date - Year * 10000L - Month * 100L );
	
	// Now get the time
	p = strtok( NULL , " \t\n" );
	if( p == NULL )
		return FALSE;
	long Time = atol( p );
	int Hour = int( Time / 10000L );
	int Minute = int( ( Time - Hour * 10000L ) / 100L );
	int Secs = 0;
	m_LastTime = CTime( Year + 1900, Month , Day , Hour , Minute , Secs );
	
	// Roll the time back 5 minutes to ensure we get everything.
	CTimeSpan FiveMinutes( 0 , 0 , 5 , 0 );
	m_LastTime = m_LastTime - FiveMinutes;	
	// Right, just have to load the newsgroups now
	while( NewsDat.ReadString( Buffer , 255 ) )
	{
		// Strip off any spaces
		p = strtok( Buffer , " \t\n" );
		
		if( p == NULL || strlen( p ) == 0 )
			continue;
		CString Tmp = p;			
		m_NewsGroups.AddTail( Tmp );
		
		// We stick the groups in another list so that when we
		// delete stuff from the m_NewsGroups list, there's
		// still a copy of what was in the file.
		// EMK could be done better.
		m_SaveGroups.AddTail( Tmp );
	}             
	
	// Close the Nntpdata file
	NewsDat.Close();
	// All done!!	            
	
	m_Parent->Trace( 1 , "Data Loaded" );
	return TRUE;
}
	 

void CNntpSocket::SaveNntpDat()
{
	if( !m_GotSomething )
		return;

	CString TimeStr = gConfig->GetNewsServer();
	TimeStr += m_ThisTime.Format( " %y%m%d %H%M00\n" );
	CStdioFile NewsDat;
	if( !NewsDat.Open( m_NntpFileName , CFile::modeCreate | CFile::typeText ) )
	{
		::MsgBox( MB_OK | MB_ICONHAND , IDS_MSG_SAVE_NNTP );
		return;
	}                                                 
	NewsDat.WriteString( TimeStr );
//	Following line removed by steve@one47.demon.co.uk
//	Not required.	
//	NewsDat.WriteString( "\n" ); // A blank link
	// Now write the groups;                 
	while( !m_SaveGroups.IsEmpty() )
	{
		NewsDat.WriteString( " " );// Write the space.
		NewsDat.WriteString( m_SaveGroups.RemoveHead() );
		NewsDat.WriteString( "\n" );
	}
	NewsDat.Close();
}
	
	
	
