// ========================================================================
//  TODB LIBRARY
//    fblock.h
//
//    Key set file's data block
//    Block class, ObjHeader structure
//
//    Version: see TODB.H file
//
//    Copyright 1993 Christian Thrien
//    All rights reserved
// ========================================================================

#ifndef _FBLOCK_H
#define _FBLOCK_H

#include "ptodb.h"
#include "todb.h"
#include <stdio.h>
#include <memory.h>

class String;
class ErrReporter;
class TODB;

//-------------------------------------------------------------------------
//  ObjHeader structure
//-------------------------------------------------------------------------

struct ObjHeader
{
    IdType classid;		// class identification
    BlockNbr blknbr;		// relative address of an object
    ObjHeader( int cid = 0, BlockNbr bn = 0 );
};

inline ObjHeader::ObjHeader( int cid, BlockNbr bn ) :
    classid( cid ), blknbr( bn )
{
}

//-------------------------------------------------------------------------
//  Block class
//-------------------------------------------------------------------------

enum BlockError
{
    BLK_APPEND
};

static const int fheadersize = sizeof(BlockNbr) * 2;

class Block
{
public:
    Block( TODB * todb = NULL, BlockNbr block = 0 );
    Block( const Block & block );

    ~Block();

    Block & operator = ( const Block & block );
    void SetNextBlock( BlockNbr block );
    BlockNbr NextBlock() const;
    BlockNbr GetBlockNbr() const;
    void MarkBlockDeleted();
    void Save();
    void Get( void * buf, int length, int pos );
    void Put( void * buf, int length, int pos );

    // error control
    static void SetErrOut( ErrReporter * er );

private:
    TODB * todb_;		// current database
    FILE * dataf_;		// file control block
    BlockNbr blocknbr_;		// this block number
    long blockaddr_;		// disk address of this block
    Boolean deleteblock_;	// true if this block is being deleted
    BlockNbr nextblock_;	// relative address of the next block
    char buf_[blocklength];	// block contents

    // error control
    static ErrReporter * ErrOut;
    static void ErrorHandler( BlockError err );
};

inline void Block::SetNextBlock( BlockNbr block )
{
    memcpy( buf_, &block, sizeof( nextblock_ ) );
    nextblock_ = block;
}

inline BlockNbr Block::NextBlock() const
{
    return nextblock_;
}

inline BlockNbr Block::GetBlockNbr() const
{
    return blocknbr_;
}

inline void Block::MarkBlockDeleted()
{
    deleteblock_ = True;
}

inline void Block::Save()
{
    fseek( dataf_, blockaddr_, SEEK_SET );
    fwrite( buf_ , 1, blocklength, dataf_ );
}

inline void Block::Get( void * buf, int length, int pos )
{
    memcpy( buf, buf_ + pos, length );
}

inline void Block::Put( void * buf, int length, int pos )
{
    memcpy( buf_ + pos, buf, length );
}

#endif // _FBLOCK_H
