/*
     Program PM-Man: a program for viewing BSD-style Manual Pages.
     Copyright (C) 1992,1993,1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#define INCL_WIN
#define INCL_GPI
#include <os2.h>

#include "font.h"
#include "pmman.h"
#include "bugme.h"
#include "stdwin.h"
#include "attr.h"
#include "view.h"
#include "manfile.h"
#include "diaman.h"

extern const char *embedded_text_COPYING[], *embedded_text_Copyman[];
extern int embedded_text_COPYING_count, embedded_text_Copyman_count;

#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))
#define LIMIT(v,a,b) (v) = max((a),min((b),(v)))

class MyWin: public StdWinPlus
{
    int char_width, char_height, char_descender;
    int client_height;
    int client_char_height;
    int scroll_position;
    int max_scroll_position;
    HWND vscrollbar;
    View *view;
    static char *error_view[];
    void recalc_max_scroll_position(void)
    {
	max_scroll_position = view->length() - client_char_height;
	if (max_scroll_position < 0) max_scroll_position = 0;
	if (scroll_position > max_scroll_position) {
	    scroll_position = max_scroll_position;
	}
#if 0
	NOTE(client_char_height);
	NOTE(view->length());
	NOTE(max_scroll_position);
	NOTE(scroll_position);
#endif
    }
    void recalc_scrollbar(void);
    void FreeView(void);
    void SelectStaticView(const char **static_view, int static_view_count);
    void GetFontInfo();
    void view_printf(char *buffer, const char *fmt, ...);
    void SelectMan(const char *manname);
public:
    virtual MRESULT MsgPaint(int *);
    virtual MRESULT MsgSize(SHORT, SHORT, SHORT, SHORT, int *);
    virtual MRESULT MsgCreate(PVOID, PCREATESTRUCT, int *);
    virtual MRESULT MsgVscroll(USHORT, SHORT, USHORT, int *);
    virtual MRESULT MsgButton1Down(POINTS, USHORT, USHORT, int *);
    virtual MRESULT MsgCommand(USHORT, USHORT, USHORT, int *);
    virtual MRESULT MsgPresParamChanged(MPARAM, MPARAM, int *);
    void GotoSelect();
    void MenuSelect();
    void SelectLine(int line, int xpos);
    void RefreshLine(int line);
    void Init(HAB);
    MyWin();
};

char *MyWin::error_view[] = { "Error", "" };

MyWin::MyWin()
    : StdWinPlus(),
      view(NULL)
{
    SelectStaticView(embedded_text_Copyman, embedded_text_Copyman_count);
}

void MyWin::SelectStaticView(const char **static_view, int static_view_count)
{
    FreeView();
    view = new StaticView(static_view_count, static_view);

    scroll_position = 0;
    recalc_max_scroll_position();
    recalc_scrollbar();
    WinInvalidateRect(window, NULL, FALSE);
}

void MyWin::SelectMan(const char *manname)
{
    View *newview = getmanual(manname);
    if (newview == NULL) return;
    FreeView();
    view = newview;
    scroll_position = 0;
    recalc_max_scroll_position();
    recalc_scrollbar();
    WinInvalidateRect(window, NULL, FALSE);
}


void MyWin::FreeView(void)
{
    if (view == NULL) return;
    delete view;
}

void MyWin::Init(HAB hab)
{
    StdWin::Init(hab, "StdWin", CS_SIZEREDRAW,
		 FCF_TITLEBAR | FCF_SYSMENU | FCF_SIZEBORDER | FCF_MINMAX
		 | FCF_SHELLPOSITION | FCF_TASKLIST | FCF_VERTSCROLL | FCF_MENU
		 | FCF_ICON | FCF_ACCELTABLE);
    scroll_position = 0;
}

MRESULT MyWin::MsgCommand(USHORT cmd, USHORT, USHORT, int *dodefault)
{
  switch(cmd) {
  case IDM_ABOUT_ABOUT:
    SelectStaticView(embedded_text_Copyman, embedded_text_Copyman_count);
    break;
  case IDM_ABOUT_COPYING:
    SelectStaticView(embedded_text_COPYING, embedded_text_COPYING_count);
    break;
  case IDM_ABOUT_WARRANTY:
    SelectStaticView(embedded_text_COPYING, embedded_text_COPYING_count);
    scroll_position = 277; /* Start of Warantee section */
    recalc_max_scroll_position();
    recalc_scrollbar();
    WinInvalidateRect(window, NULL, FALSE);
    break;
  case IDM_FILE_FIND:
    const char *manfile = manual_select_dialog(window);
    if (manfile)
      {
	BUGME(("selecting new manual: '%s'", manfile));
	SelectMan (manfile);
      }
    break;
  case IDM_KBD_PGUP:
    MsgVscroll(0, 0, SB_PAGEUP, NULL);
    break;
  case IDM_KBD_PGDN:
    MsgVscroll(0, 0, SB_PAGEDOWN, NULL);
    break;
  case IDM_KBD_LINEDOWN:
    MsgVscroll(0, 0, SB_LINEDOWN, NULL);
    break;
  case IDM_KBD_LINEUP:
    MsgVscroll(0, 0, SB_LINEUP, NULL);
    break;
  case IDM_TEST2:
  case IDM_TEST3:
  case IDM_TEST4:
  default:
    *dodefault = 1; /* Just in case */
  }
  return 0;
}


MRESULT MyWin::MsgButton1Down(POINTS pos, USHORT, USHORT, int *)
{
    BUGME(("MyWin::MsgButton1Down((%d,%d)...)",pos.x,pos.y));
    // If we are in background, set active and do not select an item
    if (WinQueryActiveWindow(HWND_DESKTOP) 
	!= WinQueryWindow(window, QW_PARENT)) {
	WinSetActiveWindow(HWND_DESKTOP, window);
	return 0;
    }
    int x,y;
    x = pos.x;
    y = pos.y;
    y = (client_height - y) / char_height + scroll_position;

    // Now translate the x
    const char *line = view->line(y);
    int newx;
    if (line == NULL) {
	newx=1;
    } else {
	int len = strlen(line);
	BUGME(("length of line %d \"%s\" is %d", y, line, len));
	POINTL pts[TXTBOX_COUNT];
	HPS hps = WinGetPS(window);
	for (newx=1; newx <= len; newx++) {
	    GpiQueryTextBox(hps, newx, line, TXTBOX_COUNT, pts);
	    BUGME(("size of \"%*s\" is %d", newx, line, 
		   pts[TXTBOX_TOPRIGHT].x));
	    if (pts[TXTBOX_TOPRIGHT].x >= x) break;
	}
	WinReleasePS(hps);
    }

    SelectLine(y, newx-1);
    return 0;
}

void MyWin::RefreshLine(int line)
{
    WinInvalidateRect(window, NULL, FALSE);
}

void MyWin::SelectLine(int line, int pos)
{
    BUGME(("MyWin::SelectLine(%d,%d)", line, pos));
}

MRESULT MyWin::MsgVscroll(USHORT, SHORT slider, USHORT cmd, int *)
{
    SHORT new_scroll_position = scroll_position;

    switch(cmd) {
    case SB_LINEUP:
	new_scroll_position--;
	break;
    case SB_LINEDOWN:
	new_scroll_position++;
	break;
    case SB_PAGEUP:
	new_scroll_position -= max(1,client_char_height);
	break;
    case SB_PAGEDOWN:
	new_scroll_position += max(1,client_char_height);
	break;
    case SB_SLIDERTRACK:
	new_scroll_position = slider;
	break;
    }
    LIMIT(new_scroll_position, 0, max_scroll_position);
    if (new_scroll_position != scroll_position) {
	WinScrollWindow(window, 0, 
	    (new_scroll_position - scroll_position) * char_height,
	    NULL,NULL,NULL,NULL, SW_INVALIDATERGN);
	scroll_position = new_scroll_position;
	WinSendMsg(vscrollbar, SBM_SETPOS,
	    MPFROMSHORT(scroll_position), NULL);
	WinUpdateWindow(window);
    }
    return 0;
}

MRESULT MyWin::MsgPresParamChanged(MPARAM, MPARAM, 
				   int *dodefault)
{
    GetFontInfo();
    client_char_height = client_height / char_height;
    recalc_max_scroll_position();
    recalc_scrollbar();
    WinInvalidateRect(window, NULL, FALSE);
    *dodefault = 1;
    return 0;
}


MRESULT MyWin::MsgPaint(int *)
{
    RECTL rect;
    HPS hps = WinBeginPaint(window, NULL, &rect);
    POINTL pt;
    int top,bottom;
    int i;

    VisibleAttribute::setup_hps (hps, &view->attr.text_fonts);
    GpiErase(hps);
    top = (client_height - rect.yTop) / char_height;
    bottom = (client_height - rect.yBottom) / char_height + 1;
    pt.x = 0;
    pt.y = client_height - char_height * (1+top)+ char_descender;
    BUGME(("Repainting lines %d thru %d", top, bottom));
    NOTE(scroll_position);
    top += scroll_position;
    bottom += scroll_position;
#if 0
    NOTE(top);
    NOTE(bottom);
#endif
    LIMIT(top,scroll_position,view->length());
    LIMIT(bottom,scroll_position,view->length());
    for (i = top; i < bottom; i++) {
	GpiMove(hps, &pt);
	const char *line = view->line(i);
	int length = strlen(line);
	int total_length = length;
	int segment_length;
	TextPoint pos(i,0), next;
	while (length > 0) {
	    VisibleAttribute::set_default_attributes(hps,
						     &view->attr.text_fonts);
	    view->attr.set_attributes(pos);
	    next = view->attr.next_transition(pos);
	    if (next.line != i || next.position >= total_length) {
		segment_length = length;
	    } else {
		segment_length = next.position - pos.position;
	    }
	    GpiCharString(hps, segment_length, line);
	    length -= segment_length;
	    line += segment_length;
	    pos = next;
	}
	pt.y -= char_height;
    }
    WinEndPaint(hps);
    return 0;
}

void MyWin::recalc_scrollbar(void)
{
    WinSendMsg(vscrollbar, SBM_SETSCROLLBAR, MPFROM2SHORT(scroll_position, 0), 
	MPFROM2SHORT(0, max_scroll_position));
    WinSendMsg(vscrollbar, SBM_SETTHUMBSIZE, 
	       MPFROM2SHORT(client_char_height, view->length()), 0);
    WinEnableWindow(vscrollbar, (max_scroll_position == 0) ? FALSE : TRUE);
}

MRESULT MyWin::MsgSize(SHORT, SHORT, SHORT, SHORT scynew, int *)
{
    client_height = scynew;
    client_char_height = client_height / char_height;
    recalc_max_scroll_position();
    BUGME(("Size: Scrollbar set to %d of %d", scroll_position,
		  max_scroll_position));
    recalc_scrollbar();
#if 0
    NOTE(client_height);
    NOTE(client_char_height);
    NOTE(max_scroll_position);
#endif
    return 0;
}

void MyWin::GetFontInfo()
{
    HPS hps;
    FONTMETRICS fm;

    hps = WinGetPS(window);
    VisibleAttribute::setup_hps (hps, &view->attr.text_fonts);
    GpiQueryFontMetrics(hps, sizeof(fm), &fm);
    WinReleasePS(hps);

    char_width = fm.lEmInc;
    char_height = fm.lMaxBaselineExt;
    char_descender = fm.lMaxDescender;
#if 0
    NOTE(char_width);
    NOTE(char_height);
    NOTE(char_descender);
#endif
}

MRESULT MyWin::MsgCreate(PVOID, PCREATESTRUCT, int *)
{
    vscrollbar = WinWindowFromID(WinQueryWindow(window, QW_PARENT), 
				 FID_VERTSCROLL);
    GetFontInfo();
    return 0;
}

void MyWin::view_printf(char *buffer, const char *fmt, ...)
{
    BUGME(("MyWin::view_printf(..., \"%s\", ...)", buffer, fmt));
    va_list va;
    va_start(va, fmt);
    vsprintf(buffer, fmt, va);
    va_end(va);
    BUGME(("MyWin::view_printf: Message is \"%s\"", buffer));

    FreeView();
    view = new SimpleView;
    char *p;
    for (p = strtok(buffer, "\n\r"); p != NULL; p = strtok(NULL, "\n\r")) {
	BUGME(("MyWin::view_printf:  Line: \"%s\"", p));
	view->append(p);
    }
}

int main(int argc, char **argv)
{
    int i;
    for (i = 1; i < argc; i++) {
#ifdef DEBUG
	if (!strcmp(argv[i], "-debug")) {
	    debug_open_file();
	    continue;
	} else if (!strcmp(argv[i], "-debug_tail")) {
	    debug_open_cycle_file();
	    continue;
	}
#endif
    }
    BUGME(("main"));
    StdWin::StdStartup();
    MyWin win;
    win.Init(StdWin::hab);
    StdWin::StdMainLoop();
    win.Destroy();
    StdWin::StdExit();
#ifdef DEBUG
    if (fnote != NULL) fclose(fnote);
#endif
    return 0;
}

TextFonts::TextFonts ()
{
  normal_id = make_font ("Times New Roman", 120);
  bold_id = make_bold_font ("Times New Roman Bold", 120);
  italic_id = make_italic_font ("Times New Roman Italic", 120);
}

