/*
     Library PMIO: A library for stdio/pcio on a PM screen
     Copyright (C) 1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#define INCL_DOSSEMAPHORES
#define INCL_WIN
#define INCL_DOSPROCESS
#include <os2.h>

#include "stdwin.h"
#include "font.h"
#include "frame.h"
#include "pmio3.h"
#include "pmio4.h"


#include "pmio.h"

enum PMIO_ENUM {
  PMIO_WRITE,
  PMIO_TTYWRITE,
  PMIO_STDIOWRITE,
  PMIO_SCROLL,
  PMIO_GOTOXY,
  PMIO_CLREOL,
  PMIO_CLRSCR,
  PMIO_GETEVENT,
  PMIO_TEXTCOLOR,
  PMIO_TEXTBACKGROUND,
  PMIO_GETTEXTINFO,
  PMIO_SETTEXTINFO,
  PMIO_WINDOW,
  PMIO_DELAYOUTPUT,
  PMIO_NODELAYOUTPUT,
  PMIO_NL,
  PMIO_CR
};

static void winmsg (PMIO_ENUM e, MPARAM data)
{
  WinSendMsg (interface->window, WM_APP1, (MPARAM) e, data);
}

static void msgp (PMIO_ENUM e, const void *data) { winmsg (e, (MPARAM) data); }
static void msg (PMIO_ENUM e) { winmsg (e, 0); }
static void msgi (PMIO_ENUM e, int i) { winmsg (e, (MPARAM) i); }

/*****************************************/
/**** Events returned from the screen ****/
/*****************************************/

int pmio_getevent(Event *e)
{
  msgp (PMIO_GETEVENT, &e);
  if (e == 0)
    return 0;
  return 1;
}

static int event_availible = 0;
static Event event;

void pmio_poll()
{
  if (event_availible)
    return;
  Event *e = &event;
  if (pmio_getevent(e))
    event_availible = 1;
}

void pmio_kbdpoll()
{
  char c1, c2;
  for (;;) {
    pmio_poll();
    if (!event_availible)
      return;
    cvt_event (event, &c1, &c2);
    if (c1 != -1 || c2 != -1)
      return;
    event_availible = 0;
  }
}

int kbhit()
{
  pmio_kbdpoll();
  if (event_availible)
    return 1;
  return 0;
}


Event getevent (void)
{
  for(;;) {
    interface->reset ();
    pmio_poll ();
    if (event_availible)
      {
	event_availible = 0;
	return event;
      }
    else
      interface->wait ();
  }
}

char getch()
{
  static int second_key_availible = 0;
  static char second_key;
  char c;

  if (second_key_availible)
    {
      second_key_availible = 0;
      return second_key;
    }
  for(;;) 
    {
      interface->reset ();
      pmio_kbdpoll();
      if (event_availible)
	{
	  event_availible = 0;
	  cvt_event (event, &c, &second_key);
	  if (c == 0)
	    second_key_availible = 1;
	  return c;
	}
      else
	interface->wait ();
    }
}

/*****************************************/
/**** Other types of control messages ****/
/*****************************************/

struct pmio_write_struct
{
  int count;
  const char *msg;
};

void pmio_write(int count, const char *msg)
{
  pmio_write_struct pmio;
  pmio.count = count;
  pmio.msg = msg;
  msgp (PMIO_WRITE, &pmio);
}

void pmio_ttywrite (int count, const char *msg)
{
  pmio_write_struct pmio;
  pmio.count = count;
  pmio.msg = msg;
  msgp (PMIO_TTYWRITE, &pmio);
}

void pmio_stdiowrite (int count, const char *msg)
{
  pmio_write_struct pmio;
  pmio.count = count;
  pmio.msg = msg;
  msgp (PMIO_STDIOWRITE, &pmio);
}

void pmio_scroll (int count)
{
  if (!count)
    return;
  msgi (PMIO_SCROLL, count);
}

struct pmio_gotoxy_struct
{
  int x;
  int y;
};

void gotoxy(int x, int y)
{
  pmio_gotoxy_struct pmio;
  pmio.x = x;
  pmio.y = y;
  msgp (PMIO_GOTOXY, &pmio);
}

void clreol()
{
  msg (PMIO_CLREOL);
}

void clrscr ()
{
  msg (PMIO_CLRSCR);
}

void delayoutput ()
{
  msg (PMIO_DELAYOUTPUT);
}

void nodelayoutput ()
{
  msg (PMIO_NODELAYOUTPUT);
}

void textcolor (int color)
{
  msgi (PMIO_TEXTCOLOR, color);
}

void textbackground (int color)
{
  msgi (PMIO_TEXTBACKGROUND, color);
}

void gettextinfo (struct text_info *i)
{
  if (!i)
    return;
  msgp (PMIO_GETTEXTINFO, i);
}

void settextinfo (struct text_info *i)
{
  if (!i)
    return;
  msgp (PMIO_SETTEXTINFO, i);
}

struct wininfo
{
  int xmin, xmax, ymin, ymax;
};

void window (int xmin, int ymin, int xmax, int ymax)
{
  struct wininfo info;
  info.xmin = xmin;
  info.ymin = ymin;
  info.xmax = xmax;
  info.ymax = ymax;
  msgp (PMIO_WINDOW, &info);
}

/******************************************/
/**** Window Interface Message Handler ****/
/******************************************/

MRESULT MyWin::MsgApp1(MPARAM mp1, MPARAM mp2, int *)
{
  switch ((int) mp1)
    {
    case PMIO_WRITE:
      pmio_write_struct *pmio = (pmio_write_struct *) mp2;
      view->write (pmio->count, pmio->msg);
      break;

    case PMIO_TTYWRITE:
      pmio = (pmio_write_struct *) mp2;
      view->write_tty (pmio->count, pmio->msg);
      break;

    case PMIO_STDIOWRITE:
      pmio = (pmio_write_struct *) mp2;
      view->write_stdio (pmio->count, pmio->msg);
      break;

    case PMIO_SCROLL:
      view->scroll ((int) mp2);
      break;

    case PMIO_GOTOXY:
      pmio_gotoxy_struct *gotoxy = (pmio_gotoxy_struct *) mp2;
      view->gotoxy (gotoxy->x, gotoxy->y);
      break;

    case PMIO_CLREOL:
      view->clreol();
      break;

    case PMIO_CLRSCR:
      view->clrscr();
      break;

    case PMIO_DELAYOUTPUT:
      delay_output++;
      break;

    case PMIO_NODELAYOUTPUT:
      if (delay_output)
	delay_output--;
      break;

    case PMIO_GETEVENT:
      Event **e = (Event **) mp2;
      if (eventq_size == 0)
	*e = 0;
      else
	{
	  **e = eventq[eventq_tail++];
	  eventq_size--;
	  if (eventq_tail >= MAXEVENTS)
	    eventq_tail = 0;
	}
      break;

    case PMIO_TEXTCOLOR:
      view->textcolor ((int) mp2);
      break;

    case PMIO_TEXTBACKGROUND:
      view->textbackground ((int) mp2);
      break;

    case PMIO_GETTEXTINFO:
      view->gettextinfo ((struct text_info *) mp2);
      break;

    case PMIO_SETTEXTINFO:
      view->settextinfo ((const struct text_info *) mp2);
      break;

    case PMIO_WINDOW:
      struct wininfo *wininfo = (struct wininfo *) mp2;
      view->set_window (wininfo->xmin, wininfo->ymin,
			wininfo->xmax, wininfo->ymax);
      break;

    case PMIO_CR:
      view->cr ();
      break;

    case PMIO_NL:
      view->nl ();
      break;
    }

  flush_dirty_lines();
  return 0;
}
