/***********************************************************************/
/* UTIL.C - Utility routines                                           */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\util.c 1.4 1993/09/01 16:27:26 MH Interim MH $
*/

#include <stdio.h>
#include "the.h"
#include "proto.h"

/*--------------------------- common data -----------------------------*/
#define MAX_RECV 20
 static char *recv[MAX_RECV];
 static int recv_len[MAX_RECV];
 static int add_recv=(-1);
 static int retr_recv=(-1);
 static int num_recv=0;
/*-------------------------- external data ----------------------------*/
extern LINE *next_line,*curr_line;
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
extern char number_of_views;                   /* number of open files */
extern char current_file;                   /* pointer to current file */
extern WINDOW *foot,*error_window;
extern bool error_on_screen;
extern char *rec;
extern unsigned short rec_len;
extern char *pre_rec;
extern unsigned short pre_rec_len;
extern char mode_insert;        /* defines insert mode toggle */
extern char in_profile;    /* indicates if processing profile */
extern char *temp_cmd;
/*man***************************************************************************
NAME
     memrevne - search buffer reversed for NOT character

SYNOPSIS
     #include "the.h"

     short memrevne(buffer,known_char,max_len)
     char *buffer;
     char known_char;
     short max_len;

DESCRIPTION
     The memrevne function searches the buffer from the right for first
     character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne, strzne
*******************************************************************************/
#ifdef PROTO
short memrevne(char *buffer,char known_char,short max_len)
#else
short memrevne(buffer,known_char,max_len)
char *buffer;
char known_char;
short max_len;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int len=max_len;
/*--------------------------- processing ------------------------------*/
 for (--len; len>=0 && buffer[len]==known_char; len--);
 return(len);
}
/*man***************************************************************************
NAME
     meminschr - insert character into buffer

SYNOPSIS
     #include "the.h"

     char *meminschr(buffer,chr,location,max_length,curr_length)
     char *buffer;
     char chr;
     short location,max_length,curr_length;

DESCRIPTION
     The meminschr inserts the supplied 'chr' into the buffer 'buffer'
     before the 'location' specified. 'location' is an offset (0 based)
     from the start of 'buffer'.
     The 'buffer' will not be allowed to have more than 'max_length'
     characters, so if the insertion of the character causes the
     'max_length' to be exceeded, the last character of 'buffer' will
     be lost.

RETURN VALUE
     A pointer to the same 'buffer' as was supplied.

SEE ALSO
    meminsstr, memdelchr

*******************************************************************************/
#ifdef PROTO
char *meminschr(char *buffer,char chr,short location,
                short max_length,short curr_length)
#else
char *meminschr(buffer,chr,location,max_length,curr_length)
char *buffer;
char chr;
short location,max_length,curr_length;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
 for (i=curr_length;i > location;i--)
     if (i < max_length)
       buffer[i] = buffer[i-1];
 if (location < max_length)
    buffer[location] = chr;
 return(buffer);
}
/*man***************************************************************************
NAME
     meminsmem - insert memory into buffer

SYNOPSIS
     #include "the.h"

     char *meminsmem(buffer,str,len,location,max_length,curr_length)
     char *buffer;
     char *str;
     short len,location,max_length,curr_length;

DESCRIPTION
     The meminsmem function inserts the supplied 'str' into the buffer 'buffer'
     before the 'location' specified. 'location' is an offset (0 based)
     from the start of 'buffer'.
     The 'buffer' will not be allowed to have more than 'max_length'
     characters, so if the insertion of the string causes the
     'max_length' to be exceeded, the last character(s) of 'buffer' will
     be lost.

RETURN VALUE
     A pointer to the same 'buffer' as was supplied.

SEE ALSO
    meminschr

*******************************************************************************/
#ifdef PROTO
char *meminsmem(char *buffer,char *str,short len,short location,
                short max_length,short curr_length)
#else
char *meminsmem(buffer,str,len,location,max_length,curr_length)
char *buffer,*str;
short len,location,max_length,curr_length;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
 for (i=curr_length;i > location;i--)
     if (i+len-1 < max_length)
       buffer[i+len-1] = buffer[i-1];
 for (i=0;i<len;i++)
     if (location+i < max_length)
       buffer[location+i] = str[i];
 return(buffer);
}
/*man***************************************************************************
NAME
     memdelchr - delete character(s) from buffer

SYNOPSIS
     #include "the.h"

     char *memdelchr(buffer,location,curr_length,num_chars)
     char *buffer;
     short location,curr_length,num_chars;

DESCRIPTION
     The memdelchr deletes the supplied number of characters from the
     buffer starting at the 'location' specified. 'location' is an offset (0 based)
     from the start of 'buffer'.
     For each character deleted, what was the last character in buffer;
     based on 'curr_length' will be replaced with a space.

RETURN VALUE
     A pointer to the same 'buffer' as was supplied.

SEE ALSO
    meminschr

*******************************************************************************/
#ifdef PROTO
char *memdelchr(char *buffer,short location,short curr_length,
                short num_chars)
#else
char *memdelchr(buffer,location,curr_length,num_chars)
char *buffer;
short location,curr_length,num_chars;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
 for (i=location;i <curr_length;i++)
     if (i+num_chars >= curr_length)
        buffer[i] = ' ';
      else
        buffer[i] = buffer[i+num_chars];
 return(buffer);
}
/*man***************************************************************************
NAME
     strzne - search string for NOT character

SYNOPSIS
     #include "the.h"

     short strzne(str,chr)
     char *str;
     char ch;

DESCRIPTION
     The strzne function searches the string from the left for the first
     character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne, memrevne
*******************************************************************************/
#ifdef PROTO
short strzne(char *str,char ch)
#else
short strzne(str,ch)
char *str;
char ch;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int len;
 register int  i = 0;
/*--------------------------- processing ------------------------------*/
 len = strlen(str);
 for (; i<len && str[i]==ch; i++);
 if (i>=len)
    i = (-1);
 return(i);
}
/*man***************************************************************************
NAME
     memne - search buffer for NOT character

SYNOPSIS
     #include "the.h"

     short memne(buffer,chr,length)
     char *buffer;
     char chr;
     short length;

DESCRIPTION
     The memne function searches the buffer from the left for the first
     character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne, memrevne, strzne
*******************************************************************************/
#ifdef PROTO
short memne(char *buffer,char chr,short length)
#else
short memne(buffer,chr,length)
char *buffer;
char chr;
short length;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int  i = 0;
/*--------------------------- processing ------------------------------*/
 for (; i<length && buffer[i]==chr; i++);
 if (i>=length)
    i = (-1);
 return(i);
}
/*man***************************************************************************
NAME
     strzrevne - search string reversed for NOT character

SYNOPSIS
     #include "the.h"

     short strzrevne(str,chr)
     char *str;
     char ch;

DESCRIPTION
     The strzrevne function searches the string from the right for the
     first character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzne, memrevne
*******************************************************************************/
#ifdef PROTO
short strzrevne(char *str,char ch)
#else
short strzrevne(str,ch)
char *str;
char ch;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int len;
/*--------------------------- processing ------------------------------*/
 len = strlen(str);
 for (--len; len>=0 && str[len]==ch; len--);
 return(len);
}
/*man***************************************************************************
NAME
     strzreveq - search string reversed for character

SYNOPSIS
     short strzreveq(str,chr)
     char *str;
     char ch;

DESCRIPTION
     The strzreveq function searches the string from the right for the
     first character equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne
*******************************************************************************/
#ifdef PROTO
short strzreveq(char *str,char ch)
#else
short strzreveq(str,ch)
char *str,ch;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short len;
/*--------------------------- processing ------------------------------*/
 len = strlen(str);
 for (--len; len>=0 && str[len]!=ch; len--);
 return(len);
}
/*man***************************************************************************
NAME
     strtrunc - truncate leading and trailing spaces from string

SYNOPSIS
     #include "the.h"

     char *strtrunc(string)
     char *string;

DESCRIPTION
     The strtrunc truncates all leading and trailing spaces from 
     the supplied string.

RETURN VALUE
     A pointer to the original string, now truncated.

SEE ALSO

*******************************************************************************/
#ifdef PROTO
char *strtrunc(char *string)
#else
char *strtrunc(string)
char *string;
#endif
{
/*--------------------------- local data ------------------------------*/
 register int i;
 int pos;
/*--------------------------- processing ------------------------------*/
 pos = strzrevne(string,' ');
 if (pos == (-1))
    strcpy(string,"");
 else
    *(string+pos+1) = '\0';
 pos = strzne(string,' ');
 if (pos != (-1))
   {
    for (i=0;*(string+i)!='\0';i++)
       *(string+i) = *(string+i+pos);
    *(string+i) = '\0';
   }
 return(string);
}
/*man***************************************************************************
NAME
     memfind - finds a needle in a haystack respecting case and arbitrary
               characters if set.

SYNOPSIS
     int memfind(haystack,needle,hay_len,nee_len,case_ignore,arbsts,arb)
     char *haystack;                             string to be searched
     char *needle;         string to search for - may contain arbchars
     int hay_len;                                   length of haystack
     int nee_len;                                     length of needle
     bool case_ignore;                   TRUE if search to ignore case
     bool arbsts;         ON if need to check for arbitrary characters
     char arb;                 the actual arbitrary character

DESCRIPTION
     The memfind function locates a needle in a haystack. Both the needle
     and haystack may contain null characters. If case_ignore is TRUE,
     then upper and lower case characters are treated equal. If arbsts
     is ON, any arbitrary character, specified by arb, in needle, will
     match ANY character in the haystack.

RETURN VALUE
     The first occurrence (0 based) of needle in haystack, or (-1) if
     the needle does not appear in the haystack.
*******************************************************************************/
#ifdef PROTO
int memfind(char *haystack,char *needle,int hay_len,int nee_len,
            bool case_ignore,bool arbsts,char arb)
#else
int memfind(haystack,needle,hay_len,nee_len,case_ignore,arbsts,arb)
char *haystack;
char *needle;
int hay_len;
int nee_len;
bool case_ignore;
bool arbsts;
char arb;
#endif
/*--------------------------- local data ------------------------------*/
{
 register char c1,c2;
 register char *buf1,*buf2;
 register int i,j;
 int matches;
/*--------------------------- processing ------------------------------*/
 for (i=0;i<(hay_len-nee_len+1);i++)
    {
     buf1 = haystack+i;
     buf2 = needle;
     matches=0;
     for (j=0;j<nee_len;j++)
        {
         if (case_ignore)
           {
            if (isupper(*buf1))
               c1 = tolower(*buf1);
            else
               c1 = *buf1;
            if (isupper(*buf2))
               c2 = tolower(*buf2);
            else
               c2 = *buf2;
           }
         else
           {
            c1 = *buf1;
            c2 = *buf2;
           }
         if (arbsts)
           {
            if (c1 != c2 && c2 != arb)
               break;
            else
               matches++;
           }
         else
           {
            if (c1 != c2)
               break;
            else
               matches++;
           }
         ++buf1;
         ++buf2;
        }
     if (matches == nee_len)
        return(i);
    }
 return(-1);
}
/***********************************************************************/
#ifdef PROTO
short memcmpi(char *buf1,char *buf2,short len)
#else
short memcmpi(buf1,buf2,len)
char *buf1,*buf2;
short len;
#endif
/***********************************************************************/
/* Function  : Compares two memory buffers for equality;               */
/*             case insensitive. Same as memicmp() Microsoft C.        */
/* Parameters: buf1     - first buffer                                 */
/*             buf2     - second buffer                                */
/*             len      - number of characters to compare.             */
/* Return    : <0 if buf1 < buf2,                                      */
/*             =0 if buf1 = buf2,                                      */
/*             >0 if buf1 > buf2,                                      */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i;
/*--------------------------- processing ------------------------------*/
 for(i=0;i<len;i++)
   {
    char c1,c2;
    if (isupper(*buf1))
       c1 = tolower(*buf1);
    else
       c1 = *buf1;
    if (isupper(*buf2))
       c2 = tolower(*buf2);
    else
       c2 = *buf2;
    if (c1 != c2)
       return(c1-c2);
    ++buf1;
    ++buf2;
   }
 return(0);
}
/***********************************************************************/
#ifdef PROTO
char *make_upper(char *str)
#else
char *make_upper(str)
char *str;
#endif
/***********************************************************************/
/* Function  : Makes the supplied string uppercase.                    */
/*             Equivalent to strupr() on some platforms.      .        */
/* Parameters: str      - string to uppercase                          */
/* Return    : str uppercased                                          */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
 while(*str)
   {
    if (islower(*str))
       *str = toupper(*str);
    ++str;
   }
 return(str);
}
/*man***************************************************************************
NAME
     equal - determine if strings are equal up to specified length

SYNOPSIS
     unsigned short equal(con,str,min_len)
     char *con,*str;
     short min_len;

DESCRIPTION
     The equal function determines if a two strings are equal, irrespective
     of case, up to the length of the second string. The length of the
     second string must be greater than or equal to the specified minimum 
     length for the strings to be considered equal.

RETURN VALUE
     If 'equal' TRUE else FALSE.
*******************************************************************************/
#ifdef PROTO
unsigned short equal(char *con,char *str,short min_len)
#else
unsigned short equal(con,str,min_len)
char *con,*str;
short min_len;
#endif
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    equal");
#endif
 if (min_len == 0)
    return(FALSE);
/* if (memcmpi(con,str,min(strlen(str),strlen(con))) == 0*/
 if (memfind(con,str,min(strlen(str),strlen(con)),
     min(strlen(str),strlen(con)),TRUE,OFF,'\0') == 0
 &&  strlen(str) >= min_len
 &&  strlen(con) >= strlen(str))
   {
#ifdef TRACE
    trace_return();
#endif
    return(TRUE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(FALSE);
}
/***********************************************************************/
#ifdef PROTO
int valid_integer(char *str)
#else
int valid_integer(str)
char *str;
#endif
/***********************************************************************/
/* Function  : Checks that string contains only 0-9,- or +.            */
/* Parameters: *str     - string to be checked                         */
/* Return    : YES or NO                                               */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
   register int  i;
   int num_signs=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    valid_integer");
#endif
 for (i=0; i<strlen(str); i++)
    {
     if (*(str+i) == '-' || *(str+i) == '+')
        num_signs++;
     else
        if (!isdigit(*(str+i)))
          {
#ifdef TRACE
           trace_return();
#endif
           return(NO);
          }
    }
 if (num_signs > 1)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NO);
   }
#ifdef TRACE
 trace_return();
#endif
 return(YES);
}
/***********************************************************************/
#ifdef PROTO
int valid_positive_integer(char *str)
#else
int valid_positive_integer(str)
char *str;
#endif
/***********************************************************************/
/* Function  : Checks that string contains only 0-9, or +.             */
/* Parameters: *str     - string to be checked                         */
/* Return    : YES or NO                                               */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
   register int  i;
   int num_signs=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    valid_positive_integer");
#endif
 for (i=0; i<strlen(str); i++)
    {
     if (*(str+i) == '+')
        num_signs++;
     else
        if (!isdigit(*(str+i)))
          {
#ifdef TRACE
           trace_return();
#endif
           return(NO);
          }
    }
 if (num_signs > 1)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NO);
   }
#ifdef TRACE
 trace_return();
#endif
 return(YES);
}
/***********************************************************************/
#ifdef PROTO
int strzeq(char *str,char ch)
#else
int strzeq(str,ch)
char *str;
char ch;
#endif
/***********************************************************************/
/* Function  : Locate in ASCIIZ string, character                      */
/* Parameters: *str     - string to be searched                        */
/*             ch       - character to be searched for                 */
/* Return    : position in string of character - (-1) if not found     */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int len;
 register int  i = 0;
/*--------------------------- processing ------------------------------*/
 len = strlen(str);
 for (; i<len && str[i]!=ch; i++);
 if (i>=len)
    i = (-1);
 return(i);
}

/***********************************************************************/
#ifdef PROTO
char *strtrans(char *str,char oldch,char newch)
#else
char *strtrans(str,oldch,newch)
char *str;
char oldch,newch;
#endif
/***********************************************************************/
/* Function  : Translate all occurrences of oldch to newch in str      */
/* Parameters: *str     - string to be amendedd                        */
/*             oldch    - character to be replaced                     */
/*             newch    - character to replace oldch                   */
/* Return    : same string but with characters translated              */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int len;
 register int  i;
/*--------------------------- processing ------------------------------*/
 len = strlen(str);
 for (i=0;i<strlen(str); i++)
   {
    if (*(str+i) == oldch)
       *(str+i) = newch;
   }
 return(str);
}

#ifdef USE_VOID
/***********************************************************************/
#ifdef PROTO
void *ll_add(void *first,void *curr,unsigned short size)
#else
void *ll_add(first,curr,size)
void *first;
void *curr;
unsigned short size;
#endif
/***********************************************************************/
/* Adds a LINE to the current linked list after the current member.    */
/* PARAMETERS:                                                         */
/* first      - pointer to first LINE in linked list                   */
/* curr       - pointer to current LINE in linked list                 */
/* size       - size of a LINE item                                    */
/* RETURN:    - pointer to next LINE item                              */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 void *next=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    ll_add");
#endif

 if ((next=(void *)malloc(size)) != (void *)NULL)
    {
     if (curr == NULL)
        {
         first = next;
         next->next = NULL;
        }
     else
        {
         if (curr->next != NULL)
            curr->next->prev = next;
         next->next = curr->next;
         curr->next = next;
        }
     next->prev = curr;
    }
#ifdef TRACE
 trace_return();
#endif
 return(next);
}
/***********************************************************************/
#ifdef PROTO
void *ll_del(void *first,void *curr,int direction)
#else
void *ll_del(first,curr,direction)
void *first;
void *curr;
int direction;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 void *new_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    ll_del");
#endif
/*---------------------------------------------------------------------*/
/* Delete the only record                                              */
/*---------------------------------------------------------------------*/

 if (curr->prev == NULL && curr->next == NULL)
    {
     free(curr);
#ifdef TRACE
     trace_return();
#endif
     return(NULL);
    }
/*---------------------------------------------------------------------*/
/* Delete the first record                                             */
/*---------------------------------------------------------------------*/

 if (curr->prev == NULL)
    {
     curr->next->prev = NULL;
     new_curr = first = curr->next;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* Delete the last  record                                             */
/*---------------------------------------------------------------------*/

 if (curr->next == NULL)
    {
     curr->prev->next = NULL;
     new_curr = curr->prev;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }

/*---------------------------------------------------------------------*/
/* All others                                                          */
/*---------------------------------------------------------------------*/
 curr->prev->next = curr->next;
 curr->next->prev = curr->prev;
 if (direction == DIRECTION_FORWARD)
   new_curr = curr->next;
 else
   new_curr = curr->prev;

 free(curr);
 curr = new_curr;
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef PROTO
void ll_free(void *first)
#else
void ll_free(first)
void *first;
#endif
/***********************************************************************/
/* Free up all allocated memory until the last item in the linked-list */
/* PARAMETERS:                                                         */
/* first      - pointer to first line for the file                     */
/* RETURN:    - void                                                   */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 void *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    ll_free");
#endif
 curr = first;
 while(curr != NULL)
      {
       free(curr->line);
       if (curr->name != (char *)NULL)
          free(curr->name);
#if 0
       curr = curr->next;
       free(curr);
#else
       new_curr = curr->next;
       free(curr);
       curr = new_curr;
#endif
      }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void *ll_find(void *first,unsigned long row)
#else
void *ll_find(first,row)
void *first;
unsigned long row;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 void *curr;
 long i;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    ll_find");
#endif
 curr = first;
 if (curr != NULL)
    for(i=0L;i<row && curr->next != NULL; i++, curr=curr->next);
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
#else
/***********************************************************************/
#ifdef PROTO
LINE *lll_add(LINE *first,LINE *curr,unsigned short size)
#else
LINE *lll_add(first,curr,size)
LINE *first;
LINE *curr;
unsigned short size;
#endif
/***********************************************************************/
/* Adds a LINE to the current linked list after the current member.    */
/* PARAMETERS:                                                         */
/* first      - pointer to first LINE in linked list                   */
/* curr       - pointer to current LINE in linked list                 */
/* size       - size of a LINE item                                    */
/* RETURN:    - pointer to next LINE item                              */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 LINE *next=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    lll_add");
#endif

 if ((next=(LINE *)malloc(size)) != (LINE *)NULL)
    {
     if (curr == NULL)
        {
         first = next;
         next->next = NULL;
        }
     else
        {
         if (curr->next != NULL)
            curr->next->prev = next;
         next->next = curr->next;
         curr->next = next;
        }
     next->prev = curr;
    }
#ifdef TRACE
 trace_return();
#endif
 return(next);
}
/***********************************************************************/
#ifdef PROTO
LINE *lll_del(LINE **first,LINE **last,LINE *curr,int direction)
#else
LINE *lll_del(first,last,curr,direction)
LINE **first;
LINE **last;
LINE *curr;
int direction;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 LINE *new_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    lll_del");
#endif
/*---------------------------------------------------------------------*/
/* Delete the only record                                              */
/*---------------------------------------------------------------------*/
 if (curr->prev == NULL && curr->next == NULL)
    {
     free(curr);
     *first = NULL;
     if (last != NULL)
        *last = NULL;
#ifdef TRACE
     trace_return();
#endif
     return(NULL);
    }
/*---------------------------------------------------------------------*/
/* Delete the first record                                             */
/*---------------------------------------------------------------------*/
 if (curr->prev == NULL)
    {
     curr->next->prev = NULL;
     *first = new_curr = curr->next;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* Delete the last  record                                             */
/*---------------------------------------------------------------------*/
 if (curr->next == NULL)
    {
     curr->prev->next = NULL;
     new_curr = curr->prev;
     if (last != NULL)
        *last = curr->prev;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* All others                                                          */
/*---------------------------------------------------------------------*/
 curr->prev->next = curr->next;
 curr->next->prev = curr->prev;
 if (direction == DIRECTION_FORWARD)
   new_curr = curr->next;
 else
   new_curr = curr->prev;

 free(curr);
 curr = new_curr;
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef PROTO
LINE *lll_free(LINE *first)
#else
LINE *lll_free(first)
LINE *first;
#endif
/***********************************************************************/
/* Free up all allocated memory until the last item in the linked-list */
/* PARAMETERS:                                                         */
/* first      - pointer to first line for the file                     */
/* RETURN:    - NULL                                                   */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 LINE *curr;
 LINE *new_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    lll_free");
#endif
 curr = first;
 while(curr != NULL)
      {
       free(curr->line);
       if (curr->name != (char *)NULL)
          free(curr->name);
       new_curr = curr->next;
       free(curr);
       curr = new_curr;
      }
#ifdef TRACE
 trace_return();
#endif
 return((LINE *)NULL);
}
/***********************************************************************/
#ifdef PROTO
LINE *lll_find(LINE *first,unsigned long row)
#else
LINE *lll_find(first,row)
LINE *first;
unsigned long row;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 LINE *curr;
 long i;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    lll_find");
#endif
 curr = first;
 if (curr != NULL)
    for(i=0L;i<row && curr->next != NULL; i++, curr=curr->next);
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef PROTO
VIEW_DETAILS *vll_add(VIEW_DETAILS *first,VIEW_DETAILS *curr,unsigned short size)
#else
VIEW_DETAILS *vll_add(first,curr,size)
VIEW_DETAILS *first;
VIEW_DETAILS *curr;
unsigned short size;
#endif
/***********************************************************************/
/* Adds a VIEW_DETAILS to the current linked list after the current member.    */
/* PARAMETERS:                                                         */
/* first      - pointer to first VIEW_DETAILS in linked list                   */
/* curr       - pointer to current VIEW_DETAILS in linked list                 */
/* size       - size of a VIEW_DETAILS item                                    */
/* RETURN:    - pointer to next VIEW_DETAILS item                              */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 VIEW_DETAILS *next=(VIEW_DETAILS *)NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    vll_add");
#endif

 if ((next=(VIEW_DETAILS *)malloc(size)) != (VIEW_DETAILS *)NULL)
    {
     if (curr == (VIEW_DETAILS *)NULL)
        {
         first = next;
         next->next = (VIEW_DETAILS *)NULL;
        }
     else
        {
         if (curr->next != (VIEW_DETAILS *)NULL)
            curr->next->prev = next;
         next->next = curr->next;
         curr->next = next;
        }
     next->prev = curr;
    }
#ifdef TRACE
 trace_return();
#endif
 return(next);
}
/***********************************************************************/
#ifdef PROTO
VIEW_DETAILS *vll_del(VIEW_DETAILS **first,VIEW_DETAILS **last,VIEW_DETAILS *curr,int direction)
#else
VIEW_DETAILS *vll_del(first,last,curr,direction)
VIEW_DETAILS **first;
VIEW_DETAILS **last;
VIEW_DETAILS *curr;
int direction;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 VIEW_DETAILS *new_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    vll_del");
#endif
/*---------------------------------------------------------------------*/
/* Delete the only record                                              */
/*---------------------------------------------------------------------*/
 if (curr->prev == NULL && curr->next == NULL)
    {
     free(curr);
#ifdef TRACE
     trace_return();
#endif
     *first = curr = NULL;
     if (last != NULL)
        *last = NULL;
     return(NULL);
    }
/*---------------------------------------------------------------------*/
/* Delete the first record                                             */
/*---------------------------------------------------------------------*/
 if (curr->prev == NULL)
    {
     curr->next->prev = NULL;
     *first = new_curr = curr->next;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* Delete the last  record                                             */
/*---------------------------------------------------------------------*/
 if (curr->next == NULL)
    {
     curr->prev->next = NULL;
     new_curr = curr->prev;
     if (last != NULL)
        *last = curr->prev;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* All others                                                          */
/*---------------------------------------------------------------------*/
 curr->prev->next = curr->next;
 curr->next->prev = curr->prev;
 if (direction == DIRECTION_FORWARD)
   new_curr = curr->next;
 else
   new_curr = curr->prev;

 free(curr);
 curr = new_curr;
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef PROTO
DEFINE *dll_add(DEFINE *first,DEFINE *curr,unsigned short size)
#else
DEFINE *dll_add(first,curr,size)
DEFINE *first;
DEFINE *curr;
unsigned short size;
#endif
/***********************************************************************/
/* Adds a DEFINE to the current linked list after the current member.  */
/* PARAMETERS:                                                         */
/* first      - pointer to first DEFINE in linked list                 */
/* curr       - pointer to current DEFINE in linked list               */
/* size       - size of a DEFINE item                                  */
/* RETURN:    - pointer to next DEFINE item                            */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 DEFINE *next=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    dll_add");
#endif

 if ((next=(DEFINE *)malloc(size)) != (DEFINE *)NULL)
    {
     if (curr == NULL)
        {
         first = next;
         next->next = NULL;
        }
     else
        {
         if (curr->next != NULL)
            curr->next->prev = next;
         next->next = curr->next;
         curr->next = next;
        }
     next->prev = curr;
    }
#ifdef TRACE
 trace_return();
#endif
 return(next);
}
/***********************************************************************/
#ifdef PROTO
DEFINE *dll_del(DEFINE **first,DEFINE **last,DEFINE *curr,int direction)
#else
DEFINE *dll_del(first,last,curr,direction)
DEFINE **first;
DEFINE **last;
DEFINE *curr;
int direction;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 DEFINE *new_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    dll_del");
#endif
/*---------------------------------------------------------------------*/
/* Delete the only record                                              */
/*---------------------------------------------------------------------*/

 if (curr->prev == NULL && curr->next == NULL)
    {
     free(curr);
     *first = NULL;
     if (last != NULL)
        *last = NULL;
#ifdef TRACE
     trace_return();
#endif
     return(NULL);
    }
/*---------------------------------------------------------------------*/
/* Delete the first record                                             */
/*---------------------------------------------------------------------*/
 if (curr->prev == NULL)
    {
     curr->next->prev = NULL;
     *first = new_curr = curr->next;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* Delete the last  record                                             */
/*---------------------------------------------------------------------*/
 if (curr->next == NULL)
    {
     curr->prev->next = NULL;
     new_curr = curr->prev;
     if (last != NULL)
        *last = curr->prev;
     free(curr);
     curr = new_curr;
#ifdef TRACE
     trace_return();
#endif
     return(curr);
    }
/*---------------------------------------------------------------------*/
/* All others                                                          */
/*---------------------------------------------------------------------*/
 curr->prev->next = curr->next;
 curr->next->prev = curr->prev;
 if (direction == DIRECTION_FORWARD)
   new_curr = curr->next;
 else
   new_curr = curr->prev;

 free(curr);
 curr = new_curr;
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef PROTO
DEFINE *dll_free(DEFINE *first)
#else
DEFINE *dll_free(first)
DEFINE *first;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 DEFINE *curr;
 DEFINE *new_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    dll_free");
#endif
 curr = first;
 while(curr != (DEFINE *)NULL)
  {
   if (curr->def_params != NULL)
      free(curr->def_params);
   new_curr = curr->next;
   free(curr);
   curr = new_curr;
  }
#ifdef TRACE
 trace_return();
#endif
 return((DEFINE *)NULL);
}
#endif
/***********************************************************************/
#ifdef PROTO
LINE *add_line(LINE *first,LINE *curr,char *line,
               unsigned short len)
#else
LINE *add_line(first,curr,line,len)
LINE *first;
LINE *curr;
char *line;
unsigned short len;
#endif
/***********************************************************************/
/* Adds a member of the linked list for the specified file containing  */
/* the line contents and length.                                       */
/* PARAMETERS:                                                         */
/* first      - pointer to first line for the file                     */
/* curr       - pointer to current line for the file                   */
/* line       - contents of line to be added                           */
/* len        - length of line to be added                             */
/* RETURN:    - pointer to current item in linked list or NULL if error*/
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    add_line");
#endif
#ifdef USE_VOID
 next_line = (LINE *)ll_add((void *)first,(void *)curr,sizeof(LINE));
#else
 next_line = lll_add(first,curr,sizeof(LINE));
#endif
 if (next_line == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NULL);
   }
 curr_line = next_line;

 curr_line->line = (char *)malloc((len+1)*sizeof(char));
 if (curr_line->line == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NULL);
   }
 memcpy(curr_line->line,line,len);
 *(curr_line->line+len) = '\0'; /* for functions that expect ASCIIZ string */
 curr_line->length = len;
 curr_line->display = 0;
 curr_line->pre = (-1);
 curr_line->name = NULL;
#ifdef TRACE
 trace_return();
#endif
 return(curr_line);
}
/***********************************************************************/
#ifdef PROTO
LINE *delete_line(LINE *first,LINE *curr,int direction)
#else
LINE *delete_line(first,curr,direction)
LINE *first,*curr;
int direction;
#endif
/***********************************************************************/
/* Deletes a member of the linked list for the specified file.         */
/* PARAMETERS:                                                         */
/* first      - pointer to first line for the file                     */
/* curr       - pointer to current line for the file                   */
/* direction  - direction in which to delete.                          */
/* RETURN:    - pointer to current item in linked list or NULL if error*/
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    delete_line");
#endif
 if (curr->name != (char *)NULL)
    free(curr->name);
 free(curr->line);
#ifdef USE_VOID
 curr = (LINE *)ll_del((void *)first,(void *)curr,direction);
#else
 curr = lll_del(&first,NULL,curr,direction);
#endif
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef PROTO
long find_string(long start_line,short *start_col,char *needle,
                 int len,int direction,int offset,char kase)
#else
long find_string(start_line,start_col,needle,len,direction,offset,kase)
long start_line;
short *start_col;
char *needle;
int len;
int direction;
int offset;
char kase;
#endif
/***********************************************************************/
/* Finds a string from the line after/before the current line/col      */
/* containing needle. Zone settings are considered as is case.         */
/* PARAMETERS:                                                         */
/* start_line - current line number                                    */
/* start_col  - current column-offset from very beginning of buffer    */
/* needle     - string to search for                                   */
/* len        - length of needle (could contain nulls)                 */
/* direction  - forward or backward                                    */
/*              value of 1 if forward, -1 if backward                  */
/* offset     - offset from current line. For searches, this is the    */
/*              same as direction. For changes, this is 0. ie look at  */
/*              the current line before going on to the next/prev line.*/
/* kase       - indicates if case is to be considered when matching    */
/* RETURN:    - TARGET_NOT_FOUND if not found                          */
/*            - line number if found and column                        */
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINE *curr;
 long i;
 short real_start,real_end,loc;
 bool use_rec=FALSE;
 char *ptr;
 unsigned short haystack_len;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    find_string");
#endif

 post_process_line(CURRENT_VIEW->focus_line);
 if (len == 0)
    use_rec = TRUE;
 else
    if (*(needle+(len-1)) == ' ')
       use_rec = TRUE;
#ifdef USE_VOID
 curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,start_line+offset);
#else
 curr = lll_find(CURRENT_FILE->first_line,start_line+offset);
#endif
 for (i=0;;i+=(long)direction)
   {
    if ((curr->next == NULL && direction == DIRECTION_FORWARD)
    ||  (curr->prev == NULL && direction == DIRECTION_BACKWARD))
      break;
    if (use_rec)
      {
       memset(rec,' ',max_line_length);
       memcpy(rec,curr->line,curr->length);
       ptr = rec;
       haystack_len = max_line_length;
      }
    else
      {
       ptr = curr->line;
       haystack_len = curr->length;
      }
    real_end = min(haystack_len,CURRENT_VIEW->zone_end-1);
    real_start = max(*start_col,CURRENT_VIEW->zone_start-1);

    loc = memfind(ptr+real_start,needle,(real_end-real_start+1),
                  len,(kase == CASE_IGNORE) ? TRUE : FALSE,
                  CURRENT_VIEW->arbchar_status,
                  CURRENT_VIEW->arbchar_char);
    if (loc != (-1))
      {
       *start_col = loc+real_start;
       pre_process_line(CURRENT_VIEW->focus_line);
#ifdef TRACE
       trace_return();
#endif
       return(i+offset);
      }

/*---------------------------------------------------------------------*/
/* Once we get here, we have gone on to a new line. For searches, the  */
/* start_col has to be set back to 0 so that it can look from the      */
/* beginning of a new line.                                            */
/*---------------------------------------------------------------------*/
    *start_col = 0;
    if (direction == DIRECTION_FORWARD)
       curr = curr->next;
    else
       curr = curr->prev;
   }
#ifdef TRACE
 trace_return();
#endif
 pre_process_line(CURRENT_VIEW->focus_line);
 return(TARGET_NOT_FOUND);
}
/***********************************************************************/
#ifdef PROTO
void put_string(WINDOW *win,int row,int col,char *string,int len)
#else
void put_string(win,row,col,string,len)
WINDOW *win;
int row,col;
char *string;
int len;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    put_string");
#endif
 for (i=0;i<len;i++)
   {
    wmove(win,row,i+col);
    put_char(win,*(string+i),ADDCHAR);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void put_char(WINDOW *win,chtype ch,char add_ins)
#else
void put_char(win,ch,add_ins)
WINDOW *win;
chtype ch;
char add_ins;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern char NONDISPx;
/*--------------------------- local data ------------------------------*/
 chtype chr_attr,chr,attr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    put_char");
#endif
 chr_attr = ch;

 chr = chr_attr & A_CHARTEXT;
 if (chr > 126 || chr < 32)
   {
    attr = chr_attr & A_ATTRIBUTES;
    if (attr == colour[ATTR_FILEAREA])
       attr = colour[ATTR_BLOCK];
    else
       if (attr == colour[ATTR_CURLINE])
          attr = colour[ATTR_CBLOCK];
       else
          if (attr == colour[ATTR_CBLOCK])
             attr = colour[ATTR_CURLINE];
          else
             if (attr == colour[ATTR_BLOCK])
                attr = colour[ATTR_FILEAREA];
            else
                attr = colour[ATTR_MSGLINE];
    if (chr > 126)
       chr_attr = NONDISPx | attr;
    else
       if (chr < 32)
          chr_attr = ('@' + chr) | attr;
   }

 if (add_ins == ADDCHAR)
    waddch(win,chr_attr);
 else
    winsch(win,chr_attr);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
int set_up_windows(void)
#else
int set_up_windows()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
 char command_location;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    set_up_windows");
#endif
/*---------------------------------------------------------------------*/
/* This is a real kludge; to be fixed with revamp of screen handling...*/
/* If we already have windows for the view, delete them before         */
/* recreating them.                                                    */
/*---------------------------------------------------------------------*/
 for (i=0;i<VIEW_WINDOWS;i++)
    {
     if (CURRENT_VIEW->win[i] != (WINDOW *)NULL)
        delwin(CURRENT_VIEW->win[i]);
    }

 if (CURRENT_VIEW->cmd_line == 'T')   /* command top */
   {
    CURRENT_VIEW->command_row = CURRENT_SCREEN.origin_y+1;
    command_location = 1;
   }
 else
   {
    CURRENT_VIEW->command_row = CURRENT_SCREEN.origin_y+CURRENT_SCREEN.screen_rows-1;
    command_location = 0;
   }

 CURRENT_SCREEN.rows = CURRENT_SCREEN.screen_rows-2;
 CURRENT_SCREEN.top_row = CURRENT_SCREEN.origin_y+1+command_location;
 if (CURRENT_VIEW->prefix)
   {
    CURRENT_SCREEN.cols = CURRENT_SCREEN.screen_cols-PREFIX_WIDTH;
    if (CURRENT_VIEW->prefix == PREFIX_LEFT)
      {
       CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x+PREFIX_WIDTH;
       CURRENT_WINDOW_PREFIX = newwin(CURRENT_SCREEN.rows,PREFIX_WIDTH,
                                 CURRENT_SCREEN.top_row,CURRENT_SCREEN.origin_x);
      }
    else
      {
       CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x;
       CURRENT_WINDOW_PREFIX = newwin(CURRENT_SCREEN.rows,PREFIX_WIDTH,
                                 CURRENT_SCREEN.top_row,
                                 CURRENT_SCREEN.origin_x+CURRENT_SCREEN.cols);
      }
/*---------------------------------------------------------------------*/
/* Check that there was enough memory for the prefix window.           */
/*---------------------------------------------------------------------*/
    if (CURRENT_WINDOW_PREFIX == (WINDOW *)NULL)
      {
       display_error(30,(char *)"creating prefix window");
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY);
      }
   }
 else
   {
    CURRENT_SCREEN.cols = CURRENT_SCREEN.screen_cols;
    CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x;
   }
 
 CURRENT_WINDOW_MAIN = newwin(CURRENT_SCREEN.rows,
                               CURRENT_SCREEN.cols,
                               CURRENT_SCREEN.top_row,
                               CURRENT_SCREEN.top_col);
/*---------------------------------------------------------------------*/
/* Check that there was enough memory for the main window.             */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_MAIN == (WINDOW *)NULL)
   {
    display_error(30,(char *)"creating main window");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 CURRENT_WINDOW_COMMAND = newwin(1,CURRENT_SCREEN.screen_cols-PREFIX_WIDTH,
                                   CURRENT_VIEW->command_row,
                                   CURRENT_SCREEN.origin_x+PREFIX_WIDTH);
/*---------------------------------------------------------------------*/
/* Check that there was enough memory for the command window.          */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_COMMAND == (WINDOW *)NULL)
   {
    display_error(30,(char *)"creating command window");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 CURRENT_WINDOW_ARROW = newwin(1,PREFIX_WIDTH,
                               CURRENT_VIEW->command_row,
                               CURRENT_SCREEN.origin_x);
/*---------------------------------------------------------------------*/
/* Check that there was enough memory for the arrow window.            */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_ARROW == (WINDOW *)NULL)
   {
    display_error(30,(char *)"creating arrow window");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 CURRENT_WINDOW_IDLINE = newwin(1,CURRENT_SCREEN.screen_cols,
                                  CURRENT_SCREEN.origin_y,
                                  CURRENT_SCREEN.origin_x);
/*---------------------------------------------------------------------*/
/* Check that there was enough memory for the idline window.           */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_IDLINE == (WINDOW *)NULL)
   {
    display_error(30,(char *)"creating idline window");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 wattrset(CURRENT_WINDOW_ARROW,colour[ATTR_ARROW]);
 wattrset(CURRENT_WINDOW_MAIN,colour[ATTR_FILEAREA]);
 wattrset(CURRENT_WINDOW_IDLINE,colour[ATTR_IDLINE]);
 wmove(CURRENT_WINDOW_IDLINE,0,0);
 my_wclrtoeol(CURRENT_WINDOW_IDLINE);
#ifdef SUN
 notimeout(CURRENT_WINDOW_MAIN,TRUE);
 notimeout(CURRENT_WINDOW_COMMAND,TRUE);
#endif
 wattrset(CURRENT_WINDOW_COMMAND,colour[ATTR_CMDLINE]);
 wmove(CURRENT_WINDOW_COMMAND,0,0);
 my_wclrtoeol(CURRENT_WINDOW_COMMAND);
 if (CURRENT_VIEW->prefix)
   {
#ifdef SUN
    notimeout(CURRENT_WINDOW_PREFIX,TRUE);
#endif
    wattrset(CURRENT_WINDOW_PREFIX,colour[ATTR_PENDING]);
#if !defined(NO_KEYPAD)
    keypad(CURRENT_WINDOW_PREFIX,TRUE);
#endif
   }
#if !defined(NO_KEYPAD)
 keypad(CURRENT_WINDOW_COMMAND,TRUE);
 keypad(CURRENT_WINDOW_MAIN,TRUE);
#endif
 CURRENT_VIEW->current_window = WINDOW_COMMAND;
 CURRENT_VIEW->previous_window = WINDOW_MAIN;
 mvwaddstr(CURRENT_WINDOW_ARROW,0,0,"====> ");
 wnoutrefresh(CURRENT_WINDOW_ARROW);
 wnoutrefresh(CURRENT_WINDOW_COMMAND);
 wmove(CURRENT_WINDOW_COMMAND,0,0);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
void delete_windows(void)
#else
void delete_windows()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    delete_windows");
#endif

 if (CURRENT_VIEW->prefix)
    delwin(CURRENT_WINDOW_PREFIX);
 delwin(CURRENT_WINDOW_MAIN);
 delwin(CURRENT_WINDOW_COMMAND);
 delwin(CURRENT_WINDOW_IDLINE);
 delwin(CURRENT_WINDOW_ARROW);

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void pre_process_line(long line_number)
#else
void pre_process_line(line_number)
long line_number;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    pre_process_line");
#endif
#ifdef USE_VOID
 curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,line_number);
#else
 curr = lll_find(CURRENT_FILE->first_line,line_number);
#endif
 memset(rec,' ',max_line_length);
 memcpy(rec,curr->line,curr->length);
 rec_len = curr->length;
/*---------------------------------------------------------------------*/
/* Now set up the prefix command from the linked list...               */
/*---------------------------------------------------------------------*/
 if (curr->pre != (-1))
   {
    memset(pre_rec,' ',PREFIX_WIDTH);
    strcpy(pre_rec,CURRENT_VIEW->ppc[curr->pre].ppc_command);
    pre_rec_len = strlen(pre_rec);
    pre_rec[pre_rec_len] = ' ';
    pre_rec[PREFIX_WIDTH] = '\0';
   }
 else
   {
    memset(pre_rec,' ',PREFIX_WIDTH);
    pre_rec_len = 0;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void post_process_line(long line_number)
#else
void post_process_line(line_number)
long line_number;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool prefix_changed;
/*--------------------------- local data ------------------------------*/
 LINE *curr;
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    post_process_line");
#endif
/*---------------------------------------------------------------------*/
/* Find the specified line in the linked list...                       */
/*---------------------------------------------------------------------*/
#ifdef USE_VOID
 curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,line_number);
#else
 curr = lll_find(CURRENT_FILE->first_line,line_number);
#endif
/*---------------------------------------------------------------------*/
/* First copy the pending prefix command to the linked list.           */
/* Only do it if the prefix command has a value or there is already a  */
/* pending prefix command for that line.                               */
/*---------------------------------------------------------------------*/
/* if (pre_rec_len != 0
 || curr->pre != (-1)) */
 if (prefix_changed)
    add_prefix_command(curr,line_number,FALSE);
/*---------------------------------------------------------------------*/
/* If the line hasn't changed, return.                                 */
/*---------------------------------------------------------------------*/
 if (rec_len == curr->length && (memcmp(rec,curr->line,curr->length) == 0))
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Increment the alteration counters...                                */
/*---------------------------------------------------------------------*/
 if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return; /* should return rc */
   }
/*---------------------------------------------------------------------*/
/* Add the old line contents to the line recovery list.                */
/*---------------------------------------------------------------------*/
 add_to_recovery_list(curr->line,curr->length);
/*---------------------------------------------------------------------*/
/* Realloc the dynamic memory for the line if the line is now longer.  */
/*---------------------------------------------------------------------*/
 if (rec_len > curr->length)
                              /* what if realloc fails ?? */
    curr->line = (char *)realloc((void *)curr->line,(rec_len+1)*sizeof(char));

/*---------------------------------------------------------------------*/
/* Copy the contents of rec into the line.                             */
/*---------------------------------------------------------------------*/
 memcpy(curr->line,rec,rec_len);
 curr->length = rec_len;
 *(curr->line+rec_len) = '\0';
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
short blank_field(char *field)
#else
short blank_field(field)
char *field;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    blank_field");
#endif
 if (strzne(field,' ') == (-1))
   {
#ifdef TRACE
    trace_return();
#endif
    return(TRUE);                /* field is NULL or just contains spaces */
   }
#ifdef TRACE
 trace_return();
#endif
 return(FALSE);
}
/***********************************************************************/
#ifdef PROTO
void adjust_marked_lines(bool insert_line,long base_line,long num_lines)
#else
void adjust_marked_lines(insert_line,base_line,num_lines)
bool insert_line;
long base_line;
long num_lines;
#endif
/***********************************************************************/
{
/*---------------------------------------------------------------------*/
/* When lines are deleted, the base line is the first line in the file */
/* irrespective of the direction that the delete is done.              */
/*---------------------------------------------------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    adjust_marked_lines");
#endif
/*---------------------------------------------------------------------*/
/* If there are no marked lines in the current view, return.           */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW != CURRENT_VIEW)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 switch(insert_line)
   {
    case TRUE:/* INSERT */
         if (base_line < CURRENT_VIEW->mark_start_line)
           {
            CURRENT_VIEW->mark_start_line += num_lines;
            CURRENT_VIEW->mark_end_line += num_lines;
            break;
           }
         if (base_line >= CURRENT_VIEW->mark_start_line
         &&  base_line < CURRENT_VIEW->mark_end_line)
           {
            CURRENT_VIEW->mark_end_line += num_lines;
            break;
           }
         break;
    case FALSE:  /* DELETE */
         if (base_line <= CURRENT_VIEW->mark_start_line
         &&  base_line+num_lines-1L >= CURRENT_VIEW->mark_end_line)
           {
            CURRENT_VIEW->mark_start_line = (-1L);
            CURRENT_VIEW->mark_end_line = (-1L);
            MARK_VIEW = (VIEW_DETAILS *)NULL;
            break;
           }
         if (base_line+num_lines-1L < CURRENT_VIEW->mark_start_line)
           {
            CURRENT_VIEW->mark_start_line -= num_lines;
            CURRENT_VIEW->mark_end_line -= num_lines;
            break;
           }
         if (base_line > CURRENT_VIEW->mark_end_line)
           {
            break;
           }
         if (base_line+num_lines-1L > CURRENT_VIEW->mark_end_line)
           {
            CURRENT_VIEW->mark_end_line = base_line - 1L;
            break;
           }
         if (base_line < CURRENT_VIEW->mark_start_line)
           {
            CURRENT_VIEW->mark_start_line = base_line;
            CURRENT_VIEW->mark_end_line = base_line + 
                                         (CURRENT_VIEW->mark_end_line -
                                          (base_line + num_lines));
            break;
           }
         CURRENT_VIEW->mark_end_line -= num_lines;
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void adjust_pending_prefix(VIEW_DETAILS *view,bool insert_line,long base_line,long num_lines)
#else
void adjust_pending_prefix(view,insert_line,base_line,num_lines)
VIEW_DETAILS *view;
bool insert_line;
long base_line;
long num_lines;
#endif
/***********************************************************************/
{
/*---------------------------------------------------------------------*/
/* When lines are deleted, the base line is the first line in the file */
/* irrespective of the direction that the delete is done.              */
/*---------------------------------------------------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    adjust_pending_prefix");
#endif
/*---------------------------------------------------------------------*/
/* If there are no pending prefix commands in the view, return.        */
/*---------------------------------------------------------------------*/
 if (view->prefix_command_index == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 for (i=0;i<view->prefix_command_index;i++)
   {
    switch(insert_line)
      {
       case TRUE:/* INSERT */
            if (base_line < view->ppc[i].ppc_line_number)
              {
               view->ppc[i].ppc_line_number += num_lines;
               break;
              }
            break;
       case FALSE:  /* DELETE */
            if (base_line+num_lines-1L < view->ppc[i].ppc_line_number)
              {
               view->ppc[i].ppc_line_number -= num_lines;
               break;
              }
            if (base_line > view->ppc[i].ppc_line_number)
               break;
            clear_pending_prefix_command(i,(LINE *)NULL);
            break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
char case_translate(char key)
#else
char case_translate(key)
char key;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    case_translate");
#endif
 if (CURRENT_VIEW->case_enter == CASE_UPPER
 && islower(key))
   {
#ifdef TRACE
    trace_return();
#endif
    return(toupper(key));
   }
 if (CURRENT_VIEW->case_enter == CASE_LOWER
 && isupper(key))
   {
#ifdef TRACE
    trace_return();
#endif
    return(tolower(key));
   }
#ifdef TRACE
 trace_return();
#endif
 return(key);
}
/***********************************************************************/
#ifdef PROTO
void add_to_recovery_list(char *line,int len)
#else
void add_to_recovery_list(line,len)
char *line;
int len;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    add_to_recovery_list");
#endif
/*---------------------------------------------------------------------*/
/* Ignore if in profile.                                               */
/*---------------------------------------------------------------------*/
 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Ignore empty lines.                                                 */
/*---------------------------------------------------------------------*/
 if (len == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* First time through, set length array to (-1) to indicated unused.   */
/* This setup MUST occur before the freeing up code.                   */
/*---------------------------------------------------------------------*/
 if (add_recv == (-1))
   {
    for (i=0;i<MAX_RECV;i++)
       recv_len[i] = (-1);
    add_recv = 0;               /* set to point to next available slot */
   }
/*---------------------------------------------------------------------*/
/* Special case at end of program to free up dynamic memory allocated. */
/*---------------------------------------------------------------------*/
 if (len == (-1) && line == NULL)
   {
    for (i=0;i<MAX_RECV;i++)
       {
        if (recv_len[i] != (-1))
           free(recv[i]);
       }
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Now we are here, lets add to the array.                             */
/*---------------------------------------------------------------------*/
 if (recv_len[add_recv] == (-1))  /* haven't malloced yet */
   {
    if ((recv[add_recv] = (char *)malloc((len+1)*sizeof(char))) == NULL)
      {
       display_error(30,(char *)"");
#ifdef TRACE
       trace_return();
#endif
       return;
      }
   }
 else
   {
    if ((recv[add_recv] = (char *)realloc(recv[add_recv],(len+1)*sizeof(char))) == NULL)
      {
       display_error(30,(char *)"");
#ifdef TRACE
       trace_return();
#endif
       return;
      }
   }
 memcpy(recv[add_recv],line,len);
 recv_len[add_recv] = len;
 retr_recv = add_recv;
 add_recv = (++add_recv >= MAX_RECV) ? 0 : add_recv;
 num_recv = (++num_recv > MAX_RECV) ? MAX_RECV : num_recv;

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void get_from_recovery_list(int num)
#else
void get_from_recovery_list(num)
int num;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
 int num_retr = min(num,num_recv);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    get_from_recovery_list");
#endif
/*---------------------------------------------------------------------*/
/* Return error if nothing to recover.                                 */
/*---------------------------------------------------------------------*/
 if (retr_recv == (-1))
   {
    display_error(0,(char *)"0 line(s) recovered");
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Retrieve each allocated recovery line and put back into the body.   */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW->focus_line);
 for (i=0;i<num_retr;i++)
   {
    if (recv_len[retr_recv] != (-1))
      {
       insert_new_line(recv[retr_recv],recv_len[retr_recv],1L,get_true_line(),TRUE,FALSE);
       retr_recv = (--retr_recv < 0) ? num_recv-1 : retr_recv;
      }
   }

 sprintf(temp_cmd,"%d line(s) recovered",num_retr);
 display_error(0,(char *)temp_cmd);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
WINDOW *adjust_window(WINDOW *win,int tr,int tc,int lines,int cols)
#else
WINDOW *adjust_window(win,tr,tc,lines,cols)
WINDOW *win;
int tr;
int tc;
int lines;
int cols;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 WINDOW *neww;
 register int i;
 int begy,begx,maxy,maxx,y,x;
 int cl,cr,rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    adjust_window");
#endif
/*---------------------------------------------------------------------*/
/* Get existing details about the current window.                      */
/*---------------------------------------------------------------------*/
 getbegyx(win,begy,begx);
 getmaxyx(win,maxy,maxx);
 if (maxy == lines && maxx == cols)  /* same size */
   {
    if (begy == tr && begx == tc)   /* same position */
      {
#ifdef TRACE
       trace_return();
#endif
       return(win); /* nothing to do, return same window */
      }
    else /* need to move window */
      {
       rc = mvwin(win,tr,tc);
#ifdef TRACE
       trace_return();
#endif
       return(win);
      }
   }
/*---------------------------------------------------------------------*/
/* To get here the window needs to be resized.                         */
/*---------------------------------------------------------------------*/
 getyx(win,y,x);
 delwin(win);
 neww = newwin(lines,cols,tr,tc);
 if (neww != (WINDOW *)NULL)
    wmove(neww,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(neww);
}
/***********************************************************************/
#ifdef PROTO
void draw_cursor(int visible)
#else
void draw_cursor(visible)
int visible;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    draw_cursor");
#endif
#if !defined(SYSVR31curses) || defined(USE_EXTCURSES)
 return;
#else
 if (visible)
   {
    if (mode_insert)
       curs_set(2);   /* block cursor */
    else
       curs_set(1);   /* underline cursor */
   }
 else
    curs_set(0);      /* cursor off */
#ifdef TRACE
 trace_return();
#endif
 return;
#endif
}
/***********************************************************************/
#ifdef PROTO
int my_wclrtoeol(WINDOW *win)
#else
int my_wclrtoeol(win)
WINDOW *win;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
 short x,y,maxx,maxy;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    my_wclrtoeol");
#endif
 getyx(win,y,x);
 getmaxyx(win,maxy,maxx);
 for (i=x;i<maxx;i++)
   mvwaddch(win,y,i,' ');
 wmove(win,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(0);
}
