/*  $Id$
 *  
 *  File	file.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	File manipulation
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	16/10/93  (Created)
 *  		03/11/93  (Last modified)
 */ 


/*  This file contains a number of file manipulation functions
 *  that may or may not be different depending on the hardware or
 *  software environment being used.  At the moment the ANSI
 *  library functions are used.
 */

/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"


/*------------------------------------------------------------
 *  Opening files with extensions
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node fopenExtension
@deftypefun {FILE *} fopenExtension (char *@var{base}, char *@var{ext}, char *@var{mode}, bool @var{overwrite})
Returns a @code{FILE *} for file specified by @var{base} and @var{ext}.
The handle is returned by calling @code{fopen} with the file name
and the @var{mode} argument.  If @var{overwrite} is @code{FALSE} the
file may not exist before this call (an error message is generated
if the file exists).  Examples:
@example
fopenExtension("mybase", "cbi", "rb", TRUE)
fopenExtension("newbase", "cbi", "wb", FALSE)
@end example
The first example opens @code{mybase.cbi} for reading.  The second creates
@code{newbase.cbi} for writing, it may not yet exist.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

FILE *
fopenExtension(char *base, char *ext, char *mode, bool overwrite)
{ FILE *fd;
  char buf[MAX_FILE_NAME_SIZE+1];	/* Real file name */

  if ((strlen(base)+strlen(".")+strlen(ext)) > MAX_FILE_NAME_SIZE)
  { fprintf(stderr, "File name %s.%s too long\n", base, ext);
    exit(1);
  }

  strcpy(buf, base);
  strcat(buf, ".");
  strcat(buf, ext);

  if (overwrite == FALSE)
  { if ((fd = fopen(buf, "r")))		/* Determine if file exists */
    { setNameError(buf);
      setError(ERR_COULD_NOT_OVERWRITE_FILE);
      fclose(fd);
      return NULL;
    }
  }
  
  if ((fd = fopen(buf, mode)) == NULL)
  { setNameError(buf);
    if (mode[0] == 'r')
      setError(ERR_COULD_NOT_OPEN_READ);
    else
      setError(ERR_COULD_NOT_OPEN_WRITE);
    return NULL;
  }
  
  return fd;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node fopenCbuffFile
@deftypefun {FILE *} fopenCbuffFile (char *@var{name}, char *@var{mode})
Returns a @code{FILE *} for the file specified by @var{name}.
@code{fopenCbuffFile} is used for ``standard'' CBUFF files.  It first
tries to find the file in the current directory and, if not present,
searches the CBUFF directory (as specified by @code{CBUFFDIR} environment
variables).
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

FILE *
fopenCbuffFile(char *name, char *mode)
{ FILE *fd;
  char buf[MAX_FILE_NAME_SIZE+1];	/* Real file name */
  char *dir;				/* CBUFF directory */

  if (strlen(name) > MAX_FILE_NAME_SIZE)
  { fprintf(stderr, "File name %s too long\n", name);
    exit(1);
  }

  if ((fd = fopen(name, mode)))		/* Find file in current dir */
    return fd;

  if ((dir = getCbuffDirectory()))	/* Find file in CBUFF directory */
  { strcpy(buf, dir);
    strcat(buf, name);
  
    if ((fd = fopen(buf, mode)))
      return fd;
  }
  
  setNameError(name);
  if (mode[0] == 'r')
    setError(ERR_COULD_NOT_OPEN_READ);
  else
    setError(ERR_COULD_NOT_OPEN_WRITE);

  return NULL;
}


/*------------------------------------------------------------
 *  Reading and writing binary data
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node readLong
@deftypefun long readLong (FILE *@var{fd})
Read a @code{long} from the given file.  The function takes the ChessBase
byte order into account.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

long
readLong(FILE *fd)
{ unsigned char d[4];
  long val;

  if (fread(d,1,4,fd) != 4)
  { setError(ERR_UNEXPECTED_EOF);
    reportError(stderr);
  }

             val  = d[0];
  val <<= 8; val |= d[1];
  val <<= 8; val |= d[2];
  val <<= 8; val |= d[3];

  return val;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node writeLong
@deftypefun void writeLong (long val, FILE *@var{fd})
Writes a @code{long} to the given @var{fd}, taking into account the
byte order used by ChessBase.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
writeLong(long val, FILE *fd)
{ unsigned char d[4];

  d[0] = (unsigned char) (val >> 24);
  d[1] = (unsigned char) (val >> 16);
  d[2] = (unsigned char) (val >> 8);
  d[3] = (unsigned char) (val >> 0);

  fwrite(d, 1, 4, fd);
}


/*------------------------------------------------------------
 *  CBUFF directory
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node getCbuffDirectory
@deftypefun {char *} getCbuffDirectory ()
Returns a string containing the name of the CBUFF directory (as specified
with the @code{CBUFFDIR} environment variable.  @xref{CBUFF directory}.
Some environments may not define the @code{getenv} function that is
necessary to implement this function (see @code{HAS_GETENV} in the file
@code{machine.h} for details).
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

char *
getCbuffDirectory()
{
#if HAS_GETENV
  return getenv("CBUFFDIR");
#else
  return NULL;
#endif
}
