

Please note:

	This "SAMPLE.C" file is a concatenation of a few source code files
	from the Hobbit House Software hhstring library source code. It is
	presented as part of the demo package to demonstrate the detailed
	documentation which comes with the source code when you buy the
	commercial version of hhstring.

	The documentation portion of the source code files is presented
	in the manual.doc file for the demo version of hhstring. For the
	commercial version, the source code itself will also, at the
	owner's option, be included in the manual.doc file.

	Print out this file, or examine it with an on-line file viewer, to
	see the quantity/quality of commenting which comes with the source
	code for hhstring.



/*                       Ŀ            strcenternew
Ĵ Hobbit House Software Ŀ
			 
			  copyright(c) 1992, 1993

function:	strcenternew (STRing CENTER to NEW string)

KWIC:      	%center a string

syntax: 	#include "hhstring.h"
		char *strcenternew(char *newstring, char *instring)

description:	evens out the blanks around a string so that the non-blank
		text is centered. TABs are treated as non-blank characters,
		so this function is not necessarily useful for strings with
		leading and/or trailing TABs. The result is created in the
		new string with the input string being left untouched.

returns:	a pointer to the new string

comments:	this function assumes that the input string contains
		whitespace on one or both sides but there is no problem
		if it doesn't. If there is no whitespace at all, this
		function just moves instring to newstring.

		The string pointed to by the return value of this function
		is in the calling function's space. The return value is
		provided as a convenience, not a necessity.

		Setting newstring = instring will not work with this
		function and overlapping newstring/instring may or may
		not work so if you must overlap, do it cautiously.

keywords:	string, center, justify, new

key sentence:	centers a string and puts the result in a new string

see also:	str< lf | rt >just{new}		strcenter{n}{new}



usage example:	compiled/executed/verified on 12/13/92

	char *Astring = "  2 left blanks, 11 right blanks           ";
	char *Bstring = "            12 left blanks,  0 right blanks";
	char Cstring[80];
	char Dstring[80];
	strcenternew(Cstring, Astring);
	strcenternew(Dstring, Bstring);
	printf("\n-->%s<--\n-->%s<--", Cstring, Dstring);

	-->       2 left blanks, 11 right blanks      <--
	-->      12 left blanks,  0 right blanks      <--
			 Ŀ
Ĵ Hobbit House Software 
			 			*/


#include "hhstring.h"

char *strcenternew(			/* return a pointer to newstring*/
	char *newstring,		/* the new string		*/
	char *instring)			/* the string to be centered	*/
{ char *ptr1;				/* destination pointer		*/
  char *ptr2;				/* source pointer		*/
  int ii;				/* index			*/
  int k1;				/* extra shift on left		*/
  int k2;				/* extra shift on right		*/
  int nleft;				/* # of blanks on the left	*/
  int nright;				/* # of blanks on the right	*/
  int ncent;				/* # of blanks needed on the	*/
					/*   left to center text	*/
  int correct;				/* correction for ncent "error"	*/
  int nshifts;				/* number of shifts needed	*/
  int slen;				/* slen = the length of		*/
  slen = strlen(instring);		/*   the input string		*/
  for (ii = 0; ii < slen; ii++)		/* get nleft = the		*/
    if (instring[ii] != ' ')		/*   number of blanks on	*/
      break;				/*     the left of the		*/
  nleft = ii;				/*	 input string		*/
  for (ii = 0; ii < slen; ii++)		/* get nright = the		*/
    if (instring[slen - ii - 1] != ' ')	/*   number of blanks on	*/
      break;				/*     the right of the		*/
  nright = ii;				/*	 input string		*/
  ncent = (nleft + nright) / 2;		/* ncent = evenly divided space	*/

  /*									*/
  /* If the left side has more blanks than the right side, then the	*/
  /* string needs to be shifted to the left as it is moved into the new	*/
  /* string. Otherwise, it needs to be shifted to the right as it is	*/
  /* moved into the new string. The number of characters which have to	*/
  /* be moved from instring into newstring (calculated below as		*/
  /* "nshifts") is the string length (slen) minus the number of blanks	*/
  /* on the sides (nleft + nright). Once the body of the string has	*/
  /* been moved into newstring, an appropriate portion of newstring	*/
  /* must be blanked out on each side. This requires a correction	*/
  /* factor to accommodate the fact that ncent may or may not have	*/
  /* "thrown away half of a character" due to integer arithmetic        */
  /* roundoff error. This correction factor is the total number of	*/
  /* blanks (from both sides) taken modulo 2, and is calculated below	*/
  /* as "correct". As can be seen in the code, the correction is	*/
  /* applied to the left margin if the shift was to the	right and is	*/
  /* applied to the right margin if the shift was to the left.		*/
  /*									*/
  nshifts = slen - (nleft + nright);	/* get number of shifts needed	*/
  correct = (nleft + nright) % 2;	/* to correct for ncent "error"	*/
  /*	-	-	-	-	-	-	-	-	*/
  if (ncent > nleft)			/* if shift right is needed:	*/
  { ptr1 = newstring + slen - ncent - 1;/* destination is to the right	*/
    ptr2 = instring + slen - nright - 1;/*   of the source		*/
    for (ii = 0; ii < nshifts; ii++)	/* move the "text" from		*/
      *(ptr1 - ii) = *(ptr2 - ii);	/*   instring to the proper	*/
					/*     place in the new string	*/
    k1 = correct;			/* set the margin correction	*/
    k2 = 0;				/*   for the left side		*/
  }					/* end if			*/
  /*	-	-	-	-	-	-	-	-	*/
  else					/* since shift left is needed:	*/
  { ptr1 = newstring + ncent;		/* destination is to the left	*/
    ptr2 = instring + nleft;		/*   of the source		*/
    for (ii = 0; ii < nshifts; ii++)	/* move the "text" from		*/
      *(ptr1 + ii) = *(ptr2 + ii);	/*   instring to the proper	*/
					/*     place in the new string	*/
    k1 = 0;				/* set the margin correction	*/
    k2 = correct;			/*   for the right side		*/
  }					/* end else			*/
  /*	-	-	-	-	-	-	-	-	*/
  for (ii = 0; ii < ncent + k1; ii++)	/* create the left margin	*/
    newstring[ii] = ' ';		/*   of blanks			*/
  for (ii = slen-ncent-k2;              /* create the			*/
		ii < slen;ii++)		/*   right margin		*/
    newstring[ii] = ' ';		/*     of blanks		*/
  newstring[slen] = '\0';		/* terminate the new string	*/
  return(newstring);			/* return pointer to new string */
}					/* end function strcenternew	*/


/*                       Ŀ              isfilename
Ĵ Hobbit House Software Ŀ
			 
			  copyright(c) 1992, 1993

function:	isfilename (IS this a valid FILE NAME) (not a path name)

KWIC:      	determine the %validity of a DOS %file name (but not a
		DOS path name)

syntax:		#include <stddef.h>
		#include "hhstring.h"
		int isfilename(char *filename)

description:	examines the filename to see whether or not it is a
		valid DOS file name. Paths are NOT allowed. This merely
		checks for illegal characters or too-long names or
		extents.

returns:	 0 for valid file name
		-1 if name contains * or ? but is otherwise OK
		-2 if name contains invalid characters
		-3 if name is too long
		-4 if extent is too long
		-5 if name is OK other than a meaningless
			use of '*'

comments:	none

keywords:	filename

key sentence:	examines a DOS file name (NOT a path name) and reports
		on whether or not it is valid and if not, gives some
		information about why not

see also:	none



usage example: 	compiled/executed/verified on 2/1/93

		printf("\n%d", isfilename("12345678.9ab"));
		printf("\n%d", isfilename("*.bak"));
		printf("\n%d", isfilename("123<5>78.9ab"));
		printf("\n%d", isfilename("123456789.ab"));
		printf("\n%d", isfilename("1234567.89ab"));
		printf("\n%d", isfilename("*234*.abc"));

		0
		-1
		-2
		-3
		-4
		-5

			 Ŀ
Ĵ Hobbit House Software 
			 			*/



#include <stddef.h>
#include "hhstring.h"

int isfilename(			/* return the results flag		*/
	char *filename)		/* file name string to check		*/
{ char *badchars;		/* pointer to string of invalid chars	*/
  char *dotptr;			/* pointer to the '.' in the file name	*/
  char *tempp;			/* scratch pointer variable		*/
  size_t tempi; 		/* scratch integer variable		*/
  size_t ii;			/* index				*/
  int wildflag = 0;		/* flag for whether or not wild card	*/
				/*   characters are present in the	*/
				/*     file name			*/
  size_t len;			/* the length of the file name		*/
  /*									*/
  /* executable code begins						*/
  /*									*/
  badchars = " =;\t\n\"\\"	/* this is a list of the characters     */
		"/|<>+[],.:";	/*   which should NOT be in file names  */
  len = strlen(filename);		/* get the length of the name	*/
  for (ii = 0; ii < len; ii++)		/* if any of the characters	*/
    if (filename[ii] < 33 ||		/*   in the file name are	*/
      (unsigned char)filename[ii] > 127)/*     outside the acceptable	*/
      return(-2);			/*	 range, return -2	*/


  if ((dotptr = strchr(filename, '.'))	/* if there is no period in the	*/
	 == NULL)			/*   file name then:		*/
  { tempi = strcspn(filename, badchars);/* if there are any bad		*/
    if (tempi != len)			/*   characters, then		*/
      return(-2);			/*     return -2		*/
    if (len > 8)			/* if the name portion is too	*/
      return(-3);			/*   long, return -3		*/
  }					/* end no period in name	*/
  else					/* there is a '.' in the file	*/
					/*   name, so:			*/
  { *dotptr = '\0';			/* end the string at the '.'	*/
    if (strchr(dotptr+1, '.') != NULL)	/* if there is ANOTHER		*/
    { *dotptr = '.';			/*   '.', restore the '.'	*/
      return(-2);			/*     and return -2		*/
    }					/*	 to show bad char	*/
    len = strlen(filename);		/* if the name is too		*/
    if (len > 8)			/*   long, then			*/
    { *dotptr = '.';			/*     restore the '.' and	*/
      return(-3);			/*	 return -3		*/
    }					/* end if name too long		*/
    ii = strindex(dotptr, filename);	/* get the length of the	*/
					/*   name portion		*/
    if (strcspn(filename,		/* if the name portion has	*/
		badchars) != ii)	/*   any bad characters,	*/
    { *dotptr = '.';			/*     restore the dot		*/
      return(-2);			/*	 and return -2		*/
    }					/* end if name has bad chars	*/
    if (strlen(dotptr+1) > 3)		/* if the extent portion is	*/
    { *dotptr = '.';			/*   too long, restore		*/
      return(-4);			/*     the dot and		*/
    }					/*	 return -4		*/
    *dotptr = '.';			/* put the '.' back in the name	*/
  }					/* end else period in name	*/
  if (strchr(filename, '?') != NULL)	/* if the file name contains a	*/
    wildflag = 1;			/*   '?', then set wildflag	*/
  tempp = strchr(filename, '*');	/* check for '*' in name	*/
  if (tempp != NULL)			/* if there is one:		*/
  { wildflag = 1;			/* set the wildcard flag	*/
    if (*(tempp + 1) != '.'		/* if the following character	*/
     && *(tempp + 1) != '\0')		/*   is not the '.' or the end	*/
      return(-5);			/*     of the string, return -5	*/
    else				/* the first '*' was OK, so:	*/
    { tempp = strchr(tempp+1, '*');	/* if there is a second		*/
      if (tempp != NULL)		/*   '*' in the name and it is	*/
	if (*(tempp + 1) != '\0')	/*     not the last character	*/
	  return(-5);			/*	 in the name, then	*/
    }					/*	   return -5		*/
  }					/* end if there is a '*'	*/


  if (wildflag)				/* if we got this far with no	*/
    return(-1);				/*   error exit, but there was	*/
					/*     a wildcard, return -1	*/
  return(0);				/* alles ist gut; flag it happy	*/
}					/* end function isfilename	*/


/*                       Ŀ            strmidsetnew
Ĵ Hobbit House Software Ŀ
			 
			  copyright(c) 1992, 1993

function:	strmidsetnew (STRing MIDdle SET to NEW string)

KWIC:      	%set the %middle of a string to a given character

syntax: 	#include "hhstring.h"
		char *strmidsetnew(char *newstring, char *instring,
				    int start, int nchar, char setchar)

description:	create newstring equal to instring except that from
		"start" characters beyond the beginning to "start" +
		"nchar" characters beyond the beginning, newstring
		consists of "setchar".

returns:	a pointer to newstring

comments:	if start+nchar would go beyond the end of instring, the
		entire right side of newstring will consist of setchar,
		but newstring will have the same length as instring. If
		start itself would start beyond the end of the instring,
		then newstring will be identical to instring.

		The string pointed to by the return value of this function
		is in the calling function's space. The return value is
		provided as a convenience, not a necessity.

		Setting newstring = instring WILL NOT WORK with this
		function and overlapping newstring/instring will not work
		in general although in some special cases it will. Analyse
		the source code if you need more details.

keywords:	string, middle, set, new

key sentence:	changes a specified number of characters in a string,
		starting at a specified point in the string, to a
		specified set character, and puts the results into a
		new string

see also:	str{ lf | mid | rt }set{new}



usage example:	compiled/executed/verified on 2/4/93

		char Bstring[80];
		char *Astring = "Name Date Amount Number";
		strmidsetnew(Bstring, Astring, 10, 6, '-');
		printf("%s", Bstring);

		Name Date ------ Number

			 Ŀ
Ĵ Hobbit House Software 
			 			*/


#include "hhstring.h"

char *strmidsetnew(			/* return a pointer to newstring*/
	    char *newstring,		/* new string to be created	*/
	    char *instring,		/* string to be set		*/
	    int start,			/* where the "middle" starts	*/
	    int nchar,			/* number of set chars to use	*/
	    char setchar)		/* the set character		*/
{ int ii;				/* index			*/
  int slen;				/* slen = the length		*/
  slen = strlen(instring);		/*   of the input string	*/
  if (start > slen)			/* if start is beyond the end	*/
  { strcpy(newstring, instring);	/*   of instring, put instring	*/
    return(newstring);			/*     into newstring and return*/
  }					/*	 a pointer to newstring	*/
  nchar = (start + nchar > slen)	/* limit nchar so that the set	*/
	? slen-start : nchar;		/*   does exceed end of string	*/
  for (ii = 0; ii < start; ii++)	/* fill left of newstring	*/
    newstring[ii] = instring[ii];	/*   from instring		*/
  for (ii = 0; ii < nchar; ii++)	/* set from start to start +	*/
    newstring[start + ii] = setchar;	/*   nchar to setchar		*/
  for (ii=start+nchar; ii < slen; ii++)	/* fill right of newstring	*/
    newstring[ii] = instring[ii];	/*   from instring		*/
  newstring[slen] = '\0';		/* put terminator in newstring	*/
  return(newstring);			/* return pointer to newstring	*/
}					/* end function strmidsetnew	*/


/*                       Ŀ                strxcatn
Ĵ Hobbit House Software Ŀ
			 
			  copyright(c) 1992, 1993

function:	strxcatn (STRing eXtended conCATenate, N-char limit)

KWIC:      	%concatenate %multiple strings but limit the %size of the
		result to n characters

syntax: 	#include <stddef.h>
		#include <stdarg.h>
		#include "hhstring.h"
		char *strxcatn(char *instring, int len, ... )

		note: ... is a variable list of string pointers which,
		for this function, must be terminated with a pointer to
		a NULL string. The constant NULL is recommended; see the
		code example below.

description:	concatenates a variable number of strings onto the
		input string, up to n characters. If the n chars are
		reached, then a terminating '\0' is placed on the string
		at the nth position.

returns:	a pointer to the input string

comments:	The input string must have enough memory allocated to
		handle all of the concatenation, else there be dragons.

keywords:	string, concatenate, size

key sentence:	concatenates multiple strings but limits the result to a
		fixed size

see also:	strxcat{n}



usage example:	compiled/executed/verified on 2/25/93

		char str1[80] = "";
		char stra[10] = "this, ";
		char strb[10] = "that, ";
		char strc[20] = "and the other";
		strxcatn(str1, 20, stra, strb, strc, NULL);
		printf("\n%s<--", str1);

		this, that, and the <--		(note 20-char size limit)

			 Ŀ
Ĵ Hobbit House Software 
			 			*/

#include <stddef.h>
#include <stdarg.h>
#include "hhstring.h"

char *strxcatn(				/* return pointer to string	*/
	char *instring,			/* string to get it all		*/
	int len,			/* how long it can be		*/
	... )				/* variable arg list		*/
{ va_list va_argslist;			/* declare local arg list	*/
  int ii;				/* index			*/
  int inlen;				/* length of accumulated string	*/
  char *thisarg;			/* declare storage for		*/
					/*   individual arguments	*/
  va_start(va_argslist, len);		/* start the va_arg utility	*/
  while (NULL !=  (thisarg =		/* as long as there are		*/
	va_arg(va_argslist, char*)))	/*   more arguments left:	*/
  { if (strlen(instring)		/* if the string won't get	*/
	+ strlen(thisarg) < (size_t)len)/*   too long, concatenate	*/
      strcat(instring, thisarg);	/*     this one			*/
    else				/* if the string would get too	*/
					/*   long, then:		*/
    { inlen = strlen(instring);		/* get length so far		*/
      for (ii = 0; ii < len -		/* put in enough chars		*/
		inlen; ii++)		/*   to just fill up the	*/
	instring[inlen+ii] =		/*     resulting string to	*/
			thisarg[ii];	/*	 the allowed length,	*/
      instring[inlen + ii] = '\0';	/*	   terminate the	*/
      break;				/*	     string and then	*/
    }					/*	       let's get the	*/
  }					/*		 hell gone	*/
  va_end(va_argslist);			/* terminate va_arg as required	*/
  return(instring);			/* return pointer to string	*/
}					/* end function strxcatn	*/

/*                       Ŀ             fnconv_1to2
Ĵ Hobbit House Software Ŀ
			 
			  copyright(c) 1992, 1993

function:	fnconv_1to2 (File Name CONVersion, type 1 TO type 2)

KWIC:      	%converts %file name format

syntax:		#include "hhstring.h"
		void fnconv_1to2(char *fname);

description:	converts fname from a type 1 format to a type 2 format.
		The designation of "type 1" and "type 2" to these two
		"types" is purely arbitrary and has no relationship to
		any naming convention outside this library

		file format 1 is "name.ext" with a period and no embedded
			      spaces
		file format 2 is "name    ext", with exactly 12 characters
			      including however many spaces it takes in
			      the middle to pad the name and however many
			      it takes at the end to pad the extent

		examples:
			format 1:	"sub1.h"
			format 2:	"sub2     h  "

returns:	a pointer to the converted file name

comments:	the file name must have enough space reserved to
		accommodate the expanded name under the type 2 format.

		Since the file name string is modified in the calling
		program's space, use of this function's return value
		is optional.

keywords:	filename, convert

key sentence:   converts a file name from the format containing a period
		but no blanks to the one with exactly 12 characters and
		no period

see also:	fnconv_2to1



usage example:	compiled/executed/verified on 1/17/93

		char *filename = "12345678.123";
		printf("\n-->%s<--", filename);
		strcpy(filename, "sub1.h");
		fnconv_1to2(filename);
		printf("\n-->%s<--", filename);

		-->12345678.123<--
		-->sub1     h  <--

			 Ŀ
Ĵ Hobbit House Software 
			 			*/


#include "hhstring.h"

char *fnconv_1to2(			/* return pointer to file name	*/
	char *fname)			/* file name to convert		*/
{ int length;				/* length of name		*/
  int tempi;				/* scratch integer variable	*/
  char *tempp;				/* scratch pointer		*/
  length = strlen(fname);		/* get the length of the name	*/
  tempp = strchr(fname, '.');		/* get the '.' position		*/
  if (tempp == NULL)			/* if there is no '.', then	*/
    strrtpad(fname, 12-length, ' ');	/*   just pad name to 12 chars	*/
  else					/* there is a '.', so		*/
  { tempi = strindex(tempp, fname);	/* get the index of the '.'	*/
    if (tempi != 8)			/* if the '.' is not in		*/
      strmidpad(fname, tempi,		/*   the 9th position,		*/
			8-tempi, ' ');	/*     put it there		*/
    fname[8] = ' ';			/* blank out the '.'		*/
    tempi = strlen(fname);		/* pad the extent		*/
    if (tempi != 12)			/*   as much as necessary	*/
      strrtpad(fname, 12-tempi, ' ');	/*     to make it 3 chars	*/
  }					/* end else there is a '.'	*/
  return(fname);			/* return pointer to name	*/
}					/* end function fnconv_1to2	*/


