/*
 * Program to print various statistics about lines of code and comments
 * in a C source file (or any source file that uses /'s and *'s to delimit
 * comments).
 *
 * Original written by Keith Glennan, January 1984.
 *
 * Almost completely rewritten by Clifford Heath, September 1988.
 *
 * $Header: clc.c,v 1.11 90/10/09 10:11:23 cjh Exp $
 */

#include    <stdio.h>
#include    <string.h>
#include    <ctype.h>

#define     TRUE        1
#define     FALSE       0

#define     LINE_LENGTH 1024    /* Max length of lines in source file */
#define     NON_COMMENT_TEXT " \t*" /* Chars that aren't comment 'text' */

int     Nbr_files = 0;      /* Number of filename arguments */

struct  counts
{
    int line;               /* Total line count             */
    int code;               /* Total blank line count       */
    int blank;              /* Total comment line count     */
    int comment;            /* Total in line comment count  */
    int inline;             /* Total lines of code          */
    int unpadded;           /* Total unpadded comment lines */
}
    subtotal,
    total;

int     Fnwidth = 0;            /* Max width of file name           */
int     fflag = FALSE;          /* Function flag                    */
int     nflag = FALSE;          /* Comments for -o, -c nest         */
int     pflag = FALSE;          /* Count unpadded comment lines     */
int     tflag = FALSE;          /* True means don't give headings   */
char    *ostr, *cstr, *eolstr;  /* Open and close comment sym args  */
char    *filename;              /* Name of current file             */
char    *myname;                /* argv[0]                          */
char    *open_str, *close_str;  /* Open and close comment symbols   */
char    *eolcomm;               /* Comment to end of line symbol    */
int     brace_comm = FALSE;     /* Braces enclose comments          */
int     nest = FALSE;           /* Comments nest; not implemented   */
char    *non_comment_text = NON_COMMENT_TEXT;   /* Default value    */

void    calc_stats(), write_stats(), write_total(), write_subtotal();

main(argc, argv)
int argc;
char    *argv[];
{
    int     c, i, w;
    extern  char    *optarg;    /* args returned by getopt          */
    extern  int optind;         /* getopt's current arg number      */

    if (myname = strrchr(argv[0], '/'))
        myname++;
    else
        myname = argv[0];

    while ((c = getopt(argc, argv, ".:o:c:fnb:pte:")) != EOF)
    {
        switch (c)
        {
        case '.':
            /*
             *  This must be a filename "-.suffix".
             *  Break from the getopt loop.
             */
            optind--;
            break;

        case 'o':
            ostr = optarg;      /* Open comment symbol */
            continue;

        case 'c':
            cstr = optarg;      /* Close comment symbol */
            continue;

        case 'e':
            eolstr = optarg;    /* Open comment to eol */
            continue;

        case 'n':
            nflag = TRUE;       /* Comments nest for -o/-c (NOT IMPL) */
            continue;

        case 'b':
            non_comment_text = optarg;
            continue;

        case 'p':
            pflag = TRUE;       /* Count unpadded comment lines */
            continue;

        case 't':
            tflag = TRUE;       /* Don't give headings */
            continue;

        case 'f':
            fflag = TRUE;       /* count functions (NOT IMPL) */
            continue;

        case '?':               /* Unknown options used */
            usage();
            continue;
        }
        break;
    }

    Nbr_files = argc - optind;

    if (Nbr_files < 1)
        usage();

    for (i = optind; i < argc; i++)
    {
        char*   cp;

        if ((cp = strrchr(argv[i], ',')) && cp[1] == '\0' && Fnwidth <8)
            Fnwidth = 8;    /* For "Subtotal" */

        w = strlen(argv[i]);
        if (w > Fnwidth)
            Fnwidth = w;
    }
    if (Fnwidth != 0)
        Fnwidth = (Fnwidth+8) & ~07;
    if (Nbr_files < 2)
        Fnwidth = 0;

    argc -= optind;
    argv += optind;
    while (argc-- > 0)
    {
        filename = *argv++;
        count_file();
    }

    if (Nbr_files > 1)
        write_total();

    exit(0);
}

usage()
{
    register i;
    static char *help[] =
    {
        "Usage: %s options files...\n",
        "-o openstr -c closestr\topen/close for unknown files\n",
        "-e eolstr\t\tcomment to eol for unknown files\n",
        /*
        "-n\t\t\t\t-o/-c comments nest (NOT IMPLEMENTED)\n",
        "-f\t\t\t\tcount functions (NOT IMPLEMENTED)\n",
         */
        "-p\t\t\tcount unpadded lines in comments\n",
        "-t\t\t\tomit the headings\n",
        "-b padstr\t\tchange comment pad characters\n",
        "files can include - or -.suffix for stdin of given type\n",
        0
    };

    for (i = 0; help[i]; i++)
        fprintf(stderr, help[i], myname);
    exit(1);
}

struct
{
    char    *suffix;    /* Filename suffix */
    char    *openstr;   /* Open comment */
    char    *closestr;  /* Closing commment */
    char    *eolcomm;   /* Comment to end of line */
    int nest;       /* Whether comments nest (not implemented) */
    int brcomm;     /* Pascal kluge; braces are also comments */
}
    file_types[] =
{
    { ".c",       "/*", "*/", NULL, FALSE, FALSE }, /* C source file */
    { ".h",       "/*", "*/", "//", FALSE, FALSE }, /* C or C++ header file */
    { ".C",       "/*", "*/", "//", FALSE, FALSE }, /* C++ */
    { ".H",       "/*", "*/", "//", FALSE, FALSE }, /* C++ headers */
    { ".cc",      "/*", "*/", "//", FALSE, FALSE }, /* C++ */
    { ".hh",      "/*", "*/", "//", FALSE, FALSE }, /* C++ headers */
    { ".cxx",     "/*", "*/", "//", FALSE, FALSE }, /* C++ */
    { ".hxx",     "/*", "*/", "//", FALSE, FALSE }, /* C++ headers */
    { ".p",       "(*", "*)", NULL, FALSE, TRUE  }, /* Pascal */
    { ".sh",      NULL, NULL, "#",  FALSE, FALSE }, /* Shell script */
    { ".mk",      NULL, NULL, "#",  FALSE, FALSE }, /* Makefile */
    { "Makefile", "/*", "*/", "#",  FALSE, FALSE }, /* Makefile */
    { "makefile", "/*", "*/", "#",  FALSE, FALSE }, /* Makefile */
#ifdef  hp9000s300
    { ".s",       NULL, NULL, "#",  FALSE, FALSE }, /* Assembly code */
#endif
#ifdef  hp9000s800
    { ".s",       NULL, NULL, ";",  FALSE, FALSE },  /* Assembly code */
#endif
    { 0, 0, 0, 0, 0 },
};

count_file()
{
    register int    i;
    FILE        *fp;
    register char   *cp;

    if (*filename == '-'
     && (filename[1] == '\0' || filename[1] == '.'))
    {
        /*
         *  Filename is "-" or "-.suffix".
         *  If you want to count a file called "-.foo.c",
         *  use "./-.foo.c"
         */
        fp = stdin;
    }
    else if ((fp = fopen(filename, "r")) == NULL)
    {
        if ((cp = strrchr(filename, ',')) && cp[1] == '\0')
        {
            *cp = '\0';     /* Zap the comma */
            if (cp != filename)
                count_file();   /* Oops, there's still a file */

            write_subtotal();
        }
        else
            perror(filename);
        return;
    }

    /*
     *  Identify file type
     */
    for (i = 0; file_types[i].suffix; i++)
    {
        if (suffix(filename, file_types[i].suffix))
        {
            open_str = file_types[i].openstr;
            close_str = file_types[i].closestr;
            eolcomm = file_types[i].eolcomm;
            nest = file_types[i].nest;
            brace_comm = file_types[i].brcomm;
            break;
        }
    }

    /*
     *  Unknown file type, use -o and -c values if any.
     */
    if (!file_types[i].suffix)
    {
        eolcomm = eolstr;
        if (ostr && cstr)
        {
            open_str = ostr;
            close_str = cstr;
            nest = nflag;
            brace_comm = FALSE;
        }
        else if (eolstr)
        {
            open_str = "";
            close_str = "";
        }
        else
            fprintf(
                stderr,
                "comment symbols unknown for %s\n",
                filename
            );
    }

    calc_stats(fp);

    if (fp != stdin)
        fclose(fp);
}

/*
 *  Return true if suff is a suffix of str.
 */
suffix(str, suff)
char    *str, *suff;
{
    register    len, sl;

    len = strlen(str);
    sl = strlen(suff);

    if (len < sl)
        return 0;
    return strcmp(str+len-sl, suff) == 0;
}

int     line_cnt;           /* Lines in file */
int     code_cnt;           /* Code lines in file */
int     blank_cnt;          /* Blank lines in file */
int     comm_cnt;           /* Comment lines in file */
int     inl_comm_cnt;       /* In-line comments */
int     blank;              /* Is the line blank? */
int     comments_this_line; /* Number of comments ended this line */
int     code_this_line;     /* code text in this line? */
int     unpadded_lines;     /* # of unpadded lines */
int     padding;            /* Comment padding on this line? */

#define NO_PADDING  0       /* No padding seen yet */
#define PAD_CHARS   1       /* Padding chars seen, but no blanks */
#define PAD_BLANK   2       /* Padding and following blanks seen */

void
calc_stats(fp)
FILE    *fp;
{
    register int    c, quote;
    register char   *cp;
    register char   *closep;        /* Pointer to close str*/

    /*
     *  Initialize counters.
     */
    line_cnt = 0;           /* Lines in file */
    code_cnt = 0;           /* Code lines in file */
    blank_cnt = 0;          /* Blank lines in file */
    comm_cnt = 0;           /* Comment lines in file */
    inl_comm_cnt = 0;       /* In-line comments */
    blank = TRUE;
    comments_this_line = 0; /* Number of comments ended this line */
    code_this_line = 0;
    unpadded_lines = 0;

    c = getc(fp);
    while (c != EOF)
    {
        if (c == *open_str)
        {
            /*
             * Found the first character of a start comment symbol.
             * Try to match the rest of the symbol.
             */
            blank = FALSE;      /* The line isn't blank */

            for (cp = open_str+1; *cp != '\0'; cp++)
                if ((c = getc(fp)) != *cp)
                    break;
            if (*cp != '\0')
            {
                /*
                 *  The input didn't match the open symbol.
                 *  We put back the last character (since
                 *  we can only put one back) and continue
                 *  processing with the first character.
                 *  Note that if the open symbol is more
                 *  than two characters long, that means
                 *  that we won't see the middle characters
                 *  again.
                 */
                ungetc(c, fp);
                c = *open_str;  /* Restore the first character*/
            }
            else
            {
                scan_comment(fp, close_str);
                c = getc(fp);
                continue;
            }
        }

        if (c == *eolcomm)
        {
            for (cp = eolcomm+1; *cp != '\0'; cp++)
                if ((c = getc(fp)) != *cp)
                    break;
            if (*cp != 0)
            {
                /*
                 *  Not a comment-to-eol.
                 *  Same again, put back the last char.
                 */
                ungetc(c, fp);
                c = *eolcomm;   /* Restore the first character*/
            }
            else
            {
                scan_comment(fp, "\n");
                c = getc(fp);
                continue;
            }
        }

        switch (c)
        {
        case ' ':
        case '\t':
        case '\f':
            break;

        case '\n':
            newline();
            break;
        
        case '{':   /* Help vi match braces... } */
            blank = FALSE;
            if (!brace_comm)
            {
                /*
                 * Brace doesn't open a comment for this file.
                 */
                code_this_line++;
                break;
            }

            scan_comment(fp, "}");
            break;

        case '\'':
        case '\"':
            blank = FALSE;

            /*
             *  Found a quoted string. Find other end.
             */
            code_this_line++;
            quote = c;
            while ((c = getc(fp)) != quote)
            {
                if (c == '\n' || c == EOF)
                {
                    if (c == '\n')
                        newline();
                    nonterm("literal");
                    break;
                }
                else if (c == '\\' && (c = getc(fp)) == '\n')
                    newline();
            }
            break;

        default:
            blank = FALSE;

            /*
             *  Any comments on this line are in-line comments,
             *  not block comments as we assumed.
             */
            inl_comm_cnt += comments_this_line;
            comments_this_line = 0;     /* Restart counter */
            code_this_line++;
            break;
        }
        c = getc(fp);
    }
    write_stats(
        line_cnt,
        code_cnt,
        blank_cnt,
        comm_cnt,
        inl_comm_cnt,
        unpadded_lines,
        TRUE
    );
}

/*
 *  An open comment has been seen.
 *  Scan through to a closing comment symbol.
 */
scan_comment(fp, closep)
FILE    *fp;
char    *closep;
{
    register    c;
    register char   *cp;
    int     text_this_comm = FALSE;

    padding = PAD_BLANK;
    c = getc(fp);
    while (c != EOF)
    {
        if (c == *closep)
        {
            int ctext = FALSE;

            /*
             *  When the close comment symbol is end of line,
             *  we need to adjust the line counts appropriately.
             *
             *  We can't have multi-character comment close
             *  symbols that begin with \n.
             */
            if (c == '\n')
            {
                if (text_this_comm)
                    comments_this_line++;
                else if (!code_this_line)
                    blank = TRUE;

                newline();
                return;
            }

            /*
             *  We've seen the start of the
             *  closing comment; see if we
             *  can match the rest.
             */
            for (cp = closep+1; *cp != '\0'; cp++)
            {
                if (strchr(non_comment_text, c) == (char *)0)
                    ctext = TRUE;
                else if (pflag)
                    pad(c);
                if ((c = getc(fp)) != *cp)
                    break;
            }
            if (*cp != '\0')
            {
                if (ctext)
                    text_this_comm = TRUE;
                continue; /* without getc */
            }

            if (text_this_comm)
                comments_this_line++;
            return;
        }
        else if (c == '\n')
        {
            comments_this_line = 0;
            line_cnt++;

            if (code_this_line)
            {
                /*
                 *  First line of a multi line comment
                 *  which started on a code line.
                 */
                code_cnt++;
                if (text_this_comm)
                    inl_comm_cnt++; /* Call it inline text*/
            }
            else if (text_this_comm)
            {
                if (pflag && padding != PAD_BLANK)
                    unpadded_lines++;
                else
                    comm_cnt++; /* Line had comment text */
            }
            else
                blank_cnt++; /* Neither code nor comment text */

            text_this_comm = FALSE;
            code_this_line = FALSE;
            blank = TRUE;       /* So far it's a blank line */
            padding = NO_PADDING;   /* And we've seen no padding */
        }
        else if (strchr(non_comment_text, c) == (char *)0)
        {
            text_this_comm = TRUE;  /* Aha some real comment text */
            blank = FALSE;
        }
        else if (pflag)
            pad(c);
        c = getc(fp);
    }

    if (c == EOF)
        nonterm("comment");
}

newline()
{
    if (blank)
        blank_cnt++;    /* The line was blank; count it. */
    blank = TRUE;       /* The new line is blank so far! */

    line_cnt++;
    if (code_this_line)
        code_cnt++;

    if (comments_this_line)
        if (code_this_line)
            inl_comm_cnt += comments_this_line;
        else if (pflag && padding != PAD_BLANK)
            unpadded_lines++;
        else
            comm_cnt++; /* Line had comment text */

    comments_this_line = 0;
    code_this_line = FALSE;
}

/*
 *  We've seen a comment pad char of c.
 *  Work out where we're up to in identifying a correctly padded line.
 */
pad(c)
char    c;
{
    register int    bl = (c == ' ' || c == '\t');

    switch (padding)
    {
    case NO_PADDING:
        if (!bl)
            padding = PAD_CHARS;
        break;

    case PAD_CHARS:
        if (bl)
            padding = PAD_BLANK;
        break;
    }
}

nonterm(what)
char    *what;
{
    fprintf(
        stderr,
        "%s: non-terminated %s, file %s line %d\n", 
        myname,
        what,
        filename,
        line_cnt
    );
}

/* Write statistics to output stream */

void
write_stats(lines, code, blanks, comls, inl_comms, unpadded, accumulate)
int lines;              /* Total number of lines          */
int code;               /* Number of lines of code            */
int blanks;             /* Number of blank lines found        */
int comls;              /* Number of comment lines found      */
int inl_comms;          /* Number of in line comments found   */
int unpadded;           /* Number of unpadded comment lines   */
int accumulate;         /* Whether to accumulate statistics   */
{
    static int  title = FALSE; /* Flag to denote whether tile printed */

    if (!title)
    {
        do_title();
        title = TRUE;
    }

    if (Nbr_files > 1)
        printf("%*s", -Fnwidth, filename);

    printf(
        "%d\t%d\t%d\t%d\t%d\t %4.1f",
        lines,
        code,
        blanks,
        comls,
        inl_comms,
        (code+comls+inl_comms)
        ? (comls+inl_comms)*100.0/(code+comls+inl_comms)
        : 0 /* Don't divide by zero */
    );
    if (pflag)
    {
        printf("\t  %d\n", unpadded);
    }
    else
        printf("\n");

    if (accumulate)
    {
        subtotal.line += lines;
        subtotal.code += code;
        subtotal.blank += blanks;
        subtotal.comment += comls;
        subtotal.inline += inl_comms;
        subtotal.unpadded += unpadded;

        total.line += lines;
        total.code += code;
        total.blank += blanks;
        total.comment += comls;
        total.inline += inl_comms;
        total.unpadded += unpadded;
    }
}

do_title()
{
    if (tflag)
        return;

    printf("%*sTotal\tCode\tBlank\tComment\tIn-line\t Percent",Fnwidth,"");
    if (pflag)
        printf("  Unpadded\n");
    else
        printf("\n");
    printf("%*sLines\tLines\tLines\tLines\tComments Comments",Fnwidth,"");
    if (pflag)
        printf(" Comments\n");
    else
        printf("\n");
}

void
write_total()
{
    if (subtotal.line && subtotal.line != total.line)
        write_subtotal();

    if (!tflag)
        printf(
             "%*s-------------------------------------------------%s\n",
             Fnwidth,
             "",
             pflag ? "---------" : ""
        );

    filename = "Total";

    write_stats(
        total.line,
        total.code,
        total.blank,
        total.comment,
        total.inline,
        total.unpadded,
        FALSE
    );
    if (!tflag)
        printf(
             "%*s-------------------------------------------------%s\n",
             Fnwidth,
             "",
             pflag ? "---------" : ""
        );
}

void
write_subtotal()
{
    filename = "Subtotal";

    write_stats(
        subtotal.line,
        subtotal.code,
        subtotal.blank,
        subtotal.comment,
        subtotal.inline,
        subtotal.unpadded,
        FALSE
    );
    if (!tflag)
        printf("\n");
    memset(&subtotal, 0, sizeof(subtotal));
}



