; MODSTUF3.ASM
;
; This is the third part of the assembler "half" of the player module.
; Here we handle time-critical things like actual resampling and mixing of
; mod samples.  This file is broken into four parts to get around the
; assembler's 64k limit on source files.
;
; RESAMPLERS (CONTINUED) *********************************************
;
; These routines may modify any register, including DS, but should leave
; DF clear.  On entry, DS addresses the default data segment, and BX
; addresses the channel data structure.  They return nothing.
;
; RSBZ routine, "resamples" a "subsequent" channel on the left or right.  This
; is a special-case resampler for a zero-volume sample.  For any size sample.
;
	EVEN
RSBZ:
	;
	; Figure out where the sample pointer is after "generating" silence
	; samples.
	;
	MOV	AX,[BX].CHANSTEPFRAC	; multiply sample step by # samples
	MUL	_samplespertick
	MOV	BP,AX			; (accumulate in DX:DI.BP)
	MOV	DI,DX
	MOV	AX,[BX].CHANSTEPINT
	MUL	_samplespertick
	ADD	DI,AX
	ADC	DX,0
	ADD	BP,[BX].CHANSAMPFRAC	; add in current position in sample
	ADC	DI,[BX].CHANSAMPOFFS
	ADC	DX,0
	MOV	SI,[BX].CHANSAMPPTR	; SI addresses sample data
	CMP	[SI].SAMPEMSFLAG,1	; sample in EMS?
	JE	RSBZ_INEMS
	MOV	AX,[SI].SAMPSEG		; set CF if over 64k
	CMP	AX,[BX].CHANSAMPSEG
	JMP	SHORT RSBZ_ADDSEG
	EVEN
RSBZ_INEMS:
	CMP	[BX].CHANSAMPHALF,1	; set CF if 2nd 64k
	CMC
RSBZ_ADDSEG:
	ADC	DX,0			; add 1 if past 64k
	SHR	DX,1			; convert to words
	RCR	DI,1
	RCR	BP,1
	;
	; The current position in words relative to the start of the sample
	; is now in DX:DI.BP.  See if we're past the end of the sample.
	;
	OR	DX,DX
	JNZ	RSBZ_PASTEND
	CMP	DI,[SI].SAMPLEN
	JAE	RSBZ_PASTEND
	;
	; We're not past the end.  Save the new position.
	;
	CMP	[SI].SAMPEMSFLAG,1	; sample in EMS?
	JE	RSBZ_INEMS2
	SHL	BP,1			; convert back to bytes
	RCL	DI,1
#IF M_I286
	RCR	DX,4
#ELSE
	RCR	DX,1
	SHR	DX,1
	SHR	DX,1
	SHR	DX,1
#ENDIF
	ADD	DX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,DX
	JMP	SHORT RSBZ_SETOFFSET
	EVEN
RSBZ_INEMS2:
	SHL	BP,1			; convert back to bytes
	RCL	DI,1
	RCL	DL,1
	MOV	[BX].CHANSAMPHALF,DL
RSBZ_SETOFFSET:
	MOV	[BX].CHANSAMPFRAC,BP
	MOV	[BX].CHANSAMPOFFS,DI
	RET
	;
	; We're past the end.  If the sample is not looped, mark it done
	; and set the last sample to zero.
	;
	EVEN
RSBZ_PASTEND:
	CMP	[SI].SAMPLLEN,0
	JNE	RSBZ_LOOPED
	MOV	[BX].CHANSAMPDONE,1
	MOV	[BX].CHANLASTSAMP,0
	RET
	;
	; Sample is looped.  Determine new position (inside the loop) and
	; save it.
	;
	EVEN
RSBZ_LOOPED:
	SUB	DI,[SI].SAMPLSTART	; subtract loop start from position
	SBB	DX,0
	MOV	AX,DX			; divide high word of offset by loop
	XOR	DX,DX			;   length
	DIV	[SI].SAMPLLEN		; keep remainder, discard quotient
	MOV	AX,DI			; divide remainder:low word
	DIV	[SI].SAMPLLEN
	ADD	DX,[SI].SAMPLSTART	; add loop start to remainder
	XOR	AX,AX
	CMP	[SI].SAMPEMSFLAG,1	; sample in EMS?
	JE	RSBZ_INEMS3
	SHL	BP,1			; convert to bytes
	RCL	DX,1
#IF M_I286
	RCR	AX,4
#ELSE
	RCR	AX,1
	SHR	AX,1
	SHR	AX,1
	SHR	AX,1
#ENDIF
	ADD	AX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,AX
	JMP	SHORT RSBZ_SETOFFSET2
	EVEN
RSBZ_INEMS3:
	SHL	BP,1			; convert back to bytes
	RCL	DX,1
	RCL	AL,1
	MOV	[BX].CHANSAMPHALF,AL
RSBZ_SETOFFSET2:
	MOV	[BX].CHANSAMPFRAC,BP
	MOV	[BX].CHANSAMPOFFS,DX
	RET
;
; RS64BN routine, resamples a "subsequent" channel on the left or right.
; This resampler assumes that the sample is shorter than 64k and that we
; will not get to the end of it during the tick - a good but fairly common
; case.
;
	EVEN
RS64BN:
#IF M_I286
	; (286 version)
	;
	; Save the channel data pointer on the stack and put a copy in BP.
	;
	PUSH	BX
	MOV	BP,BX
	;
	; We know that the sample will not be played through.  The limit in
	; BX is the offset from the start of the partial-mix buffer of the
	; end of the output for the tick on this channel.  DI is a negative
	; number which added to that limit gives the offset where the
	; current sample goes.
	;
	MOV	DI,_samplespertick
	SHL	DI,1
	SHL	DI,1
	LES	BX,DWORD PTR [BP].CHANPARTOFFS
	ADD	BX,DI
	NEG	DI
	;
	; Patch in the volume.  If we have more than 8 channels, we need
	; to adjust the volume downward to avoid overflowing 16 bits in the
	; partial-mix buffer.
	;
	MOV	AL,[BP].CHANMIXVOL
	MUL	_globalvol
	SHL	AX,1
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64BN_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64BN_SETVOL
	SHR	AH,1
RS64BN_SETVOL:
	MOV	BYTE PTR CS:RS64BN_VOLPT1,AH
	;
	; Now set up remaining register values.  We have:
	;
	;    ES:BX+DI      addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.CX         current byte offset in sample
	;    BP.DX         sample step in bytes
	;
	MOV	CX,[BP].CHANSAMPFRAC	; BX = fraction of byte offset
	LDS	SI,DWORD PTR [BP].CHANSAMPOFFS ; DS:SI = segment:offset
	MOV	DX,[BP].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	BP,[BP].CHANSTEPINT	; BP = integer part of sample step
	EVEN
RS64BN_MAINLOOP:
	DB	0B0h			; MOV AL,immed MIXVOL
RS64BN_VOLPT1:
	DB	0
	IMUL	BYTE PTR [SI]
	ADD	ES:[BX+DI],AX
	ADD	CX,DX
	ADC	SI,BP
	ADD	DI,4
	JNZ	RS64BN_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,CX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
#ELSE
	; (8086 version)
	;
	; Save the channel data pointer on the stack.
	;
	PUSH	BX
	;
	; We know that the sample will not be played through.  The limit in
	; BP is the offset from the start of the partial-mix buffer of the
	; end of the output for the tick on this channel.  DI is a negative
	; number which added to that limit gives the offset where the
	; current sample goes.
	;
	MOV	DI,_samplespertick
	SHL	DI,1
	SHL	DI,1
	LES	BP,DWORD PTR [BX].CHANPARTOFFS
	ADD	BP,DI
	NEG	DI
	;
	; Patch in the volume.  In the 8086 version, we use the offset in
	; the precomputed multiplication table.  If there are more than 8
	; channels in the mod, we have to adjust the volume downward to
	; avoid overflowing 16 bits in the partial-mix buffer.
	;
	MOV	AH,[BX].CHANMIXVOL	; "multiply" mixing volume by global
	MOV	AL,_globalvol		;   volume
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,MULTBL[SI]
	SHL	AX,1			; "divide" by 64
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64BN_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64BN_SETVOL
	SHR	AH,1
RS64BN_SETVOL:
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:RS64BN_VOLPT1,AX
	;
	; Now set up remaining register values.  We have:
	;
	;    ES:BP+DI      addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.CX         current byte offset in sample
	;    AX.DX         sample step in bytes
	;
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	AX,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	CX,[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	LDS	SI,DWORD PTR [BX].CHANSAMPOFFS ; DS:SI = segment:offset
	EVEN
RS64BN_MAINLOOP:
	MOV	BL,[SI]
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,9Fh		; MOV BX,SS:[BX+immed]
RS64BN_VOLPT1:
	DW	0
	ADD	ES:[BP+DI],BX
	ADD	CX,DX
	ADC	SI,AX
	ADD	DI,4
	JNZ	RS64BN_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,CX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
#ENDIF
;
; RS64B routine, resamples a "subsequent" channel on the left or right.  This
; resampler assumes that the sample is shorter than 64k, though we may get
; to the end of it during the tick.
;
	EVEN
RS64B:
#IF M_I286
	; (286 version)
	;
	; Patch in the volume.  If we have a more than 8 channels, we need
	; to adjust the volume downward to avoid overflowing 16 bits in the
	; partial-mix buffer.  Do these patches first since we will need
	; them even if the sample has played through.
	;
	MOV	AL,[BX].CHANMIXVOL
	MUL	_globalvol
	SHL	AX,1
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64B_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64B_SETVOL
	SHR	AH,1
RS64B_SETVOL:
	MOV	BYTE PTR CS:RS64B_VOLPT1,AH
	MOV	BYTE PTR CS:RS64B_VOLPT2,AH
	;
	; First check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS64B_DOPATCH
	JMP	RS64B_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS64B_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS64B_LIMITPT1,DI
	MOV	WORD PTR CS:RS64B_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Get the sample step in BP.DX and patch in the fractional part.
	;
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	WORD PTR CS:RS64B_STFRACPT1,DX ; patch in fraction
	;
	; Patch in the sample length, loop start, and loop length, leaving
	; the loop length in AX.  Convert these to bytes.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	CX,[SI].SAMPLEN		; get sample length in CX
	SHL	CX,1
	MOV	AX,[SI].SAMPLSTART	; patch in start of sample loop
	SHL	AX,1
	MOV	WORD PTR CS:RS64B_LSTARTPT1,AX
	MOV	WORD PTR CS:RS64B_LSTARTPT2,AX
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	SHL	AX,1
	MOV	WORD PTR CS:RS64B_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS64B_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS64B_LOOPED
	MOV	WORD PTR CS:RS64B_SAMPEND,RS64B_JUMPINSTR
	;
	; Now set up remaining register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.BX         current byte offset in sample
	;    BP.DX         sample step in bytes (already set)
	;    CX            sample length in bytes (already set)
	;
RS64B_LOOPED:
	PUSH	BX			; save pointer to channel data
	LDS	SI,DWORD PTR [BX].CHANSAMPOFFS ; DS:SI = segment:offset
	MOV	BX,SS:[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	EVEN
RS64B_MAINLOOP:
	DB	0B0h			; MOV AL,immed MIXVOL
RS64B_VOLPT1:
	DB	0
	IMUL	BYTE PTR [SI]
	DB	26h,01h,85h		; ADD ES:[DI+immed],AX
RS64B_LIMITPT1:
	DW	0
	ADD	BX,DX
	ADC	SI,BP
	JC	RS64B_SAMPEND
	CMP	SI,CX
	JAE	RS64B_SAMPEND		; (JAE = JNC)
RS64B_MAINLPEND:
	ADD	DI,4
	JNZ	RS64B_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,BX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
	;
	; We got to the end of the sample.  CF:SI.BX is the current offset
	; in bytes from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS64B_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS64B_JUMPSIZE
	MOV	DX,0		; LSB of DX = CF
	RCL	DX,1
	MOV	AX,SI		; DX:AX = integer of offset in bytes
	DB	2Dh		; SUB AX,immed SAMPLSTART (in bytes)
RS64B_LSTARTPT1:
	DW	0
	SBB	DX,0
	DB	0BEh		; MOV SI,immed SAMPLLEN (in bytes)
RS64B_LLENPT1:
	DW	0
	DIV	SI
	MOV	SI,DX
	DB	81h,0C6h	; ADD SI,immed SAMPLSTART (in bytes)
RS64B_LSTARTPT2:
	DW	0
	DB	0BAh		; MOV DX,immed STEPFRAC
RS64B_STFRACPT1:
	DW	0
	JMP	SHORT RS64B_MAINLPEND
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.BX
	; is the current offset in bytes from the start of the sample.  BP.DX
	; is the sample step in bytes.
	;
	EVEN
RS64B_JUMPSIZE		EQU	$ - (OFFSET RS64B_SAMPEND + 2)
RS64B_JUMPINSTR		EQU	(RS64B_JUMPSIZE*256) + 0EBh
	SUB	BX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 64k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS64B_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS64B_LIMITPT2:
	DW	0
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	;
RS64B_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS64B_DONE		; no ramping if already at baseline
	DB	0B7h			; MOV BH,immed MIXVOL
RS64B_VOLPT2:
	DB	0
	JS	RS64B_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS64B_ABOVE:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	ADD	ES:[DI],AX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	DEC	AL		; decrement sample
	LOOPNZ	RS64B_ABOVE	; go again if more and not at baseline
	JMP	SHORT RS64B_DONE
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS64B_BELOW:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	ADD	ES:[DI],AX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	INC	AL		; increment sample
	LOOPNZ	RS64B_BELOW	; go again if more and not at baseline
	;
	; All done for now.
	;
RS64B_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ELSE
	; (8086 version)
	;
	; Patch in the volume.  In the 8086 version, we use the offset in
	; the precomputed multiplication table.  These patches need to be
	; done even if the sample has been played through, so do them here.
	; If there are more than 8 channels in the mod, the volume needs to
	; be adjusted downward to avoid overflowing 16 bits in the partial-
	; mix buffer.
	;
	MOV	AH,[BX].CHANMIXVOL	; "multiply" mixing volume by global
	MOV	AL,_globalvol		;   volume
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,MULTBL[SI]
	SHL	AX,1			; "divide" by 64
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64B_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64B_SETVOL
	SHR	AH,1
RS64B_SETVOL:
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:RS64B_VOLPT1,AX
	MOV	WORD PTR CS:RS64B_VOLPT2,AX
	MOV	WORD PTR CS:RS64B_VOLPT3,AX
	;
	; Check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS64B_DOPATCH
	JMP	RS64B_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS64B_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS64B_LIMITPT1,DI
	MOV	WORD PTR CS:RS64B_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Patch in the sample length and loop length, leaving the loop
	; length in AX.  Convert these to bytes.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	CX,[SI].SAMPLEN		; get sample length in CX
	SHL	CX,1
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	SHL	AX,1
	MOV	WORD PTR CS:RS64B_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS64B_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS64B_LOOPED
	MOV	WORD PTR CS:RS64B_SAMPEND,RS64B_JUMPINSTR
	;
	; Now set up register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.AX         current byte offset in sample
	;    BP.DX         sample step in bytes
	;    CX            sample length in bytes (already set)
	;
RS64B_LOOPED:
	PUSH	BX			; save pointer to channel data
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	AX,[BX].CHANSAMPFRAC	; AX = fraction of byte offset
	LDS	SI,DWORD PTR [BX].CHANSAMPOFFS ; DS:SI = segment:offset
	EVEN
RS64B_MAINLOOP:
	MOV	BL,[SI]
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,9Fh		; MOV BX,SS:[BX+immed]
RS64B_VOLPT1:
	DW	0
	DB	26h,01h,9Dh		; ADD ES:[DI+immed],BX
RS64B_LIMITPT1:
	DW	0
	ADD	AX,DX
	ADC	SI,BP
	JC	RS64B_SAMPEND
	CMP	SI,CX
	JAE	RS64B_SAMPEND		; (JAE = JNC)
RS64B_MAINLPEND:
	ADD	DI,4
	JNZ	RS64B_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,AX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
	;
	; We got to the end of the sample.  CF:SI.AX is the current offset
	; in bytes from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS64B_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS64B_JUMPSIZE
	MOV	BL,0		; LSB of BL = CF
	RCL	BL,1		; BL:SI = integer of offset in bytes
RS64B_ADJLOOP:
	DB	81h,0EEh	; SUB SI,immed SAMPLLEN (in bytes)
RS64B_LLENPT1:
	DW	0
	SBB	BL,0
        JNZ     RS64B_ADJLOOP
	CMP	SI,CX
        JAE     RS64B_ADJLOOP
	JMP	SHORT RS64B_MAINLPEND
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.AX
	; is the current offset in bytes from the start of the sample.  BP.DX
	; is the sample step in bytes.
	;
	EVEN
RS64B_JUMPSIZE		EQU	$ - (OFFSET RS64B_SAMPEND + 2)
RS64B_JUMPINSTR		EQU	(RS64B_JUMPSIZE*256) + 0EBh
	SUB	AX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 64k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS64B_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS64B_LIMITPT2:
	DW	0
	MOV	BX,DGROUP		; DS addresses local data again
	MOV	DS,BX
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	; DS addresses the default data segment.
	;
RS64B_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS64B_DONE		; no ramping if already at baseline
	JS	RS64B_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS64B_ABOVE:
	MOV	BL,AL		; get sample in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS64B_VOLPT2:
	DW	0
	ADD	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	DEC	AL		; decrement sample
	LOOPNZ	RS64B_ABOVE	; go again if more and not at baseline
	JMP	SHORT RS64B_DONE
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS64B_BELOW:
	MOV	BL,AL		; save copy in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS64B_VOLPT3:
	DW	0
	ADD	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	INC	AL		; increment sample
	LOOPNZ	RS64B_BELOW	; go again if more and not at baseline
	;
	; All done for now.
	;
RS64B_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ENDIF
;
; RS128B routine, resamples a "subsequent" channel on the left or right.
; This is the worst-case resampler for a "subsequent" channel.  It can handle
; samples longer than 64k where we may get to the end of the sample during
; the tick.
;
	EVEN
RS128B:
#IF M_I286
	; (286 version)
	;
	; Patch in the volume.  If we have more than 8 channels, we need
	; to adjust the volume downward to avoid overflowing 16 bits in the
	; partial-mix buffer.  Do these patches first since we will need
	; them even if the sample has played through.
	;
	MOV	AL,[BX].CHANMIXVOL
	MUL	_globalvol
	SHL	AX,1
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS128B_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS128B_SETVOL
	SHR	AH,1
RS128B_SETVOL:
	MOV	BYTE PTR CS:RS128B_VOLPT1,AH
	MOV	BYTE PTR CS:RS128B_VOLPT2,AH
	;
	; Check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS128B_DOPATCH
	JMP	RS128B_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS128B_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS128B_LIMITPT1,DI
	MOV	WORD PTR CS:RS128B_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Get the sample step in BP.DX, convert it to words, and patch in
	; the fractional part.
	;
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	SHR	BP,1			; convert to words
	RCR	DX,1
	MOV	WORD PTR CS:RS128B_STFRACPT1,DX ; patch in fraction
	;
	; Patch in the sample length, loop start, and loop length, leaving
	; the loop length in AX.  These are word quantities already.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	AX,[SI].SAMPLEN		; patch in sample length
	MOV	WORD PTR CS:RS128B_LENPT1,AX
	MOV	AX,[SI].SAMPLSTART	; patch in start of sample loop
	MOV	WORD PTR CS:RS128B_LSTARTPT1,AX
	MOV	WORD PTR CS:RS128B_LSTARTPT2,AX
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	MOV	WORD PTR CS:RS128B_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS128B_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS128B_LOOPED
	MOV	WORD PTR CS:RS128B_SAMPEND,RS128B_JUMPINSTR
	;
	; Now set up register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.BX         current word offset in sample
	;    BP.DX         sample step in words (already set)
	;    CX            8000h if 64k or more past sample start, 0 otherwise
	;
RS128B_LOOPED:
	PUSH	BX			; save pointer to channel data
	XOR	CX,CX			; assume first 64k
	MOV	AX,[BX].CHANSAMPSEG	; get segment in AX
	CMP	[SI].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	RS128B_INEMS
	MOV	BYTE PTR CS:RS128B_SEGJMP,RS128B_SEGJMP1
	CMP	[SI].SAMPSEG,AX		; set CF if 2nd 64k
	JMP	SHORT RS128B_SETCX
	EVEN
RS128B_INEMS:
	MOV	BYTE PTR CS:RS128B_SEGJMP,RS128B_SEGJMP2
	MOV	SI,[SI].SAMPEMSHANDLE	; get EMS handle and patch into code
	MOV	WORD PTR CS:RS128B_EMSHANDLE,SI
	CMP	[BX].CHANSAMPHALF,1	; set CF if in upper 64k
	CMC
RS128B_SETCX:				; CF set if 2nd 64k
	RCR	CX,1
	MOV	SI,[BX].CHANSAMPOFFS	; SI = integer part of byte offset
	MOV	BX,[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	SHR	SI,1			; convert offset to words
	RCR	BX,1
	MOV	DS,AX			; DS = current segment
	EVEN
RS128B_MAINLOOP:
	DB	0B0h			; MOV AL,immed MIXVOL
RS128B_VOLPT1:
	DB	0
	SHL	BX,1
	RCL	SI,1
	IMUL	BYTE PTR [SI]
	SHR	SI,1
	RCR	BX,1
	DB	26h,01h,85h		; ADD ES:[DI+immed],AX
RS128B_LIMITPT1:
	DW	0
	OR	SI,CX
	ADD	BX,DX
	ADC	SI,BP
	JC	RS128B_SAMPEND
	DB	81h,0FEh		; CMP SI,immed SAMPLEN
RS128B_LENPT1:
	DW	0
	JAE	RS128B_SAMPEND		; (JAE = JNC)
RS128B_SEGTEST:
	XOR	SI,CX
	DB	78h			; JS segchange_code
RS128B_SEGJMP:
	DB	0
RS128B_MAINLPEND:
	ADD	DI,4
	JNZ	RS128B_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset and the "over 64k" flag.  There is at least one more
	; sample to be output on this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	SHL	BX,1		; SI.BX is position in bytes relative to
	RCL	SI,1		;   current segment
	MOV	[BP].CHANSAMPFRAC,BX
	MOV	[BP].CHANSAMPOFFS,SI
	MOV	[BP].CHANSAMPSEG,DS
	ROL	CH,1
	MOV	[BP].CHANSAMPHALF,CH
	RET
	;
	; Different 64k segment now.  Adjust DS, CX, and SI.  The most
	; significant bit of SI is set when we get here and needs to be
	; cleared in any case.  If CX is 0, we're going from the lower 64k
	; to the upper 64k, and we need to set DS to SAMPSEG+1000h and CX
	; to 8000h.  If CX is 8000h, we're going from the upper 64k to the
	; lower 64k (a looped sample longer than 64k, and we just looped).
	; In that case, we need to set DS to SAMPSEG and CX to 0.  This is
	; for a sample loaded in conventional RAM.
	;
	EVEN
RS128B_SEGJMP1		EQU	$ - (OFFSET RS128B_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	MOV	AX,DS			; DS value in AX for adjustment
	JZ	RS128B_SEGLOW		; if CX = 0, CX was 8000h, so low 64k
	ADD	AH,10h			; low to high, add 1000h to DS
	MOV	DS,AX
	JMP	SHORT RS128B_MAINLPEND
	EVEN
RS128B_SEGLOW:
	SUB	AH,10h			; high to low, subtract 1000h from DS
	MOV	DS,AX
	JMP	SHORT RS128B_MAINLPEND
	;
	; Different 64k segment now.  Adjust CX and SI, and map in the
	; correct half of the EMS sample.  The most significant bit of SI
	; is set when we get here and needs to be cleared in any case.  If
	; CX is 0, we're going from the lower 64k to the upper 64k, and we
	; need to set CX to 8000h and map in the upper 64k.  If CX is 8000h,
	; we're going from the upper 64k to the lower 64k (a looped sample
	; longer than 64k, and we just looped).  In that case, we need to
	; set CX to 0 and map in the lower 64k.  This is for a sample loaded
	; in EMS RAM.
	;
	EVEN
RS128B_SEGJMP2		EQU	$ - (OFFSET RS128B_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	DB	68h			; PUSH immed SAMPEMSHANDLE
RS128B_EMSHANDLE:
	DW	0
	JZ	RS128B_SEGLOW2		; if CX = 0, CX was 8000h, so low 64k
	CALL	MAPEMS2ND		; low to high, map in the second 64k
	JMP	SHORT RS128B_MAINLPEND
	EVEN
RS128B_SEGLOW2:
	CALL	MAPEMS1ST		; high to low, map in the first 64k
	JMP	SHORT RS128B_MAINLPEND
	;
	; We got to the end of the sample.  CF:SI.BX is the current offset
	; in words from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS128B_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS128B_JUMPSIZE
	MOV	DX,0		; LSB of DX = CF
	RCL	DX,1
	MOV	AX,SI		; DX:AX = integer of offset in words
	DB	2Dh		; SUB AX,immed SAMPLSTART
RS128B_LSTARTPT1:
	DW	0
	SBB	DX,0
	DB	0BEh		; MOV SI,immed SAMPLLEN
RS128B_LLENPT1:
	DW	0
	DIV	SI
	MOV	SI,DX
	DB	81h,0C6h	; ADD SI,immed SAMPLSTART
RS128B_LSTARTPT2:
	DW	0
	DB	0BAh		; MOV DX,immed STEPFRAC
RS128B_STFRACPT1:
	DW	0
	JMP	SHORT RS128B_SEGTEST
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.BX
	; is the current offset in words from the start of the sample.  BP.DX
	; is the sample step in words.
	;
	EVEN
RS128B_JUMPSIZE		EQU	$ - (OFFSET RS128B_SAMPEND + 2)
RS128B_JUMPINSTR	EQU	(RS128B_JUMPSIZE*256) + 0EBh
	SUB	BX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 128k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS128B_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS128B_LIMITPT2:
	DW	0
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	;
RS128B_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS128B_DONE		; no ramping if already at baseline
	DB	0B7h			; MOV BH,immed MIXVOL
RS128B_VOLPT2:
	DB	0
	JS	RS128B_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS128B_ABOVE:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	ADD	ES:[DI],AX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	DEC	AL		; decrement sample
	LOOPNZ	RS128B_ABOVE	; go again if more and not at baseline
	JMP	SHORT RS128B_DONE
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS128B_BELOW:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	ADD	ES:[DI],AX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	INC	AL		; increment sample
	LOOPNZ	RS128B_BELOW	; go again if more and not at baseline
	;
	; All done for now.
	;
RS128B_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ELSE
	; (8086 version)
	;
	; Patch in the volume.  In the 8086 version, we use the offset in
	; the precomputed multiplication table.  These patches need to be
	; done even if the sample has been played through, so do them here.
	; If there are more than 8 channels in the mod, the volume needs to
	; be adjusted downward to avoid overflowing 16 bits in the partial-
	; mix buffer.
	;
	MOV	AH,[BX].CHANMIXVOL	; "multiply" mixing volume by global
	MOV	AL,_globalvol		;   volume
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,MULTBL[SI]
	SHL	AX,1			; "divide" by 64
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS128B_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS128B_SETVOL
	SHR	AH,1
RS128B_SETVOL:
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:RS128B_VOLPT1,AX
	MOV	WORD PTR CS:RS128B_VOLPT2,AX
	MOV	WORD PTR CS:RS128B_VOLPT3,AX
	;
	; Check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS128B_DOPATCH
	JMP	RS128B_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS128B_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS128B_LIMITPT1,DI
	MOV	WORD PTR CS:RS128B_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Patch in the sample length and loop length, leaving the loop
	; length in AX.  These are word quantities already.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	AX,[SI].SAMPLEN		; patch in sample length
	MOV	WORD PTR CS:RS128B_LENPT1,AX
	MOV	WORD PTR CS:RS128B_LENPT2,AX
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	MOV	WORD PTR CS:RS128B_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS128B_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS128B_LOOPED
	MOV	WORD PTR CS:RS128B_SAMPEND,RS128B_JUMPINSTR
	;
	; Now set up register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.AX         current word offset in sample
	;    BP.DX         sample step in words
	;    CX            8000h if 64k or more past sample start, 0 otherwise
	;
RS128B_LOOPED:
	PUSH	BX			; save pointer to channel data
	XOR	CX,CX			; assume first 64k
	MOV	AX,[BX].CHANSAMPSEG	; get segment in AX
	CMP	[SI].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	RS128B_INEMS
	MOV	BYTE PTR CS:RS128B_SEGJMP,RS128B_SEGJMP1
	CMP	[SI].SAMPSEG,AX		; set CF if 2nd 64k
	JMP	SHORT RS128B_SETCX
	EVEN
RS128B_INEMS:
	MOV	BYTE PTR CS:RS128B_SEGJMP,RS128B_SEGJMP2
	MOV	SI,[SI].SAMPEMSHANDLE	; get EMS handle and patch into code
	MOV	WORD PTR CS:RS128B_EMSHANDLE,SI
	CMP	[BX].CHANSAMPHALF,1	; set CF if in upper 64k
	CMC
RS128B_SETCX:				; CF set if 2nd 64k
	RCR	CX,1
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	SHR	BP,1			; convert to words
	RCR	DX,1
	MOV	SI,[BX].CHANSAMPOFFS	; SI = integer part of byte offset
	MOV	BX,[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	SHR	SI,1			; convert offset to words
	RCR	BX,1
	MOV	DS,AX			; DS = current segment
	MOV	AX,BX			; AX = fraction of word offset
	EVEN
RS128B_MAINLOOP:
	SHL	AX,1
	RCL	SI,1
	MOV	BL,[SI]
	SHR	SI,1
	RCR	AX,1
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,9Fh		; MOV BX,SS:[BX+immed]
RS128B_VOLPT1:
	DW	0
	DB	26h,01h,9Dh		; ADD ES:[DI+immed],BX
RS128B_LIMITPT1:
	DW	0
	OR	SI,CX
	ADD	AX,DX
	ADC	SI,BP
	JC	RS128B_SAMPEND
	DB	81h,0FEh		; CMP SI,immed SAMPLEN
RS128B_LENPT1:
	DW	0
	JAE	RS128B_SAMPEND		; (JAE = JNC)
RS128B_SEGTEST:
	XOR	SI,CX
	DB	78h			; JS segchange_code
RS128B_SEGJMP:
	DB	0
RS128B_MAINLPEND:
	ADD	DI,4
	JNZ	RS128B_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset and the "over 64k" flag.  There is at least one more
	; sample to be output on this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	SHL	AX,1		; SI.AX is position in bytes relative to
	RCL	SI,1		;   current segment
	MOV	[BP].CHANSAMPFRAC,AX
	MOV	[BP].CHANSAMPOFFS,SI
	MOV	[BP].CHANSAMPSEG,DS
	ROL	CH,1
	MOV	[BP].CHANSAMPHALF,CH
	RET
	;
	; Different 64k segment now.  Adjust DS, CX, and SI.  The most
	; significant bit of SI is set when we get here and needs to be
	; cleared in any case.  If CX is 0, we're going from the lower 64k
	; to the upper 64k, and we need to set DS to SAMPSEG+1000h and CX
	; to 8000h.  If CX is 8000h, we're going from the upper 64k to the
	; lower 64k (a looped sample longer than 64k, and we just looped).
	; In that case, we need to set DS to SAMPSEG and CX to 0.  This is
	; for a sample loaded in conventional RAM.
	;
	EVEN
RS128B_SEGJMP1		EQU	$ - (OFFSET RS128B_MAINLPEND)
	MOV	BX,8000h
	XOR	SI,BX			; clear MSB of SI
	XOR	CX,BX			; invert MSB of CX
	MOV	BX,DS			; DS value in BX for adjustment
	JZ	RS128B_SEGLOW		; if CX = 0, CX was 8000h, so low 64k
	ADD	BH,10h			; low to high, add 1000h to DS
	MOV	DS,BX
	JMP	SHORT RS128B_MAINLPEND
	EVEN
RS128B_SEGLOW:
	SUB	BH,10h			; high to low, subtract 1000h from DS
	MOV	DS,BX
	JMP	SHORT RS128B_MAINLPEND
	;
	; Different 64k segment now.  Adjust CX and SI, and map in the
	; correct half of the EMS sample.  The most significant bit of SI
	; is set when we get here and needs to be cleared in any case.  If
	; CX is 0, we're going from the lower 64k to the upper 64k, and we
	; need to set CX to 8000h and map in the upper 64k.  If CX is 8000h,
	; we're going from the upper 64k to the lower 64k (a looped sample
	; longer than 64k, and we just looped).  In that case, we need to
	; set CX to 0 and map in the lower 64k.  This is for a sample loaded
	; in EMS RAM.
	;
	EVEN
RS128B_SEGJMP2		EQU	$ - (OFFSET RS128B_MAINLPEND)
	MOV	BX,8000h
	XOR	SI,BX			; clear MSB of SI
	XOR	CX,BX			; invert MSB of CX
	DB	0BBh			; MOV BX,immed SAMPEMSHANDLE
RS128B_EMSHANDLE:
	DW	0
	PUSH	BX
	JZ	RS128B_SEGLOW2		; if CX = 0, CX was 8000h, so low 64k
	CALL	MAPEMS2ND		; low to high, map in the second 64k
	JMP	SHORT RS128B_MAINLPEND
	EVEN
RS128B_SEGLOW2:
	CALL	MAPEMS1ST		; high to low, map in the first 64k
	JMP	SHORT RS128B_MAINLPEND
	;
	; We got to the end of the sample.  CF:SI.AX is the current offset
	; in words from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS128B_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS128B_JUMPSIZE
	MOV	BL,0		; LSB of BL = CF
	RCL	BL,1		; BL:SI = integer of offset in words
RS128B_ADJLOOP:
	DB	81h,0EEh	; SUB SI,immed SAMPLLEN
RS128B_LLENPT1:
	DW	0
	SBB	BL,0
        JNZ     RS128B_ADJLOOP
	DB	81h,0FEh	; CMP SI,immed SAMPLEN
RS128B_LENPT2:
	DW	0
        JAE     RS128B_ADJLOOP
	JMP	SHORT RS128B_SEGTEST
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.AX
	; is the current offset in words from the start of the sample.  BP.DX
	; is the sample step in words.
	;
	EVEN
RS128B_JUMPSIZE		EQU	$ - (OFFSET RS128B_SAMPEND + 2)
RS128B_JUMPINSTR	EQU	(RS128B_JUMPSIZE*256) + 0EBh
	SUB	AX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 128k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS128B_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS128B_LIMITPT2:
	DW	0
	MOV	BX,DGROUP		; DS addresses local data again
	MOV	DS,BX
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	; DS addresses the default data segment.
	;
RS128B_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS128B_DONE		; no ramping if already at baseline
	JS	RS128B_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS128B_ABOVE:
	MOV	BL,AL		; get sample in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS128B_VOLPT2:
	DW	0
	ADD	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	DEC	AL		; decrement sample
	LOOPNZ	RS128B_ABOVE	; go again if more and not at baseline
	JMP	SHORT RS128B_DONE
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS128B_BELOW:
	MOV	BL,AL		; save copy in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS128B_VOLPT3:
	DW	0
	ADD	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	INC	AL		; increment sample
	LOOPNZ	RS128B_BELOW	; go again if more and not at baseline
	;
	; All done for now.
	;
RS128B_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ENDIF
;
; RESAMPLEB routine, resamples a "subsequent" channel on the left or right
; (i.e., it uses ADD rather than MOV to put samples in the partial-mix
; buffer).  This is a "shell" routine that selects the fastest available
; subroutine to resample the channel.  If the sample is in EMS, this
; routine maps in the current 64k of it.
;
	EVEN
RESAMPLEB:
	MOV	SI,[BX].CHANSAMPPTR	; SI addresses sample data
	CMP	[SI].SAMPLEN,0		; is the sample less than 64k?
	JS	RESAMPLEB_OVER64
	;
	; Sample less than 64k.  See if the sample is already played through.
	; The normal resampler handles that as a special case.
	;
	CMP	[BX].CHANSAMPDONE,1	; is the sample played through?
	JE	RESAMPLEB_64NORMAL
	;
	; If the sample has zero volume, we have a special resampler for
	; that.
	;
	CMP	[BX].CHANMIXVOL,0
	JE	RESAMPLEB_ZEROVOL
	;
	; Sample has nonzero volume and hasn't been played through.  Map
	; it in if it's EMS.
	;
	CMP	[SI].SAMPEMSFLAG,0
	JE	RESAMPLEB_64CHKEND
	PUSH	[SI].SAMPEMSHANDLE
	CALL	MAPEMS1ST
	;
	; See if we will get to the end of it this time.
	;
RESAMPLEB_64CHKEND:
	MOV	AX,[BX].CHANSTEPFRAC	; multiply sample step by # samples
	MUL	_samplespertick
	MOV	CX,AX			; (accumulate in DI.CX)
	MOV	DI,DX
	MOV	AX,[BX].CHANSTEPINT
	MUL	_samplespertick
	JC	RESAMPLEB_64NORMAL
	ADD	DI,AX
	JC	RESAMPLEB_64NORMAL
	ADD	CX,[BX].CHANSAMPFRAC	; add in current position in sample
	ADC	DI,[BX].CHANSAMPOFFS
	JC	RESAMPLEB_64NORMAL
	MOV	AX,[SI].SAMPLEN		; AX is length of sample in bytes
	SHL	AX,1
	CMP	DI,AX
	JAE	RESAMPLEB_64NORMAL
	JMP	RS64BN
	EVEN
RESAMPLEB_64NORMAL:
	JMP	RS64B
	EVEN
RESAMPLEB_ZEROVOL:
	JMP	RSBZ
	;
	; Sample 64k or longer.  We won't make any special resamplers for
	; this rare case, but maybe we can use a 64k resampler.  First
	; check whether the sample has already played through.  The normal
	; resampler handles that as a special case.
	;
	EVEN
RESAMPLEB_OVER64:
	CMP	[BX].CHANSAMPDONE,1	; is the sample played through?
	JE	RESAMPLEB_128NORMAL
	;
	; If the sample has zero volume, we have a special resampler for
	; that.
	;
	CMP	[BX].CHANMIXVOL,0
	JE	RESAMPLEB_ZEROVOL
	;
	; Sample has nonzero volume and has not played through.  Map it in
	; if it's in EMS.
	;
	CMP	[SI].SAMPEMSFLAG,0
	JE	RESAMPLEB_128CHKEND
	PUSH	[SI].SAMPEMSHANDLE
	CMP	[BX].CHANSAMPHALF,1
	JE	RESAMPLEB_2NDHALF
	CALL	MAPEMS1ST
	JMP	SHORT RESAMPLEB_GETSPOT
	EVEN
RESAMPLEB_2NDHALF:
	CALL	MAPEMS2ND
	JMP	SHORT RESAMPLEB_128NORMAL	; already past 64k
	;
	; We can use a 64k resampler if we're guaranteed to stay within 64k
	; of the sample start, so try that.
	;
	EVEN
RESAMPLEB_128CHKEND:
	MOV	AX,[SI].SAMPSEG		; if already above 64k, forget it
	CMP	AX,[BX].CHANSAMPSEG
	JB	RESAMPLEB_128NORMAL
RESAMPLEB_GETSPOT:
	MOV	AX,[BX].CHANSTEPFRAC	; multiply sample step by # samples
	MUL	_samplespertick
	MOV	CX,AX			; (accumulate in DI.CX)
	MOV	DI,DX
	MOV	AX,[BX].CHANSTEPINT
	MUL	_samplespertick
	JC	RESAMPLEB_128NORMAL
	ADD	DI,AX
	JC	RESAMPLEB_128NORMAL
	ADD	CX,[BX].CHANSAMPFRAC	; add in current position in sample
	ADC	DI,[BX].CHANSAMPOFFS
	JC	RESAMPLEB_128NORMAL
	JMP	RS64BN
	EVEN
RESAMPLEB_128NORMAL:
	JMP	RS128B
