; vt220v/asm	- DEC VT220 series terminal emulator
; Kim J. Watt	- 27-jan-87
;
;	clear video display
;
CLS	LD	HL,VIDEO	; start of video area
	LD	BC,24*128	; char length
CLSLOOP	LD	(HL),' '	; blank to character
	INC	HL		; bump pointer
	LD	(HL),0		; clear attribute
	INC	HL		; bump pointer
	DEC	BC		; less counter
	LD	A,B		; get msb
	OR	C		; or with lsb
	JR	NZ,CLSLOOP	; go for count
	RET			; and exit
;
;	refresh the video display
;
REFRESH	LD	HL,(WINDOW)	; get table offset
	LD	DE,VIDEO	; start of table
	ADD	HL,DE		; HL -> current data
	LD	DE,3C00H	; display memory
;
;	loop to refresh each line
;
REFHLP	LD	B,64		; char / row counter
REFVLP	LD	A,(HL)		; get display data
	INC	HL		; skip by 2 for next
	INC	HL
	LD	(DE),A		; to display
	INC	DE		; bump video pointer
	DJNZ	REFVLP		; go vertical looper
	LD	BC,128		; offset to next row
	ADD	HL,BC		; next data
	LD	A,D		; get video address
	CP	40H		; end of video?
	JR	C,REFHLP	; go next line if more
	RET			; and exit OR
;
;	if cursor is on monitor, save the current char
;	if the cursor is enabled, turn it on
;
UPDCURS	PUSH	BC		; save
	CALL	CALCVID		; get video address
	POP	BC		; restore
	RET	C		; not visible
	LD	A,(HL)		; get video data
	LD	(VCDATA),A	; save the char
	BIT	1,(IY+1)	; cursor off?
	RET	NZ		; go if yes
	LD	(HL),143	; load cursor char
	RET			; and exit
;
;	if cursor is on monitor, turn it off
;	and restore the saved character
;
FIXCURS	PUSH	BC		; save
	CALL	CALCVID		; check if displayable
	POP	BC		; restore
	RET	C		; not visible
	LD	(HL),A		; restore character
	RET			; and exit
;
;	execute a reverse linefeed/carriage return
;
DORCR	CALL	FIXCURS		; restore cursor
	LD	BC,(CURSOR)	; get current cursor
	LD	C,0		; home it
	LD	(CURSOR),BC	; update
	CALL	UPDCURS		; update cursor
	RET			; and exit
;
;	do a reverse linefeed
;
DORLF	LD	BC,(CURSOR)	; get current cursor
	DEC	B		; decrement row
	LD	A,B		; get row
	CP	-1		; at bottom row?
	JR	NZ,CKUNL	; check for newline
;
	BIT	5,(IY+1)	; auto wrap?
	LD	B,23		; reset to bottom
	JR	NZ,CKUNL	; go if wrap
	CALL	SCROLLU		; else scroll up
	LD	BC,(CURSOR)	; keep the cursor
CKUNL	BIT	4,(IY+0)	; newline?
	JR	Z,CKUNU		; nope, continue
	LD	C,0		; reset to home
CKUNU	CALL	FIXCURS		; restore current char
	LD	(CURSOR),BC	; set new cursor
	CALL	UPDCURS		; write the cursor
	RET			; and exit
;
;	execute a linefeed/carriage return
;
DOCR	CALL	FIXCURS		; restore cursor
	LD	BC,(CURSOR)	; get current cursor
	LD	C,0		; home it
	LD	(CURSOR),BC	; restore
	CALL	UPDCURS		; update new cursor
	RET			; and exit
;
DOLF	LD	BC,(CURSOR)	; get current cursor
	INC	B		; bump row
	LD	A,B		; get row
	CP	24		; at bottom row?
	JR	C,CKDNL		; check for newline
;
	BIT	5,(IY+1)	; auto wrap?
	LD	B,0		; reset to top
	JR	NZ,CKDNL	; go if wrap
	CALL	SCROLLD		; else scroll down
	LD	BC,(CURSOR)	; keep the cursor
CKDNL	BIT	4,(IY+0)	; newline?
	JR	Z,CKDNU		; nope, continue
	LD	C,0		; reset to home
CKDNU	CALL	FIXCURS		; restore current char
	LD	(CURSOR),BC	; set new cursor
	CALL	UPDCURS		; write the cursor
	RET			; and exit
;
;	scroll the display down
;
SCROLLD	LD	HL,(TSCROLL)	; get top of scroll area
	LD	A,(BSCROLL+1)	; get bottom row
	SUB	L		; bot - top = lines -1
	LD	B,A		; B = line count
	LD	DE,VIDEO	; start video memory
	ADD	HL,DE		; HL -> first line
	LD	D,H		; pass to DE
	LD	E,L
	INC	H		; next source line
	LD	C,0		; init row count
;
SCROLD1	LD	A,(HL)		; get source
	LD	(DE),A		; copy it
	INC	DE		; bump dest
	INC	HL		; bump source
	DEC	C		; less row count
	JR	NZ,SCROLD1	; go for col counter
	DJNZ	SCROLD1		; go for row counter
;
;	clear the last line
;
	EX	DE,HL		; HL -> last line
	LD	B,128		; init counter
;
SCROLD2	LD	(HL),' '	; space char
	INC	HL		; bump pointer
	LD	(HL),0		; clear attribute
	INC	HL		; bump pointer
	DJNZ	SCROLD2		; go for 128 chars
	CALL	REFRESH		; refresh the video
	RET			; and exit
;
;	scroll the display up
;
SCROLLU	LD	HL,(TSCROLL)	; get top of scroll area
	LD	A,(BSCROLL+1)	; get bottom row
	SUB	L		; bot - top = lines -1
	LD	HL,(BSCROLL)	; get bottom line
	LD	DE,VIDEO	; start of video table
	ADD	HL,DE		; HL = source
	LD	BC,255		; offset to end of line
	ADD	HL,BC		; HL -> source end
	LD	D,H		; pass to DE
	LD	E,L
	DEC	H		; adjust source
	LD	B,A		; pass row count
	LD	C,0		; init row count
;
SCROLU1	LD	A,(HL)		; get source
	LD	(DE),A		; copy it
	DEC	DE		; dec dest
	DEC	HL		; dec source
	DEC	C		; less row count
	JR	NZ,SCROLU1	; go for col counter
	DJNZ	SCROLU1		; go for row counter
;
;	clear the last line
;
	EX	DE,HL		; HL -> first line
	LD	B,128		; init counter
;
SCROLU2	LD	(HL),0		; clear attribute
	DEC	HL		; bump pointer
	LD	(HL),' '	; load data
	DEC	HL		; bump pointer
	DJNZ	SCROLU2		; go for 128 chars
	CALL	REFRESH		; refresh the video
	RET			; and exit
;
;	output character in C to data buffer
;	update video display to reflect new char
;
OUTCHAR	LD	HL,VIDEO	; point to buffer
	LD	DE,(CURSOR)	; get cursor address
	ADD	HL,DE		; HL => character
	LD	(HL),C		; to buffer
	INC	HL		; bump pointer
	LD	A,(ATTRIB)	; get current attribute
	LD	(HL),A		; load to memory
	CALL	CALCVID		; get video address
	RET	C		; not displayable
;
	LD	A,C		; get data
	LD	(VCDATA),A	; save the character
	BIT	1,(IY+1)	; cursor on?
	RET	Z		; go if yes
	LD	(HL),A		; else load to monitor
	RET			; and exit
;
;	check if character appears on the monitor
;	adjust cursors to actual row/col
;
CALCVID	LD	HL,(WINDOW)	; get window offset
	LD	DE,(CURSOR)	; get cursor
;
;	check if cursor column is valid
;
	LD	A,E		; get cursor col
	SUB	L		; compare to display strt
	JR	C,MOVL		; < st of disply, move?
	CP	64*2		; displayable?
	JR	NC,MOVR		; > end of dsply, move?
	SRL	A		; /2
	LD	L,A		; save col offset
;
;	column is valid
;
	LD	A,D		; get cursor row
	SUB	H		; compare to display strt
	JR	C,MOVU		; < 1st row, move?
	CP	16		; displayable?
	JR	NC,MOVD		; > lst row, move?
;
;	valid row and column
;	calculate physical display address
;
	LD	H,A		; pass in H
	XOR	A		; clear byte
	SRL	H		; H/2
	RRA			; catch the bit
	SRL	H		; H/4
	RRA			; catch the bit
	OR	L		; merge column
	LD	L,A		; update
	LD	A,H		; get row
	ADD	A,3CH		; add video offset
	LD	H,A		; update
;
;	cursor is displayable
;
OUTCHR2	LD	A,(VCDATA)	; get saved char
	OR	A		; set NCy for OK
	RET			; and exit
;
;	cursor is outside of display area
;
OUTMON	SCF			; set carry for non video
	RET			; and exit
;
;	cursor has moved out of range of the monitor
;	check if autocursor is on
;	if yes, move window and refresh the display
;
MOVR	BIT	6,(IY+2)	; on?
	JR	NZ,OUTMON	; go if not
	LD	A,(WINDOW)	; get offset column
	ADD	A,32		; jumps possible
	LD	(WINDOW),A	; update column
	CALL	REFRESH		; refresh the video
	JP	CALCVID		; try again
;
MOVL	BIT	6,(IY+2)	; on?
	JR	NZ,OUTMON	; outside of monitor
	LD	A,(WINDOW)	; get offset column
	SUB	32		; jumps possible
	LD	(WINDOW),A	; update column
	CALL	REFRESH		; refresh video
	JP	CALCVID		; try again
;
MOVU	BIT	6,(IY+2)	; on?
	JR	NZ,OUTMON	; outside of monitor
	LD	A,(WINDOW+1)	; get offset row
	SUB	8		; jumps gap
	LD	(WINDOW+1),A	; update
	CALL	REFRESH		; update monitor
	JP	CALCVID		; try again
;
MOVD	BIT	6,(IY+2)	; on?
	JR	NZ,OUTMON	; go if not
	LD	A,(WINDOW+1)	; get offset row
	ADD	A,8		; jump size
	LD	(WINDOW+1),A	; update
	CALL	REFRESH		; update display
	JP	CALCVID		; try again
;
;	move cursor to right
;
CURR	LD	A,(CURSOR)	; get column
	CP	78*2		; valid?
	JR	C,CURR0		; 0-78 can move!
	BIT	1,(IY+0)	; 80 column?
	JR	Z,CURR2		; yes, out of bounds
	CP	127*2		; valid?
	JR	NC,CURR2	; nope, out of range
CURR0	CALL	FIXCURS		; restore cursor char
	INC	E		; bump cursor
	INC	E		; skip attribute
	JR	CURCOM		; go common
CURR2	SCF			; set carry for no
	RET			; and exit
;
CURL	LD	A,(CURSOR)	; get cursor column
	OR	A		; end already?
	SCF			; set carry for yes
	RET	Z		; do not move
	CALL	FIXCURS		; restore cursor char
	DEC	E		; decrement cursor
	DEC	E
	JR	CURCOM		; go common
;
CURU	LD	A,(CURSOR+1)	; get cursor row
	OR	A		; at 0?
	SCF			; preset carry
	RET	Z		; go if yes, do not move
	CALL	FIXCURS		; fix cursor char
	DEC	D		; decrement row
	JR	CURCOM		; and go common
;
CURD	LD	A,(CURSOR+1)	; get cursor row
	CP	23		; in range?
	JR	C,CURD0		; go if yes
	SCF			; else set carry for no
	RET			; and exit
CURD0	CALL	FIXCURS		; restore cursor char
	INC	D		; bump row
	JR	CURCOM		; go common
;
CURCOM	LD	(CURSOR),DE	; update
	CALL	UPDCURS		; update cursor
	OR	A		; clear carry for OK
	RET			; and exit
;
;	read char from comms device and save in
;	the input spool buffer
;
READCI	LD	IX,CIDCB	; comms input DCB
;
;	read the device status to see if char available
;
	IN	A,(COSTAT)	; read comms status
	BIT	7,A		; RX data ready?
	RET	Z		; go if no input data
;
;	check for any hardware errors
;
;	AND	00111000B	; test for errors
;	JR	NZ,INHARD	; hardware input error
;
;	read the data and save in the input buffer
;
	IN	A,(CODATA)	; read comms char
	CALL	PUTBUFF		; add to input buffer
;
;	check if > 192 chars in buffer
;	and will send XOFF
;
	LD	L,(IX+2)	; get length
	LD	H,(IX+2)
	LD	E,(IX+4)	; get chars in buffer
	LD	D,(IX+5)
	SRL	H		; HL / 2 = half of buff
	RR	L
	OR	A		; clear carry
	SBC	HL,DE		; compare
	JR	NC,NOHALF	; not half full yet
;
;	have more than half, check if already sent
;	the XOFF already
;
	BIT	7,(IX+10)	; holding?
	RET	NZ		; go if yes
	LD	A,XOFF		; load data
	CALL	PUTBUFF		; add to output buffer
	RET	NC		; could not send
	SET	7,(IX+10)	; tag as sent
	RET			; and exit
;
;	less than half full
;	check if less than 1/4 and have sent
;	a XOFF and need to clear it
;
NOHALF	BIT	7,(IX+10)	; paused sender?
	RET	Z		; go if not
;
	LD	L,(IX+2)	; get length
	LD	H,(IX+3)
	SRL	H		; HL / 2
	RR	L
	SRL	H		; HL / 4
	RR	L
	OR	A		; clear carry
	SBC	HL,DE		; compare
	RET	NC		; more than 1/4
;
	LD	A,XON		; can turn back on
	CALL	PUTBUFF		; add to output buffer
	RET	NC		; could not send
	RES	7,(IX+10)	; clear holding
	RET			; and exit
;
;	write character from output SPOOL buffer
;	to the communication interface
;
WRITCO	LD	IX,CODCB	; comms output DCB
	BIT	7,(IX+10)	; paused?
	RET	NZ		; go if yes
;
;	check if transmitter can send a char
;
	IN	A,(COSTAT)	; read comms status
	BIT	6,A		; TX ready?
	RET	Z		; exit if not
;
;	see if any chars available in buffer
;
	CALL	GETBUFF		; any data?
	RET	NC		; go if not
	OUT	(CODATA),A	; else output the char
	RET			; and exit
;
;	check if have received any input chars
;	and process
;
WRITDO	LD	IX,DODCB	; display DCB
	CALL	GETBUFF		; any data?
	RET	NC		; go if not
	JR	PUTCDP		; else continue
;
EXECCI	LD	IX,CIDCB	; comms input DCB
	CALL	GETBUFF		; any data?
	RET	NC		; go if not
	LD	C,A		; save char
;
;	check for a control character
;
PUTCDO	LD	A,C		; get char
PUTCDP	CP	7FH		; DEL?
	JR	Z,PUTDEL	; go if yes
	CP	20H		; 01-1F?
	JR	C,EXCC0		; go if yes, special
	CP	80H		; C1 range?
	JR	C,EXCOM1	; go if not
	CP	0A0H		; 80-9F?
	JR	C,EXCC1		; go if yes
;
;	test if ESC/CSI sequence pending
;
EXCOM1	BIT	6,(IX+10)	; ESC pending?
	JR	NZ,ESCPND	; go if yes
	BIT	5,(IX+10)	; CSI pending?
	JR	NZ,CSIPND	; go if yes
;
;	normal output character
;
DISPLAY	CALL	OUTCHAR		; output the character
	CALL	CURR		; move cursor right
	RET			; do not linefeed
;
;	DEL character
;
PUTDEL	LD	C,' '		; output space at cursor
	CALL	OUTCHAR		; output character
	CALL	CURL		; cursor left
	RET			; and exit
;
;	have an escape sequence pending
;	lookup next character in table and branch
;
ESCPND	LD	HL,ESCTBL	; escape lookup table
	RES	6,(IX+10)	; clear escape pending
	CALL	LOOKUP		; lookup the character
	LD	C,191		; unknown data
	JR	C,DISPLAY	; send if unknown
	JP	(HL)		; else go service
;
;	have a CSI sequence pending
;	save data and check for action
;
CSIPND	RES	5,(IX+10)	; clear pending
	RET			; and exit
;
;	have a new control character, go table entry
;
EXCC0	LD	HL,INC0TBL	; c0 table
	JR	EXCCC		; go common
EXCC1	LD	HL,INC1TBL	; c1 table
;
EXCCC	ADD	A,A		; *2 for word table
	LD	E,A		; pass to E
	LD	D,0		; clear msb
	ADD	HL,DE		; HL -> vector
	LD	A,(HL)		; get lsb
	INC	HL		; bump pointer
	LD	H,(HL)		; get msb
	LD	L,A		; HL = vector
	JP	(HL)		; go table!
;
;	services for control char input
;
ICNUL	RET			; ignore data
;
ICENQ	RET			; ignore for now
;
ICBEL	LD	A,'B'		; received data
	LD	(3C3FH),A	; load it
	RET			; and exit
;
ICBS	CALL	CURL		; move cursor left
	RET			; no wrap
;
ICHT	CALL	CURR		; move right
	RET	C		; cannot move
	LD	A,(CURSOR)	; get column
	AND	07H		; tab stop?
	JR	NZ,ICHT		; continue if not
	RET			; else stop
;
ICLF	CALL	DOLF		; do a linefeed
	RET			; and exit
;
ICVT	JR	ICLF		; do as LF
;
ICFF	JR	ICLF		; do as LF
;
ICCR	CALL	DOCR		; do a carriage return
	RET			; and exit
;
ICSO	RET			; finish
;
ICSI	RET			; finish
;
ICDC1	LD	IX,CODCB	; comms output DCB
	SET	7,(IX+10)	; mark as HOLD
	RET			; and exit
;
ICDC3	LD	IX,CODCB	; comms output DCB
	RES	7,(IX+10)	; release HOLD
	RET			; and exit
;
ICCAN	RES	5,(IY+2)	; clear ESC
	RES	4,(IY+2)	; clear CSI
	RET			; and exit
;
ICSUB	RES	5,(IY+2)	; clear ESC
	RES	4,(IY+2)	; clear CSI
	LD	C,'?'		; should be reverse
	CALL	OUTCHAR		; at current cursor
	RET			; and exit
;
ICESC	SET	5,(IY+2)	; set ESC sequence
	RET			; and exit
;
ICIND	CALL	DOLF		; linefeed only
	RET			; and exit
;
ICNEL	CALL	DOCR		; first col
	CALL	DOLF		; and linefeed
	RET			; and exit
;
ICHTS	RET			; set horiz tab
;
ICRI	CALL	DORLF		; reverse linefeed
	RET			; and exit
;
ICSS2	RET			; G2 char set
;
ICSS3	RET			; G3 char set
;
ICDCS	SET	2,(IY+2)	; set DCS start
	RET			; and exit
;
ICCSI	SET	4,(IY+2)	; set CSI sequence
	LD	HL,CSIBUF	; CSI buffer
	LD	(HL),0		; clear char counter
	RET			; and exit
;
ICST	RES	2,(IY+2)	; clear DCS string
	RET			; and exit
;
;	input control key vector table
;
INC0TBL	DW	ICNUL	; 00
	DW	ICNUL	; 01
	DW	ICNUL	; 02
	DW	ICNUL	; 03
	DW	ICNUL	; 04
	DW	ICENQ	; 05
	DW	ICNUL	; 06
	DW	ICBEL	; 07
	DW	ICBS	; 08
	DW	ICHT	; 09
	DW	ICLF	; 0A
	DW	ICLF	; 0B
	DW	ICLF	; 0C
	DW	ICCR	; 0D
	DW	ICSO	; 0E
	DW	ICSI	; 0F
	DW	ICNUL	; 10
	DW	ICDC1	; 11
	DW	ICNUL	; 12
	DW	ICDC3	; 13
	DW	ICNUL	; 14
	DW	ICNUL	; 15
	DW	ICNUL	; 16
	DW	ICNUL	; 17
	DW	ICCAN	; 18
	DW	ICNUL	; 19
	DW	ICSUB	; 1A
	DW	ICESC	; 1B
	DW	ICNUL	; 1C
	DW	ICNUL	; 1D
	DW	ICNUL	; 1E
	DW	ICNUL	; 1F
;
INC1TBL	DW	ICNUL	; 80
	DW	ICNUL	; 81
	DW	ICNUL	; 82
	DW	ICNUL	; 83
	DW	ICIND	; 84
	DW	ICNEL	; 85
	DW	ICNUL	; 86
	DW	ICNUL	; 87
	DW	ICHTS	; 88
	DW	ICNUL	; 89
	DW	ICNUL	; 8A
	DW	ICNUL	; 8B
	DW	ICNUL	; 8C
	DW	ICRI	; 8D
	DW	ICSS2	; 8E
	DW	ICSS3	; 8F
	DW	ICDCS	; 90
	DW	ICNUL	; 91
	DW	ICNUL	; 92
	DW	ICNUL	; 93
	DW	ICNUL	; 94
	DW	ICNUL	; 95
	DW	ICNUL	; 96
	DW	ICNUL	; 97
	DW	ICNUL	; 98
	DW	ICNUL	; 99
	DW	ICNUL	; 9A
	DW	ICCSI	; 9B
	DW	ICST	; 9c
	DW	ICNUL	; 9D
	DW	ICNUL	; 9E
	DW	ICNUL	; 9F
;
;	escape sequence lookup table
;
ESCTBL	DB	'D'	; IND
	DW	ICIND
	DB	'E'	; NEL
	DW	ICNEL
	DB	'H'	; HTS
	DW	ICHTS
	DB	'M'	; RI
	DW	ICRI
	DB	'N'	; SS2
	DW	ICSS2
	DB	'O'	; SS3
	DW	ICSS3
	DB	'P'	; DCS
	DW	ICDCS
	DB	5BH	; CSI
	DW	ICCSI
	DB	5CH	; ST
	DW	ICST
	DB	-1
;
