; vt100v/asm	- DEC VT100 series terminal emulator
; Kim J. Watt	/ 28-Mar-87
;
;	WRITDO - write data to display output
;
WRITDO	LD	IX,DODCB	; get output DCB
	CALL	GETBUFF		; get character
	RET	NC		; go if no data
;
;	check if display or control char
;
	OR	A		; data/command
	JP	M,SPECIAL	; go if special
	CP	20H		; control?
	JR	C,CONTROL	; go if yes
;
;	normal display character
;
	LD	HL,(BOFF)	; get buffer offset
	LD	DE,BUFFER	; start of buffer
	ADD	HL,DE		; to buffer
	LD	(HL),A		; write data
	LD	(VDATA),A	; video data also
	CALL	SPEC86		; cursor right
	RET			; and exit
;
;	have a special char, lookup in table
;
SPECIAL	LD	DE,SPCTBL	; lookup table
	JR	CTLSPC		; go common
CONTROL	LD	DE,CTLTBL	; lookup table
;
CTLSPC	ADD	A,A		; *2 for word table
	LD	L,A		; pass to L
	LD	H,0		; clear msb
	ADD	HL,DE		; hl -> vector
	LD	A,(HL)		; get lsb
	INC	HL		; bump pointer
	LD	H,(HL)		; get msb
	LD	L,A		; HL = vector
	JP	(HL)		; go service!
;
;	control code lookup table
;
CTLTBL	DW	CTRL00		; 00 - ignore
	DW	CTRL00		; 01 - ignore
	DW	CTRL00		; 02 - ignore
	DW	CTRL00		; 03 - ignore
	DW	CTRL00		; 04 - ignore
	DW	CTRL00		; 05 - ignore
	DW	CTRL00		; 06 - ignore
	DW	CTRL07		; 07 - bell
	DW	CTRL08		; 08 - backspace
	DW	CTRL09		; 09 - tab
	DW	CTRL0A		; 0A - linefeed
	DW	CTRL0B		; 0B - vertical tab
	DW	CTRL0C		; 0C - form feed
	DW	CTRL0D		; 0D - carriage return
	DW	CTRL00		; 0E - ignore
	DW	CTRL00		; 0F - ignore
	DW	CTRL00		; 10 - ignore
	DW	CTRL00		; 11 - ignore
	DW	CTRL00		; 12 - ignore
	DW	CTRL00		; 13 - ignore
	DW	CTRL00		; 14 - ignore
	DW	CTRL00		; 15 - ignore
	DW	CTRL00		; 16 - ignore
	DW	CTRL00		; 17 - ignore
	DW	CTRL00		; 18 - ignore
	DW	CTRL00		; 19 - ignore
	DW	CTRL1A		; 1A - reverse linefeed
	DW	CTRL00		; 1B - ignore
	DW	CTRL00		; 1C - ignore
	DW	CTRL1D		; 1D - reverse CR
	DW	CTRL00		; 1E - ignore
	DW	CTRL00		; 1F - ignore
;
;	special lookup table of vectors
;
SPCTBL	DW	SPEC80		; home
	DW	SPEC81		; clear BOF
	DW	SPEC82		; clear EOF
	DW	SPEC83		; cursor down
	DW	SPEC84		; cursor up
	DW	SPEC85		; cursor left
	DW	SPEC86		; cursor right
	DW	SPEC87		; next line
	DW	SPEC88		; set tab
	DW	SPEC89		; clear tab
	DW	SPEC8A		; insert line
	DW	SPEC8B		; delete line
	DW	SPEC8C		; insert char
	DW	SPEC8D		; delete char
	DW	SPEC8E		; erase line
	DW	SPEC8F		; erase display
	DW	SPEC90		; save cursor
	DW	SPEC91		; restore cursor
	DW	SPEC92		; cursor up x lines
	DW	SPEC93		; cursor down x lines
	DW	SPEC94		; cursor right x cols
	DW	SPEC95		; cursor left x cols
	DW	SPEC96		; cursor line x col y
	DW	SPEC97		; clear eol
	DW	SPEC98		; clear bol
	DW	SPEC99		; clear line
	DW	SPEC9A		; clear fram
;
;*****************
;
;	service handlers for special display controls
;
CTRL00
SPEC00	RET			; unused
;
;	clear from cursor to beginning of frame
;
SPEC81	LD	BC,(BOFF)	; get buffer offset
	LD	HL,BUFFER	; start of buffer
SPEC81A	LD	(HL),' '	; to buffer
	INC	HL		; bump it
	DEC	BC		; less counter
	LD	A,B		; get msb
	OR	C		; or with lsb
	JR	NZ,SPEC81A	; go till end
	CALL	REFRESH		; update display
	JP	CURUPD		; update cursor
;
;	clear from cursor to end of frame
;
SPEC82	LD	BC,(BOFF)	; get buffer offset
	LD	HL,24*128	; buffer size
	OR	A		; clear carry
	SBC	HL,BC		; distance to move
	LD	B,H		; pass to BC
	LD	C,L		; BC = length
	LD	HL,(BOFF)	; get offset
	LD	DE,BUFFER	; start of data
	ADD	HL,DE		; point to it
	JR	SPEC81A		; fill
;
;	cursor down
;
SPEC83	LD	A,(BROW)	; get buffer row
	CP	23		; at bottom?
	RET	Z		; go if yes
	INC	A		; +1
	LD	(BROW),A	; update
	LD	HL,(BOFF)	; get buffer offset
	LD	DE,128		; row length
	ADD	HL,DE		; next row
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; update new
;
;	cursor up
;
SPEC84	LD	A,(BROW)	; get buffer row
	OR	A		; at top?
	RET	Z		; go if yes, no change
	DEC	A		; decrement
	LD	(BROW),A	; update
	LD	HL,(BOFF)	; get buffer offset
	LD	DE,128		; row offset
	OR	A		; clear carry
	SBC	HL,DE		; HL = new
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; set new
;
;	cursor right
;
SPEC86	LD	A,(BCOL)	; get buffer column
	INC	A		; bump it
	CP	80		; can move?
	JR	C,SPEC05A	; go if yes
	BIT	4,(IY+0)	; 132 col on?
	RET	Z		; go if not
	CP	128		; at end already?
	RET	NC		; go if not
;
;	ok to move right
;
SPEC05A	LD	(BCOL),A	; update buffer column
	LD	HL,(BOFF)	; get buffer offset
	INC	HL		; next position
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; set new
;
;	cursor left
;
CTRL08
SPEC85	LD	A,(BCOL)	; get buffer column
	OR	A		; at beginning?
	RET	Z		; go if yes
	DEC	A		; else move back
	LD	(BCOL),A	; update
	LD	HL,(BOFF)	; get buffer offset
	DEC	HL		; move back
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	bell
;
CTRL07	LD	HL,VIDEO	; display
	LD	DE,VIDEO+1	; start +1
	LD	BC,16*64-1	; length -1
	LD	(HL),191	; all pixels on
	LDIR			; white out video
	CALL	REFRESH		; update display
	JP	CURUPD		; and update cursor
;
;	tab
;
CTRL09	LD	HL,TABTBL	; tab lookup table
	LD	DE,(BCOL)	; get buffer column
	LD	D,0		; clear msb
	ADD	HL,DE		; point to current col
	LD	A,E		; save column
;
CTRL09A	CP	79		; 0 - 78?
	JR	C,CTRL09B	; go if yes, ok to move
	BIT	4,(IY+0)	; 132 col?
	JR	Z,CTRL09C	; go if not, at end
	CP	127		; 80 - 126?
	JR	NC,CTRL09C	; go if not, stop here
;
CTRL09B	INC	HL		; bump pointer
	INC	A		; bump column
	BIT	7,(HL)		; empty?
	JR	NZ,CTRL09A	; yes, continue
;
CTRL09C	LD	BC,(BCOL)	; get current column
	LD	(BCOL),A	; update new
	SUB	C		; new - old = gap
	LD	C,A		; pass to C
	LD	B,0		; BC = gap
	LD	HL,(BOFF)	; get buffer offset
	ADD	HL,BC		; add gap
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	linefeed
;
CTRL0B
CTRL0A	LD	A,(BROW)	; get buffer row
	CP	23		; at bottom?
	JP	Z,SCROLLU	; go if yes, scroll
	JP	SPEC83		; else cursor down
;
;	newline
;
SPEC87	LD	A,(IY+0)	; get flags
	PUSH	AF		; save
	SET	5,(IY+0)	; force newline
	CALL	CTRL0D		; carriage return
	POP	AF		; restore
	LD	(IY+0),A	; update
	RET			; and exit
;
;	cursor to beginning of line
;
CTRL0D	LD	A,(BCOL)	; get current column
	LD	C,A		; and save data
	XOR	A		; load zero
	LD	(BCOL),A	; set new column
	LD	B,A		; BC = length moved
	LD	HL,(BOFF)	; get buffer offset
	SBC	HL,BC		; offset moved
	LD	(BOFF),HL	; update new offset
	BIT	5,(IY+0)	; newline?
	JP	NZ,CTRL0A	; go if yes
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	cursor to home
;
SPEC80	LD	HL,(BOFF)	; get buffer offset
	LD	A,H		; get msb
	OR	L		; at home?
	RET	Z		; go if yes, no change
	LD	HL,0		; else clear
	LD	(BCOL),HL	; update
	LD	(BOFF),HL	; buffer offset
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	cursor to end of line
;
SPEC0E	LD	BC,(BCOL)	; get current column
	LD	A,79		; max 80 col
	BIT	4,(IY+0)	; 132 col?
	JR	Z,SPEC0E1	; go if not
	LD	A,127		; max 132 col
;
SPEC0E1	LD	(BCOL),A	; set new column
	SUB	C		; A = distance moved
	RET	Z		; no change!
	LD	C,A		; pass to C
	LD	B,0		; BC = gap
	LD	HL,(BOFF)	; get buffer offset
	ADD	HL,BC		; HL = new posit
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	insert line
;
SPEC8A	RET
;
;	delete line
;
SPEC8B	RET
;
;	insert char
;
SPEC8C	RET
;
;	delete char
;
SPEC8D	RET
;
;	reverse tab
;
CTRL19	LD	HL,TABTBL	; tab lookup table
	LD	DE,(BCOL)	; get buffer column
	LD	D,0		; clear msb
	ADD	HL,DE		; point to current col
	LD	A,E		; save column
;
CTRL19A	OR	A		; at end?
	JR	Z,CTRL19C	; go if yes, save
CTRL19B	DEC	HL		; backspace tab
	DEC	A		; less column
	BIT	7,(HL)		; tab stop set?
	JR	Z,CTRL19A	; continue if not
CTRL19C	LD	BC,(BCOL)	; get current column
	LD	(BCOL),A	; update new
	SUB	C		; new - old = gap
	LD	C,A		; pass to C
	LD	B,-1		; BC = negative gap
	LD	HL,(BOFF)	; get buffer offset
	ADD	HL,BC		; subtract gap
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	reverse linefeed
;
CTRL1A	LD	A,(BROW)	; get buffer row
	OR	A		; at top?
	CALL	Z,SCROLLD	; scroll down if yes
	RET			; leave cursor
;
;	reverse carriage return
;
CTRL1D	RET
;
;	erase line
;
SPEC8E	RET
;
;	save cursor position
;
SPEC90	LD	HL,(BCOL)	; get buffer row/col
	LD	(SBCOL),HL	; save it
	LD	HL,(BOFF)	; get buffer offset
	LD	(SBOFF),HL	; save it
	RET			; and exit
;
;	restore cursor position
;
SPEC91	LD	HL,(SBCOL)	; saved row/col
	LD	(BCOL),HL	; restore
	LD	HL,(SBOFF)	; saved offset
	LD	(BOFF),HL	; restore
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	cursor up x lines
;
SPEC92	CALL	GETBUFF		; get count
	LD	B,A		; save
	LD	A,(BROW)	; get buffer row
SPEC92A	OR	A		; at top?
	JR	Z,SPEC92B	; go if yes
	DEC	A		; less one
	DJNZ	SPEC92A		; go for count
SPEC92B	LD	(BROW),A	; update new row
	JP	CURCALC		; cal new cursor address
;
;	cursor down x lines
;
SPEC93	CALL	GETBUFF		; get count
	LD	B,A		; save
	LD	A,(BROW)	; get buffer row
SPEC93A	CP	23		; at bottom?
	JR	Z,SPEC93B	; go if yes
	INC	A		; one more
	DJNZ	SPEC93A		; go for count
SPEC93B	LD	(BROW),A	; update
	JP	CURCALC		; calc new cursor
;
;	cursor right x cols
;
SPEC94	CALL	GETBUFF		; get count
	LD	B,A		; save
	LD	A,(BCOL)	; get column
SPEC94A	CP	79		; can move?
	JR	C,SPEC94B	; go if yes
	BIT	4,(IY+0)	; 132 cols?
	JR	Z,SPEC94C	; go if not
	CP	127		; can move?
	JR	NC,SPEC94C	; go if not
SPEC94B	INC	A		; bump it
	DJNZ	SPEC94A		; and continue
SPEC94C	LD	(BCOL),A	; update new column
	JP	CURCALC		; calc new cursor
;
;	cursor	left x cols
;
SPEC95	CALL	GETBUFF		; get count
	LD	B,A		; save it
	LD	A,(BCOL)	; get buffer column
SPEC95A	OR	A		; at end?
	JR	Z,SPEC95B	; go if yes
	DEC	A		; else can move
	DJNZ	SPEC95A		; go for count
SPEC95B	LD	(BCOL),A	; update new col
	JP	CURCALC		; new cursor
;
;	move cursor to line x col y
;
SPEC96	CALL	GETBUFF		; get line
SPEC96A	SUB	24		; 0 to 23?
	JR	NC,SPEC96A	; go till in range
	ADD	A,24		; correct
	LD	(BROW),A	; update
	CALL	GETBUFF		; get column
SPEC96B	SUB	128		; 0 to 127?
	JR	NC,SPEC96B	; go till in range
	ADD	A,128		; correct
	LD	(BCOL),A	; update
;
;	update new row/col and offset
;
CURCALC	LD	HL,(BCOL)	; get current row/col
	SLA	L		; left 1 bit
	SRL	H		; row /2
	RR	L		; to low byte
	LD	(BOFF),HL	; update
	CALL	CUROFF		; cursor off
	JP	CURUPD		; and update new
;
;	erase to end of line
;
SPEC97	LD	HL,(BOFF)	; get buffer offset
	LD	BC,BUFFER	; start of data
	ADD	HL,BC		; point to buffer
	LD	A,128		; line length
	LD	BC,(BCOL)	; get current col
	SUB	C		; less length
SPEC97A	LD	(HL),' '	; clear char
	INC	HL		; bump pointer
	DEC	A		; less counter
	JR	NZ,SPEC97A	; go for count
	CALL	REFRESH		; update display
	JP	CURUPD		; update cursor
;
SPEC99	LD	BC,(BCOL)	; get current column
	LD	B,0		; BC = offset
	LD	HL,(BOFF)	; current offset
	LD	DE,BUFFER	; start of buffer
	ADD	HL,DE		; point to place
	OR	A		; clear carry
	SBC	HL,BC		; beginning of line
	LD	A,128		; line length
	JR	SPEC97A		; fill and refresh
;
;	clear from cursor to beginning of line
;
SPEC98	LD	HL,(BOFF)	; get buffer offset
	LD	BC,BUFFER	; start of data
	ADD	HL,BC		; point to position
	LD	A,(BCOL)	; get current column
	INC	A		; 1 relative
SPEC98A	LD	(HL),' '	; clear char
	DEC	HL		; backspace
	DEC	A		; less counter
	JR	NZ,SPEC98A	; go for count
	CALL	REFRESH		; update display
	JP	CURUPD		; update cursor
;
;	clear display and home cursor
;
CTRL0C
SPEC8F	LD	HL,VIDEO	; start of video ram
	LD	DE,VIDEO+1	; next pos
	LD	BC,16*64-1	; length -1
	LD	A,' '		; blank
	LD	(HL),A		; blank
	LDIR			; blank screen
;
	LD	HL,BUFFER	; start of buffer ram
	LD	DE,BUFFER+1	; next pos
	LD	BC,24*128-1	; length -1
	LD	(HL),A		; blank
	LDIR			; blank buffer
;
;	reset pointers to home
;
	LD	(BCOL),BC	; buffer row/col
	LD	(BOFF),BC	; buffer offset
	LD	(VCOL),BC	; video row/col
	LD	(VOFF),BC	; video offset
	LD	(WCOL),BC	; window row/column
	RES	0,(IY+0)	; cursor visible
	JP	CURUPD		; cursor update
;
;	clear frame, leave cursor where it is
;
SPEC9A	LD	HL,VIDEO	; start of display ram
	LD	DE,VIDEO+1	; start +1
	LD	BC,16*64-1	; length -1
	LD	(HL),' '	; blank
	LDIR			; fill
;
	LD	HL,BUFFER	; start of image
	LD	DE,BUFFER+1	; next
	LD	BC,24*128-1	; length -1
	LD	(HL),' '	; space
	LDIR			; fill it
	JP	CURUPD		; draw the cursor
;
;	set tab stop
;
SPEC88	LD	HL,TABTBL	; start of tab table
	LD	DE,(BCOL)	; get buffer column
	LD	D,0		; clear msb
	ADD	HL,DE		; current entry
	LD	(HL),0		; set tab
	RET			; and exit
;
;	clear tab
;
SPEC89	LD	HL,TABTBL	; start of tab table
	LD	DE,(BCOL)	; get buffer col
	LD	D,0		; clear msb
	ADD	HL,DE		; current entry
	LD	(HL),80H	; clear tab
	RET			; and exit
;
;	turn off the cursor if on
;
CUROFF	BIT	0,(IY+0)	; visible?
	RET	NZ		; go if not
	BIT	1,(IY+0)	; cursor off?
	RET	NZ		; go if yes
;
;	is visible and on
;
	LD	HL,(VOFF)	; get video offset
	LD	DE,VIDEO	; start of video memory
	ADD	HL,DE		; HL -> cursor char
	LD	A,(VDATA)	; get data
	LD	(HL),A		; restore real data
	RET			; and exit
;
;	turn the cursor on if off
;
CURON	BIT	0,(IY+0)	; visible?
	RET	NZ		; go if not
	BIT	1,(IY+0)	; on?
	RET	NZ		; go if not
;
;	is visible, write the cursor character
;
CURONA	LD	HL,(VOFF)	; get video offset
	LD	DE,VIDEO	; start of video
	ADD	HL,DE		; HL -> video data
	LD	A,(HL)		; get current char
	LD	(VDATA),A	; save for restore
	LD	(HL),143	; write cursor
	RET			; and exit
;
;	update the cursor position to the display
;
CURUPD	RES	0,(IY+3)	; clear refresh needed
;
UPDCURA	LD	A,(BCOL)	; get buffer column
	LD	BC,(WCOL)	; get window column
	SUB	C		; compare
	JR	C,MWINL		; move window left!
	CP	64		; col 64+?
	JR	NC,MWINR	; move window right!
	LD	(VCOL),A	; set new video column!
;
	LD	A,(BROW)	; get buffer row
	SUB	B		; compare
	JR	C,MWINU		; move window up!
	CP	16		; 0 to 15?
	JR	NC,MWIND	; move window down!
	LD	(VROW),A	; set new video row!
;
;	calculate video offset
;	row*64+column
;
	LD	HL,(VCOL)	; get row column
	SLA	L		; col shift 2 bits
	SLA	L
	SRL	H		; row right
	RR	L		; to low byte
	SRL	H		; row right
	RR	L		; to low byte
	LD	(VOFF),HL	; update video offset
	RES	0,(IY+0)	; cursor is visible
	BIT	0,(IY+3)	; refresh needed?
	CALL	NZ,REFRESH	; refresh if yes
	JP	CURONA		; turn cursor on and ret
;
;	attempt to move window to visible area
;
MWINL	BIT	2,(IY+0)	; auto window on?
	JR	NZ,CURNO	; no cursor if not
	LD	A,(WCOL)	; get window column
	SUB	16		; back 16
	LD	(WCOL),A	; update
	SET	0,(IY+3)	; refresh needed!
	JP	UPDCURA		; try again
;
MWINR	BIT	2,(IY+0)	; auto window on?
	JR	NZ,CURNO	; no cursor if not
	LD	A,(WCOL)	; get window column
	ADD	A,16		; forward 16
	LD	(WCOL),A	; update
	SET	0,(IY+3)	; refresh needed!
	JP	UPDCURA		; try again
;
MWINU	BIT	2,(IY+0)	; auto window on?
	JR	NZ,CURNO	; no cursor if not
	LD	A,(WROW)	; get window row
	SUB	8		; less 8
	LD	(WROW),A	; update
	SET	0,(IY+3)	; refresh needed!
	JP	UPDCURA		; try again
;
MWIND	BIT	2,(IY+0)	; auto window on?
	JR	NZ,CURNO	; no cursor if not
	LD	A,(WROW)	; get window row
	ADD	A,8		; 8 more
	LD	(WROW),A	; update
	SET	0,(IY+3)	; refresh needed
	JP	UPDCURA		; try again
;
;	cursor is not visible
;
CURNO	SET	0,(IY+0)	; invisible cursor
	RET			; nothing to do!
;
;	refresh the video display
;
REFRESH	LD	HL,(WCOL)	; get row/col of window
	SLA	L		; cols left
	SRL	H		; rows right
	RR	L		; catch high bit
	LD	DE,BUFFER	; start of buffer
	ADD	HL,DE		; hl -> display data
	LD	DE,VIDEO	; start of display ram
;
	LD	C,16		; 16 columns
REFLA	LD	B,64		; 64 characters
REFLB	LD	A,(HL)		; get source
	LD	(DE),A		; to video
	INC	HL		; bump source
	INC	DE		; bump dest
	DJNZ	REFLB		; go for this row
	LD	A,L		; get lsb
	ADD	A,64		; offset to next data
	LD	L,A		; update
	JR	NC,REFLC	; go if no carry
	INC	H		; else bump msb
REFLC	DEC	C		; less row counter
	JR	NZ,REFLA	; go outer loop
	RET			; and exit
;
;	scroll down one line
;
SCROLLD	RET
;
;	scroll up one line
;
SCROLLU	CALL	CUROFF		; restore cursor char
	LD	HL,BUFFER+128	; second line
	LD	DE,BUFFER	; first line
	LD	BC,23*128	; lines -1
	LDIR			; move the buffer
	LD	H,D		; pass last line to HL
	LD	L,E		; both at last line
	INC	DE		; bump pointer
	LD	(HL),' '	; blank here
	LD	BC,127		; row length -1
	LDIR			; fill last line
;
;	scroll the video
;
	LD	HL,VIDEO+64	; start line 2
	LD	DE,VIDEO	; start line 1
	LD	BC,64*15	; cols*rows-1
	LDIR			; scroll the video
;
;	check if blank or copy bottom line
;
	LD	A,(WROW)	; get window row
	OR	A		; at top?
	JR	Z,SCROLU1	; go if load data
;
;	clear last line of display
;
	LD	HL,15*64+VIDEO	; last line
	LD	D,H		; pass to DE
	LD	E,L		; both last line
	INC	DE		; bump dest
	LD	(HL),' '	; load blank
	LD	BC,63		; length -1
	LDIR			; fill last line
	JP	CURUPD		; turn cursor on
;
;	copy data to last line of display
;
SCROLU1	LD	DE,15*64+VIDEO	; last display line
	LD	HL,23*128+BUFFER	; last buffer
	LD	BC,64		; length
	LDIR			; move to display
	JP	CURUPD		; turn cursor on
