; vt100k/asm	- DEC VT100 series terminal emulator
; Kim J. Watt	/ 25-Mar-87
;
;	this file performs 2 basic functions
; READKI	read keyboard hardware and place input
;		data into KIBUFF in form of key code
; EXECKI	extract and execute any keyboard input
;
; NOTE:	read keyboard hardware is stand-alone and
;	may be placed into an interrupt
;	if registers are preserved
;
;
;	READKI	- read keyboard hardware
;	place keyboard code into output buffer
;	first byte is keyboard key # shown below
;	values with no character will never be loaded
;	second byte is shift/clear/break key status
;	bit 0=shift, bit 1=clear, bit 2=break
;	data must always be stored with byte pairs
;
;	00 @	01 A	02 B	03 C	04 D
;	05 E	06 F	07 G	08 H	09 I
;	10 J	11 K	12 L	13 M	14 N
;	15 O	16 P	17 Q	18 R	19 S
;	20 T	21 U	22 V	23 W	24 X
;	25 Y	26 Z	27 	28 	29
;	30 	31 	32 0	33 1	34 2
;	35 3	36 4	37 5	38 6	39 7
;	40 8	41 9	42 :	43 ;	44 ,
;	45 -	46 .	47 /	48 ent	49 
;	50    	51 up	52 dn	53 le	54 ri
;	55 	56 	57 	58 	59 
;	60 	61 	62 	63
;
;
READKI	LD	IX,KIDCB	; point to DCB
	BIT	7,(IY+1)	; repeat on/off?
	JR	NZ,GONWKEY	; go if off!
;
;	check if any keys available
;
	LD	A,(387FH)	; get all row key
	OR	A	 	; set flags
	JR	NZ,ADDRPT	; yes, go!
;
;	reset first key counter
;
GONWKEY	LD	HL,600		; double speed value
	LD	(DLY1),HL	; insert delay
;
;	reset between key delay
;
CONTRPT	LD	HL,100		; double speed time
	LD	(DLY2),HL	; update
	JR	KIGO		; go single key
;
;	check for current same as last key
;
ADDRPT	CP	0		; same as last key?
LSTKEY	EQU	$-1
	LD	(LSTKEY),A	; update
	JR	NZ,GONWKEY	; go if not same key!
;
;	same key as last, check for first repeat time
;
	LD	HL,0		; get first key delay
DLY1	EQU	$-2
	LD	A,H		; get msb
	OR	L		; timed out?
	JR	Z,CKDLY2	; go if yes
	DEC	HL		; else decrement it
	LD	(DLY1),HL	; put it back
	JR	KIGO		; and continue
;
;	key is repeating, check for between key delay
;
CKDLY2	LD	HL,0		; get between key count
DLY2	EQU	$-2
	DEC	HL		; decrement
	LD	A,H		; get msb
	OR	L		; check if zero
	LD	(DLY2),HL	; update it
	JR	NZ,KIGO		; go if not time yet
;
;	time to repeat the character
;	clear out the key mask
;
	LD	HL,KEYBRD	; keyboard mask area
	LD	DE,KEYBRD+1	; 7 byte last key mask
	LD	BC,6		; length -1
	LD	(HL),0		; clear a zero
	LDIR			; clear all zeroes
	JR	CONTRPT		; reset counters
;
;	mask area all set for repeat, scan key matrix
;
KIGO	LD	HL,KEYBRD	; work area
	LD	BC,3801H	; keyboard memory
	LD	E,0		; row counter
;
NEXROW	LD	A,(BC)		; get key byte
	BIT	6,C		; last row?
	JR	Z,NLSROW	; go if not
	AND	11111001B	; strip break/clear
NLSROW	LD	D,A		; save unmasked
;
;	compare to mask area to see if it's a new key
;
	XOR	(HL)		; compare to last scan
	LD	(HL),D		; update new key
	AND	D		; same as last key?
	JR	NZ,HAVEIT	; have a new one!
;
;	not a new key, continue to next matrix row
;
	INC	E		; bump row count
	INC	HL		; bump mask byte
	RLC	C		; move key memory
	JP	P,NEXROW	; go if more
	RET			; done, Z flag set
;
;	have a new key, decode into unique keycode
;
HAVEIT	LD	A,E		; get row counter
	ADD	A,A		; *2
	ADD	A,A		; *4
	ADD	A,A		; *8
MKT1	RRC	D		; shift out a bit
	JR	C,MKT2		; go if have a bit
	INC	A		; else bump column #
	JR	MKT1		; and continue
;
;	have a key in A of 00 to 55
;	save character into keyboard input buffer
;
MKT2	CALL	PUTBUFF		; load char to buffer
;
;	load shift/clear/break key status
;
	LD	A,(3880H)	; get shift
	AND	3		; either pressed?
	LD	E,A		; save if zero
	JR	Z,MKT3		; go if neither
	LD	E,1		; else return 1
;
MKT3	LD	A,(3840H)	; get clear/break
	AND	00000110B	; keep break/clear
	OR	E		; merge shift bit
	CALL	PUTBUFF		; put into buffer
	RET			; and exit
;
;	execute any key input data
;
EXECKI	BIT	4,(IY+1)	; KAM off?
	RET	NZ		; yes, leave in buffer
;
;	first byte data, second shift status
;	lookup data in appropriate table
;
	LD	IX,KIDCB	; keyboard DCB
	CALL	GETBUFF		; get data char
	RET	NC		; go if no data
	LD	C,A		; save char
	CALL	GETBUFF		; get shift char
	RET	NC		; should never happen!
;
;	check for valid key state
;
	CP	6		; 0-5 valid only
	RET	NC		; illegal key!
	LD	L,C		; pass data
	SLA	L		; shift out 2 bits
	SLA	L		; high 6 bits = key
	RRA			; 1 bit to carry
	RR	L		; catch 1
	RRA			; next bit to carry
	RR	L		; catch 2
	LD	H,A		; BC=keycode+shift(0-511)
	LD	E,L		; pass to DE
	LD	D,H		; DE = *1
	ADD	HL,HL		; *2
	ADD	HL,HL		; *4
	ADD	HL,DE		; *5
	LD	DE,KEYS		; lookup table
	ADD	HL,DE		; HL -> data
;
;	if first byte is negative, command vector follows
;	if not, byte is data and continues for up
;	to 5 bytes max per key, 00 terminator
;
	LD	A,(HL)		; get lookup data
	OR	A		; check sign bit
	JP	M,KIVECT	; go if control vector
	LD	C,5		; load key count max
;
;	display characters until terminator found
;
KEYLOOP	LD	A,(HL)		; get char
	OR	A		; terminator?
	RET	Z		; yes, done
;
	PUSH	HL		; save table pointer
	LD	IX,CODCB	; comms output DCB
	CALL	PUTBUFF		; write to buffer
	POP	HL		; restore pointer
;
	BIT	6,(IY+0)	; echo?
	JR	NZ,KEYNEXT	; go if not
;
	PUSH	HL		; save
	LD	IX,CIDCB	; comms input DCB
	LD	A,(HL)		; get char
	CALL	PUTBUFF		; write to buffer
	POP	HL		; restore pointer
;
KEYNEXT	INC	HL		; bump it
	DEC	C		; less counter
	RET	Z		; go if 5 done
	JR	KEYLOOP		; and continue
;
;	keyboard command found, go vector
;
KIVECT	INC	HL		; bump to vector
	LD	A,(HL)		; get lsb
	INC	HL		; bump to next
	LD	H,(HL)		; get msb
	LD	L,A		; HL = vector
	JP	(HL)		; go service!
;
;	ring buffer services
;	IX -> current DCB to service
;
PUTBUFF	LD	E,(IX+6)	; get add offset
	LD	D,(IX+7)
	LD	L,(IX+2)	; get length
	LD	H,(IX+3)
	INC	DE		; bump pointer
	SCF			; set for adjust
	SBC	HL,DE		; check for ring end
	JR	NC,PUTBUF1	; go if not
	LD	DE,0		; else reset to beginning
;
PUTBUF1	LD	L,(IX+8)	; get take offset
	LD	H,(IX+9)
	OR	A		; clear carry
	SBC	HL,DE		; compare
	RET	Z		; NC = no room!
;
	LD	L,(IX+0)	; get buffer start
	LD	H,(IX+1)
	ADD	HL,DE		; add ring offset
	LD	(HL),A		; write the data
	LD	(IX+6),E	; update add offset
	LD	(IX+7),D
	SET	6,(IX+10)	; set data available
	LD	E,(IX+4)	; get lsb count
	LD	D,(IX+5)	; get msb
	INC	DE		; bump it
	LD	(IX+4),E	; update lsb
	LD	(IX+5),D	; update msb
	SCF			; set for char OK
	RET			; and exit
;
;	attempt to remove a character from a buffer
;	IX -> DCB to operate on
;
GETBUFF	XOR	A		; set Z/NCy
	BIT	6,(IX+10)	; data available?
	RET	Z		; go if not
;
	LD	E,(IX+8)	; get take offset
	LD	D,(IX+9)
	LD	L,(IX+2)	; get length
	LD	H,(IX+3)
	INC	DE		; bump ring pointer
	SCF			; set for adjust
	SBC	HL,DE		; check for ring end
	JR	NC,GETBUF1	; go if not wrap
	LD	DE,0		; else reset to beginning
;
GETBUF1	LD	L,(IX+0)	; get buffer start
	LD	H,(IX+1)
	ADD	HL,DE		; point to data
	LD	(IX+8),E	; update offset
	LD	(IX+9),D
;
	LD	E,(IX+4)	; get lsb count
	LD	D,(IX+5)	; get msb count
	DEC	DE		; decrement
	LD	(IX+4),E	; update lsb
	LD	(IX+5),D	; update msb
	LD	A,D		; check if empty
	OR	E		; yes?
	JR	NZ,GETBUF2	; go if more
	RES	6,(IX+10)	; else set empty
;
GETBUF2	LD	A,(HL)		; get char data
	SCF			; set carry for OK
	RET			; and exit
;
;	special key handling vectors
;
;	shift 0 - caps lock
;	alter lower case keyboard table
;
KINS32	LD	A,'A'		; upcase
	BIT	5,(IY+1)	; caps lock on?
	JR	NZ,KINS32A	; go if not
	LD	A,'a'		; locase
;
KINS32A	LD	HL,NU01		; start of data
	LD	DE,5		; data length (offset)
	LD	B,26		; data count
;
KINS32B	LD	(HL),A		; load data
	ADD	HL,DE		; next entry
	INC	A		; next data
	DJNZ	KINS32B		; go for count
;
;	toggle new state bit
;
	LD	A,(IY+1)	; shift zero
	XOR	00100000B	; caps lock
	LD	(IY+1),A	; update
	RET			; and exit
;
;	shift space - send/clear pause
;
KINS55	BIT	7,(IX+10)	; already sent?
	LD	A,'S'-'A'+1	; control S
	JR	Z,KINS55A	; go if not
	LD	A,'Q'-'A'+1	; control Q
;
KINS55A	LD	IX,CODCB	; comms output DCB
	CALL	PUTBUFF		; send the data
	LD	A,(KIDCB+10)	; get flag
	XOR	10000000B	; toggle bit
	LD	(KIDCB+10),A	; update
	RET			; and exit
;
;	toggle auto window
;
KINB00	LD	A,(IY+0)	; get flags
	XOR	00000100B	; toggle
	LD	(IY+0),A	; update
	RET			; and exit
;
;	window move commands
;
KINB51	LD	A,(WROW)	; get row
	OR	A		; at top?
	RET	Z		; go if not
	SUB	8		; move it
	LD	(WROW),A	; update
	JP	NEWVID		; new video
;
KINB52	LD	A,(WROW)	; get row
	CP	8		; at top?
	RET	Z		; go if yes
	ADD	A,8		; else ok to move
	LD	(WROW),A	; update
	JP	NEWVID		; new video
;
KINB53	LD	A,(WCOL)	; get column
	OR	A		; at start?
	RET	Z		; go if yes
	SUB	16		; else move it
	LD	(WCOL),A	; update
	JP	NEWVID		; new video
;
KINB54	LD	A,(WCOL)	; get column
	CP	64		; max possible?
	RET	Z		; go if yes
	OR	A		; at left?
	JR	Z,KINB54A	; go if yes, move
	BIT	4,(IY+0)	; 132 col?
	RET	Z		; go if not
KINB54A	ADD	A,16		; ok to move
	LD	(WCOL),A	; update
	JP	NEWVID		; new video
;
;	update new window image
;
NEWVID	CALL	REFRESH		; update display
	LD	A,(IY+0)	; get table
	PUSH	AF		; save it
	SET	1,(IY+0)	; auto window off
	CALL	CURUPD		; update cursor
	POP	AF		; restore flag
	LD	(IY+0),A	; update
	RET			; and exit
;
;	keyboard lookup tables
;	6 tables of 64*5 bytes each
;	nu = normal unshifted
;	ns = normal shifted
;	cu = clear unshifted
;	cs = clear shift
;	bu = break unshifted
;	bs = break shifted
;
KEYS
NU00	DB	'@',0,0,0,0	; nu 00
NU01	DB	'a',0,0,0,0	; nu 01
	DB	'b',0,0,0,0	; nu 02
	DB	'c',0,0,0,0	; nu 03
	DB	'd',0,0,0,0	; nu 04
	DB	'e',0,0,0,0	; nu 05
	DB	'f',0,0,0,0	; nu 06
	DB	'g',0,0,0,0	; nu 07
	DB	'h',0,0,0,0	; nu 08
	DB	'i',0,0,0,0	; nu 09
	DB	'j',0,0,0,0	; nu 10
	DB	'k',0,0,0,0	; nu 11
	DB	'l',0,0,0,0	; nu 12
	DB	'm',0,0,0,0	; nu 13
	DB	'n',0,0,0,0	; nu 14
	DB	'o',0,0,0,0	; nu 15
	DB	'p',0,0,0,0	; nu 16
	DB	'q',0,0,0,0	; nu 17
	DB	'r',0,0,0,0	; nu 18
	DB	's',0,0,0,0	; nu 19
	DB	't',0,0,0,0	; nu 20
	DB	'u',0,0,0,0	; nu 21
	DB	'v',0,0,0,0	; nu 22
	DB	'w',0,0,0,0	; nu 23
	DB	'x',0,0,0,0	; nu 24
	DB	'y',0,0,0,0	; nu 25
	DB	'z',0,0,0,0	; nu 26
	DB	0,0,0,0,0	; nu 27
	DB	0,0,0,0,0	; nu 28
	DB	0,0,0,0,0	; nu 29
	DB	0,0,0,0,0	; nu 30
	DB	0,0,0,0,0	; nu 31
	DB	'0',0,0,0,0	; nu 32
	DB	'1',0,0,0,0	; nu 33
	DB	'2',0,0,0,0	; nu 34
	DB	'3',0,0,0,0	; nu 35
	DB	'4',0,0,0,0	; nu 36
	DB	'5',0,0,0,0	; nu 37
	DB	'6',0,0,0,0	; nu 38
	DB	'7',0,0,0,0	; nu 39
	DB	'8',0,0,0,0	; nu 40
	DB	'9',0,0,0,0	; nu 41
	DB	':',0,0,0,0	; nu 42
	DB	';',0,0,0,0	; nu 43
	DB	',',0,0,0,0	; nu 44
	DB	'-',0,0,0,0	; nu 45
	DB	'.',0,0,0,0	; nu 46
	DB	'/',0,0,0,0	; nu 47
	DB	0DH,0,0,0,0	; nu 48
	DB	0,0,0,0,0	; nu 49
	DB	0,0,0,0,0	; nu 50
NU51	DB	1BH,5BH,41H,0,0	; nu 51
	DB	1BH,5BH,42H,0,0	; nu 52
	DB	1BH,5BH,44H,0,0	; nu 53
	DB	1BH,5BH,43H,0,0	; nu 54
	DB	' ',0,0,0,0	; nu 55
	DB	0,0,0,0,0	; nu 56
	DB	0,0,0,0,0	; nu 57
	DB	0,0,0,0,0	; nu 58
	DB	0,0,0,0,0	; nu 59
	DB	0,0,0,0,0	; nu 60
	DB	0,0,0,0,0	; nu 61
	DB	0,0,0,0,0	; nu 62
	DB	0,0,0,0,0	; nu 63
;
NS00	DB	1BH,5BH,32H,31H,7EH ; ns 00
NS01	DB	'A',0,0,0,0	; ns 01
	DB	'B',0,0,0,0	; ns 02
	DB	'C',0,0,0,0	; ns 03
	DB	'D',0,0,0,0	; ns 04
	DB	'E',0,0,0,0	; ns 05
	DB	'F',0,0,0,0	; ns 06
	DB	'G',0,0,0,0	; ns 07
	DB	'H',0,0,0,0	; ns 08
	DB	'I',0,0,0,0	; ns 09
	DB	'J',0,0,0,0	; ns 10
	DB	'K',0,0,0,0	; ns 11
	DB	'L',0,0,0,0	; ns 12
	DB	'M',0,0,0,0	; ns 13
	DB	'N',0,0,0,0	; ns 14
	DB	'O',0,0,0,0	; ns 15
	DB	'P',0,0,0,0	; ns 16
	DB	'Q',0,0,0,0	; ns 17
	DB	'R',0,0,0,0	; ns 18
	DB	'S',0,0,0,0	; ns 19
	DB	'T',0,0,0,0	; ns 20
	DB	'U',0,0,0,0	; ns 21
	DB	'V',0,0,0,0	; ns 22
	DB	'W',0,0,0,0	; ns 23
	DB	'X',0,0,0,0	; ns 24
	DB	'Y',0,0,0,0	; ns 25
	DB	'Z',0,0,0,0	; ns 26
	DB	0,0,0,0,0	; ns 27
	DB	0,0,0,0,0	; ns 28
	DB	0,0,0,0,0	; ns 29
	DB	0,0,0,0,0	; ns 30
	DB	0,0,0,0,0	; ns 31
	DB	80H		; ns 32
	DW	KINS32
	DB	0,0
	DB	'!',0,0,0,0	; ns 33
	DB	'"',0,0,0,0	; ns 34
	DB	'#',0,0,0,0	; ns 35
	DB	'$',0,0,0,0	; ns 36
	DB	'%',0,0,0,0	; ns 37
	DB	'&',0,0,0,0	; ns 38
	DB	27H,0,0,0,0	; ns 39
	DB	'(',0,0,0,0	; ns 40
	DB	')',0,0,0,0	; ns 41
	DB	'*',0,0,0,0	; ns 42
	DB	'+',0,0,0,0	; ns 43
	DB	'<',0,0,0,0	; ns 44
	DB	'=',0,0,0,0	; ns 45
	DB	'>',0,0,0,0	; ns 46
	DB	'?',0,0,0,0	; ns 47
	DB	0DH,0AH,0,0,0	; ns 48
	DB	0,0,0,0,0	; ns 49
	DB	0,0,0,0,0	; ns 50
	DB	0,0,0,0,0	; ns 51
	DB	0,0,0,0,0	; ns 52
	DB	7FH,0,0,0,0	; ns 53
	DB	09H,0,0,0,0	; ns 54
	DB	80H		; ns 55
	DW	KINS55
	DB	0,0
	DB	0,0,0,0,0	; ns 56
	DB	0,0,0,0,0	; ns 57
	DB	0,0,0,0,0	; ns 58
	DB	0,0,0,0,0	; ns 59
	DB	0,0,0,0,0	; ns 60
	DB	0,0,0,0,0	; ns 61
	DB	0,0,0,0,0	; ns 62
	DB	0,0,0,0,0	; ns 63
;
NC00	DB	0,0,0,0,0	; nc 00
	DB	01H,0,0,0,0	; nc 01
	DB	02H,0,0,0,0	; nc 02
	DB	03H,0,0,0,0	; nc 03
	DB	04H,0,0,0,0	; nc 04
	DB	05H,0,0,0,0	; nc 05
	DB	06H,0,0,0,0	; nc 06
	DB	07H,0,0,0,0	; nc 07
	DB	08H,0,0,0,0	; nc 08
	DB	09H,0,0,0,0	; nc 09
	DB	0AH,0,0,0,0	; nc 10
	DB	0BH,0,0,0,0	; nc 11
	DB	0CH,0,0,0,0	; nc 12
	DB	0DH,0,0,0,0	; nc 13
	DB	0EH,0,0,0,0	; nc 14
	DB	0FH,0,0,0,0	; nc 15
	DB	10H,0,0,0,0	; nc 16
	DB	11H,0,0,0,0	; nc 17
	DB	12H,0,0,0,0	; nc 18
	DB	13H,0,0,0,0	; nc 19
	DB	14H,0,0,0,0	; nc 20
	DB	15H,0,0,0,0	; nc 21
	DB	16H,0,0,0,0	; nc 22
	DB	17H,0,0,0,0	; nc 23
	DB	18H,0,0,0,0	; nc 24
	DB	19H,0,0,0,0	; nc 25
	DB	1AH,0,0,0,0	; nc 26
	DB	0,0,0,0,0	; nc 27
	DB	0,0,0,0,0	; nc 28
	DB	0,0,0,0,0	; nc 29
	DB	0,0,0,0,0	; nc 30
	DB	0,0,0,0,0	; nc 31
	DB	1BH,4FH,'p',0,0	; nc 32
	DB	1BH,4FH,'q',0,0	; nc 33
	DB	1BH,4FH,'r',0,0	; nc 34
	DB	1BH,4FH,'s',0,0	; nc 35
	DB	1BH,4FH,'t',0,0	; nc 36
	DB	1BH,4FH,'u',0,0	; nc 37
	DB	1BH,4FH,'v',0,0	; nc 38
	DB	1BH,4FH,'w',0,0	; nc 39
	DB	1BH,4FH,'x',0,0	; nc 40
	DB	1BH,4FH,'y',0,0	; nc 41
	DB	0,0,0,0,0	; nc 42
	DB	0,0,0,0,0	; nc 43
	DB	1BH,4FH,'l',0,0	; nc 44
	DB	1BH,4FH,'m',0,0	; nc 45
	DB	1BH,4FH,'n',0,0	; nc 46
	DB	0,0,0,0,0	; nc 47
	DB	1BH,4FH,'M',0,0	; nc 48
	DB	0,0,0,0,0	; nc 49
	DB	0,0,0,0,0	; nc 50
	DB	1BH,4FH,'P',0,0	; nc 51
	DB	1BH,4FH,'S',0,0	; nc 52
	DB	1BH,4FH,'Q',0,0	; nc 53
	DB	1BH,4FH,'R',0,0	; nc 54
	DB	1BH,0,0,0,0	; nc 55
	DB	0,0,0,0,0	; nc 56
	DB	0,0,0,0,0	; nc 57
	DB	0,0,0,0,0	; nc 58
	DB	0,0,0,0,0	; nc 59
	DB	0,0,0,0,0	; nc 60
	DB	0,0,0,0,0	; nc 61
	DB	0,0,0,0,0	; nc 62
	DB	0,0,0,0,0	; nc 63
;
SC00	DB	1AH,0,0,0,0	; sc 00
	DB	1BH,0,0,0,0	; sc 01
	DB	1CH,0,0,0,0	; sc 02
	DB	1DH,0,0,0,0	; sc 03
	DB	1EH,0,0,0,0	; sc 04
	DB	1FH,0,0,0,0	; sc 05
	DB	0,0,0,0,0	; sc 06
	DB	0,0,0,0,0	; sc 07
	DB	0,0,0,0,0	; sc 08
	DB	0,0,0,0,0	; sc 09
	DB	0,0,0,0,0	; sc 10
	DB	0,0,0,0,0	; sc 11
	DB	0,0,0,0,0	; sc 12
	DB	0,0,0,0,0	; sc 13
	DB	0,0,0,0,0	; sc 14
	DB	0,0,0,0,0	; sc 15
	DB	0,0,0,0,0	; sc 16
	DB	0,0,0,0,0	; sc 17
	DB	0,0,0,0,0	; sc 18
	DB	0,0,0,0,0	; sc 19
	DB	0,0,0,0,0	; sc 20
	DB	0,0,0,0,0	; sc 21
	DB	0,0,0,0,0	; sc 22
	DB	0,0,0,0,0	; sc 23
	DB	0,0,0,0,0	; sc 24
	DB	0,0,0,0,0	; sc 25
	DB	0,0,0,0,0	; sc 26
	DB	0,0,0,0,0	; sc 27
	DB	0,0,0,0,0	; sc 28
	DB	0,0,0,0,0	; sc 29
	DB	0,0,0,0,0	; sc 30
	DB	0,0,0,0,0	; sc 31
	DB	0,0,0,0,0	; sc 32
	DB	7CH,0,0,0,0	; sc 33
	DB	7EH,0,0,0,0	; sc 34
	DB	5EH,0,0,0,0	; sc 35
	DB	0,0,0,0,0	; sc 36
	DB	0,0,0,0,0	; sc 37
	DB	0,0,0,0,0	; sc 38
	DB	5CH,0,0,0,0	; sc 39
	DB	5BH,0,0,0,0	; sc 40
	DB	5DH,0,0,0,0	; sc 41
	DB	0,0,0,0,0	; sc 42
	DB	0,0,0,0,0	; sc 43
	DB	7BH,0,0,0,0	; sc 44
	DB	0,0,0,0,0	; sc 45
	DB	7DH,0,0,0,0	; sc 46
	DB	5CH,0,0,0,0	; sc 47
	DB	0,0,0,0,0	; sc 48
	DB	0,0,0,0,0	; sc 49
	DB	0,0,0,0,0	; sc 50
	DB	0,0,0,0,0	; sc 51
	DB	0,0,0,0,0	; sc 52
	DB	0,0,0,0,0	; sc 53
	DB	0,0,0,0,0	; sc 54
	DB	0,0,0,0,0	; sc 55
	DB	0,0,0,0,0	; sc 56
	DB	0,0,0,0,0	; sc 57
	DB	0,0,0,0,0	; sc 58
	DB	0,0,0,0,0	; sc 59
	DB	0,0,0,0,0	; sc 60
	DB	0,0,0,0,0	; sc 61
	DB	0,0,0,0,0	; sc 62
	DB	0,0,0,0,0	; sc 63
;
NB00	DB	80H
	DW	KINB00
	DB	0,0
	DB	0,0,0,0,0	; nb 01
	DB	0,0,0,0,0	; nb 02
	DB	0,0,0,0,0	; nb 03
	DB	0,0,0,0,0	; nb 04
	DB	0,0,0,0,0	; nb 05
	DB	1BH,5BH,'1',7EH,0	; nb 06
	DB	0,0,0,0,0	; nb 07
	DB	0,0,0,0,0	; nb 08
	DB	1BH,5BH,'2',7EH,0	; nb 09
	DB	0,0,0,0,0	; nb 10
	DB	0,0,0,0,0	; nb 11
	DB	0,0,0,0,0	; nb 12
	DB	0,0,0,0,0	; nb 13
	DB	1BH,5BH,'6',7EH,0	; nb 14
	DB	0,0,0,0,0	; nb 15
	DB	1BH,5BH,'5',7EH,0	; nb 16
	DB	0,0,0,0,0	; nb 17
	DB	1BH,5BH,'3',7EH,0	; nb 18
	DB	1BH,5BH,'4',7EH,0	; nb 19
	DB	0,0,0,0,0	; nb 20
	DB	0,0,0,0,0	; nb 21
	DB	0,0,0,0,0	; nb 22
	DB	0,0,0,0,0	; nb 23
	DB	0,0,0,0,0	; nb 24
	DB	0,0,0,0,0	; nb 25
	DB	0,0,0,0,0	; nb 26
	DB	0,0,0,0,0	; nb 27
	DB	0,0,0,0,0	; nb 28
	DB	0,0,0,0,0	; nb 29
	DB	0,0,0,0,0	; nb 30
	DB	0,0,0,0,0	; nb 31
	DB	1BH,5BH,'21',7EH	; nb 32
	DB	0,0,0,0,0	; nb 33
	DB	0,0,0,0,0	; nb 34
	DB	0,0,0,0,0	; nb 35
	DB	0,0,0,0,0	; nb 36
	DB	0,0,0,0,0	; nb 37
	DB	1BH,5BH,'17',7EH	; nb 38
	DB	1BH,5BH,'18',7EH	; nb 39
	DB	1BH,5BH,'19',7EH	; nb 40
	DB	1BH,5BH,'20',7EH	; nb 41
	DB	0,0,0,0,0	; nb 42
	DB	0,0,0,0,0	; nb 43
	DB	0,0,0,0,0	; nb 44
	DB	0,0,0,0,0	; nb 45
	DB	0,0,0,0,0	; nb 46
	DB	0,0,0,0,0	; nb 47
	DB	0,0,0,0,0	; nb 48
	DB	0,0,0,0,0	; nb 49
	DB	0,0,0,0,0	; nb 50
	DB	80H
	DW	KINB51
	DB	0,0
	DB	80H
	DW	KINB52
	DB	0,0
	DB	80H
	DW	KINB53
	DB	0,0
	DB	80H
	DW	KINB54
	DB	0,0
	DB	0,0,0,0,0	; nb 55
	DB	0,0,0,0,0	; nb 56
	DB	0,0,0,0,0	; nb 57
	DB	0,0,0,0,0	; nb 58
	DB	0,0,0,0,0	; nb 59
	DB	0,0,0,0,0	; nb 60
	DB	0,0,0,0,0	; nb 61
	DB	0,0,0,0,0	; nb 62
	DB	0,0,0,0,0	; nb 63
;
SB00	DB	0,0,0,0,0	; sb 00
	DB	0,0,0,0,0	; sb 01
	DB	0,0,0,0,0	; sb 02
	DB	0,0,0,0,0	; sb 03
	DB	0,0,0,0,0	; sb 04
	DB	0,0,0,0,0	; sb 05
	DB	0,0,0,0,0	; sb 06
	DB	0,0,0,0,0	; sb 07
	DB	0,0,0,0,0	; sb 08
	DB	0,0,0,0,0	; sb 09
	DB	0,0,0,0,0	; sb 10
	DB	0,0,0,0,0	; sb 11
	DB	0,0,0,0,0	; sb 12
	DB	0,0,0,0,0	; sb 13
	DB	0,0,0,0,0	; sb 14
	DB	0,0,0,0,0	; sb 15
	DB	0,0,0,0,0	; sb 16
	DB	0,0,0,0,0	; sb 17
	DB	0,0,0,0,0	; sb 18
	DB	0,0,0,0,0	; sb 19
	DB	0,0,0,0,0	; sb 20
	DB	0,0,0,0,0	; sb 21
	DB	0,0,0,0,0	; sb 22
	DB	0,0,0,0,0	; sb 23
	DB	0,0,0,0,0	; sb 24
	DB	0,0,0,0,0	; sb 25
	DB	0,0,0,0,0	; sb 26
	DB	0,0,0,0,0	; sb 27
	DB	0,0,0,0,0	; sb 28
	DB	0,0,0,0,0	; sb 29
	DB	0,0,0,0,0	; sb 30
	DB	0,0,0,0,0	; sb 31
	DB	1BH,5BH,'34',7EH	; sb 32
	DB	1BH,5BH,'23',7EH	; sb 33
	DB	1BH,5BH,'24',7EH	; sb 34
	DB	1BH,5BH,'25',7EH	; sb 35
	DB	1BH,5BH,'26',7EH	; sb 36
	DB	1BH,5BH,'28',7EH	; sb 37
	DB	1BH,5BH,'29',7EH	; sb 38
	DB	1BH,5BH,'31',7EH	; sb 39
	DB	1BH,5BH,'32',7EH	; sb 40
	DB	1BH,5BH,'33',7EH	; sb 41
	DB	0,0,0,0,0	; sb 42
	DB	0,0,0,0,0	; sb 43
	DB	0,0,0,0,0	; sb 44
	DB	0,0,0,0,0	; sb 45
	DB	0,0,0,0,0	; sb 46
	DB	0,0,0,0,0	; sb 47
	DB	0,0,0,0,0	; sb 48
	DB	0,0,0,0,0	; sb 49
	DB	0,0,0,0,0	; sb 50
	DB	0,0,0,0,0	; sb 51
	DB	0,0,0,0,0	; sb 52
	DB	0,0,0,0,0	; sb 53
	DB	0,0,0,0,0	; sb 54
	DB	0,0,0,0,0	; sb 55
	DB	0,0,0,0,0	; sb 56
	DB	0,0,0,0,0	; sb 57
	DB	0,0,0,0,0	; sb 58
	DB	0,0,0,0,0	; sb 59
	DB	0,0,0,0,0	; sb 60
	DB	0,0,0,0,0	; sb 61
	DB	0,0,0,0,0	; sb 62
	DB	0,0,0,0,0	; sb 63
