; vt100c/asm	- DEC VT100 series terminal emulator
; Kim J. Watt	/ 28-Mar-87
;
;	READCI	- read comms input and place in buffer
;	WRITCO	- write comms output from buffer
;	NOTE: both these can be placed in interrupts
;	EXECCI	- execute any input comms characters
;
;	READCI - read comms input char
;
READCI	LD	IX,CIDCB	; comms input DCB
;
;	read the device status to see if char available
;
	IN	A,(COSTAT)	; read comms status
	BIT	7,A		; RX data ready?
	RET	Z		; go if no input data
;
;	check for any hardware errors
;
;	AND	00111000B	; test for errors
;	JR	NZ,INHARD	; hardware input error
;
;	read the data and save in the input buffer
;
	IN	A,(CODATA)	; read comms char
	CALL	PUTBUFF		; add to input buffer
	RET			; and exit %%%%%%
;
	RET	NC		; go if could not add
;
;	check if > 192 chars in buffer
;	and will send XOFF
;
	LD	A,(KIDCB+10)	; get key flags
	AND	10000000B	; already paused?
	RET	NZ		; yes, ignore
;
	LD	L,(IX+2)	; get length
	LD	H,(IX+3)
	LD	E,(IX+4)	; get chars in buffer
	LD	D,(IX+5)
	SRL	H		; HL / 2 = half of buff
	RR	L
	OR	A		; clear carry
	SBC	HL,DE		; compare
	JR	NC,NOHALF	; not half full yet
;
;	have more than half, check if already sent
;	the XOFF already
;
	BIT	7,(IX+10)	; holding?
	RET	NZ		; go if yes
	LD	IX,CODCB	; output DCB
	LD	A,XOFF		; load data
	CALL	PUTBUFF		; add to output buffer
	RET	NC		; could not add
	LD	IX,CIDCB	; input DCB
	SET	7,(IX+10)	; tag as sent
	RET			; and exit
;
;	less than half full
;	check if less than 1/4 and have sent
;	a XOFF and need to clear it
;
NOHALF	BIT	7,(IX+10)	; paused sender?
	RET	Z		; go if not
;
	LD	L,(IX+2)	; get length
	LD	H,(IX+3)
	SRL	H		; HL / 2
	RR	L
	SRL	H		; HL / 4
	RR	L
	OR	A		; clear carry
	SBC	HL,DE		; compare
	RET	NC		; more than 1/4
;
	LD	IX,CODCB	; output DCB
	LD	A,XON		; can turn back on
	CALL	PUTBUFF		; add to output buffer
	RET	NC		; could not send
	LD	IX,CIDCB	; input DCB
	RES	7,(IX+10)	; clear holding
	RET			; and exit
;
;	write character from output SPOOL buffer
;	to the communication interface
;
WRITCO	LD	IX,CODCB	; comms output DCB
	BIT	7,(IX+10)	; paused?
	RET	NZ		; go if yes
;
;	check if transmitter can send a char
;
	IN	A,(COSTAT)	; read comms status
	BIT	6,A		; TX ready?
	RET	Z		; exit if not
;
;	see if any chars available in buffer
;
	CALL	GETBUFF		; any data?
	RET	NC		; go if not
	OUT	(CODATA),A	; else output the char
	RET			; and exit
;
;	check if have received any input chars
;	and process
;
EXECCI	LD	IX,CIDCB	; comms input DCB
	CALL	GETBUFF		; any data?
	RET	NC		; go if not
	LD	HL,EXECCI	; return vector
	PUSH	HL		; leave on stack
;
;	test if any special sequences in progress
;
	LD	IX,DODCB	; display DCB
	BIT	7,(IY+2)	; CSI pending?
	JR	NZ,CSIPND	; go if yes
	BIT	6,(IY+2)	; ESC pending?
	JR	NZ,ESCPND	; go if yes
	BIT	5,(IY+2)	; DCS pending?
	JP	NZ,DCSPND	; go if yes
	BIT	4,(IY+2)	; STRIP pending?
	JP	NZ,ESCRCAN	; reverse cancel
;
;	no pending, check if start of one
;
	CP	20H		; control code?
	JR	C,DOCTRL	; go if yes
	CP	80H		; high bit set?
	JR	NC,DOCS1	; nope, ctrl 1 set
	CALL	PUTBUFF		; else output data
	RET			; and exit
DOCS1	CP	9BH		; CSI introducer
	JP	Z,ESCLS		; go if yes
	RET			; %% else exit
;
;
DOCTRL	LD	C,A		; save char
	ADD	A,A		; *2 for word table
	LD	L,A		; to L
	LD	H,0		; HL = offset
	LD	DE,DVECTS	; display vectors
	ADD	HL,DE		; HL -> vector
	LD	A,(HL)		; get lsb
	INC	HL		; bump pointer
	LD	H,(HL)		; get msb
	LD	L,A		; hl = vector
	LD	A,C		; get char back
	JP	(HL)		; go table
;
;	ignore next character
;
ESCCAN	SET	4,(IY+2)	; set flag
	RET			; and exit
;
;	restore
;
ESCRCAN	RES	4,(IY+2)	; enable
	RET			; and exit
;
;	escape sequence pending
;
ESCPND	RES	6,(IY+2)	; clear pending
	LD	HL,ESCVEC	; escape lookup table
	CALL	LOOKUP		; lookup table entry
	JR	NZ,ESCSHOW	; go if not found
	JP	(HL)		; else go table entry
;
ESCSHOW	LD	C,A		; save char
	RET
	LD	A,'<'		; ESC string
	CALL	PUTBUFF		; to buffer
	LD	A,C		; get data
	CALL	PUTBUFF		; to output buffer
	LD	A,'>'		; close delim
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
;	control string introducer pending
;
CSIPND	LD	HL,CSIBUF	; start of table
	INC	(HL)		; bump count
	LD	E,(HL)		; get char count
	LD	D,0		; clear msb
	ADD	HL,DE		; hl -> position
	LD	(HL),A		; save data
;
;	check for terminator
;
	CP	'A'		; terminator?
	JR	C,CSILEN	; nope, check length
;
;	interpret the CSI string
;
	RES	7,(IY+2)	; clear pending
	LD	HL,CSIVEC	; lookup table
	CALL	FIND		; find the entry
	JR	NZ,CSISPCL	; check if special
	JP	(HL)		; else go!
;
;	check if CSI string is special
;
CSISPCL	LD	HL,CSIBUF	; start of buffer
	LD	C,(HL)		; get length
	LD	B,0		; clear msb
	ADD	HL,BC		; point to last char
	LD	A,(HL)		; get last char
	LD	HL,CSISPC	; lookup table
	CALL	LOOKUP		; find it?
	JR	NZ,CSISHOW	; nope, display data!
	JP	(HL)		; else go service!
;
CSISHOW	LD	A,'('		; delim
	RET
	CALL	PUTBUFF		; to DO buffer
	LD	HL,CSIBUF	; start of buffer
	LD	B,(HL)		; get data length
	INC	HL		; skip length
;
CSISHOL	LD	A,(HL)		; get char
	PUSH	HL		; save pointer
	CALL	PUTBUFF		; to display
	POP	HL		; restore
	INC	HL		; bump pointer
	DJNZ	CSISHOL		; go for count
	LD	A,')'		; delim
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
;	check for CSI string overflow
;
CSILEN	LD	A,(CSIBUF)	; get start of table
	CP	32		; max length
	RET	C		; go if less
	RES	7,(IY+2)	; else clear the string!
	RET			; and exit
;
;	DCS string pending
;
DCSPND	RES	5,(IY+2)	; clear flag
	RET			; and exit
;
;	escape sequence lookup table
;
ESCVEC	DB	'A'
	DW	ESCA
	DB	'B'
	DW	ESCB
	DB	'D'
	DW	ESCD
	DB	'E'
	DW	ESCE
	DB	'H'
	DW	ESCH
	DB	'I'
	DW	ESCI
	DB	'J'
	DW	ESCJ
	DB	'K'
	DW	ESCK
	DB	'M'
	DW	ESCM
	DB	'N'
	DW	ESCN
	DB	'O'
	DW	ESCO
	DB	'P'
	DW	ESCO
	DB	'Y'
	DW	ESCY
	DB	'7'
	DW	ESC7
	DB	'8'
	DW	ESC8
	DB	5BH
	DW	ESCLS
	DB	'/'
	DW	ESCSL
	DB	'Z'
	DW	CSI27
	DB	'<'
	DW	ESCLT
	DB	'('
	DW	ESCCAN
	DB	')'
	DW	ESCCAN
	DB	'*'
	DW	ESCCAN
	DB	'+'
	DW	ESCCAN
	DB	-1
;
;	lookup table for special CSI strings
;	last character in string is searched
;
CSISPC	DB	'A'		; up x lines
	DW	CSIA
	DB	'B'		; down x lines
	DW	CSIB
	DB	'C'		; right x cols
	DW	CSIC
	DB	'D'		; left x cols
	DW	CSID
	DB	'H'		; cursor posit
	DW	CSIH
	DB	'f'		; cursor posit
	DW	CSIF
	DB	-1		; terminator
;
;	handler for special CSI strings
;
CSIA	LD	A,92H		; display leader
	CALL	PUTBUFF		; to buffer
	LD	HL,CSIBUF+1	; up x lines
	CALL	VALUE		; get number
	LD	A,C		; get data
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
CSIB	LD	A,93H		; special
	CALL	PUTBUFF		; to DO buff
	LD	HL,CSIBUF+1	; down x lines
	CALL	VALUE		; get number
	LD	A,C		; get data
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
CSIC	LD	A,94H		; special
	CALL	PUTBUFF		; to buff
	LD	HL,CSIBUF+1	; right x cols
	CALL	VALUE		; get number
	LD	A,C		; get data
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
CSID	LD	A,95H		; special
	CALL	PUTBUFF		; to buffer
	LD	HL,CSIBUF+1	; left x cols
	CALL	VALUE		; get number
	LD	A,C		; get data
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
CSIH
CSIF	LD	A,96H		; special
	CALL	PUTBUFF		; to buffer
	LD	HL,CSIBUF+1	; start of number
	LD	A,(HL)		; get data
	SUB	';'		; anything?
	JR	Z,CSIH1		; go if not
	CALL	VALUE		; get number
	LD	A,C		; get data
	BIT	0,(IY+1)	; DECCOM?
	JR	Z,CSIH1		; go if not
	DEC	A		; else correct
CSIH1	PUSH	HL		; save pointer
	CALL	PUTBUFF		; to buffer
	POP	HL		; restore
	LD	A,(HL)		; get term
	CP	';'		; correct?
	JR	NZ,CSIH2	; default if not
	INC	HL		; skip term
	LD	A,(HL)		; get next
	CP	'H'		; term?
	JR	Z,CSIH2		; go if yes
	CP	'f'		; term?
	JR	NZ,CSIH3	; go if yes
CSIH2	LD	A,0		; default
	JR	CSIH4		; and go
CSIH3	CALL	VALUE		; get number
	LD	A,C		; get char
	BIT	0,(IY+1)	; DECCOM?
	JR	Z,CSIH4		; go if not
	DEC	A		; 0 relative
CSIH4	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
;	service table for escape sequences
;
ESCD	LD	A,0AH		; linefeed
	CALL	PUTBUFF		; to display buffer
	RET			; and exit
;
ESCE	LD	A,87H		; next line
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCA	LD	A,84H		; cursor up
	CALL	PUTBUFF		; to buffer
	RET			; adn exit
;
ESCB	LD	A,83H		; code
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCI	LD	A,1AH		; code
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCJ	LD	A,82H		; code
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCK	LD	A,97H		; code
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCY	LD	A,96H		; line/col follows
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCH	LD	HL,TABTBL	; tab lookup table
	LD	DE,(BCOL)	; get buffer column
	ADD	HL,DE		; bump pointer
	LD	(HL),0		; set tab
	RET			; and exit
;
ESCM	LD	A,1AH		; reverse linefeed
	CALL	PUTBUFF		; to display buffer
	RET			; and exit
;
;	save cursor position
;
ESC7	LD	A,90H		; special
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESC8	LD	A,91H		; special
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
ESCN	RET			; shift in G2 char set
;
ESCO	RET			; shift in G3 char set
;
ESCP	SET	5,(IY+2)	; device control string
	RET			; and exit
;
ESCLS	SET	7,(IY+2)	; control sequence intro
	XOR	A		; byte zero
	LD	(CSIBUF),A	; clear char count
	RET			; and exit
;
;	enter ANSI mode
;
ESCLT	SET	0,(IY+1)	; set DECCOM
	SET	3,(IY+1)	; set ANSI
	RET			; and exit
;
ESCSL	RES	5,(IY+2)	; end of DCS string
	RET			; and exit
;
;	CSI command lookup table
;
CSIVEC	DB	2,'2K'		; erase line
	DW	CSI01
	DB	3,'?2K'		; ""
	DW	CSI01
	DB	2,'2J'		; erase frame
	DW	CSI02
	DB	3,'?2J'		; ""
	DW	CSI02
	DB	1,'K'		; erase EOL
	DW	CSI03
	DB	2,'0K'		; ""
	DW	CSI03
	DB	2,'?K'		; ""
	DW	CSI02
	DB	3,'?0K'		; ""
	DW	CSI02
	DB	1,'J'		; erase EOF
	DW	CSI04
	DB	2,'0J'		; ""
	DW	CSI04
	DB	2,'?J'		; ""
	DW	CSI04
	DB	3,'?0J'		; ""
	DW	CSI04
	DB	2,'1K'		; erase BOL
	DW	CSI05
	DB	3,'?1K'		; ""
	DW	CSI05
	DB	2,'1J'		; erase BOF
	DW	CSI06
	DB	3,'?1J'		; ""
	DW	CSI06
	DB	4,'?25h'	; cursor on
	DW	CSI07
	DB	4,'?25l'	; cursor off
	DW	CSI08
	DB	3,'?3l'		; 80 col
	DW	CSI09
	DB	3,'?3h'		; 132 col
	DW	CSI10
	DB	2,'4h'		; insert
	DW	CSI11
	DB	2,'4l'		; replace
	DW	CSI12
	DB	2,'2h'		; lock keyboard
	DW	CSI13
	DB	2,'2l'		; unlock
	DW	CSI14
	DB	1,'g'		; clear tab
	DW	CSI15
	DB	2,'0g'		; ""
	DW	CSI15
	DB	2,'3g'		; clear all tab
	DW	CSI16
	DB	3,'?7h'		; auto wrap on
	DW	CSI17
	DB	3,'?7l'		; auto wrap off
	DW	CSI18
	DB	3,'?1h'		; application
	DW	CSI19
	DB	3,'?1l'		; cursor
	DW	CSI20
	DB	3,'20h'		; newline
	DW	CSI21
	DB	3,'20l'		; linefeed
	DW	CSI22
	DB	3,'12h'		; send/receive off
	DW	CSI23
	DB	3,'12l'		; on
	DW	CSI24
	DB	3,'?8h'		; auto repeat on
	DW	CSI25
	DB	3,'?8l'		; off
	DW	CSI26
	DB	1,'c'		; primary DA
	DW	CSI27
	DB	2,'0c'		; ""
	DW	CSI27
	DB	2,'>c'		; secondary DA
	DW	CSI28
	DB	3,'>0c'		; ""
	DW	CSI28
	DB	2,'5n'		; term status
	DW	CSI29
	DB	4,'?15n'	; print stat
	DW	CSI30
	DB	4,'?25n'	; UR stat
	DW	CSI31
	DB	4,'?26n'	; key lang
	DW	CSI32
	DB	2,'!p'		; reset term
	DW	CSI33
	DB	2,'6n'		; cursor pos
	DW	CSI34
	DB	-1
;
;	CSI sequence handlers
;
;	special clear commands
;
CSI03	LD	A,97H		; special
	JP	PUTBUFF		; to buffer
;
CSI05	LD	A,98H		; special
	JP	PUTBUFF		; to buffer
;
CSI06	LD	A,81H		; special
	JP	PUTBUFF		; to buffer
;
CSI01	LD	A,99H		; special
	JP	PUTBUFF		; to buffer
;
CSI02	LD	A,9AH		; special
	JP	PUTBUFF		; to buffer
;
CSI04	LD	A,82H		; special
	JP	PUTBUFF		; to buffer
;
;	host to VT100 primary DA request
;
CSI27	LD	HL,PDAR		; primary DA reply
	JR	COSEND		; send it
;
CSI28	LD	HL,SDAR		; secondary DA reply
	JR	COSEND		; send it
;
CSI29	LD	HL,TSTA		; terminal status
	JR	COSEND		; send it
;
CSI30	LD	HL,PSTA		; printer status
	JR	COSEND		; send it
;
CSI31	LD	HL,USTA		; UDK status
	JR	COSEND		; send it
;
CSI32	LD	HL,KSTA		; keyboard status
	JR	COSEND		; send
;
CSI33	LD	A,8FH		; cls
	JP	PUTBUFF		; to display
;
CSI34	LD	HL,RCPOS+2	; data table
	LD	A,(BROW)	; get row
	INC	A		; 1 relative
	CALL	ASCII		; to dec ascii
	LD	(HL),';'	; sep
	INC	HL		; bump pointer
	LD	A,(BCOL)	; get column
	INC	A		; 1 relative
	CALL	ASCII		; to dec ascii
	LD	(HL),'R'	; term
	INC	HL		; bump it
	LD	(HL),0		; term string
	LD	HL,RCPOS	; start of message
;
COSEND	LD	C,-1		; less 00 term string
	JP	KEYLOOP		; send the string
;
;	clear tab at current position
;
CSI15	LD	HL,TABTBL	; tab table
	LD	DE,(BCOL)	; get current column
	LD	D,0		; clear msb
	ADD	HL,DE		; current pos
	LD	(HL),80H	; clear tab
	RET			; and exit
;
;	clear all tabs
;
CSI16	LD	HL,TABTBL	; tab table
	LD	DE,TABTBL+1	; start +1
	LD	BC,127		; length -1
	LD	(HL),80H	; clear one
	LDIR			; clear all
	RET			; and exit
;
CSI13	SET	4,(IY+1)	; KAM
	RET
;
CSI14	RES	4,(IY+1)	; KAM off
	RET
;
CSI11	RES	7,(IY+0)	; insert
	RET			; and exit
;
CSI12	SET	7,(IY+0)	; replace
	RET			; and exit
;
CSI23	SET	6,(IY+0)	; send
	RET
;
CSI24	RES	6,(IY+0)	; receive
	RET
;
CSI21	SET	1,(IY+0)	; newline
	RET
;
CSI22	RES	1,(IY+0)	; linefeed
	RET
;
CSI19	RES	6,(IY+1)	; application
	RET
;
CSI20	SET	6,(IY+1)	; cursor
	RET
;
CSI10	SET	4,(IY+0)	; 132 col
	RET
;
CSI09	RES	4,(IY+0)	; 80 col
	RET
;
CSI17	RES	3,(IY+0)	; auto wrap off
	RET
;
CSI18	SET	3,(IY+0)	; auto off
	RET
;
CSI25	RES	7,(IY+1)	; repeat on
	RET
;
CSI26	SET	7,(IY+1)	; repeat off
	RET
;
CSI07	RES	1,(IY+0)	; cursor on
	RET
;
CSI08	SET	1,(IY+0)	; cursor off
	RET
;
;	convert A to decimal ascii at (HL)
;
ASCII	LD	B,'0'		; start ascii
ASC1	SUB	100		; remove a hundres
	JR	C,ASC2		; go if done
	INC	B		; bump ascii
	JR	ASC1		; and continue
ASC2	LD	C,A		; save count
	LD	A,B		; get value
	CP	'0'		; any?
	JR	Z,ASC3		; go if yes
	LD	(HL),B		; else load digit
	INC	HL		; bump pointer
ASC3	LD	A,C		; get count
	ADD	A,100		; adjust back
	LD	B,'0'		; init data
ASC4	SUB	10		; less 10
	JR	C,ASC5		; go if found
	INC	B		; else bump data
	JR	ASC4		; and continue
ASC5	ADD	A,'0'+10	; last digit
	LD	C,A		; save
	LD	A,B		; get this one
	CP	'0'		; any?
	JR	Z,ASC6		; go if not
	LD	(HL),B		; else save it
	INC	HL		; bump
ASC6	LD	(HL),C		; lsb
	INC	HL		; bump
	RET			; and exit
;
;	reply tables for Host enquires
;
;	primary DA reply
;
PDAR	DB	1BH,5BH,'?1;2c',0
SDAR	DB	1BH,5BH,'>1;0;0c',0
TSTA	DB	1BH,5BH,'0n',0
PSTA	DB	1BH,5BH,'?13n',0
USTA	DB	1BH,5BH,'?20n',0
KSTA	DB	1BH,5BH,'?27;1n',0
RCPOS	DB	1BH,5BH,'00;00R',0
;
;	display vector table for control chars
;
DVECTS	DW	NULL		; 00 ignore
	DW	NULL		; 01 ignore
	DW	NULL		; 02 ignore
	DW	NULL		; 03 ignore
	DW	NULL		; 04 ignore
	DW	CI05		; 05 enquire
	DW	NULL		; 06 ignore
	DW	PUTBUFF		; 07 bell
	DW	PUTBUFF		; 08 backspace
	DW	PUTBUFF		; 09 tab
	DW	PUTBUFF		; 0A linefeed
	DW	CI0B		; 0B vertical tab
	DW	CI0C		; 0C form feed
	DW	PUTBUFF		; 0D carriage return
	DW	NULL		; 0E ignore
	DW	NULL		; 0F ignore
	DW	NULL		; 10 ignore
	DW	CI11		; 11 DC1
	DW	NULL		; 12 ignore
	DW	CI13		; 13 DC3
	DW	NULL		; 14 ignore
	DW	NULL		; 15 ignore
	DW	NULL		; 16 ignore
	DW	NULL		; 17 ignore
	DW	CI18		; 18 CAN
	DW	NULL		; 19 ignore
	DW	CI1A		; 1A SUB
	DW	CI1B		; 1B ESC
	DW	NULL		; 1C ignore
	DW	NULL		; 1D ignore
	DW	NULL		; 1E ignore
	DW	NULL		; 1F ignore
;
;*******************
;
;	services for received comms input codes
;
NULL	RET			; ignore
;
;	05 ENQ - generate answerback message
;
CI05	RET
;
;	0B VT - vertical tab
;	0C FF - form feed
;
CI0C
CI0B	LD	A,0AH		; processes as LF
	CALL	PUTBUFF		; to output buffer
	RET			; and exit
;
;	11 DC1 - XON
;
CI11	LD	IX,CODCB	; comms output DCB
	RES	7,(IX+10)	; enable
	RET			; and exit
;
;	13 DC3 - XOFF
;
CI13	LD	IX,CODCB	; comms output DCB
	SET	7,(IX+10)	; disable
	RET			; and exit
;
;	18 CAN - cancel
;
CI18	RES	7,(IY+2)	; clear CSI
	RES	6,(IY+2)	; clear ESC
	RES	5,(IY+2)	; clear DCS
	RET			; and exit
;
;	1A SUB - substitute
;
CI1A	RES	7,(IY+2)	; clear CSI
	RES	6,(IY+2)	; clear ESC
	RES	5,(IY+2)	; clear DCS
	LD	A,'?'		; reverse question mark?
	CALL	PUTBUFF		; to buffer
	RET			; and exit
;
;	1B ESC - start escape sequence
;
CI1B	SET	6,(IY+2)	; set ESC
	RET			; and exit
;
;	support for comms input table searches
;
;	lookup entry in table
;	byte char (FF term) followed by word vector
;
LOOKUP	INC	(HL)		; term?
	JR	Z,LOOKNO	; yes, not found
	DEC	(HL)		; restore
	CP	(HL)		; match?
	JR	Z,LOOKYES	; go if yes
	INC	HL		; bump to next entry
	INC	HL
	INC	HL
	JR	LOOKUP		; and continue
;
LOOKNO	DEC	(HL)		; correct table
	OR	A		; set NZ for not found
	RET			; and exit
;
LOOKYES	INC	HL		; bump to vector
	LD	A,(HL)		; get lsb
	INC	HL		; bump pointer
	LD	H,(HL)		; get msb
	LD	L,A		; HL = vector
	XOR	A		; set Z for found
	RET			; and exit
;
;	find a CSI string in table
;	byte length followed by data and vector
;	length FF terminates
;
FIND	INC	(HL)		; term?
	JR	Z,FINDNO	; go if yes, not ok
	DEC	(HL)		; correct
	LD	E,L		; save pointer
	LD	D,H		; HL = current entry
	LD	A,(CSIBUF)	; get length to compare
	CP	(HL)		; match?
	JR	NZ,FINDNXT	; go if not
	LD	BC,CSIBUF+1	; start of buffer
	INC	HL		; CSI data
;
FINDTST	EX	AF,AF'		; save count
	LD	A,(BC)		; get input data
	CP	(HL)		; compare to table
	JR	NZ,FINDNXT	; go next if not match
	INC	BC		; bump input
	INC	HL		; bump output
	EX	AF,AF'		; swap count back
	DEC	A		; less counter
	JR	NZ,FINDTST	; go if not end
;
;	found the CSI string!
;
FOUND	LD	A,(HL)		; get lsb vector
	INC	HL		; bump pointer
	LD	H,(HL)		; get msb
	LD	L,A		; HL = vector
	XOR	A		; set Z for found
	RET			; and exit
;
;	no match, advance to next entry
;
FINDNXT	EX	DE,HL		; hl -> current entry
	LD	E,(HL)		; get lsb length
	LD	D,0		; clear msb
	ADD	HL,DE		; HL = almost end
	INC	HL		; skip length
	INC	HL		; skip lsb vector
	INC	HL		; skip msb vector
	JR	FIND		; and go next entry
;
;	not found
;
FINDNO	DEC	(HL)		; correct terminator
	OR	-1		; set NZ for not found
	RET			; and exit
;
;	calculate numeric value from CSI string
;
VALUE	LD	BC,0		; start value
VALLP	LD	A,(HL)		; get data
	CP	'0'		; term?
	RET	C		; go if yes
	CP	'9'+1		; term?
	RET	NC		; go if yes
	SUB	'0'		; remove ascii
;
	PUSH	HL		; save pointer
	LD	H,B		; get subtotal
	LD	L,C
	ADD	HL,HL		; *2
	ADD	HL,HL		; *4
	ADD	HL,BC		; *5
	ADD	HL,HL		; *10
	LD	C,A		; new digit
	LD	B,0		; clear msb
	ADD	HL,BC		; add new
	LD	B,H		; restore temp
	LD	C,L
	POP	HL		; restore input pointer
	INC	HL		; bump it
	JR	VALLP		; and continue
