;	disas/asm
;
;  DISAS - An HD64180 disassembler
;
;	Get the current cursor location into HL
;
GET_CURSOR	MACRO
	PUSH	BC
	LD	B,4
	SVC	@VDCTL
	POP	BC
	ENDM
;
;	Set the current cursor location to that indicated in HL
;
PUT_CURSOR	MACRO
	PUSH	BC
	PUSH	DE
	PUSH	HL
	LD	B,3
	SVC	@VDCTL
	POP	HL
	POP	DE
	POP	BC
	ENDM
;
*GET	EQUATE4/EQU
;
	ORG	03000H
;
;	Check for a filename or address as an argument
;
START	EQU	$
	LD	BC,0		;Get a zero
	LD	(PCSAVE),BC	;Set the initial address
	PUSH	HL		;Save command line
	CALL	DOINTRO		;Print intro message
	POP	HL		;Get command line back
	LD	A,(HL)		;Get a character
	CP	' '		;Space?
	JR	Z,START_2	;Jump if so
	JR	C,START_5	;Jump if control
	JR	START_3		;Process the argument
START_2:
	INC	HL		;Point at next character
	JR	START		;Loop
;
;	Check for a numeric value
;
START_3:
	LD	A,(HL)		;Get a character
	CP	'0'		;Check for numeric
	JR	C,START_4	;Jump if too small
	CP	'9'+1
	JR	NC,START_4	;Jump if too large
	CALL	HEXTOBIN	;Convert to numeric
	JR	NZ,START_4	;Jump if conversion error
	LD	(PCSAVE),HL	;Set new PC value
	JR	START_5		;Go to start
;
;	Process the filename
;
START_4:
	LD	DE,FNBUF	;Get buffer
	LD	BC,32		;Get the count
	LDIR			;Copy the name in
	CALL	GETFILE_2	;Open input file
	LD	(PCSAVE),HL
START_5:
	LD	B,1
	LD	HL,(PCSAVE)
	JR	PAGE
;
;	Top of main processing loop
;
TOP	EQU	$
	LD	B,19		; Disassemble 19 instructions
PAGE	PUSH	BC		; Save count before it's destroyed
	LD	(PCSAVE),HL	; Save value of PC at start of instr
	CALL	Z80DIS		; Disassemble one instruction
	CALL	PRNTINST	; Print out instruction
	POP	BC		; Get count back into B
	SVC	@CKBRKC
	JR	NZ,ENDPG
	DJNZ	PAGE
;
;	Fall through to get user input
;
ENDPG	CALL	CONT		; Ask user if he wants to continue
	CP	128
	JP	Z,EXIT		; Go if BREAK pressed
	CP	0DH
	JR	Z,LINE		; If ENTER, then do another line
	CP	' '		; If space then do a page
	JR	Z,TOP
	CP	'-'		; If minus then backup one byte
	JR	Z,REVONE
	AND	0DFH		; Conv. to upper case
	CP	'P'		; Toggle printing?
	JR	Z,PRT
	CP	'D'		; Toggle data display?
	JR	Z,DATA
	CP	'T'		; Toggle text display?
	JR	Z,TEXT
	CP	'F'
	JR	Z,DOGETFILE
	CP	'C'
	JR	Z,DOCLOSEFILE
	CP	'L'
	JR	Z,DOLISTFILE
	JR	ENDPG		; If none of above, ask question again
;
DOGETFILE:
	CALL	GETFILE
	LD	B,1
	JP	PAGE
;
DOCLOSEFILE:
	CALL	CLOSEFILE
	JR	ENDPG
;
DOLISTFILE:
	CALL	SHOWLIST
	JR	ENDPG
;
LINE	LD	B,1		;Only one lines worth
	JR	PAGE
;
REVONE:
	LD	HL,(PCSAVE)	; Get old PC
	DEC	HL		; Back up one
	JR	LINE		; Print the instruction
;
PRT	EQU	$
	LD	A,(PRFLAG)	; Toggle printing flag
	XOR	1
	LD	(PRFLAG),A
	PUSH	HL
	LD	HL,PRTMSG
	JR	PRTIT
;
DATA	EQU	$
	LD	A,(DATFLG)	; Toggle data display flag
	XOR	1
	LD	(DATFLG),A
	PUSH	HL
	LD	HL,DATMSG
	JR	PRTIT
;
TEXT	EQU	$
	LD	A,(TEXTFLAG)	; Toggle text display flag
	XOR	1
	LD	(TEXTFLAG),A
	PUSH	HL
	LD	HL,TEXTMSG
	JR	PRTIT
;
;
;
DOINTRO:
	LD	BC,0703H	; Set scroll region to 3 lines
	SVC	@VDCTL
	LD	HL,INTRO	; Print the intro message
	SVC	@DSPLY
	RET
;
PRTIT	PUSH	AF		;Save ON/OFF value in A
	PUSH	HL		;Save dest cursor pos
	GET_CURSOR		;Get the current cursor in HL
	EX	(SP),HL		;Save it, and get dest back
	PUT_CURSOR		;Put the cursor there
	POP	HL		;Restore old pos
	POP	AF		;Restore flag value in A
	PUSH	HL		;Save old cursor loc
	LD	HL,ONSTR	;Get ON initially
	OR	A		;Check flag value
	JR	NZ,PRTIT1	;Jump if ON
	LD	HL,OFFSTR	;Get OFF
PRTIT1	EQU	$
	SVC	@DSPLY		;Display the message
	POP	HL		;Restore old cursor pos
	PUT_CURSOR		;Set cursor back
	POP	HL		;Restore old HL
	JP	ENDPG		;Join other code
;
EXIT	EQU	$
	LD	BC,0700H	; Set scroll region back to zero lines
	SVC	@VDCTL
	LD	A,(FROMFILE)
	OR	A
	CALL	NZ,CLOSEFILE
	LD	HL,0		; Exit normally
	SVC	@EXIT		; Go back to TRSDOS
;
;	Convert ASCII HEX string to a binary value in HL.
;	DE points to string.
;
HEXTOBIN EQU	$
	LD	DE,00H		; Zero DE
	EX	DE,HL		; Put buffer into DE
HEXLOOP	LD	A,(DE)		; Get a character from buffer
	CP	0DH		; Is that all?
	RET	Z		; If so, return
	ADD	HL,HL		; Shift
	ADD	HL,HL		;  HL left
	ADD	HL,HL		;   one
	ADD	HL,HL		;    nybble
	SUB	'0'		; Remove ASCII
	JR	C,HEXABRT	; Abort if character < '0'
	CP	10
	JR	C,HEXCONT	; Continue if 0-9
	CP	17
	JR	C,HEXABRT	; Abort if :;<=>?@
	AND	0DFH		; Fold lower -> upper case
	SUB	7		; Bring values down to 10-15
	CP	16
	JR	NC,HEXABRT	; Abort if > 'F'
HEXCONT	OR	L
	LD	L,A		; Add in the new bits
	INC	DE
	JR	HEXLOOP		; Go back and do the next character
HEXABRT	XOR	A		; Set NZ condition (aborted)
	RET
;
;	Print a message and read the users next command.
;
CONT	EQU	$
	PUSH	HL		; Save the PC address
	PUSH	BC		; Save what we use
CONT_1	EQU	$		; Secondary reentry point
	LD	HL,CONTMSG	; Get addr of continue message
	SVC	@DSPLY		; Print it on the screen
	SVC	@KEY		; Get a key from *KI
	PUSH	AF		; Save it
	CP	'0'		; Check for numeric and jump if not
	JR	C,NONUM
	CP	'9'+1
	JR	NC,NONUM
	LD	C,A		; Get it into C
	SVC	@DSP		; and display it
	LD	HL,BUFFER	; Start at buffer plus one
	LD	(HL),C		; Use C, A is destroyed
	INC	HL		; Point to next spot
	PUSH	HL
C_LOOP	EQU	$
	SVC	@KEY		; Get the next key
	CP	13		; Enter means end
	JR	Z,END_C_LOOP	; Jump if got it
	CP	' '		; Space also means end
	JR	Z,END_C_LOOP	; Jump if got that
	CP	24		; Check for erase all
	JR	NZ,C_LOOP_1	; Jump if not
	LD	HL,CONTMSG	; Get the message
	SVC	@DSPLY		; Display it, it does the work
C_LOOP_A	EQU	$
	POP	HL		; Restore stack
	POP	HL
	JR	CONT_1		; Restart input
C_LOOP_1	EQU	$
	CP	8		; Check for backspace
	JR	NZ,C_LOOP_2	; Jump if not
	LD	C,A		; Display the backspace to erase one
	SVC	@DSP		; character from the screen
	POP	HL		; Restore stack
	DEC	HL		; Back up pointer to buffer
	PUSH	HL		; Save it back
	LD	BC,BUFFER	; Get left margin value
	OR	A		; Clear carry
	SBC	HL,BC		; Check for at margin
	JR	Z,C_LOOP_A	; Jump if at margin
	JR	C_LOOP		; Get next character
;
C_LOOP_2	EQU	$
	CP	'0'		; Check for numeric
	JR	C,C_LOOP	; Jump if too small
	CP	'9'+1		; Check for too big
	JR	C,GOT_NUM	; Jump if not
	AND	0DFH		; Toupper() the character
	CP	'A'		; Check for HEX A-F
	JR	C,C_LOOP	; Jump if too small
	CP	'F'+1		; Check for F
	JR	NC,C_LOOP	; Jump if out of range
;
GOT_NUM	EQU	$
	POP	HL		; Get pointer back
	LD	(HL),A		; Store the character
	INC	HL		; Move pointer forward
	PUSH	HL		; Save the pointer
	LD	C,A		; Display the character
	SVC	@DSP
	JR	C_LOOP		; Get next key
;
END_C_LOOP	EQU	$
	POP	HL		; Done, get buffer pointer
	POP	BC
	PUSH	AF
	LD	C,13		; Get the CR character
	LD	(HL),C		; Terminate the string
	SVC	@DSP		; Display a newline
	LD	HL,BUFFER	; Get start of buffer
	CALL	HEXTOBIN	; Convert the value
	POP	AF		; Restore rest of regs
	POP	BC
	EX	(SP),HL		; Hide value back on stack
	PUSH	BC		; Fake stack to share code below
	PUSH	AF
NONUM	EQU	$
	LD	C,29		;Move to beginning of line
	SVC	@DSP
	LD	C,30		;Erase to end of line
	SVC	@DSP
	POP	AF		;Restore the regs
	POP	BC
	POP	HL		; Retrieve (possibly new) PC address
	RET
;
;	Disassemble one instruction at the current PC
;
Z80DIS	EQU	$
	XOR	A
	LD	(IFLAGS),A	; Zero index register flags
Z80TOP:
	CALL	GETBYTE
	JR	Z,BYTE_OK		;Jump if not out of range of file
	PUSH	AF
	PUSH	BC
	LD	B,8+5
	SVC	@SOUND
	POP	BC
	POP	AF
BYTE_OK:
	INC	HL		; Increment PC
	CP	0DDH		; IX type instruction?
	JP	Z,DOIX		; Check and set IX flag
	CP	0FDH		; IY type instruction?
	JP	Z,DOIY		; Check and set IY flag
	CP	0CBH		; Is it a bit level instruction
	JP	Z,DOBIT		; Do the bit instruction elsewhere
	CP	0EDH		; Is it an extended instruction
	JP	Z,Z80EXT	; Do the extended instr
;
	PUSH	HL		; Save PC address
	LD	E,A		; Put A into DE
	LD	D,0
	PUSH	DE		; Save opcode
	LD	HL,TABLE1	; Get address of table 1
	ADD	HL,DE		; Index into table for opcode number
	LD	A,(HL)		; Get the number from the table
	LD	HL,OP_1		; Start of linked list for table 1
	OR	A		; Set flags
	JR	Z,Z80DONE	; Fall thru if already zero
	LD	D,0		; Zero high byte of DE
Z80LL	LD	E,(HL)
	ADD	HL,DE		; Calc. addr. of next entry
	DEC	A
	JR	NZ,Z80LL	; Continue till A=0
Z80DONE	EQU	$
	INC	HL		; Point at name of opcode
	CALL	COPYOP		; Copy into buffer
	CALL	IXCHECK		; Change HL to IX,IY if necessary
	LD	C,(HL)
	POP	DE		; Get opcode back
	POP	HL		; Get PC back off stack
	LD	A,E		; Get real opcode back
	AND	0FH		; Strip upper nybble
	BIT	7,C		; Test for 'g' field
	CALL	NZ,ADDG1
	BIT	6,C		; Test for 'f' field
	CALL	NZ,ADDF
	BIT	5,C		; Test for 'ww' field
	CALL	NZ,ADDWW
	BIT	4,C		; Test for 'v' field
	CALL	NZ,ADDV
	BIT	3,C		; Test for 'j' field
	CALL	NZ,ADDJ
	BIT	2,C		; Test for 'm' field
	CALL	NZ,ADDM
	BIT	1,C		; Test for 's' field
	CALL	NZ,ADDG2
	BIT	0,C		; Test for 'mn' field
	CALL	NZ,ADDMN
	RET
;
DOIX	EQU	$
	LD	A,(IFLAGS)	; Get IFLAGS
	OR	A		; Test IFLAGS
	JR	NZ,ILLEGAL	; If not zero, then illegal instruction
	SET	0,A		; Set the IX bit
	LD	(IFLAGS),A	; Store new IFLAGS value
	JP	Z80TOP		; ... and get the next byte of the instr.
;
DOIY	EQU	$
	LD	A,(IFLAGS)	; Get IFLAGS
	OR	A
	JR	NZ,ILLEGAL	; Go on illegal instruction
	SET	1,A		; Set the IY bit
	LD	(IFLAGS),A	; Store new IFLAGS
	JP	Z80TOP		; ... and get the next byte of the instr.
;
ILLEGAL	EQU	$
	PUSH	HL		; Save PC
	LD	HL,ILLMESG
	CALL	COPYOP		; Copy illegal opcode message into buffer
	CALL	PRNTINST	; Print it out
	POP	HL		; Retrieve PC
	JP	Z80TOP		; Try to get the rest of the instruction
;
IXCHECK	EQU	$
	PUSH	AF		; Save ACC
	PUSH	HL		; Save PC
	PUSH	BC
	LD	A,(IFLAGS)
	OR	A
	JR	Z,IXOUT
	LD	HL,BUFFER
	LD	B,15
IXLOOP	EQU	$
	LD	A,(HL)
	CP	'H'
	INC	HL
	JR	Z,IXFOUND
	DJNZ	IXLOOP
IXOUT	POP	BC
	POP	HL
	POP	AF
	RET
IXFOUND	EQU	$
	LD	A,(HL)
	CP	'L'
	JR	NZ,IXLOOP
	LD	C,'X'
	LD	A,(IFLAGS)
	RRCA
	JR	C,COK
	INC	C
COK	EQU	$
	LD	(HL),C
	DEC	HL
	LD	(HL),'I'
	JR	IXOUT
;
DOBIT	EQU	$
	CALL	GETBYTE		; Get second byte of opcode
	LD	E,A		; Store second byte in E also
	INC	HL		; Bump PC
	AND	0F8H		; Strip lower 3 bits
	CP	30H
	JR	Z,ILLEGAL	; This is where SRR g would be
	LD	A,E		; Get second byte back
	AND	0F0H		; Mask off lower nybble
	CP	40H
	JR	C,BITOK		; Go if left of BIT instruction on map
	SRL	A		; Divide A
	SRL	A		; ... by four
	AND	0F0H		; Kill the lower nybble again
	ADD	A,30H		; Add three to upper nybble to adj.
BITOK	EQU	$
	RRCA
	RRCA
	RRCA
	RRCA			; Rotate into lower nybble
	BIT	3,E		; Test for lower half of map
	JR	Z,BITAR		; Value OK if bit = 0
	ADD	A,7		; Adjust for lower half of table
BITAR	EQU	$		; Now multiply Ax8 to get offset into
				; opcode table for opcode name.
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	ADD	A,A		; Ax2
	ADD	A,A		; Ax4
	ADD	A,A		; Ax8
	LD	E,A
	LD	D,0		; A into DE
	LD	HL,OP_2		; Addr of table
	ADD	HL,DE		; Calc addr of opcode
	CALL	COPYOP		; Put it into buffer
	POP	DE		; Retrieve opcode
	PUSH	DE		; Save it again
	LD	A,E		; Get opcode back
	CP	40H
	JR	C,NOTBIT	; Go if not a bit operation
	CALL	PUTBIT		; Add bit number on end of buffer
	CALL	ADDCOM		; Add a comma to the buffer
NOTBIT	CALL	ADDG2		; Add register spec on end of buffer
	POP	DE		; Retrieve opcode yet again
	POP	HL		; Retrieve PC
	RET
;
COPYOP	EQU	$
	LD	IX,BUFFER	; Point at line buffer
COPYL	LD	A,(HL)		; Get character from source
	LD	(IX),A		; Put in buffer
	INC	HL		; Bump source
	INC	IX		; Bump dest
	CP	03H		; <ETX>?
	JR	NZ,COPYL	; If not done, continue
	RET			; return
;
CATOP:	DEC	IX
	JR	COPYL		; And copy chars onto end of buffer
;
PUTIN	LD	DE,BUFFER	; Point to buffer
	LD	B,'*'		; Load an alphabetic wildcard
	PUSH	BC		; Save C
	LD	C,0		; Set flag to 0
TOPPUT	LD	A,(DE)		; Get a character
	CP	B		; Found a field delimiter?
	JR	Z,OUT		; If so, copy to field
	CP	03H		; Otherwise, <CR>?
	JR	Z,GOCAT		; If EOL, then just catenate
	INC	DE		; Bump pointer
	JR	TOPPUT		; Otherwise, search some more
OUT	LD	A,(HL)		; Get a character from source
	LD	B,A		; Get char into B also
	CP	03H		; End of source?
	JR	Z,PUTOUT	; If so, clean up and return
	LD	A,(DE)		; Check char in buffer
	CP	03H		; ... to see if it's EOL
	JR	NZ,PUTCONT	; If not, then continue
	LD	C,1		; Set flag if EOL
PUTCONT EQU	$
	LD	A,B		; Get char back in A
	LD	(DE),A		; Otherwise, save in field
	INC	DE		; Bump dest pointer
	INC	HL		; Bump source pointer
	JR	OUT		; Continue copying
PUTOUT	EQU	$
	LD	A,C		; Get flag into A
	POP	BC		; Retrieve C
	OR	A		; Set flags
	RET	Z		; Return if OK
	PUSH	DE
	POP	HL		; Get buffer pointer into HL
	DB	00H
	LD	(HL),03H	; Terminate buffer
	INC	HL		; Bump pointer again
	PUSH	HL		; Get HL
	POP	IX		; ... into IX
	RET
;
GOCAT	EQU	$
	POP	BC		; Retrieve C
	JP	CATOP		; Tack message on end of buffer
;
PUTNUM	LD	B,'#'		; Load a numeric wildcard
	PUSH	BC		; Save C
	LD	C,0		; Set flag to zero
	LD	DE,BUFFER	; Point at buffer
	JR	TOPPUT		; Store it an continue
;
ADDCOM	EQU	$
	LD	(IX-1),','	; Put comma in buffer
	LD	(IX+0),03H	; Put <ETX> terminator
	INC	IX		; Bump buffer pointer
	RET
;
PUTBIT	EQU	$
	LD	A,E		; Get opcode into A
	AND	30H		; Mask bit 7,6,3,2,1,0
	RRCA
	RRCA
	RRCA			; Put 2 x bits 4,5 in A
	BIT	3,E
	JR	Z,TOPTBL	; Bit zero if top of map
	INC	A		; Bottom of table adds one to bit #
TOPTBL	ADD	A,'0'		; Make ASCII
	LD	(IX-1),A	; Store bit character
	LD	(IX+0),03H	; Store terminator
	INC	IX		; Bump buffer pointer
	RET
;
ADDG2	EQU	$
	LD	A,E		; Get opcode
	AND	07H		; Kill bits 7-3
	CP	6		; Is it an (HL) type instruction?
	JR	NZ,REGTBL	; If not, then use register table
ADDIND	LD	A,(IFLAGS)	; Get IFLAGS into a
	OR	A		; Set flags
	JR	Z,ISHL		; If no IFLAGS set, then (HL)
	CALL	INDEX		; Do the indexed address business
	RET
ISHL	EQU	$
	PUSH	HL		; Save PC
	LD	HL,HLMSG	; Point at '(HL)<ETX>'
	PUSH	DE		; Save opcode
	PUSH	BC		; Save C
	CALL	PUTIN	 	; Put message in buffer
	POP	BC		; Retrieve C
	POP	DE		; Retrieve opcode
	DC	8,0
;	LD	A,C		; Get type into A
;	AND	0B0H		; Keep bits 7,5,4
;	CP	30H		; Decide whether we need a comma
;	CALL	Z,ADDCOM	; Add it if necessary
	POP	HL		; Retrieve PC
	RET
REGTBL	EQU	$
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	LD	E,A
	LD	D,0		; A into DE
	LD	HL,REGSTR	; Point at 'BCDEHL*A'
	ADD	HL,DE		; Point at the correct letter
	LD	A,(HL)		; Get it into A
	LD	(IX-1),A	; Put it into buffer
	LD	(IX+0),03H	; Terminate buffer
	INC	IX		; Bump buffer pointer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	RET
;
ADDG1	EQU	$
	LD	A,E		; Get opcode
	AND	30H		; Keep bits 4,5
	RRCA
	RRCA
	RRCA
	RRCA			; Rotate into bits 0,1
	BIT	3,E		; Test for lower part of map
	JR	Z,GOK		; Go if top of map
	ADD	A,4		; Add four for bottom of map
GOK	CP	3		; Is it an (HL) type instruction?
	JR	NZ,G1CONT	; If not, go around (HL) code
	CALL	ADDIND		; Put the indirect stuff in buffer
	LD	A,C
	AND	6		; S or M field yet to do?
	CALL	NZ,ADDCOM	; Add a comma after (HL),(IX+dd), etc.
	RET
G1CONT	EQU	$
	PUSH	DE		; Save opcode
	PUSH	HL		; Save PC
	LD	E,A
	LD	D,0		; Xfer A to DE
	LD	HL,GRSTR	; Point to list of 'g1' register names
	ADD	HL,DE		; Point at proper one
	LD	A,(HL)		; Get it
	LD	(SCRATCH),A	; Store in scratch buffer
	LD	A,03H		; <ETX>
	LD	(SCRATCH+1),A	; Terminate buffer
	LD	HL,SCRATCH	; Point at scratch buffer
	CALL	PUTIN		; Put in or tack on buffer
	POP	HL		; Retrieve PC
	POP	DE		; Retrieve opcode
	RET
;
INDEX	EQU	$
	PUSH	BC		; Save C
	BIT	0,A		; Test IX flag
	PUSH	AF
	CALL	GETBYTE
	LD	C,A
	POP	AF
	INC	HL		; Bump PC
	PUSH	HL		; Save PC on stak
	PUSH	DE		; Save opcode
	JR	Z,IYM		; Go if IY spec'd
	LD	HL,IXD		; Point at IX displacement area
	SVC	@HEX8		; Convert to hex into message
	LD	HL,IXMSG	; Point at IX message
	JR	INDXOUT
IYM	LD	HL,IYD		; Point at IY displacement area
	SVC	@HEX8		; Convert displacement to hex
	LD	HL,IYMSG	; Point at IY message
INDXOUT	CALL	PUTIN		; Put message in buffer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	POP	BC		; Retrieve C
	RET
;
ADDF	EQU	$
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	LD	A,E		; Get opcode into A
	AND	30H		; Keep bits 4,5
	RRCA
	RRCA
	RRCA
	RRCA			; Rotate into low two bits
	BIT	3,E		; Check for upper or lower map
	JR	Z,FDONE		; If zero, we're in upper part
	ADD	A,4		; Otherwise, add four
FDONE	LD	B,A		; Save A into B
	ADD	A,A		; Ax2
	ADD	A,B		; Ax3
	LD	E,A
	LD	D,0		; A into DE
	LD	HL,CONTBL	; Point to condition table
	ADD	HL,DE		; Calc addr of proper conditional
	CALL	PUTIN		; Tack it onto the buffer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	RET
;
ADDWW	EQU	$
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	LD	A,E		; Get opcode into A
	AND	30H		; Keep bits 4,5
	RRCA
	RRCA
	RRCA
	RRCA			; Rotate into bits 0,1
	LD	B,A		; Save A into B
	ADD	A,A		; Ax2
	ADD	A,B		; Ax3
	BIT	7,E		; Check for ww/zz field
	JR	Z,WWOK		; If ww field, go on
	LD	D,A		; Move A into D
	LD	A,E		; Get opcode into A
	AND	30H		; Keep bits 4,5
	CP	30H		; Check for PUSH AF
	LD	A,D		; Get old A back
	JR	NZ,WWOK
	ADD	A,3		; Move over to AF
WWOK	EQU	$
	LD	D,A		; Put offset into d
	LD	A,(IFLAGS)	; Get index flags
	LD	E,A		; Get flags into E
	LD	A,D		; Get offset back into A
	BIT	0,E		; Test for IX
	JR	Z,WWC1
	ADD	A,9		; Move over to IX
WWC1	BIT	1,E		; Test for IY
	JR	Z,WWC2
	ADD	A,12		; Move over to IY
WWC2	EQU	$
	LD	E,A
	LD	D,0		; A into DE
	LD	HL,WWTBL	; Point to condition table
	ADD	HL,DE		; Calc addr of proper pair
	CALL	PUTIN		; Put in proper place in buffer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	RET
;
ADDV	EQU	$
	LD	A,E		; Get opcode
	AND	30H		; Keep bits 4,5
	BIT	3,E		; Test for bottom of map
	JR	Z,VDONE		; Go if top of map
	ADD	A,8		; Add eight for bottom of map
VDONE	PUSH	BC		; Save C
	PUSH	HL		; Save PC
	LD	C,A		; Get restart # into C
	PUSH	IX
	POP	HL
	LD	(IX-1),'0'
	SVC	@HEX8		; Convert to ASCII in buffer
	LD	(HL),'H'
	INC	HL
	LD	(HL),03H	; Terminate buffer
	INC	HL
	PUSH	HL
	POP	IX		; Get new end of buffer pointer
	POP	HL		; Retrieve PC
	POP	BC		; Get old C back
	RET
;
ADDJ	EQU	$
	PUSH	BC		; Save C
	CALL	GETBYTE
	LD	C,A
	INC	HL		; Bump PC
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	BIT	7,C		; Test high bit of C
	LD	B,0		; Zero B
	JR	Z,JOK		; If zero, B is ok
	DEC	B		; Otherwise, sign extend C into B
JOK	ADD	HL,BC		; Calc. relative address
	EX	DE,HL		; Address into DE
	PUSH	IX
	POP	HL
	LD	(IX-1),'0'
	SVC	@HEX16		; Convert DE to hex into buffer
	LD	(HL),'H'
	INC	HL
	LD	(HL),03H	; Terminate buffer
	INC	HL
	PUSH	HL
	POP	IX		; Get new EOB pointer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	POP	BC		; Retrieve C
	RET
;
ADDM	EQU	$
	PUSH	BC		; Save C
	CALL	GETBYTE
	LD	C,A
	INC	HL		; Bump PC
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	LD	HL,SCRATCH	; Point at scratch buffer
	PUSH	HL		; Save scratch buffer
	LD	(HL),'0'
	INC	HL
	SVC	@HEX8		; Convert C to ASCII in buffer
	LD	(HL),'H'
	INC	HL
	LD	(HL),03H	; Terminate buffer
	POP	HL		; Retrieve scratch buffer addr.
	CALL	PUTNUM		; Put number into buffer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	POP	BC		; Retrieve C
	RET
;
ADDMN	EQU	$
	PUSH	BC		; Save C
	CALL	GETBYTE
	LD	E,A
	INC	HL		; Bump PC
	CALL	GETBYTE
	LD	D,A
	INC	HL		; Bump PC
	PUSH	HL		; Save PC
	PUSH	DE		; Save opcode
	LD	HL,SCRATCH	; Point at scratch buffer
	PUSH	HL		; Save scratch buffer addr
	LD	(HL),'0'
	INC	HL
	SVC	@HEX16		; Convert DE to ASCII in buffer
	LD	(HL),'H'
	INC	HL
	LD	(HL),03H	; Terminate buffer
	POP	HL		; Retrieve scratch pointer
	CALL	PUTNUM		; Put number into buffer
	POP	DE		; Retrieve opcode
	POP	HL		; Retrieve PC
	POP	BC		; Retrieve C
	RET
;
Z80EXT	EQU	$
	CALL	GETBYTE
	INC	HL		; Bump PC
	PUSH	HL		; Save PC on stak
	LD	E,A
	LD	D,0		; A --> DE
	PUSH	DE		; Save opcode in E
	LD	HL,TABLE3	; Get address of extended table
	ADD	HL,DE		; Get position of this instr in table
	LD	A,(HL)		; Get table entry
	LD	HL,OP_3		; Start of linked list for table 3
	OR	A		; Set flags
	LD	D,0
	JR	Z,EXTDONE	; Fall thru if already zero
EXTLL	LD	E,(HL)		; Get offset
	ADD	HL,DE		; Calc. addr. of next item in list
	DEC	A
	JR	NZ,EXTLL	; Continue until A=0
EXTDONE	INC	HL
	DW	00
	CALL	COPYOP		; Copy into buffer
	CALL	IXCHECK		; Change HL to IX,IY if necessary
	LD	C,(HL)
	POP	DE		; Retrieve opcode
	POP	HL		; Get PC back
	LD	A,E		; Get real opcode back
	BIT	7,C		; Test for 'g' field
	CALL	NZ,ADDG1
	BIT	5,C		; Test for 'ww' field
	CALL	NZ,ADDWW
	BIT	2,C		; Test for 'm' field
	CALL	NZ,ADDM
	BIT	0,C		; Test for 'mn' field
	CALL	NZ,ADDMN
	RET
;
PRNTINST EQU	$
	PUSH	HL		; Save PC
	LD	HL,SCRATCH	; Fill the print buffer with spaces
	PUSH	HL		; Save the start for later
	LD	DE,SCRATCH+1	; Get the destination
	LD	(HL),' '	; Store one blank
	LD	BC,80		; Number to fill
	LDIR			; Copy to rest
	POP	HL		; Restore the start
	LD	DE,(PCSAVE)	; Get PC into DE
	SVC	@HEX16		; Convert DE to hex ASCII
	INC	HL
	PUSH	HL
	POP	IX
	LD	(IX-1),':'	; Colon
	LD	(IX),9
	INC	IX
	EX	(SP),HL
	LD	DE,(PCSAVE)
	CALL	CHECKDATA	; See if data should be displayed
	EX	(SP),HL
	LD	HL,BUFFER	; Point at buffer
	CALL	COPYL
	EX	(SP),HL
	LD	DE,(PCSAVE)
	CALL	ADDTEXT		; Put in the text data if necessary
	EX	(SP),HL
	LD	(IX-1),0DH	; Load up a <CR>
	LD	HL,SCRATCH
	CALL	PRNTLINE
;
	LD	A,(PRFLAG)
	OR	A		; Set flags
	JR	Z,NOPRNT	; Go if printer off
	LD	HL,SCRATCH	; Point at buffer
	SVC	@PRINT		; Send line to printer
	JR	NZ,PRERROR
;
NOPRNT	EQU	$
	POP	HL		; Retrieve PC
	RET
;
PRERROR	EQU	$
	OR	0C0H
	PUSH	AF
	LD	HL,PRERRMSG
	SVC	@DSPLY
	POP	AF
	LD	C,A
	SVC	@ERROR
	XOR	A
	LD	(PRFLAG),A
	JR	NOPRNT
;
CHECKDATA	EQU	$
	LD	A,(DATFLG)
	OR	A
	RET	Z
	PUSH	HL		;Save the registers
	PUSH	DE
	OR	A		;Clear the carry
	SBC	HL,DE		;Compute the length
	PUSH	BC		;Save BC
	LD	B,L		;Get the length
	EX	DE,HL		;Move the source address to HL
	LD	E,4		;4 bytes max (Plus some fudge)
	PUSH	IX		;Put the buffer on the stack
DATLOOP	EQU	$
	CALL	GETBYTE
	LD	C,A
	EX	(SP),HL		;Swap buffers
	CALL	HEX8
	EX	(SP),HL		;Swap back
	INC	HL		;Point at next data
	DEC	E		;Decrement fields used
	DJNZ	DATLOOP		;Loop until done
	POP	IX		;Get the buffer back
	LD	(IX),9
	INC	IX
	LD	A,E
	OR	A
	JR	Z,DAT_1
	LD	(IX),9
	INC	IX
DAT_1:
	POP	BC		;Pop the regs
	POP	DE
	POP	HL
	RET			;Return to caller
;
ADDTEXT	EQU	$
	LD	A,(TEXTFLAG)	; See if text should be displayed
	OR	A		; Is it on?
	RET	Z		; Return if not
	PUSH	HL
	DEC	IX
	CALL	VISLEN
	LD	HL,40
	LD	A,(DATFLG)
	OR	A
	JR	NZ,ADD_1
	LD	HL,32
ADD_1	EQU	$
	OR	A
	SBC	HL,BC			;Compute the space left
	LD	(IX),9
	INC	IX
	LD	BC,8
	JR	Z,ADD_1A
	JR	NC,ADD_1
ADD_1A
	LD	(IX),';'		; Put in comment character
	INC	IX
ADD_2	EQU	$
	POP	HL
	PUSH	HL
ADD_3	EQU	$
	EX	DE,HL
	CALL	GETBYTE
	EX	DE,HL
	CP	' '
	JR	C,ADD_5
	CP	128
	JR	NC,ADD_5
	LD	(IX),A
	JR	ADD_6
ADD_5	EQU	$
	LD	(IX),'.'
ADD_6	EQU	$
	INC	IX
	OR	A
	INC	DE
	SBC	HL,DE
	JR	NZ,ADD_2
	INC	IX
	POP	HL
	RET
;
;
;
VISLEN:
	PUSH	HL
	LD	HL,SCRATCH
	LD	BC,0
VIS_1:
	LD	A,(HL)
	CP	9
	JR	NZ,VIS_3
	LD	A,8
	ADD	A,C
	AND	0F8H
	LD	C,A
	JR	VIS_4
VIS_3:
	CP	3
	JR	Z,VIS_5
	CP	13
	JR	Z,VIS_5
	INC	C
VIS_4:
	INC	HL
	JR	VIS_1
VIS_5:
	POP	HL
	RET
;
;	Convert character in A to 2 digit hex number, store (HL)
;
HEX8	EQU	$
	LD	A,C		;Get the character
	PUSH	AF		;Save the original
	SRL	A		;Divide by 16
	SRL	A
	SRL	A
	SRL	A
	CALL	HEXCHR		;Convert high nibble
	LD	(HL),A		;Store it
	INC	HL		;Point to next store location
	POP	AF		;Get the original back
	CALL	HEXCHR		;Convert lower nibble
	LD	(HL),A		;Store it
	INC	HL		;Point to next
	RET			;Return
;	Convert character in A to a Hex character, 0-F.
HEXCHR	EQU	$
	AND	15		;Mask for reality
	ADD	A,'0'		;Add initial digit conversion
	CP	'9'+1		;Less than '9'?
	RET	M		;Return if so
	ADD	A,7		;Convert to A-F
	RET			;Return
;
PRNTLINE EQU	$
	LD	B,0		; Get length
PRNT_1:
	LD	A,(HL)		; Get a character
	CP	09H		; Is it a tab?
	JR	Z,TABCHAR	; If so, convert to spaces
	LD	C,A		; Get character into C
	PUSH	BC
	SVC	@DSP		; Otherwise, put character on screen
	POP	BC
	LD	A,C		; Get char back into A
	CP	03H		; ETX?
	RET	Z		; If so, return
	CP	0DH		; <CR>?
	RET	Z		; Return on that, too
	INC	B		; Increment count
PBACK	INC	HL		; Bump buffer pointer
	JR	PRNT_1
;
TABCHAR	EQU	$
	LD	B,8
	PUSH	HL
TAB1	DEC	HL
	LD	A,(HL)
	CP	' '
	JR	C,TAB2
	DJNZ	TAB1
	LD	B,8
	JR	TAB1
;
TAB2	LD	C,' '
	SVC	@DSP
	DJNZ	TAB2
	POP	HL
	JR	PBACK
;
; Messages
;
*LIST		OFF
DATFLG	DB	1
DATMSG	EQU	004CH
TEXTMSG	EQU	0011H
PRTMSG	EQU	002BH
CONTMSG	DB	29,'Address>',30,3
PRERRMSG	DB	'Printer error, printer disabled',13
PCSAVE	DW	00H
PRFLAG	DB	00H
TEXTFLAG	DB	1
INTRO	DB	28,31,16,17
	DB	'Text display ==> ',16,'Yes',17,'  '
	DB	'Route to printer ==> ',16,'No',17,'  '
	DB	'Display instruction data ==> ',16,'Yes',17,10
	DB	'Break ==> exit to TRSDOS   '
	DB	'Enter ==> next instruction      '
	DB	'Space ==> next screen'
	DB	'________________________________________________________________________________',3
ONSTR	DB	16,'Yes',17,3
OFFSTR	DB	16,'No',17,32,3
REGSTR	DB	'BCDEHL*A'
HLMSG	DB	'(HL)',03H
IXMSG	DB	'(IX+'
IXD	DB	'**H)',03H
IYMSG	DB	'(IY+'
IYD	DB	'**H)',03H
CONTBL	DB	'NZ',03H
	DB	'NC',03H
	DB	'PO',03H
	DB	'P',03H,00H
	DB	'Z',03H,00H
	DB	'C',03H,00H
	DB	'PE',03H
	DB	'M',03H,00H
WWTBL	DB	'BC',03H
	DB	'DE',03H
	DB	'HL',03H
	DB	'SP',03H
	DB	'AF',03H
	DB	'IX',03H
	DB	'IY',03H
GRSTR	DB	'BDH*CELA'
BUFFER	DS	40		; Buffer for constructing dis'd instruc's
IFLAGS	DB	0		; Index flags. Bit 0:IX, Bit 1:IY.
ILLMESG	DB	'*** Illegal Instruction ***',03H
SCRATCH	DS	86		; Used for constructing hex numbers
;
; This is the table of the one-byte opcode map
;
TABLE1	EQU	$
	DB	00,10,12,16,21,23,25,30,35,44,49,52,21,23,25,61
	DB	01,10,12,16,21,23,25,31,36,44,49,52,21,23,25,62
	DB	02,10,13,16,21,23,25,32,37,44,50,52,21,23,25,63
	DB	03,10,14,16,21,23,25,33,38,44,51,52,21,23,25,64
	DB	04,04,04,04,04,04,04,04,04,04,04,04,04,04,04,04
	DB	04,04,04,04,04,04,04,04,04,04,04,04,04,04,04,04
	DB	04,04,04,04,04,04,04,04,04,04,04,04,04,04,04,04
	DB	04,04,04,04,04,04,39,04,04,04,04,04,04,04,04,04
	DB	05,05,05,05,05,05,05,05,40,40,40,40,40,40,40,40
	DB	06,06,06,06,06,06,06,06,41,41,41,41,41,41,41,41
	DB	07,07,07,08,07,07,07,07,42,42,42,42,42,42,42,42
	DB	08,08,08,08,08,08,08,08,43,43,43,43,43,43,43,43
	DB	09,11,15,17,22,24,26,34,09,45,15,65,22,56,57,34
	DB	09,11,15,18,22,24,27,34,09,46,15,53,22,65,58,34
	DB	09,11,15,19,22,24,28,34,09,47,15,54,22,65,59,34
	DB	09,11,15,20,22,24,29,34,09,48,15,55,22,65,60,34
OP_1	DB	OP1-$
	DB	'NOP',03H,0H
OP1	DB	OP2-$
	DB	'DJNZ	',03H,8H
OP2	DB	OP3-$
	DB	'JR	NZ,',03H,8H
OP3	DB	OP4-$
	DB	'JR	NC,',03H,8H
OP4	DB	OP5-$
	DB	'LD	*,',03H,82H
OP5	DB	OP6-$
	DB	'ADD	A,',03H,2H
OP6	DB	OP7-$
	DB	'SUB	',03H,2H
OP7	DB	OP8-$
	DB	'AND	',03H,2H
OP8	DB	OP9-$
	DB	'OR	',03H,2H
OP9	DB	OP10-$
	DB	'RET	',03H,40H
OP10	DB	OP11-$
	DB	'LD	**,',03H,21H
OP11	DB	OP12-$
	DB	'POP	',03H,20H
OP12	DB	OP13-$
	DB	'LD	(**),A',03H,20H
OP13	DB	OP14-$
	DB	'LD	(#####H),HL',03H,1H
OP14	DB	OP15-$
	DB	'LD	(#####H),A',03H,1H
OP15	DB	OP16-$
	DB	'JP	* ,',03H,41H
OP16	DB	OP17-$
	DB	'INC	',03H,20H
OP17	DB	OP18-$
	DB	'JP	',03H,1H
OP18	DB	OP19-$
	DB	'OUT	(###H),A',03H,4H
OP19	DB	OP20-$
	DB	'EX	(SP),HL',03H,0H
OP20	DB	OP21-$
	DB	'DI',03H,0H
OP21	DB	OP22-$
	DB	'INC	',03H,80H
OP22	DB	OP23-$
	DB	'CALL	* ,',03H,41H
OP23	DB	OP24-$
	DB	'DEC	',03H,80H
OP24	DB	OP25-$
	DB	'PUSH	',03H,20H
OP25	DB	OP26-$
	DB	'LD	*,',03H,84H
OP26	DB	OP27-$
	DB	'ADD	A,',03H,4H
OP27	DB	OP28-$
	DB	'SUB	',03H,4H
OP28	DB	OP29-$
	DB	'AND	',03H,4H
OP29	DB	OP30-$
	DB	'OR	',03H,4H
OP30	DB	OP31-$
	DB	'RLCA',03H,0H
OP31	DB	OP32-$
	DB	'RLA',03H,0H
OP32	DB	OP33-$
	DB	'DAA',03H,0H
OP33	DB	OP34-$
	DB	'SCF',03H,0H
OP34	DB	OP35-$
	DB	'RST	',03H,10H
OP35	DB	OP36-$
	DB	'EX	AF,AF',27H,03H,0H
OP36	DB	OP37-$
	DB	'JR	',03H,8H
OP37	DB	OP38-$
	DB	'JR	Z,',03H,8H
OP38	DB	OP39-$
	DB	'JR	C,',03H,8H
OP39	DB	OP40-$
	DB	'HALT',03H,0H
OP40	DB	OP41-$
	DB	'ADC	A,',03H,2H
OP41	DB	OP42-$
	DB	'SBC	A,',03H,2H
OP42	DB	OP43-$
	DB	'XOR	',03H,2H
OP43	DB	OP44-$
	DB	'CP	',03H,2H
OP44	DB	OP45-$
	DB	'ADD	HL,',03H,20H
OP45	DB	OP46-$
	DB	'RET',03H,0H
OP46	DB	OP47-$
	DB	'EXX',03H,0H
OP47	DB	OP48-$
	DB	'JP	(HL)',03H,0H
OP48	DB	OP49-$
	DB	'LD	SP,HL',03H,0H
OP49	DB	OP50-$
	DB	'LD	A,(**)',03H,20H
OP50	DB	OP51-$
	DB	'LD	HL,(#####H)',03H,1H
OP51	DB	OP52-$
	DB	'LD	A,(#####H)',03H,1H
OP52	DB	OP53-$
	DB	'DEC	',03H,20H
OP53	DB	OP54-$
	DB	'IN	A,(###H)',03H,4H
OP54	DB	OP55-$
	DB	'EX	DE,HL',03H,0H
OP55	DB	OP56-$
	DB	'EI',03H,0H
OP56	DB	OP57-$
	DB	'CALL	',03H,1H
OP57	DB	OP58-$
	DB	'ADC	A,',03H,4H
OP58	DB	OP59-$
	DB	'SBC	A,',03H,4H
OP59	DB	OP60-$
	DB	'XOR	',03H,4H
OP60	DB	OP61-$
	DB	'CP	',03H,4H
OP61	DB	OP62-$
	DB	'RRCA',03H,0H
OP62	DB	OP63-$
	DB	'RRA',03H,0H
OP63	DB	OP64-$
	DB	'CPL',03H,0H
OP64	DB	OP65-$
	DB	'CCF',03H,0H
OP65	DB	OP66-$
	DB	'** HELP! **',03H,0H
OP66	EQU	$
;
; Table 3
;
TABLE3	EQU	$
	DB	01,05,00,00,17,00,00,00,01,05,00,00,17,00,00,00
	DB	01,05,00,00,17,00,00,00,01,05,00,00,17,00,00,00
	DB	01,05,00,00,17,00,00,00,01,05,00,00,17,00,00,00
	DB	00,00,00,00,18,00,00,00,01,05,00,00,17,00,00,00
	DB	02,06,09,12,19,22,23,26,02,06,33,36,41,42,00,44
	DB	02,06,09,12,00,00,24,27,02,06,33,36,41,00,43,45
	DB	02,06,09,12,20,00,00,28,02,06,33,36,41,00,00,46
	DB	00,00,09,12,21,00,25,00,02,06,33,36,41,00,00,00
	DB	00,00,00,13,00,00,00,00,00,00,00,37,00,00,00,00
	DB	00,00,00,14,00,00,00,00,00,00,00,38,00,00,00,00
	DB	03,07,10,15,00,00,00,00,29,31,34,39,00,00,00,00
	DB	04,08,11,16,00,00,00,00,30,32,35,40,00,00,00,00
	DB	00,00,00,00,00,00,00,00,00,00,00,00,00,00,00,00
	DB	00,00,00,00,00,00,00,00,00,00,00,00,00,00,00,00
	DB	00,00,00,00,00,00,00,00,00,00,00,00,00,00,00,00
	DB	00,00,00,00,00,00,00,00,00,00,00,00,00,00,00,00
OP_3	DB	OP3X1-$
	DB	'** HELP! **',03H,0H
OP3X1	DB	OP3X2-$
	DB	'IN0	*,(###H)',03H,84H
OP3X2	DB	OP3X3-$
	DB	'IN	*,(C)',03H,80H
OP3X3	DB	OP3X4-$
	DB	'LDI',03H,0H
OP3X4	DB	OP3X5-$
	DB	'LDIR',03H,0H
OP3X5	DB	OP3X6-$
	DB	'OUT0	(###H),',03H,84H
OP3X6	DB	OP3X7-$
	DB	'OUT	(C),',03H,80H
OP3X7	DB	OP3X8-$
	DB	'CPI',03H,0H
OP3X8	DB	OP3X9-$
	DB	'CPIR',03H,0H
OP3X9	DB	OP3X10-$
	DB	'SBC	HL,',03H,20H
OP3X10	DB	OP3X11-$
	DB	'INI',03H,0H
OP3X11	DB	OP3X12-$
	DB	'INIR',03H,0H
OP3X12	DB	OP3X13-$
	DB	'LD	(#####H),',03H,21H
OP3X13	DB	OP3X14-$
	DB	'OTIM',03H,0H
OP3X14	DB	OP3X15-$
	DB	'OTIMR',03H,0H
OP3X15	DB	OP3X16-$
	DB	'OTI',03H,0H
OP3X16	DB	OP3X17-$
	DB	'OTIR',03H,0H
OP3X17	DB	OP3X18-$
	DB	'TST	',03H,80H
OP3X18	DB	OP3X19-$
	DB	'TST	(HL)',03H,0H
OP3X19	DB	OP3X20-$
	DB	'NEG',03H,0H
OP3X20	DB	OP3X21-$
	DB	'TST	',03H,4H
OP3X21	DB	OP3X22-$
	DB	'TSTIO	',03H,1H
OP3X22	DB	OP3X23-$
	DB	'RETN',03H,0H
OP3X23	DB	OP3X24-$
	DB	'IM0',03H,0H
OP3X24	DB	OP3X25-$
	DB	'IM1',03H,0H
OP3X25	DB	OP3X26-$
	DB	'SLP',03H,0H
OP3X26	DB	OP3X27-$
	DB	'LD	I,A',03H,0H
OP3X27	DB	OP3X28-$
	DB	'LD	A,I',03H,0H
OP3X28	DB	OP3X29-$
	DB	'RRD',03H,0H
OP3X29	DB	OP3X30-$
	DB	'LDD',03H,0H
OP3X30	DB	OP3X31-$
	DB	'LDDR',03H,0H
OP3X31	DB	OP3X32-$
	DB	'CPD',03H,0H
OP3X32	DB	OP3X33-$
	DB	'CPDR',03H,0H
OP3X33	DB	OP3X34-$
	DB	'ADC	HL,',03H,20H
OP3X34	DB	OP3X35-$
	DB	'IND',03H,0H
OP3X35	DB	OP3X36-$
	DB	'INDR',03H,0H
OP3X36	DB	OP3X37-$
	DB	'LD	**,(#####H)',03H,21H
OP3X37	DB	OP3X38-$
	DB	'OTDM',03H,0H
OP3X38	DB	OP3X39-$
	DB	'OTDMR',03H,0H
OP3X39	DB	OP3X40-$
	DB	'OUTD',03H,0H
OP3X40	DB	OP3X41-$
	DB	'OTDR',03H,0H
OP3X41	DB	OP3X42-$
	DB	'MLT	',03H,20H
OP3X42	DB	OP3X43-$
	DB	'RETI',03H,0H
OP3X43	DB	OP3X44-$
	DB	'IM2',03H,0H
OP3X44	DB	OP3X45-$
	DB	'LD	R,A',03H,0H
OP3X45	DB	OP3X46-$
	DB	'LD	A,R',03H,0H
OP3X46	DB	OP3X47-$
	DB	'RLD',03H,0H
OP3X47	EQU	$
;
; OP_table 2
;
OP_2	EQU	$
	DB	'RLC	',3,0,0,0
	DB	'RL	',3,0,0,0,0
	DB	'SLA	',3,0,0,0
	DB	'*******',3
	DB	'BIT	',3,0,0,0
	DB	'RES	',3,0,0,0
	DB	'SET	',3,0,0,0
	DB	'RRC	',3,0,0,0
	DB	'RR	',3,0,0,0,0
	DB	'SRA	',3,0,0,0
	DB	'SRL	',3,0,0,0
	DB	'BIT	',3,0,0,0
	DB	'RES	',3,0,0,0
	DB	'SET	',3,0,0,0
*LIST		ON
;
*GET		FILEIO
