#!/bin/ksh
#-------------------------------------------------------------------------------
#
# Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
#
# @(#)esm_cimom.ksh	1.1 04/04/01
#
#-------------------------------------------------------------------------------

#
# Purpose:
#	Output an error message to stderr and exit with the specified
#	return code.  A return code of 1 implies an error with the command
#	line arguments, while all other values imply something unique to
#	the application.
# Arguments:
#	1 : return code
#	2-n : error message
#
error()
{
    rc=$1 ; shift 1

    echo "$APP_NAME: ERROR $rc: $*" >&2

    if [ $rc -eq 1 ]; then
	echo "" >&2
	echo "Type \"$APP_NAME --help\" for usage information." >&2
    fi

    exit $rc
}

#
# Purpose:
#	Output a warning message to stderr.
# Arguments:
#	1-n : warning message
#
warn()
{
    echo "" >&2
    echo "$APP_NAME: WARNING: $*" >&2
    echo "" >&2
}

#
# Purpose:
#	Check for extra arguments and error if so.
# Arguments:
#	0-n : command line arguments remaining after some initial processing
#	has taken place.
# Actions:
#	When the number of arguments is greater than zero, an error occurs,
#	which will exit the application.  Otherwise, do nothing.
#
checkExtraArguments()
{
    if [ $# -gt 0 ]; then 
	error 1 "extraneous arguments: $*"
    fi
}

#
# Purpose:
#	Check for a required operand and error if not present.
# Arguments:
#	0-n : command line arguments remaining after some initial processing
#	has taken place.
# Actions:
#	When the number of arguments is greater than one, an error occurs,
#	which will exit the application (extra args).
#	When the number of arguments is equal to zero, an error occurs,
#	which will exit the application (missing operand).
#	Otherwise, do nothing.
#
checkRequiredOperand()
{
    if [ $# -gt 1 ]; then
	shift 1
	error 1 "extraneous arguments: $*"
    elif [ $# -eq 0 ]; then
	error 1 "missing required operand"
    fi
}


#
# Purpose:
#	Check to ensure the user is root and error if not.
#
checkRoot()
{
    user=`/bin/id |/bin/grep root |wc -l`
    if [ $user -eq 0 ]; then
	error 2 "Must be root to execute"
    fi
}

#
# Purpose:
#	Check to ensure the ESM packages are loaded on the system.
# Actions:
#	Set the ESM_BASE variable.
#
checkEsm()
{
    ESM_BASE=`/bin/pkgparam SUNWstm ESM_BASE 2> /dev/null`
    if [ $? -ne 0 ]; then
	error 2 "Unable to execute -- SUNWstm package not found."
    fi
}

#
# Purpose:
#	Load the common ESM script libraries.
# Actions:
#	First, checkEsm.  Then, set the ESM_LIB variable.  Finally, source
#	the common script libraries.
#
loadEsmLibs()
{
    checkEsm

    ESM_LIB=$ESM_BASE/lib
    # source the libs
    . $ESM_LIB/esm_globals.ksh
    . $ESM_LIB/esm_lib.ksh
}


#
# Purpose:
#	Find information about a host given either its IP or its hostname.
# Actions:
#	Echo an /etc/hosts style entry, such as: "IP hostname".
#
find_host()
{
    host=$1

    # First, use the NIS data to resolve the input host...
    INFO=$(ypcat hosts |grep "\<$host\>" |sort -u)

    if [ -z "$INFO" ]; then
	# Next, the host is not in the NIS domain, so use ping to resolve it...
	PING_OUT=$(/usr/sbin/ping -s $host 1 1 2>/dev/null |head -2 |tail -1)
	if [ -n "$PING_OUT" ]; then
	    FULLNAME=$(echo $PING_OUT |awk - '{print $4}')
	    HOSTNAME=$(echo $FULLNAME |cut -f1 -d'.')
	    IPSTUFF=$(echo $PING_OUT  |awk - '{print $5}')
	    IPADDR=$(echo $IPSTUFF    |cut -f2 -d'(' |cut -f1 -d')')
	
	    #DEBUG# echo HOSTNAME=$HOSTNAME
	    #DEBUG# echo IPADDR=$IPADDR
	    INFO="$IPADDR $HOSTNAME"
	fi
	# else, the host is unknown
    fi

    echo $INFO
}

# 
# Purpose:
#	Convert the specified host, either IP or hostname, into a hostname.
# Actions:
#	Echo the result.
# 
to_hostname()
{
    host=$1

    CIMOM_HOSTNAME=$(find_host $host |awk - '{print $2}')
    echo $CIMOM_HOSTNAME
}

# 
# Purpose:
#	Convert the specified host, either IP or hostname, into an IP.
# Actions:
#	Echo the result.
#
to_ipaddr()
{
    host=$1

    CIMOM_IPADDR=$(find_host $host |awk - '{print $1}')
    echo $CIMOM_IPADDR
}

