package Health;
#<copyright>
# ----------------------------------------------------------
# Sun Proprietary/Confidential Code
# Copyright 2001, Sun Microsystems, Inc. All rights reserved.
# ----------------------------------------------------------
#</copyright>

#  $Name:  $ 
#  $Id: Health.pm,v 1.89 2004/08/16 18:46:08 ccadieux Exp $
#  alarm          => mapChangeEvent
#  alarmEvent     => changeEvent
#  DiscoveryEvent => AssetEvent
#
# SEVERITY ORDER:
#    LogEvent              -0.2    # lowest severity
#    LinkEvent             -0.1
#    Others                 0.0    # include ValueChangeEvents .
#    StateEvent            +0.1
#    RemoveComponent       +0.2    # highest severity

# MANAGEMENT LEVELS:
#  C = Component (T3 in rack)
#  D = Device    (T3 stand alone)
#  DS= Rack from SP 
#  D = Rack from the outside

#  Customer View (D + DS)
#  Service Email (D + C)


use Report;
use Carp;
use Transition;
use Thresholds;
use CIM::Instance;
use Grid;
use strict;
use Message;
use State;
use TO;
use Events;
use FCRules;

sub new {
  my($class) = @_;

  my($x) = {};

  bless($x, "Health");
  return $x;
}
use vars qw ($TO $NOTOPO %DONE);

sub reset_serial {
  %DONE = ();
}

# SERIALIZE an instrumentation report into a string.

sub serial {
  my($hm, $rep, $orep) = @_;
  return "" if ($rep eq $orep);
  my $k = $rep->{"id.device_key"};
  return "" if ($k && $DONE{$k});
  $DONE{$k} = 1;

  my $out = "#INSTRUMENTATION_REPORT\n";

  foreach my $el (sort keys %$rep) {
     next if ($el =~ /fruDiskPerf/ || $el eq "FC_COUNTERS");
     if (!$orep || !exists $orep->{$el}) {
       $out .= " A\t$el\t$rep->{$el}\n";
     } elsif ($orep->{$el} ne $rep->{$el}) {
       $out .= " U\t$el\t$rep->{$el}\n";
     }
  }
  if ($orep && $rep) {
    foreach my $el (keys %$orep) {
       next if ($el =~ /fruDiskPerf/ || $el eq "FC_COUNTERS");
       if ( !exists $rep->{$el}) {
          $out .= " D\t$el\t$orep->{$el}\n";
       }
    }
  }
  return $out;
}

# RETURN "YES" IF IT'S TIME FOR A NEW AUDIT
sub auditTime {
  my($hm, $key) = @_;
  my $audit;
  my $renv = System->get_renv();

  my $freq  = $renv->{audit_freq} || 7;
  my $audit = System->get_audit() ? "YES" : Timer->isXdays("Audit$key", $freq);
  return $audit eq "YES";
}


# $hm->discoveryEvent($rep, 'NWS::T3', "YES");

sub discoveryEvent {
  my($hm, $report, $CIM, $audit, $arg) = @_;

  my $rep   = $report->value();
  my $cat   = $rep->{'id.device_type'};
  my $noReport = $arg->{noReport};
  my $etype = $audit ? "$cat.AuditEvent" : "$cat.DiscoveryEvent";
  my $title = $audit ? "Auditing a":"Discovered a new";
  my $dt    = "A" if (!$audit);
  my $mgmtLevel =  $report->value('id.mgmtLevel') || 'D';

  Grid->setCode($etype);

  my $id = $report->id('display');
  my $key= $report->deviceName();
  my $ip = $rep->{'id.device_ipno'};
  my $serial =  $hm->serial($rep) if (!$noReport);

  my $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => $etype  ],
                  [ Target      => "$cat:$key"    ],
                  [ TargetName  => $id     ],
                  [ SourceIP    => $ip     ],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ EventId     => PDM->getEventSequence()  ],
                  [ Description => "$title $cat called $id"],
                  [ Data        => $serial ],
                         ]);

  my $sd = Events->sourceDetector({ event => $ev , host => 1, rep => $rep});
     
  my $CIM_module = $CIM;
  $CIM_module =~ s/\:\:/\//g;
  require "$CIM_module.pm";
  my $p = $CIM->newSystem($rep);

  my $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $p->[0] ],
                  [ DiscoveryType => $dt],
                    ]);
  my $monitors = CIM::Instance->new('NWS_AgentMonitors', [
                  [ Agent       => $sd->[0]  ],
                  [ Element     => $p->[0] ],
                    ]);

  my $ed = Message->new( {  id     => $report->id(),
                         instances => [$ev, @$sd, @$p, $monitors, $pertains ],
                         severity  => Message::SEVERITY_NORMAL });

  PDM->saveMessage($ed);
  return $ed;
}


sub statisticEvent {
  my($hm, $report) = @_;

}
  

sub locationChangeEvent {
  my($hm, $report, $rep, $orep, $keyval) = @_;

  my($ev, $key, $sd, $pertains, $ed);
  my $renv = System->get_renv();
  return if ($renv->{solution} ne "N");
  my($cat) = $report->category();
  my($name) = $report->id('display');
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my($old) = "Contract:"   . $orep->get("location.contractNo")  . ", " . 
             "HostId:"     . $orep->get('location.hostId')      . ", " .
             "CustomerNo:" . $orep->get("location.customerNo")  . ", " .
             "Site:" .       $orep->get("location.siteName")    . ", " . 
                             $orep->get("location.siteAddress") . ", " .
                             $orep->get("location.siteCity")    . ", " .
                             $orep->get("location.siteState")   . ", " .
                             $orep->get("location.siteZip")     . ", " .
                             $orep->get("location.siteContact") . "," .
                             $orep->get("location.siteEmail") ;

  my($new) = "Contract:"   . $rep->get("location.contractNo")   . ", " . 
             "HostId:"     . $rep->get('location.hostId')       . ", " .
             "CustomerNo:" . $rep->get("location.customerNo")   . ", " . 
             "Site:" .       $rep->get("location.siteName")    . ", " . 
                             $rep->get("location.siteAddress") . ", " .
                             $rep->get("location.siteCity")    . ", " .
                             $rep->get("location.siteState")   . ", " .
                             $rep->get("location.siteZip")     . ", " .
                             $rep->get("location.siteContact") . "," .
                             $rep->get("location.siteEmail") ;

  # MAKE SURE THE SITE INFO WAS ENTERED IN THE FIRST PLACE.
  my $site  = $rep->get("location.siteName");
  my $osite = $orep->get("location.siteName");
  return if (!$site || !$osite);

  if ($old ne $new) {
    #my $desc = Events->info('LocationChange', $cat, $name)->serialize();
    my $desc = "Location of $cat $name was changed:";
    Grid->setCode("$cat.LocationChangeEvent");
    $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => lc($cat) . ".LocationChange"  ],
                  [ EventId     => PDM->getEventSequence  ],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ Target      => lc($cat) . ":" . $keyval ],
                  [ TargetName  => $report->id('display')],
                  [ Description => $desc ], 
                  [ Component   => "location"],
                  [ Data        => "Location Changed from:\n$old\nto:\n$new" ],
                         ]);
     my($key) = CIM::Key->new( ['NWS_System',
                   'Name'       => $keyval,
                    CreationClassName => 'NWS_System']);

     $sd = Events->sourceDetector({ event => $ev });

     $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

     my($loc) = NWS->location($rep);

     my($locp) = CIM::Instance->new('NWS_AgentLocation', [
                  [ Agent       => $sd->[0]  ],
                  [ Element     => $loc ],
                    ]);


     $ed = Message->new( { id        => $report->id,
                            instances => [$ev, @$sd, $pertains, $loc, $locp ],
                          });
     PDM->saveMessage($ed);
  }
     
}

# $hm->alarm($report, $orep, "BROCADE", "sensor.temperature.1", "status",
#                             $id, $id, $wwn, "sensor.temperature.status");
#      sensor.temperature.1        : used as a key in the state DB
#      sensor.temperature.1.status : used to read the agent-report
#      sensor.temperature.status   : used as a key in the Avail. Map file.

#  id = display id, use by saveState only
#  wwn= cim key
#  shortid = name
#  map_id = used to assign on topology
#  arg->{info} : extra info
#  $arg->{use_map_id}   : use the map_id for the events.
#  $arg->{severity}
#  $arg->{actionable} = Y/N
#  $arg->{state}  : use $arg->{state} for State


sub mapChangeEvent {
  my($class, @args) = @_;
  return $class->alarm(@args);
}

sub alarm {
  my($class, $report, $orep, $cat, $comp, $el, $shortid,
     $display_id, $cimkey, $map_id, $arg) = @_;

  my $rep          = $report->content();

  my $label = "$comp.$el";
  my $s2 = exists $arg->{old_status} ? $arg->{old_status} :
                   ($orep->{$label} || "[Undefined]");
  my $s1 = exists $arg->{new_status} ? $arg->{new_status} :
                   ($rep->{$label}  || "[Undefined]");

  my $extra    = "($arg->{info})" if ($arg->{info});
  my $el_extra = "($arg->{el_info})" if ($arg->{el_info});

  my $desc_display = $arg->{comp_name} || "$comp.$el";
  my $state_comp;
  if ($arg->{state}) {
    $state_comp = $arg->{state};
  } else {
    $state_comp   = ($arg->{use_map_id})? $map_id : $comp;
  }

  my($map) = PDM->getDeviceStateMap(lc($cat) . ".availability");

  $map_id = $el if (!$map_id);
  my($ostatus, $status, $map_sev, $map_action) = $map->transition("$map_id.$s2", "$map_id.$s1", $orep, $rep);
  if ($map_sev == -9) {
    Debug->print2("No event generated: $map_id.$s2 => $map_id.$s1");
    return;
  }

  my ($severity, $sign);

  if ($status eq "X") {
    $sign = "X";
    $severity = 2;
  } elsif ($status == 0) {
    $sign = "M";
    $severity = 2;
  } else {
    $sign = "P";
    $severity = 0;
  }
  $severity = $map_sev if ($map_sev >= 0);

  my $al = 0;
  # NEW TEST
  if ($s1 ne $s2 || ($rep == $orep && $status != 1)) {
     my ($st_label, $name_label) = &fix_label_name($desc_display);
        
     my $desc = $s1 ne $s2 ? 
        "The '$st_label' of '$name_label'$el_extra on $shortid is '$s1' (previous '$st_label' was '$s2') $extra":
        "The '$st_label' of '$name_label'$el_extra on $shortid is '$s1' $extra";
        
     Grid->setSign($sign);
     my $ev = $class->alarmEvent($comp, "", $report, $cimkey, $desc, $severity, 
                  "$sign.$comp.$el" , 
               { stateComp  => $state_comp, 
                 stateAvail => $status, 
                   priorValue => $s2,  currentValue => $s1,
                 actionable => $map_action, 
                      nomap => 1,
               });

     #State->saveState($cat, $cimkey, $state_comp, $display_id, $severity, $desc, $status, $ev);
  }
}

sub fix_label_name {
  my($desc_display) = @_;
  my @L = split(/\./, $desc_display);
  my($st_label, $name_label);
  if ($#L > 0 && ($L[$#L] =~ /status/i ||  $L[$#L] =~ /state/i) ) {
     $st_label = $L[$#L];
     $name_label = join(".", @L[0..$#L-1]);
  } else {
     $st_label = "State"; $name_label = $desc_display;
  }
  return ($st_label, $name_label);
}

# parent FILTER, can be implemented on each device Class

sub FILTER {
  my($hm, $rep, $ev) = @_;
  return if (!$ev);

  if ($rep->{'info.Disclaimer'}) {        # ADD TO DESCRIPTION, SAVE SEVERITY
    my $desc = $ev->value("Description");
    my $col = substr($desc, -1);
    $desc .= " (" . $rep->{'info.Disclaimer'} . ")";
    $desc .= $col if ($col eq ":");
    $ev->setValue("Description", $desc);

  } elsif ($rep->{'info.noActionable'}) { # ADD TO DESCRIPTION, LOWER SEVERITY
    $ev->setValue("Actionable", "FALSE");
    my $sev = $ev->value("Severity");
    $ev->setValue("Severity", 1) if ($sev >= 2);
    my $desc = $ev->value("Description");
    my $col = substr($desc, -1);
    $desc .= " (" . $rep->{'info.noActionable'} . ")";
    $desc .= $col if ($col eq ":");
    $ev->setValue("Description", $desc);
  }
}

sub diagEvent {
  my($hm, $report, $sev, $caption, $command, $target, $start, $end, $desc) = @_;

  my $rep = $report->content();
  my $cat = $report->category();
  my $code = "$cat.DiagnosticTest";
  $code .= "-" if ($sev);
  Grid->setCode("$code.$caption");

  my $action = $sev > 1 ? 1 : 0;

  my $ev = CIM::Instance->new('NWS_DiagnosticTestComplete', [
                  [ EventType   => "$cat.DiagnosticTest"  ],
                  [ EventId     => PDM->getEventSequence  ],
                  [ Severity    => $sev                   ],
                  [ Caption     => $caption               ],
                  [ FailureCode => 2                      ],
                  [ Target      => "$cat:$target"         ],
                  [ Actionable  => $action                ],
                  [ Severity    => $sev                   ],
                  [ StartDate   => $start                 ],
                  [ EndDate     => $end                   ],
                  [ Component   => "alarm"                ],
                  [ Description => $desc                  ],
                   ]);

  $hm->FILTER($rep, $ev);

  my $sd = Events->sourceDetector({ event => $ev , host => 1, rep => $rep});

  my $pertains = CIM::Instance->new('NWS_EventPertainsToSoftwareElement', [
                  [ Event         => $ev      ],
                  [ Element       => $sd->[0] ],
                  [ DiscoveryType => "D"      ],
                    ]);
  my $ed = Message->new( { id        => {deviceName => $target },
                           instances => [$ev, @$sd, $pertains ],
                           state     => ['alarm', 1],
                           severity  => $sev });
  PDM->saveMessage($ed);

  #State->saveState($cat,  $target, "alarm", "Alarm", $sev, $desc, 1, $ev);
  return $ev;
}


sub identificationEvent {
  my($class, $report, $old, $new, $wwn) = @_;

  return undef if (!$report);
  my $old = lc($old);
  my $new = lc($new);
  my $id = $report->id('display');
  if ($new && $old && (substr($old,-4) ne substr($new,-4)) ) {
    my $cat = $report->category();
    Grid->setCode("$cat.ValueChangeEvent.identification");
    $class->alarmEvent("system","", $report, $wwn, 
      "The identification of $id changed from '$old' to '$new'",
      1, "identification");
  }
}



# $arg->{actionable} = Y/N
# by default, warnings(1) are not actionable, errors(2) are.

sub valueChangeEvent {
  my($class, @args) = @_;
  return $class->alarmEvent(@args);
}

sub alarmEvent {
  my($hm, $comp, $data, $report, $keyval, $desc, $sev, $caption, $arg) = @_;

  my($ev, $key, $sd, $pertains, $ed);
  my($cat) = $report->category();
  my $captionIsState = $arg->{captionIsState};
  my($id)  = PDM->getEventSequence ;
  my($rep) = $report->content();
  my $mgmtLevel = $arg->{mgmtLevel} || $report->value('id.mgmtLevel') || 'D';
  $caption = "ValueChangeEvent" if (!$caption);

  my $av = ['X', -1,undef];
  if (!$arg->{nomap}) {
    my $map = PDM->getDeviceStateMap(lc($cat) . ".availability");
    $av  = $map->get2("changeEvent.$caption");
  }
  if ($av->[1] == -9) {
     Debug->print2("No event generated: $comp. $caption ");
     return undef;
  } elsif ($av->[1] >= 0) {
     $sev = $av->[1];
  } else {
     $sev = 1 if (!defined($sev)); # warning by default
  }

  my $actionable = 0;
  if ($av->[2]) {
     $actionable = $av->[2] eq "Y" ? 1:0;
  } elsif ($arg->{actionable}) {
     $actionable = $arg->{actionable} eq "Y" ? 1:0;
  } elsif ($sev >= 2 ) {
     $actionable = 1;
  }
  my $target = lc($cat) . ":" . $keyval;

  $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => lc($cat) . ".ValueChangeEvent"  ],
                  [ EventId     => $id          ],
                  [ Data        => $data        ],
                  [ Caption     => $caption     ],
                  [ Target      => $target      ],
                  [ TargetName  => $report->id('display') ],
                  [ Actionable  => $actionable  ],
                  [ MgmtLevel   => $mgmtLevel   ],
                  [ SourceIP    => $rep->{'id.ipno'}],
                  [ Severity    => $sev         ],
                  [ PriorValue  => $arg->{priorValue}],
                  [ CurrentValue=> $arg->{currentValue}],
                  [ Component   => $comp        ],
                  [ Description => $desc        ],
                         ]);
   $hm->FILTER($rep, $ev);

   $key = CIM::Key->new( ['NWS_System',
                   'Name'       => $keyval,
                    CreationClassName => 'NWS_System']);
   $sd = Events->sourceDetector({ event => $ev });

   $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

   $ed = Message->new( { id        => $report->id,
                         state     => [ 
                                exists($arg->{stateComp})  ? $arg->{stateComp} : 
                                                   ($captionIsState ? $caption : "alarm.$caption"), 
                                exists($arg->{stateAvail}) ? $arg->{stateAvail} : 1 ],
                          instances => [$ev, @$sd, $pertains ],
                        });
   PDM->saveMessage($ed);

   if (!$arg->{noState}) {
     my $c = $captionIsState ? $caption : "alarm.$caption";
     #State->saveState($cat,  $keyval, $c, "Alarm", $sev, $desc, 1, $ev);
   }
   return $ev;
}

# against any switch/3310

sub portStatEvents {
  my($hm , $report, $PORTS, $rep, $orep, $prefix, $args) = @_;
  my $wwn = $report->id('deviceName');
  my $id  = $report->id('display');
  my $cat = $report->category();
  my $info = $args->{info};
  $prefix = "port" if (!$prefix);

  my($x, $port, @P);
  for ($x=0; $x <= $PORTS; $x++) {
     $port = "$prefix.$x";
     if (exists $rep->{"$port.state"} || exists $rep->{"$port.mode"}) {
       my $state = $rep->get("$port.state") || $rep->get("$port.mode");
       next if (!$state);
       $state = lc($state);
       next if ($state =~ /offline/ || $state =~ /unknown/  || $state =~ /not-logged-in/);
     }
     my $rep1 = $rep->subset("$port.error");
     foreach my $val (keys %$rep1) {
         next if ($val eq "C3Discards");
         next if (!exists $orep->{"$port.error.$val"});
         my($old) = $orep->get("$port.error.$val");
         my($new) = $rep->get("$port.error.$val");
         if ($new > $old) {
           my $val0 = $val;
           my $pre0;
           my $ii = rindex($val, ".");
           if ($ii > 0) {
               $val0 = substr($val,$ii+1);
               $pre0 = substr($val,0,$ii) . " ";
           }
           my($level, $cnt, $desc, $mins) = Thresholds->test($cat, $val0, $x, $new-$old, $wwn);
           if ($level eq "W" || $level eq "E") {
             $P[$x] .= "Received $cnt '$pre0$desc' in $mins mins (value=$new)\n";
           }
         }
     }
  }
  for ($x=0; $x <= $PORTS; $x++) {
    if ($P[$x]) {
      my $data = $P[$x];
      my $did = $hm->getDiskId( $rep, $prefix, $x);
      my $info_t = "(" . $rep->{"$prefix.$did.$info"} . ")" if ($info);

      chop($data);
      if ($data =~ /\n/) {
         $hm->alarmEvent("$prefix.$x", $data, $report,  $wwn, 
            "Change in statistics on $cat $id, $prefix-$x $info_t:", 
            Message::SEVERITY_WARNING, "$prefix.$x.statistics", {captionIsState => 1, nomap => 1});
      } else { 
         $hm->alarmEvent("$prefix.$x", "", $report,  $wwn, 
            "$cat $id, $prefix-$x $info_t: $data",
            Message::SEVERITY_WARNING, "$prefix.$x.statistics", {captionIsState => 1, nomap => 1});
      }
    }
  }

}

sub getDiskId {
    my ($hm, $rep, $prefix, $x) = @_;
    my $port = "$prefix.$x";
    my $rep1 = $rep->subset("$port.error");
    my $disk;
    foreach my $val (keys %$rep1) {
          if ($val =~ /target/) {
             $disk = $rep->get("$port.error.$val");
             return ($disk);
          }
    }
    if (!$disk) {
        return undef;
    }
}


# {actionable => 1}

sub logEvent {
  my($hm, $log, $report, $keyval, $desc, $sev, $caption, $arg) = @_;

  my($ev, $key, $sd, $pertains, $ed);
  my($cat) = $report->category();
  my($id)  =  PDM->getEventSequence ;
  my($id2) =  $report->id('display');
  my $comp = $arg->{component};
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my($rep) = $report->content();
  my($name)= $rep->get('id.name') if ($rep);
  $caption = "MessageLog" if (!$caption);

  $desc =  "$cat.MessageLog on $id2:" if (!$desc);
  my $action= 0;
  $action = 1 if (($sev == Message::SEVERITY_ERROR) || $arg->{actionable});
  my $etype = lc($cat) . ".LogEvent" ;
  $etype .= ".$arg->{egrid}" if ($arg->{egrid});


  $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => $etype ],
                  [ EventId     => $id ],
                  [ Data        => $log ],
                  [ Caption     => $caption ],
                  [ Target      => lc($cat) . ":" . $keyval],
                  [ TargetName  => $report->id('display')],
                  [ Actionable  => $action],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ SourceIP    => $rep->{'id.ipno'}],
                  [ Component   => $comp ],
                  [ Severity    => $sev ],
                  [ Description => $desc ] 
                         ]);
   $hm->FILTER($rep, $ev);

   $key = CIM::Key->new( ['NWS_System',
                   'Name'       => $keyval,
                    CreationClassName => 'NWS_System']);
   $sd = Events->sourceDetector({ event => $ev });

   $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

   $sev -= 0.2 ;                 # logEvent < OtherEvents < StateEvent

   my $rid = $report->id();
   $ed = Message->new( { id         => {%$rid},
                          state     => [ $comp || "alarm.${cat}Log", 1],
                          severity  => $sev, 
                          instances => [$ev, @$sd, $pertains ],
                        });
   PDM->saveMessage($ed);

   #State->saveState($cat,  $keyval, $comp || "alarm.${cat}Log", "Alarm", $sev, $desc, 1, $ev);

   return $ev;
}


##########################
#     FC-EVENTS
##########################

sub fcEvent {
    my($hm, $rep, $orep) = @_;
    return [] if (!$orep || ($rep eq $orep) || !(exists $rep->{data}) || !(exists $orep->{data}));
    my $processed_list = [];

    my $new = $rep->{data};
    my $old = $orep->{data};
    my $FC = {info =>  { hba => $rep->{hba}, enc => $rep->{enc}} };
    my $cnt2 = 0;
    my $processed_list = [];

    foreach my $k (sort keys %$new) {
       my($hba, $wwn, $comp, $type) = split(/\|/, $k);
       next if (!exists $old->{$k}); # don't compare unless you have both old and new key.

       my @n1 = split(/\t/, $new->{$k});
       my @o1 = split(/\t/, $old->{$k});
       my $o_crc = $o1[3];
       my $n_crc = $n1[3];

       if ($n_crc > $o_crc) {
            next if ($type eq "switch" && ($#o1 > 5) && ($o1[6] == $n1[6]) && ($o1[7] == $n1[7]));
            $FC->{error}[$cnt2] = {key => $k, type => "CRC", count => $n_crc - $o_crc}; $cnt2++;
            Debug->print2("CRC went up by " . ($n_crc - $o_crc) . " in $k");
       }
       my $o_itw = $o1[5];
       my $n_itw = $n1[5];
       if ($n_itw > $o_itw) {
            Debug->print2("ITW went up by " . ($n_itw - $o_itw) . " in $k");
            $FC->{error}[$cnt2] = {key => $k, type => "ITW", count => $n_itw - $o_itw}; $cnt2++;
       }
       my $o_sig = $o1[1];
       my $n_sig = $n1[1];
       if ($n_sig > $o_sig) {
           Debug->print2("SIGNAL-LOSS went up by " . ($n_sig - $o_sig) . " in $k");
            $FC->{error}[$cnt2] = {key => $k, type => "SIG", count =>  $n_sig - $o_sig }; $cnt2++;
       }
    }
    if ($cnt2) {
      if ( Util->findMaster() ) {# slave
        $hm->fcRemoteEvent($FC) ;

      } else {                   # master
        my $broke_list = $hm->processFC($FC);
        foreach my $dev (@$broke_list) {
           $dev->{problems} = 1;
           push(@$processed_list, $dev);
        }
      }
    }
    return $processed_list;
}

sub fcRemoteEvent {
   my($hm, $FC) = @_;

   my($id)    =  PDM->getEventSequence ;
   my $renv = System->get_renv();
   my $tt1 = "san.fcEvent";
   require Data::Dumper;
   $Data::Dumper::Indent = 0;
   my $out = Data::Dumper::Dumper($FC);

   my $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => $tt1 ],
                  [ EventId     => $id  ],
                  [ Actionable  => 1],
                  [ Data        => $out ],
                         ]);

   my $ed = Message->new({ id  => {}, instances => [ $ev ], });

   PDM->saveMessage($ed);
}

sub processFC {
  my($hm, $FC)  = @_;

  return [] if ($NOTOPO);

  $TO = TO->readExistingTopo("MERGE-MASTER") || TO->readExistingTopo();
  if (!$TO) {
    $NOTOPO = 1;
    return [];
  }
  my $Config  = System->get_Config();
  my $rules   = FCRules->load_rules();

  my $report = Report->new({ deviceName  => "san",
                             name        => "san",
                             category    => "san"}  , {});

  my $errors = $FC->{error};
  my($x);
  my $broke_list = [];

  for ($x=0; $x <= $#$errors; $x++) {
     my $v = $errors->[$x];
     foreach my $rule (@$rules) {
        my $problems = PDM->get_last_event();
        my($error, $event) = $rule->run($TO, $hm, $report, $v->{count},
                              $v->{key}, $FC->{info}, [], [], $v->{type});
        Debug->print2($event) if ($event);
        if ($error eq "FC") { # event was generated
           my(@k) = split(/\|/, $v->{key});
           my $dev = $Config->deviceByKey($k[1]);
           if ($dev) {
              my $broke = Agent->new_events($problems, $dev, 1);
              push(@$broke_list, $dev);
           }
        }
     }
   }
   return $broke_list;
}
  

# type1:key1 : origin
# type2:key2 : destination (where the CRC increased)

sub linkEvent {
  my($hm, $report, $type1, $key1, $type2, $key2, $desc, $reads, $writes, $hba,
      $fc_type, $dev_name) = @_;

  my($ev, $key, $sd, $pertains, $ed, $ehba);
  my($id)    =  PDM->getEventSequence ;
  my $renv = System->get_renv();
  my $mgmtLevel = $renv->{solution} =~ /^se/? "C" : "D"; 

  my ($t1, $k1, $p1) = split(/\:/, $key1);
  my ($t2, $k2, $p2) = split(/\:/, $key2);
  my $keyval = $k1;

  my($e_hba, $k1, $host_hba, $f1, $f2, $f3, $hostid, $link1, $link2);
  my(@hba_list, $ix, $node1, $loc);

  Grid->setCode("$t2.LinkEvent_$fc_type");
  my $sev = Message::SEVERITY_ERROR;

  $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => "$t2.LinkEvent_$fc_type" ],
                  [ EventId     => $id  ],
                  [ Target      => $key2], 
                  [ TargetName  => $dev_name || $key2],
                  [ Caption     => "port.$p2.fc_link" ],
                  [ Actionable  => 1],
                  [ SourceIP    => System->get_ipno() ],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ Description => $desc ],
                  [ Data        => "target2=$key1" ],
                  [ Component   => "port.$p2" ],
                  [ Severity    => $sev ],
                         ]);

   $sd = Events->sourceDetector({ event => $ev });

   my $system = CIM::Key->new( ['NWS_System',
                   'Name'             => $keyval,
                    CreationClassName => 'NWS_System']);

   $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $system ],
                    ]);

   $ed = Message->new({ id        => $report->id,
                        state     => ["port.$p2.fc_link", 0, "$type2:$k2"],
                        severity  => $sev - 0.1,
                        instances =>
                        [$ev, @$sd, @hba_list, $pertains ],
                      });

   PDM->saveMessage($ed);

   #State->saveState($type2, $k2, "port.$p2.fc_link", "name", Message::SEVERITY_WARNING, $desc, 0, $ev);

}

sub thresholdEvent {
  my($class, $report, $orep, $cat, $comp, $el, $shortid, $display_id, $cimkey, $arg) = @_;

  my $rep  = $report->content();
  return if (!exists $rep->{"$comp.$el"} || !exists $orep->{"$comp.$el"});

  my $val  = $rep->{"$comp.$el"};
  my $oval = $orep->{"$comp.$el"};
  my $display_comp = $arg->{display_comp} || $comp;

  my($level, $cnt, $desc, $mins) = Thresholds->test($cat, $el, $comp, $val-$oval, $cimkey);
  if ($level) {
     my $sev = $level eq "E" ? 2 : 1;
     $class->alarmEvent($comp, "", $report, $cimkey, 
            "$el of $display_comp in $cat $display_id increased by $cnt in $mins mins.",
             $sev, "$comp.$el", {captionIsState => 1, nomap => 1});

  }
  

}
  
sub tempEvent {
  my($class, $report, $orep, $cat, $comp, $el, $shortid,
     $display_id, $cimkey, $high, $low, $arg) = @_;

  my($rep) = $report->content();
  my $temp = $rep->get("$comp.$el");
  return if ($temp !~ /\d+/ || $temp > 500);
  my $comp_info = "($arg->{comp_info})" if ($arg->{comp_info});

  my($tran, $reps) = Transition->getTransition2({      
                             key => $cimkey,
                            code => "temp_$comp",
                           in_value => ($temp > $high),
                           out_value => ($temp < $low),
                                 });
  if ($tran =~ /IN/) { #lost  or lost-repeat
     Grid->setSign("M");
     $class->alarmEvent($comp, "", $report, $cimkey, 
            "$comp.$el$comp_info in $cat $display_id: Temperature($temp) over $high",
             2, "$comp.Temp", {captionIsState => 1, nomap => 1});
 
  }
  if ($tran eq "OUT") { # can connect
     Grid->setSign("P");
     $class->alarmEvent($comp, "", $report, $cimkey, 
            "$comp.$el$comp_info in $cat $display_id: Temperature($temp) back under $low",
             0, "$comp.Temp", {captionIsState => 1, nomap => 1} );
  }
}
  
# 2=outoband, 1=inband 3=both

sub connectionEvent {
  my($hm, $keyval, $report, $args) = @_;
  my($key, $ev, $sd, $pertains, $ed, $id, $text, $rc);
  my($cat) = $report->category();
  my($rep) = $report->content();
  my $name = $rep->get('id.name') if ($rep);

  if ($args->{method} == 3) {  # BOTH
    $text = "OutOfBand";

    my($tran, $reps) = Transition->getTransition({      key => $report->deviceName(),
                                         code => 'status',
                                        value => $report->status(),
                             transition_value => Report::STATUS_CANNOT_CONNECT,
                                       repeat => '24h',
                                 });
    $rc = $hm->connection("e", $keyval, $tran, $report, $text, $reps, $args);

    if (defined($rep->{'info.IB_status'})) {  # T3 only so far.
       my $host = $rep->{'info.IB_host'}; 

       $text = "InBand(" . Util->shortHostname($host) . ")";

       my($tran, $reps) = Transition->getTransition( {  key => $report->deviceName(),
                                         code => 'statusIB',
                                        value => $rep->{'info.IB_status'},
                             transition_value => Report::STATUS_CANNOT_CONNECT,
                                       repeat => '24h',
                                       });
       $hm->connection("ib", $keyval, $tran, $report, $text, $reps, $args);
    }
    return $rc;

  } else {
    my $ho = System->hostname();
    my $ei;
    if ($args->{method} == 1) {
       $text = "InBand($ho)";
       $ei = "ib";
    } else {
       $text = "OutOfBand($ho)";
       $ei = "e";
    }

    my($tran, $reps) = Transition->getTransition({      key => $report->deviceName(),
                                         code => 'status',
                                        value => $report->status(),
                             transition_value => Report::STATUS_CANNOT_CONNECT,
                                       repeat => '24h',
                                 });
    return $hm->connection($ei, $keyval, $tran, $report, $text, $reps, $args);
  }

}

sub connection {
  my($hm, $comp, $keyval, $tran, $report, $text, $reps, $args) = @_;
  my($key, $id, $desc);
  my($ev, $sd, $pertains, $ed);
  my($cat) = $report->category();
  my($rep) = $report->content();
  my($name)= $rep->get('id.name') if ($rep);
  my($connect_errs) = $args->{connect_errs} || $rep->get('id.connect_errs');

  my $mgmtLevel  = $report->value('id.mgmtLevel') || 'D';
  my $egrid_info = $comp eq "ib" ? "ib": "oob";

  my($id2)  = $report->id('display');

  $key = CIM::Key->new( ['NWS_System',
                   'Name'             => $keyval,
                    CreationClassName => 'NWS_System']);

  return 1 if ($tran eq "LOST");
  my $comp1 = $comp;
  $comp1 = "$comp." . $rep->{'id.caption'} if ($rep && $rep->{'id.caption'});

  if ($tran =~ /IN/) { #lost  or lost-repeat
     return 1 if (!$keyval);  # never had good connection
     return 1 if ($reps > 6);
     $id =  PDM->getEventSequence;
     my $repeat = "(lost for " . ($reps*8) . " hours)" if ($reps > 0);
     $desc = "Lost communication ($text) with $cat $id2 $repeat: " .
             $connect_errs;
     Grid->setCode(lc($cat) . ".CommunicationLostEvent.$egrid_info");

     $ev = CIM::Instance->new('NWS_CommunicationLostEvent', [
                  [ EventType   => lc($cat) . ".CommunicationLostEvent"  ],
                  [ EventId     => $id  ],
                  [ Caption     => $comp1 ],
                  [ Component   => $comp  ],
                  [ Description => $desc  ],
                  [ Target      => lc($cat) . ":" . $keyval],
                  [ TargetName  => $report->id('display')],
                  [ Method      => $args->{method}  ],
                  [ MgmtLevel   => $mgmtLevel       ],
                  [ Actionable  => 1                ],
                  [ SourceIP    => $rep->{'id.ipno'}],
                  [ Severity    => Message::SEVERITY_DOWN],
                  [ Timeout     => 20],
                         ]);
     $hm->FILTER($rep, $ev);
     $sd = Events->sourceDetector({ event => $ev });

     $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

     #State->saveState($cat,  $keyval, $comp, $name, Message::SEVERITY_DOWN, "Lost Communication ($text) with $id2", 0, $ev);

     $ed = Message->new( { id        => $report->id,
                          state       => [$comp, 0],
                            instances => [$ev, @$sd, $pertains ],
                          });
     PDM->saveMessage($ed);

     return 1;
  }
  if ($tran eq "OUT") { # can connect
     $desc = "Regained communication($text) with $cat $id2";
     $id =  PDM->getEventSequence;
     Grid->setCode(lc($cat) . ".CommunicationEstablishedEvent.$egrid_info");

     $ev = CIM::Instance->new('NWS_CommunicationEstablishedEvent', [
                  [ EventType   => lc($cat) . ".CommunicationEstablishedEvent"  ],
                  [ EventId     => $id ],
                  [ Description => $desc ],
                  [ Target      => lc($cat) . ":" . $keyval],
                  [ TargetName  => $report->id('display')],
                  [ Caption     => $comp1 ],
                  [ Component   => $comp ],
                  [ SourceIP     => $rep->{'id.ipno'}],
                  [ Severity    => Message::SEVERITY_GREEN],
                  [ Method      => 1],
                         ]);
     $sd = Events->sourceDetector({ event => $ev });


     $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

     #State->saveState($cat,  $keyval, $comp, $name, Message::SEVERITY_GREEN, "Regained Communication ($text)", 1, $ev);

     $ed = Message->new( { id        => $report->id,
                            state    => [$comp, 1],
                            instances => [$ev, @$sd, $pertains ],
                            });

     PDM->saveMessage($ed);

  }
  return 0;  # keep going
}

#
# returns availability-state of the device (old,new, description) :
#    returns (1/0, 1/0, "Old=..., New=...");

sub status {
   croak("Health::status is abstract, must implement in device Health");
}

#
# arg = {
#   logical       => 1,        # use Logical component CIM class
#   descIsValue   => 1,        # skip 'The 'statusLabel'
#                         'display_comp' in id ...
#   info          => "extra info"
#   statusFunc    => "status2" # different name for the status() function
#   statusLabel   => "label2"  # change the 'State' word in the description
#                         The 'statusLabel' of ...
#   display_comp  => "comp2"   # change the component name
#                         The 'statusLabel' of 'display_comp' in id
#   comp_info     => "extra"   # extra component info.
#                         The 'statusLabel' of 'display_comp'(comp_info) in id
#  };   
#

sub stateChangeEvent {
  my($hm, $component, $status_func, $report,  $rep, $orep, $cim1, $nserial, $keyval, $arg) = @_;
  
  
  my @L = $hm->$status_func($rep, $orep, $component, $arg->{old_comp});

  $arg->{status_results} = \@L;
  return $hm->stateEvent($component, $report,  $rep, $orep, $cim1, $nserial, $keyval, $arg);
}

#
# arg = {logical => 1, descIsValue=> 1, info => "extra info"}
#
sub stateEvent {
  my($hm, $component, $report,  $rep, $orep, $cim1, $nserial, $keyval, $arg) = @_;

  my($key, $ev, $sd, $pertains, $ed , $id);
  my $func = $arg->{statusFunc} || "status";
  my $statusLabel = $arg->{statusLabel} || "State";

  my($ovalue, $nvalue, $old, $new, $map_sev, $map_act) = 
                $hm->$func($rep, $orep, $component, $arg->{old_comp});
  if ($map_sev == -9) {
    Debug->print2("No event generated: $component $ovalue => $nvalue");
    return;
  }

  my $name   = $rep->get('id.name');
  my $ix     = index($component, ".");
  my $extra  = "($arg->{info})" if ($arg->{info});
  my $no_action = $arg->{noAction};
  my $id2    = $report->id('display');
  my $cat    = $report->category();
  my $comp_info = $arg->{comp_info};
  my $catcap = uc($cat);
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';
  my $display_comp = $arg->{display_comp} || $component;
  $nserial = $report->deviceName() if (!$nserial);

  my($comp0) = $component;
  if ($ovalue ne $nvalue || ($rep == $orep && !$new) ) { 

         if ($rep == $orep) {
            $old = 1;
            #$ovalue = "unknown";
         }
         $id =  PDM->getEventSequence;
         my $xx0 = (index($component, $nserial) < 0) ? "(id=$nserial) ":"";
         my $xx  = "($comp_info)" if ($comp_info);
         my($desc);
         if ($arg->{descIsValue}) {
            $desc = "$display_comp${xx} in $catcap $id2 is '$nvalue': $extra";
         } else {
            if ($rep eq $orep) {
              $desc = "The '$statusLabel' of $display_comp${xx} in $catcap $id2 is '$nvalue' $xx0$extra";
            } else {
              $desc = "The '$statusLabel' of $display_comp${xx} in $catcap $id2 is '$nvalue' (previous '$statusLabel' was '$ovalue')$xx0$extra";
            }
         }

         if (0 && !$nserial) {
             $arg->{noState} = 1;
             my $sev0 = $map_sev >= 0 ? $map_sev : 2;
             $arg->{nomap} = 1;
             my @clist = split(/\./, Grid->getCode(), 3);
             Grid->setCode("$clist[0].ValueChangeEvent.$clist[2]");
             Grid->setSign($new != 1 ? "M": "P");
             $arg->{stateComp}  = $component; 
             $arg->{stateAvail} = $new;
             my $ev0 = $hm->alarmEvent($component, "", $report, $keyval, $desc, 
                                ($new != 1) ? $sev0 : 0, "$cat.$component", $arg);
             return $ev0;
         }
         $old = 0 if ($old eq "X");
         my $yellow;
         my $sign = "X";
         if ($new eq "X") {
            $new = $old;
            $yellow =1;
         }
         my ($severity, $action);
         if (!$new) { #  bad comp
           $severity = Message::SEVERITY_ERROR;
           $action = 1;
           $sign = "M";
         } elsif ($yellow) {
           $severity = Message::SEVERITY_ERROR;
           $action = 1;
         } else {
           $severity = Message::SEVERITY_GREEN;
           $action = 0;
           $sign = "P";
         }
         $severity = $map_sev if ($map_sev >= 0);
         if ($map_act) {
           $action   = $map_act eq "Y"? 1:0;
         }
         Grid->setSign($sign);
         if ($action && $no_action) {
            $severity = 1; $action = 0;
         }

         $ev = CIM::Instance->new('NWS_StateChangeEvent', [
             [ EventType    => lc($cat) . ".StateChangeEvent"   ],
             [ EventId      => $id ],
             [ PriorState   => $old],
             [ CurrentState => $new],
             [ MgmtLevel    => $mgmtLevel ],
             [ PriorValue   => $ovalue],
             [ CurrentValue => $nvalue],
             [ Actionable   => $action  ],
             [ Caption      => "$sign.$comp0"],
             [ Severity    => $severity ],
             [ Description => $desc ],
             [ Target      => lc($cat) . ":" . $keyval],
             [ TargetName  => $report->id('display')],
             [ Component   => $component],
             [ SourceIP    => $rep->{'id.ipno'}],
             [ Data        => $hm->serial($rep, $orep)],
                    ]);
         $hm->FILTER($rep, $ev);

         $sd = Events->sourceDetector({ event => $ev });

         $key = CIM::Key->new( [$cim1,
                  Tag               => $nserial,
                  CreationClassName => $cim1]);

         $pertains = CIM::Instance->new(
                $arg->{logical} ? 'NWS_EventPertainsToLogicalDevice': 
                                  'NWS_EventPertainsToPhysicalElement', [
             [ Event       => $ev  ],
             [ Element     => $key ],
               ]);
         my $sev2 = $severity;
         $sev2 += 0.1; # logEvent < OtherEvents < StateEvent
         $ed = Message->new( { id        => $report->id,
                               severity  => $sev2, 
                               state     => [$component, $new],
                               instances => [$ev, @$sd, $pertains ],
                       });

         PDM->saveMessage($ed);
         #State->saveState($cat,  $keyval, $component, $name, $sev2, $desc, $new, $ev);
         return $ev;
   }
   return undef;

}

# USED FOR Sun SOLUTIONS ONLY
# old/new are severities, ovalue and nvalue are descriptions
# new=1 (warning) new=2 (error) etc..

sub sevStateEvent {
  my($hm, $component, $report,  $rep, $orep, $cim1, $nserial, $keyval, $arg) = @_;
  my($key, $ev, $sd, $pertains, $ed , $id);

  my($ovalue, $nvalue, $old, $new, $oavail, $navail,$ndesc) = $hm->status($rep, $orep, $component);

  my $name   = $rep->get('id.name');
  my $data   = "";
  my $ix     = index($component, ".");
  my $id2    = $report->id('display');
  my $error  = 0;
  my $cat    = $report->category();
  my $catcap = uc($cat);
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';
  my $sign   = $navail eq "X" ? "X" : ($navail == 1 ? "P": "M");
  my($comp0) = $component;

  if ($ovalue ne $nvalue || ($rep == $orep && $new) ) { # availability change or new report  and unavailable
         $id =  PDM->getEventSequence;
         my $nvalue2 = $ndesc;
         $nvalue2 = substr($nvalue2,19) if ($nvalue2 =~ /^\d+\-\d\d/);
         my $desc = "$arg->{label} $catcap-$id2: $nvalue2";
         my $org  = State->eventHash($rep->{"$component.status-details"});

         my($g_info, $g_cause, $g_action, $g_code) = 
                      Grid->getInfoString($org->{EventType}, $org->{Caption});

         my $et = lc($cat) . ".StateChangeEvent";
         my $ac = $org->{Actionable};
         my $ca = "$sign.$comp0";
         my $co = $component;
         my $se = $new;
         my $ta = lc($cat) . ":" . $keyval;
         my $tan= $report->id('display');
         my $si = $rep->{'id.ipno'};
         if ($org->{GridCode}) {
           Grid->setCode($org->{GridCode}); 
         } else {
           Grid->setCode("$et.slot");
           Grid->setSign($se == 0 ? "P": "M");
         }

         $ev = CIM::Instance->new('NWS_StateChangeEvent', [
             [ EventType    => $et     ],
             [ EventId      => $id     ],
             [ PriorState   => $oavail ],
             [ CurrentState => $navail ],
             [ MgmtLevel    => $mgmtLevel ],
             [ PriorValue   => $ovalue ],
             [ CurrentValue => $nvalue ],
             [ Actionable   => $ac     ],
             [ Caption      => $ca     ],
             [ Severity     => $se     ],
             [ Description  => $desc   ],
             [ Target       => $ta     ], 
             [ TargetName  =>  $tan    ],
             [ Data         => $data   ],
             [ Component    => $co     ],
             [ SourceIP     => $si     ],
             [ OriginalEvent => $rep->{"$component.status-details"}. 
                                  "|grid_code=$g_code" ],
                    ]);
         $hm->FILTER($rep, $ev);

         $sd = Events->sourceDetector({ event => $ev });

         if ($arg->{logical}) {              # logical (like NWS_Slot,
           $key = CIM::Key->new( [$cim1,    
                  CreationClassName => $cim1, 
                  SystemName        => $rep->{'id.wwn'} . "." . $component,
                  DeviceID          => $nserial 
                  ]);
         } else {
           $key = CIM::Key->new( [$cim1,    # physical (like NWS_PhysicalPackage)
                  Tag               => $nserial,
                  CreationClassName => $cim1
                  ]);
         }

         $pertains = CIM::Instance->new(
                $arg->{logical} ? 'NWS_EventPertainsToLogicalDevice': 
                                  'NWS_EventPertainsToPhysicalElement', [
             [ Event       => $ev  ],
             [ Element     => $key ],
               ]);

         $ed = Message->new( { id => $report->id,
                        severity  => $se + 0.1,
                        state     => [$component, $navail, undef, 1], 
                        instances => [$ev, @$sd, $pertains ], });

         PDM->saveMessage($ed);
         my $renv = System->get_renv();
         return $ev;

   }
   return undef;

}


#ComponentRemoveEvent.volume             = 0 = I


sub removeCompEvent {
  my($hm, $component, $pf, $report,  $orep, $cim1, $oserial, $keyval, $arg) = @_;
  
  my($key, $ev, $sd, $pertains, $ed, $contain);
  my $cat = $report->category();
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';
  my $extra_desc = $arg->{desc};
  my $logical = $arg->{logicalDevice};

  my $comp0 = $component;
  my @comps  = split(/\./, $component);

  my $id    = PDM->getEventSequence;
  my $rep   = $report->content();
  my $name  = $rep->get('id.name') if ($rep);
  my $comp1 = $arg->{display_comp} || $component;
  $comp1    = "" if (index($oserial, $component) >= 0);
  my $id2   = $report->id('display');
  my $comp_info = "($arg->{comp_info})" if ($arg->{comp_info});
  my $desc  = "$comp1$comp_info was removed from $cat $id2 (sn=$oserial)$extra_desc";

  my $map = PDM->getDeviceStateMap(lc($cat) . ".availability");
  my $av  = $map->get2("ComponentRemoveEvent.$comps[0]");
  my $sev;
  if ($av->[1] == -9) {
     Debug->print2("No event generated: ComponentRemove.$component");
     return undef;
  } elsif ($av->[1] >= 0) {
     $sev = $av->[1];
  } else {
     $sev = 2;
  }
  my $act = 1 if ($sev >= 2);

  $ev = CIM::Instance->new('NWS_Event', [
             [ EventType    => lc($cat) . ".ComponentRemoveEvent"  ],
             [ Caption      => "remove.$comp0"],
             [ Description  => $desc      ],
             [ MgmtLevel    => $mgmtLevel ],
             [ PriorValue   => $oserial   ],
             [ Actionable   => $act       ],
             [ Severity     => $sev       ],
             [ SourceIP     => $rep->{'id.ipno'}],
             [ Target       => lc($cat) . ":" . $keyval],
             [ Component    => $component ],
             [ TargetName  => $report->id('display')],
             [ EventId     => $id         ],
              ]);
   $hm->FILTER($rep, $ev);

   $sd = Events->sourceDetector({ event => $ev });

   my @pert;
   if ($logical) {
      $key = CIM::Key->new( [$cim1,   # key of the disk
                  SystemName     => $oserial,
                  CreationClassName => $cim1,
                  DeviceID       => $oserial]);

      $pertains = CIM::Instance->new('NWS_EventPertainsToLogicalDevice', [
             [ Event       => $ev  ],
             [ Element     => $key ],
               ]);
     push(@pert, $pertains);

   } else {
     $key = CIM::Key->new( [$cim1,   # key of the disk
                  Tag               => $oserial,
                  CreationClassName => $cim1]);

     $contain = CIM::Instance->new( 'CIM_Container', [
                  [GroupComponent => $pf],            # frame is a package
                  [PartComponent  => $key],           # package is an element
                  [LocationWithinContainer => $component],
                   ]);

     $pertains = CIM::Instance->new('NWS_EventPertainsToContainer', [
                  [Event          => $ev   ],
                  [Association    => $contain ],
                  [DiscoveryType => "D"   ],   # delete
               ]);
     push(@pert, $contain, $pertains);
   }

   $sev += 0.2;
   #State->saveState($cat,  $keyval, $component, $name, $sev, $desc, 0, $ev);

   $ed = Message->new( { id        => $report->id,
                       instances => [$ev, @$sd, @pert ],
                       state     => [$component, 0 ],
                       severity  => $sev});

   PDM->saveMessage($ed);
   return $ed;
}

sub insertCompEvent {
  my($hm, $component, $pf, $report, $orep, $part, $nserial, $keyval, $arg) = @_;
  my($key, $ev, $sd, $pertains, $ed, $contain, $pertains2);

  my($cat) = $report->category();
  my $extra_desc = $arg->{desc};
  my $logical = $arg->{logicalDevice};
  my $comp_info = "($arg->{comp_info})" if ($arg->{comp_info});

  my $ix        = index($component, ".");
  my $comp0     = $component;
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my $rep  = $report->content();
  my $name = $rep->get('id.name') if ($rep);

  my $id  =  PDM->getEventSequence;
  my $id2 = $report->id('display');

  my $comp1 = $arg->{display_comp} || $component;
  $comp1  = "" if (index($nserial, $component) >= 0);

  my $desc = "$comp1$comp_info was added to $cat $id2 (sn=$nserial)$extra_desc";

  $ev = CIM::Instance->new('NWS_Event', [  
             [ EventType    => lc($cat) . ".ComponentInsertEvent"  ],
             [ Caption      => "add.$comp0"],
             [ Description  => $desc ],
             [ MgmtLevel    => $mgmtLevel ],
             [ CurrentValue => $nserial ],
             [ Target       => lc($cat) . ":" . $keyval],
             [ TargetName   => $report->id('display')],
             [ Severity     => Message::SEVERITY_NORMAL ],
             [ Component    => $component],
             [ SourceIP     => $rep->{'id.ipno'}],
             [ EventId      => $id ]]);

  $sd = Events->sourceDetector({ event => $ev });
  my @pert;
  if ($logical) {
     $pertains = CIM::Instance->new('NWS_EventPertainsToLogicalDevice', [
             [ Event       => $ev  ],
             [ Element     => $part->[0]],
               ]);
     push(@pert, $pertains);
  } else {
    $pertains = CIM::Instance->new('NWS_EventPertainsToPhysicalElement', [
             [ Event       => $ev  ],
             [ Element     => $part->[1]],
               ]);

    $contain = CIM::Instance->new('CIM_Container', [
              [GroupComponent           => $pf],         #PhysicalFrame
              [PartComponent            => $part->[1]],
              [LocationWithinContainer  => $component],
               ]);

    $pertains2 = CIM::Instance->new('NWS_EventPertainsToContainer', [
              [Event                    => $ev],
              [Association             => $contain  ],
              [DiscoveryType            => "A"       ],
               ]);
     push(@pert, $pertains, $contain, $pertains2);
  }

  #State->saveState($cat,  $keyval, $component, $name, Message::SEVERITY_NORMAL, $desc, 1, $ev);

  $ed = Message->new( { id        => $report->id,
                        state    => [$component, 1],
                       instances => [$ev, @$sd, @$part, @pert],
                       severity  => Message::SEVERITY_NORMAL } );

  PDM->saveMessage($ed);
}

#
# INSERT/DELETE/UPDATE Map
# ($insert, $delete, $old) = Health->ido_map($rep, $orep, "fru", "_name");
# create 3 hashes with the inserted, deleted and changed frus 
# using the _name attribute to match the frus correcly from new to old reports.
#
# INSERT and DELETE are { _name => index ...   } for entries added or deleted.
# UPDATE is a hash of { new_index => old_index } for entries present in both reports but
#                                                that may have moved.
#
sub idu_map {
  my($class, $rep, $orep, $section, $tag) = @_;
  my (%INSERT, %DELETE, %UPDATE);
  my(%MAP, %OMAP);

  foreach my $e (keys %$rep) {
    my @L = split(/\./, $e);
    if ($L[0] eq $section && $L[2] eq $tag) {
      $MAP{$rep->{$e}} = $L[1];
    }
  }
  foreach my $e (keys %$orep) {
    my @L = split(/\./, $e);
    if ($L[0] eq $section && $L[2] eq $tag) {
      $OMAP{$orep->{$e}} = $L[1];
    }
  }
  foreach my $el (keys %MAP) {
    if (!exists $OMAP{$el}) {
       $INSERT{$el} = $MAP{$el};
    } else {
       $UPDATE{$MAP{$el}} = $OMAP{$el};
    }
  }
  foreach my $el (keys %OMAP) {
    if (!exists $MAP{$el}) {
       $DELETE{$el} = $OMAP{$el};
    }
  }

  return(\%INSERT, \%DELETE, \%UPDATE);
}


1;

