#
# Copyright 2003 Sun Microsystems, Inc., All Rights Reserved.
#
package Client::Device;
use Debug;
use Client;
use Util;
use Matrix;
use Revision;
use Process;
use Socket;
use strict;
use Scheduler;

#
sub help {
  print Client->http_OK();
  print "Help: In progress...\n";
}


# Adds a device to be monitored.
# Arguments ip - Ip address of device.
#           type - hint at type of device.
#
# Returns the discovered properties for the device that have been 
#  written to the rasagent.conf file.
sub Add {
  require Logic::Subnet;
  require Agent::3310;

  my($q) = @_;
  my $in_ip = $q->{ip};
  my $intype = $q->{type} || "snmp";
  my $timeout = $q->{timeout} || "5";
  my $format = $q->{format} || "tab";
  my $userName = $q->{name};
  my $line   = "ERR|Add Failed";

  Debug->logLine("Client::Device::Add $in_ip $intype");

  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  # Check for device already added.
  foreach my $d (@$devs) {
    if (($d->{ip} eq $in_ip)||($d->{ipno} eq $in_ip)){
      Debug->logLine("Client::Device::Add Already Added");
       print Client->http_OK();
       if ( $q->{format} eq "xml"){
	 print Client->xmlHeader();
	 print Client->xmlTag("CLIENT_ADD");
	 print Client->xmlVALUE("class", $d->{class});
	 print Client->xmlVALUE("type", $d->{type});
	 print Client->xmlVALUE("ip", $d->{ip});
	 print Client->xmlVALUE("wwn", $d->{wwn});
	 print Client->xmlVALUE("key", $d->{key});
	 print Client->xmlVALUE("wwn2", $d->{wwn2});
	 print Client->xmlVALUE("name", $d->{name});
	 print Client->xmlVALUE("userLabel", $d->{userLabel});
	 print Client->xmlVALUE("time_added", $d->{time_added});
	 print Client->xmlVALUE("mgmtLevel", $d->{mgmtLevel});
	 print Client->xmlVALUE("active", $d->{active});
	 print Client->xmlEndTag("CLIENT_ADD");
       }
       else {
	 print "#<pre>\n";
	 print "class\t$d->{class}\n";
	 print "type\t$d->{type}\n";
	 print "ip\t$d->{ip}\n";
	 print "wwn\t$d->{wwn}\n";
	 print "key\t$d->{key}\n";
	 print "wwn2\t$d->{wwn2}\n" if ($d->{wwn2});
	 print "name\t$d->{name}\n";
	 print "userLabel\t$d->{userLabel}\n" if ($d->{userLabel});
	 print "time_added\t$d->{time_added}\n";
	 print "mgmtLevel\t$d->{mgmtLevel}\n";
	 print "active\t$d->{active}\n";
	 print "#DONE";
       }
       return;
    }
  }

  if (($intype eq "3510") || ($intype eq "3310")) {
    my $rc = Agent::3310->getFastWWN($in_ip);
    if ($rc->{error}){
      print Client->error($format, 103, 
			  "Device 3510 not found for $in_ip $rc->{error}");
      return;
    }
    my $t = $rc->{type};
    $line = "storage.$t|$t|$in_ip|$rc->{wwn}|$rc->{key}|$rc->{wwn2}||$in_ip|";
  }
  elsif ($intype eq "snmp"){
    my $stuff =
      Logic::Subnet->snmpget1($in_ip, undef, $timeout);
    $line = pop(@$stuff);
  }
  else {
     print Client->error($format, 102, "Unsuported type = $intype");
     return;
  }
  Debug->logLine("Client::Device::Add probe=$line");

  #  Parse line returned by discovery and add to configuration
  my($class, $type, $ip, $wwn, $key, $wwn2, $ctrl, $devname, $label) = 
    split(/\|/, $line) ;
  my($ip0, $port) = split(/\:/, $ip);

  if ($class eq "ERR") {
    print Client->error($format, 101, "$type");
    return;
  }

  if (!$key){
    print Client->error($format, 103, "Device not found for $in_ip");
    return;
  }

  # Add device to Config file
  my $new_dev = $#$devs ;
  my $cnt=1;
  my %F;
  foreach my $d (@$devs) {
    $F{$d->{key}} = $cnt;
    $cnt++;
  }

  my $timeAdded = Util->get_today();

  # Setup device name and ipno
  my $ipno   = Util->name2ip($ip);
  my $iaddr  = inet_aton($ipno);
  my ($gh_name,$gh_aliases,$gh_addrtype,$gh_length,@gh_addrs) =
    gethostbyaddr($iaddr, AF_INET);
  if ( ! $devname ){
    $devname = $userName || $gh_name || $ip;
  }

  $new_dev++;
  $devs->[$new_dev]{_name}      = "device" . $new_dev;
  $devs->[$new_dev]{name}       = $devname;
  $devs->[$new_dev]{type}       = $type;
  $devs->[$new_dev]{class}      = $class;
  $devs->[$new_dev]{userLabel}  = $label if ($label);
  $devs->[$new_dev]{ip}         = $ip;
  $devs->[$new_dev]{ipno}       = $ipno;
  $devs->[$new_dev]{wwn}        = $wwn;
  $devs->[$new_dev]{wwn2}       = $wwn2 if ($wwn2);
  $devs->[$new_dev]{key}        = $key;
  $devs->[$new_dev]{host}       = ""; #$q->{host};
  $devs->[$new_dev]{active}     = "Y";
  $devs->[$new_dev]{mgmtLevel}  = "D";
  $devs->[$new_dev]{time_added} = $timeAdded;

  # Save conf and update if device is new.
  if (!$F{$key}) {
    PDM::ConfigFile->write( $renv, $devs, $hosts,$notifs);
    my $ras_cmd  = System->get_home() . "/bin/rasagent";
    my $ras_args = "-l $key";
    my $command =  "$ras_cmd $ras_args >> /dev/null 2>&1 &";
    system($command);
  }

  print Client->http_OK();
  if ( $q->{format} eq "xml"){
    print Client->xmlHeader();
    print Client->xmlTag("CLIENT_ADD");
    print Client->xmlVALUE("class", $class);
    print Client->xmlVALUE("type", $type);
    print Client->xmlVALUE("ip", $ip);
    print Client->xmlVALUE("wwn", $wwn);
    print Client->xmlVALUE("key", $key);
    print Client->xmlVALUE("wwn2", $wwn2);
    print Client->xmlVALUE("name", $devname);
    print Client->xmlVALUE("userLabel", $label);
    print Client->xmlVALUE("time_added", $timeAdded);
    print Client->xmlVALUE("mgmtLevel", "D");
    print Client->xmlVALUE("active", "Y");
    print Client->xmlEndTag("CLIENT_ADD");
  }
  else {
    print "#<pre>\n";
    print "class\t$class\n";
    print "type\t$type\n";
    print "ip\t$ip\n";
    print "wwn\t$wwn\n";
    print "key\t$key\n";
    print "wwn2\t$wwn2\n" if ($wwn2);
    print "name\t$devname\n";
    print "userLabel\t$label\n" if ($label);
    print "time_added\t$timeAdded\n";
    print "mgmtLevel\tD\n";
    print "active\tY\n";
    print "#DONE";
  }
}


#
# Find the properties for a given device
#
sub Properties {
  my($q) = @_;
  my $format = $q->{format} || "tab";
  my($type, $key) = split(/\:/, $q->{key}) ;

  Debug->logLine("Client::Device::Remove $type $key");

  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  my $numDevs = $#$devs ;
  my $index=0;
  my %F;
  foreach my $d (@$devs) {
    if ($type eq $d->{type}) {
      if ($key eq $d->{key}){

	print Client->http_OK();
	if ( $q->{format} eq "xml"){
	  print "<?xml version=\"1.0\"?>\n";
	  print "<DEVICE_PROPERTIES>\n";
	  for my $xprop ( keys %{$d} ) {
	    print "<VALUE ID='$xprop'>$d->{$xprop}</VALUE>\n";
	  }
	  print "</DEVICE_PROPERTIES>\n";
	}
	else {
	  print "#<pre>\n";
	  for my $prop ( keys %{$d} ) {
	    print "$prop\t$d->{$prop}\n";
	  }
	  print "#DONE";
	}
	return;
      }
    }
    $index++;
  }
  print Client->error($format, 101, "No device exists for $type $key");
}


#
# Remove a device from the monitor list.
#
sub Remove {
  my($q) = @_;
  my $format = $q->{format} || "tab";
  my($type, $key) = split(/\:/, $q->{key}) ;

  Debug->logLine("Client::Device::Remove $type $key");

  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  my $numDevs = $#$devs ;
  my $index=0;
  my %F;
  foreach my $d (@$devs) {
    if ($type eq $d->{type}) {
      if ($key eq $d->{key}){
        splice(@$devs, $index, 1);
	PDM::ConfigFile->write( $renv, $devs,
				$hosts,$notifs);
	 print Client->http_OK();
	 print "OK";
	 return;
      }
    }
    $index++;
  }
  print Client->error($format, 101, "No device exists for $type $key");
}


#
# Update all devices or if a key defined a single device.
#
sub Update {
  my($q) = @_;
  my $format = $q->{format};
  my($type, $key) = split(/\:/, $q->{dev}) ;

  my @l = split(/\|/, $q->{devlist});
  foreach my $k (@l) {
    my($ktype, $kvalue) = split(/\:/, $k);
    if ($key){
      $key .= ":".$kvalue;
    }
    else {
      $key = $kvalue;
    }
  }

  my $ras_cmd = System->get_home() . "/bin/rasagent";
  my $ras_args;

  if ($key){
    $ras_args = "-l $key";
  }

  my $command =  "$ras_cmd $ras_args >> /dev/null 2>&1 &";
  system($command);

  Debug->logLine("Client::Device::Update $command");

  print Client->http_OK();
  print "OK $command";
}

#
# Run ras_revcheck on a single device
#
sub Revision {

  my($q) = @_;

  my $key     = $q->{key};
  my $format  = $q->{format} || "xml";
  my $MODULES = $q->{module} || "ALL";
  my $revpid  = $q->{pid};
  my $bg      = $q->{bg};
  my $cmd     = $q->{cmd} || "print";

  my @REPORTS;
  my $revDir = System->get_home() ."/DATA/sae_revcheck";

  Debug->logLine("Client::Device::Revision $MODULES $key $revpid");


  if ($revpid){
    if (( -f "$revDir/$revpid") && (-s "$revDir/$revpid")){
      if ($cmd eq "print"){
	open (REVOUT, "$revDir/$revpid");
	my @L = <REVOUT> ; close(REVOUT);
	print Client->http_OK();
	foreach my $l (@L) {
	  print "$l";
	}
      }
      elsif ($cmd eq "delete"){
	unlink "$revDir/$revpid";
      }
    }
    else {
      if ( -f "$revDir/$revpid"){
	print Client->error($format, 101,
			    "Output for $revpid not ready.");
      } else {
	print Client->error($format, 102,
			    "No revision process for $revpid.");
      }
    }
    return;
  }

  if ($key){
    print Client->error($format, 99,
			"Revision Check for single device not implemented.");
    return;
  }


  my $pid = fork();

  if ($pid){ #parent
    if ($bg eq "Y"){
      print Client->http_OK();
      print "OK $pid";
    }
    return;
  }

  if (defined $pid) {  # child

    my $out;
    if ($bg eq "Y"){

      if (! -d $revDir){
	mkdir $revDir, 0777;
      }
      if ( -f "$revDir/$$"){
	unlink "$revDir/$$";
      }
      open(REVFILE, ">$revDir/$$");
      print REVFILE "";
      close (REVFILE);

      $out = Client->xmlTag("REVISION_REPORT");
      close(STDIN);
      close(STDERR);
      close(STDOUT) if (!$ENV{FCGI});
    }
    else {
      print Client->http_OK();
      print Client->xmlTag("REVISION_REPORT");
    }

    my $include;
    my ($env)      = Revision->hbas_present();
    my $MATRIX     = Matrix->defaultFile();
    my $matrix     = Matrix->read($MATRIX, $env);
    my $installedp = Revision->readInstalledPatches();
    my $mods       = Modules->load("Revision");

    foreach my $x (@$mods) {
      if ($MODULES eq "ALL" || index(lc("|$MODULES|"), lc("|$x|")) >= 0) {
	$include .= " $x|";
	my $mod = "Revision::$x";
	my $db;
	eval {
	  Debug->logLine("Client::Device::Revision Running module $mod");
	  $db = $mod->RUN($matrix, $installedp);
	};
	if (!$db) {
	  #$db = Revision->warning2($x);
	  #push(@REPORTS, @$db);
	} elsif ($@) {
	  #$errors .= "ERROR in $mod: $@ \n";
	} else {
	  push(@REPORTS, @$db);
	}
	#open(O, ">>$LOG"); print O "module $x done\n"; close(O);
	chop($include);
      }
    }

    my($renv, $devs) = PDM::ConfigFile->read();
    my %KEY;
    foreach my $d (@$devs) {
      $KEY{$d->{name}} = $d->{key};
      $KEY{$d->{ipno}} = $d->{key};
    }

    foreach my $ele (@REPORTS){
      my $devKey = $ele->[8];
      if (!$devKey){
	$devKey = $KEY{$ele->[1]};
      }
      if (!$devKey){
	my @N = split(/\./, $ele->[1]);
	$devKey = $KEY{$N[0]};

	# Add smart ip parsing.
	if (!$devKey){
	  my $ipkey = $N[0];
	  my $index;
	  for ($index = 1; $index <= $#N && (!$devKey); $index++) {
	    $ipkey .= "." . $N[$index];
	    $devKey = $KEY{$ipkey};
	  }
	}

      }
      my $devType;
      foreach my $d (@$devs) {
	if ($d->{key} eq $devKey) {
	  $devType = $d->{type};
	}
      }
      my $fruKey = $ele->[9];

      $out .= Client->xmlTagOpen("ITEM");
      $out .= Client->xmlAttrib("Type", $ele->[0]);
      $out .= Client->xmlAttrib("DevType", $devType);
      $out .= Client->xmlAttrib("DevKey", $devKey);
      $out .= Client->xmlAttrib("Name", $ele->[1]);
      $out .= Client->xmlAttrib("Status", $ele->[2]);
      $out .= Client->xmlAttrib("CurrVers", $ele->[3]);
      $out .= Client->xmlAttrib("ExptVers", $ele->[4]);
      $out .= Client->xmlAttrib("CurrPatch", $ele->[5]);
      $out .= Client->xmlAttrib("ExptPatch", $ele->[6]);
      $out .= Client->xmlAttrib("Message", $ele->[10]);
      $out .= Client->xmlTagClose();

      $out .= Client->xmlEncode($ele->[7]) if $ele->[7];
      $out .= Client->xmlEndTag("ITEM");
    }
    $out .= Client->xmlEndTag("REVISION_REPORT");

    if ($bg eq "Y"){
      open(REVFILE, ">$revDir/$$");
      print REVFILE "$out";
      close (REVFILE);
      exit 0;
    }
    else {
      print "$out";
    }
  }
  else {
    print Client->error($format, 98,
			"Unable to run command.");
  }
}


#
# Lease a device
#
sub LeaseRequest {
  my($q) = @_;

  my $format   = $q->{format};

  Debug->logLine("Client::Device::LeaseRequest $format");

  print Client->error($format, 99,
		      "Lease Request not implemented.");
}

#
# Clear a device from a lease
#
sub LeaseRelease {
  my($q) = @_;

  my $format   = $q->{format};

  Debug->logLine("Client::Device::LeaseRelease $format");

  print Client->error($format, 99,
		      "Lease Release not implemented.");
}


1;
