
/*
 * Unreliable Publisher API 
 *
 * No, this does not mean that this API has bugs.  All it means is that 
 * this API does not provide any indication as to whether a published
 * event was successfully sent or received by a subscriber.
 *
 * This is a communication model that is suitable for some applications
 * but not of others.  If transfer reliability is a concern, the 
 * full-blown publisher API (publisher.h) should be used instead.
 * However, this API has the merit of being easier to use in the 
 * former case.
 */

#ifndef _INCLUDED_upub
#define _INCLUDED_upub

#ifdef _WIN32
#define __export __declspec(dllexport)
#else /* #ifdef _WIN32 */
#define __export
#endif /* #ifdef _WIN32 */

#ifdef __cplusplus
extern "C" {
#endif

/** 
 * upub_t : the unreliable publisher
 * 
 */
typedef struct upub_struct upub_t;


/** 
 * upub_publish : send a notification
 * 
 *
 * @param	upub		publisher
 *
 * @param	event_ref	event reference.  This is a URI idendifying
 *				the resource that is modified.
 *
 * @param	payload		notification payload.
 *
 * @param	payload_len	length in bytes of payload.
 *
 * @param	delete_payload 	callback to free the payload when no longer
 *				needed.
 */
__export void upub_publish(upub_t *upub, 
			   const char *event_ref,
			   void *payload,
			   unsigned int payload_len,
			   void (*delete_payload)(void *));

/**
 * upub_init : create the publisher
 *
 * This function allocates the upub object and sets up the thread 
 * dispatcher.  The thread dispatcher can be provided by the application
 * if the application already has one.  This will have the advantage 
 * of not creating a thread dedicated to this api.
 *
 * Note that the connection (if a connection-based transport is used) to
 * the notification service is not setup here.  This is deferred to the 
 * first time a notification is actually published.
 *
 * @param	gdc	GDisp context (thread dispatcher)
 *
 * @param	host	notification server host name
 *
 * @param	port	notification server port
 *
 * @param	num_workers	number of parallel sessions.
 *
 * @return	the upub ready to be used.
 */

__export upub_t *upub_init(void *gdc,
			   const char *host, unsigned short port,
			   int num_workers);

/**
 * upub_shutdown : shuts down and frees the publisher
 *
 * @param	upub		publisher 
 */
__export void upub_shutdown(upub_t *upub);


#ifdef __cplusplus
}
#endif

#endif /* _INCLUDED_upub */

