#!/bin/ksh
# Copyright 2003 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
# ident  "@(#)restore_SPAT.ksh 1.1     03/07/28 SMI"
#

#
# This script is used to set the firewall configuration to 
# the values defined in the repository file.
#

#
# Check the error codes that are not FIREWALL specific.
#
rcIsOK()
{
	rc=$1

	if [ $rc -eq 0 -o \
	     $rc -eq 41 -o \
	     $rc -eq 42 -o \
	     $rc -eq 101 -o \
	     $rc -eq 102 -o \
	     $rc -eq 103 -o \
	     $rc -eq 104 -o \
	     $rc -eq 105 -o \
	     $rc -eq 110 -o \
	     $rc -eq 130 -o \
	     $rc -eq 134 -o \
	     $rc -eq 150 ]
	then
		return 0
	else
		return 1
	fi
	
}


if [ `tty` != "/dev/console" ]
then
    myecho "You must be on the console to perform Firewall configuration."
    exit 1
fi

if [ `id | cut -f1 -d\(` != "uid=0" ] ; then
    myecho "You must be 'root' user to run this command"
    exit 1
fi

alreadyFoundSps=0
defaultFirewallAddress=10.0.0.242
firewallAddress=$defaultFirewallAddress

if [ -n "$1" ]
then
	if [ "$1" = "10.0.0.2" -o  \
	     "$1" = "10.0.0.12" -o \
	     "$1" = "10.0.0.22" -o \
	     "$1" = "10.0.0.32" -o \
	     "$1" = "10.0.0.42" -o \
	     "$1" = "10.0.0.52" -o \
	     "$1" = "10.0.0.62" -o \
	     "$1" = "10.0.0.72" ]
	then
		firewallAddress=$1
	fi
fi

if ! ping $firewallAddress 1>/dev/null 2>&1
then
	echo "ERROR : Firewall not reachable at $firewallAddress"
	exit 1
fi

#
# Step 1) Save the current firewall repository
#
echo "Step 1 : Backing up Firewall Repository file... "
oldFile=/opt/SUNWsespfw/repository.save
newFile=/opt/SUNWsespfw/repository
mv -f $newFile $oldFile

#
# Step 2) Assume new SPA tray has factory defaults? 
#   If no, do the following to reset firewall to factory defaults:
#   Verify that the firewall is back up and running.
#

echo "Step 2 : Downloading Firewall ROM file... "
cd /opt/SUNWsespfd
ftp -n $firewallAddress <<+
user root sun1
binary
put rom-0
quit
+

# 
# Step 3) Wait until Firewall is responding
#  Wait up to 5 minutes.
#
echo "Step 3 : Check for Firewall to come back online, up to 5 minutes... "
i=0
ok=0
while [ $i -lt 20 ]
do
	sleep 15
	ping $defaultFirewallAddress
	rc=$?
	if [ $rc -eq 0 ]
	then
		ok=1
		break
	fi
	i=`expr $i + 1`
done

#
# If this step fails, move the firewall repostory
# back to its original name.
#
if [ $ok -eq 0 ]
then
	echo "ERROR: Unable to ping firewall, rc = $rc."
	echo "       Manual intervention required."
        mv -f $oldFile $newFile
	exit 1
fi

#
# Step 4) Obtain the current settings from the repository file 
#       and set the following variables:
#


echo "Step 4 : Changing Firewall Settings, takes several minutes..."

nameServerAddress=`grep LanDNSSrv $oldFile | cut -f2 -d:` 
newRouterAddress=`grep LanIPAddr $oldFile | cut -f2 -d:` 
newSpAddress=`ifconfig dmfe0 | grep inet | cut -f2 -d' '`
ipAddress=`grep WanIPAddr $oldFile | cut -f2 -d:`
gatewayAddress=`grep WanGateway $oldFile | cut -f2 -d:` 
netmaskAddress=`grep WanSubnetMask $oldFile | cut -f2 -d:` 
dhcp_state=`grep WanAddrType $oldFile | cut -f2 -d:` 
spName=`hostname`



if [ "$dhcp_state" = "dynamic" ]
then
	if [ "$spName" = "sp0" ]
	then

#
# CASE 1: IF your hostname is sp0, and you are configuring your WAN IP
#         address using DCHP, run the following command :

    		/usr/local/bin/FW_config -d --dhcp_server en \
			--dhcp_dns_server $nameServerAddress \
			--lan_ip_addr $newRouterAddress \
			--port_fwd_addr $newSpAddress  \
			--wan_addr_type dynamic
	else

#
# CASE 2 : IF your hostname is NOT sp0, and you are configuring your WAN IP
#          address using DCHP, run the following command :

		/usr/local/bin/FW_config -d --dhcp_server dis \
			--lan_ip_addr $newRouterAddress \
			--port_fwd_addr $newSpAddress \
			--wan_addr_type dynamic
	fi
else
	if [ "$spName" = "sp0" ]
	then

#
# CASE 3 : IF your hostname is sp0, and you are configuring your WAN IP
#          address using a Fixed Address, run the following command :

  		/usr/local/bin/FW_config -d --dhcp_server en \
			--dhcp_dns_server $nameServerAddress \
			--lan_ip_addr $newRouterAddress \
			--port_fwd_addr $newSpAddress \
			--wan_addr_type static \
			--wan_ip_addr $ipAddress \
			--wan_gateway $gatewayAddress \
			--wan_subnet_mask $netmaskAddress

	else

#
# CASE 4 : IF your hostname is NOT sp0, and you are configuring your WAN IP
#          address using a Fixed Address, run the following command :

		/usr/local/bin/FW_config -d --dhcp_server dis \
			--lan_ip_addr $newRouterAddress \
			--port_fwd_addr $newSpAddress \
			--wan_addr_type static \
			--wan_ip_addr $ipAddress \
			--wan_gateway $gatewayAddress \
			--wan_subnet_mask $netmaskAddress

	fi
fi

rc=$?
if ! rcIsOK $rc
then
	echo "ERROR: Unable to reset firewall, rc = $rc."
	echo "       Manual intervention required."
	exit 1
fi

#
# Step 5) For all the ports that are different between the current repository 
#         and the repository.save, toggle the port state 

echo "Step 5 : Verifying Firewall Port Settings, may take several minutes..."

for i in dns esm patchpro ntp snmp cim_http svc_http sm_https svc_https ssh sm_http
do
	oldVal=`grep "^$i:" $oldFile`
	newVal=`grep "^$i:" $newFile`
	if [ "$oldVal" != "$newVal" ]
	then
                echo "Changing $i Port Setting..."
                cmd="/usr/local/bin/FW_config -d --toggle_port_state $i"
		runToggleCmd=1
		$cmd
		rc=$?
		if ! rcIsOK $rc
		then
			echo "ERROR: Unable to Toggle $i port, rc = $rc". 
			echo "       Manual intervention required."
			exit 1
		fi
	fi
done

echo "##### restore_SPAT.ksh completed successfully ######" 
exit 0
