#! /bin/sh
#
# Copyright (c) 2001-2002 by Sun Microsystems, Inc. All rights reserved.
# Use is subject to license terms.
#
#ident  "@(#)update_configDS.sh 1.0 - 02/10/31"
#
# purpose of this script is to update the Configuration Directory
# Server data (nsProductVersion, NsBuildNumber) after a DS patch 
# is installed.
#
 
#########################################
# usage
# print usage message
# input: $1 - 1: exit 1
#             0: do not exit
#########################################
usage () {
	echo "\nUsage: ${THIS}\n"
	[ $1 -eq 1 ] && exit 1
}

#########################################
# warn_and_exit
# warn user that config as not been updated
# and exit
#########################################
warn_and_exit () {
	echo "It is recommended that you fix the problem with "
	echo "the registry then perform configuration update again."
	echo "Exiting configuration update."
	exit 1
}

#########################################
# get_bindpasswd
# get bind password with echo OFF
#
# modify global variable: bind_passwd
#########################################
get_bindpasswd()
{
# Handle signals, so that echo can be turned back on if Ctrl-C.
trap "/usr/bin/stty echo; exit" 1 2 3 6 15

/usr/bin/stty -echo     # Turn echo OFF

# Endless loop that continues until passwd 
# is given
bind_passwd=""
first_try=yes
while [ "${bind_passwd}" = "" ]
do
    if [ ${first_try} = no ]
    then
	echo ""
    fi

    bind_passwd=""  

    echo "Enter bind password: \c"
    read bind_passwd
    first_try=no
done

/usr/bin/stty echo      # Turn echo ON

# Removed signal handler
trap 1 2 3 6 15

echo ""
}

#########################################
# updateCDS_4directory
# update CDS with the new data for 
# directory server
#
#########################################
updateCDS_4directory() 
{
    echo "\nUpdating Condiguration Directory Server for:"
    echo "${directory_server}"
    echo "with:"
    echo "\tnsbuildnumber: ${installed_buildnumber}"
    echo "\tnsproductversion: ${installed_productversion}"
    (/bin/cat <<EOF
dn: ${directory_server}
changetype: modify
replace: nsbuildnumber
nsbuildnumber: ${installed_buildnumber}
-
replace: nsproductversion
nsproductversion: ${installed_productversion}
EOF
) > ${TMP_LDAPMOD}
    ${LDAPMODIFY} -h ${cds_host} -p ${cds_port} -D "${admin_dn}" -j ${TMP_PASSWD_FILE} -f ${TMP_LDAPMOD} 
    err=$?
    if [ ${err}  -ne 0 ]
    then
	echo "\nError while modifying build number for:\n${directory_server}"
	warn_and_exit
    fi

}

#########################################
# updateCDS_4admin
# update CDS with the new data for 
# admin server
#
#########################################
updateCDS_4admin() 
{
    echo "Updating Condiguration Directory Server for:"
    echo "${admin_server}"
    echo "with:"
    echo "\tnsbuildnumber: ${admin_buildnumber}"
    (/bin/cat <<EOF
dn: ${admin_server}
changetype: modify
replace: nsbuildnumber
nsbuildnumber: ${admin_buildnumber}
EOF
) > ${TMP_LDAPMOD}
    ${LDAPMODIFY} -h ${cds_host} -p ${cds_port} -D "${admin_dn}" -j ${TMP_PASSWD_FILE} -f ${TMP_LDAPMOD} 
    err=$?
    if [ ${err}  -ne 0 ]
    then
	echo "\nError while modifying build number for:\n${admin_server}"
	warn_and_exit
    fi
}

#########################################
# main
##########################################
USR=/usr/iplanet
LD_LIBRARY_PATH=${USR}/ds5/lib:$LD_LIBRARY_PATH; export LD_LIBRARY_PATH
LDAPSEARCH=${USR}/ds5/shared/bin/ldapsearch
LDAPMODIFY=${USR}/ds5/shared/bin/ldapmodify
NSSLAPD=${USR}/ds5/bin/slapd/server/ns-slapd
ADM_CONF=${USR}/admserv5.1/admin-serv/config/adm.conf
SLAPD_INF=${USR}/ds5/setup/slapd/slapd.inf
ADMIN_INF=${USR}/ds5/setup/admin/admin.inf
ARG_COUNT=0

THIS=`/bin/basename $0`
echo "Starting ${THIS}"

if [ $# -ne ${ARG_COUNT} ]; then
    usage 1
fi

if [ ! -f ${ADM_CONF} ]
then
    echo "Configuration Directory Server data not found"
    warn_and_exit
fi

if [ ! -f ${SLAPD_INF} ]
then
    echo "Directory Server setup file not found"
    warn_and_exit
fi

if [ ! -f ${ADMIN_INF} ]
then
    echo "Administration Server setup file not found"
    warn_and_exit
fi

cds_host=`/bin/grep "^ldapHost:" ${ADM_CONF} 2>/dev/null | /bin/awk '{print $2}' `
if [ "x${cds_host}" = x ]
then
    echo "Configuration Directory Server host not found"
    warn_and_exit
fi
    
cds_port=`/bin/grep "^ldapPort:" ${ADM_CONF} 2>/dev/null | /bin/awk '{print $2}' `
if [ "x${cds_port}" = x ]
then
    echo "Configuration Directory Server port not found"
    warn_and_exit
fi

ds_buildnumber=`/bin/grep "^BuildNumber=" ${SLAPD_INF} | /bin/awk '{print $2}'`
if [ "x${ds_buildnumber}" = x ]
then
    echo "Directory Server build number not found"
    warn_and_exit
fi
res_ok=`echo ${ds_buildnumber} | /bin/wc -w`
if [ "$res_ok" -ne 1 ]
then
    echo "Directory Server build number undetermined (${ds_buildnumber})"
    warn_and_exit
fi
# get the installed build number from ns-slapd -v
installed_buildnumber=`${NSSLAPD} -v | /bin/awk '/^iPlanet-Directory/ {print $NF}' | /bin/sed -e "s/^B//"`
# echo "current installed build number: ${installed_buildnumber}\tbuild number from setup: ${ds_buildnumber}"
if [ ! "x${installed_buildnumber}" = "x${ds_buildnumber}" ]
then
    echo "setup build number and installed build number differ for Directory Server"
    warn_and_exit
fi

# get the installed product version from ns-slapd -v
installed_productversion=`${NSSLAPD} -v | /bin/grep "^iPlanet-Directory" | /bin/sed "s,^[^/]*/,," | /bin/awk '{for (i=1; i < NF; i++) printf("%s ",$i)}'`
if [ "x${installed_productversion=}" = x ]
then
    echo "Installed production version for Directory Server not found"
    warn_and_exit
fi


admin_buildnumber=`/bin/grep "^BuildNumber=" ${ADMIN_INF} | /bin/awk '{print $2}' | /bin/sort -u`
if [ "x${admin_buildnumber}" = x ]
then
    echo "Administration Server build number not found"
    warn_and_exit
fi
res_ok=`echo ${admin_buildnumber} | /bin/wc -w`
if [ "$res_ok" -ne 1 ]
then
    echo "Administration Server build number undetermined (${admin_buildnumber})"
    warn_and_exit
fi

# check CDS host and port
echo "Configuration Directory Server host [${cds_host}]: \c"
read tmp_host
if [ ! "x$tmp_host" = x ]
then
    cds_host=${tmp_host}
fi

echo "Configuration Directory Server host [${cds_port}]: \c"
read tmp_port
if [ ! "x$tmp_port" = x ]
then
    cds_port=${tmp_port}
fi

${LDAPSEARCH} -h ${cds_host} -p ${cds_port} -b "o=NetscapeRoot" o=NetscapeRoot > /dev/null 2>&1
err=$?
if [ ${err} -ne 0 ]
then
    echo "\nThe Configuration Directory Server at\n\t${cds_host}:${cds_port}"
    echo "is down, unreachable, or the given host and port are not valid."
    warn_and_exit
fi

# get the admin DN, if possible
admin_dn=`${LDAPSEARCH} -h ${cds_host} -p ${cds_port} -b "ou=TopologyManagement,o=NetscapeRoot" uid=admin dn 2>/dev/null | grep -v "^version" 2>/dev/null | awk '{for (i=2; i <= NF; i++) printf("%s",$i)}'`
echo "Admin DN [${admin_dn}]: \c"
read tmp_admindn
if [ ! "x$tmp_admindn" = x ]
then
    admin_dn=${tmp_admindn}
fi

TMPDIR="/tmp/${THIS}.$$"
if /bin/mkdir -m 700 ${TMPDIR}
then
	# Cleanup on exit.
	trap '/bin/rm -rf ${TMPDIR}' 0
else
	echo "ERROR: unable to create a safe temporary directory."
	exit 1
fi
TMP_PASSWD_FILE=${TMPDIR}/passdir.$$
TMP_LDAPRES=${TMPDIR}/ldap_res.$$
TMP_LDAPMOD=${TMPDIR}/ldap_mod.$$
# get passwd for admin
# in ${bind_passwd}
get_bindpasswd 
echo $bind_passwd > $TMP_PASSWD_FILE

directory_server_filter="cn=iPlanet Directory Server"
admin_server_filter="cn=Netscape Administration Server"
# get the relevant server group from adm.conf
server_group=`/bin/grep "^isie:" ${ADM_CONF} | /bin/sed "s/^[^,]*,//" 2>/dev/null"`
if [ "${server_group}"x = x ]
then
    echo "Can't find the relevant server group"
    warn_and_exit
fi
directory_server="${directory_server_filter},${server_group}"
admin_server="${admin_server_filter}, ${server_group}"

# now update CDS
updateCDS_4directory
updateCDS_4admin


echo "Configuration Directory Server successfully updated."
exit 0