#!/usr/perl5/bin/perl -w
#
# Copyright 2001-2002 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
# ident   "@(#)patch_step1.pl 1.6     03/01/07 SMI"
#
# Output the patches specified so user can verify that they are correct.

require 5.005;
use strict;
use lib '/opt/SUNWscvw/lib/perl';
use Cluster::Cgi;
use Cluster::Install;
use Sun::Solaris::Utils qw(gettext);

# Set autoflush
$| = 1;

# Create the html query object
my $q = new Cluster::Cgi;

# Create a install object
my $install = new Cluster::Install;

# Get the list of nodes
my @nodelist = $install->get_nodelist($q);

# Verify that the cdrom and patch paths are accessible from all nodes
my (@errors);

foreach my $node ($install->get_nodelist($q)) {
    my @result = $install->verify_paths($node, $q);
    
    if ($#result > -1) {
	push @errors, @result;
    }
}

# If we found errors, print them and exit
if ($#errors > -1) {

    # Start the html document
    $install->header($q, [gettext("Sun Cluster Installation"),
	"/cgi-bin/installation/step0.pl",
	gettext("Paths (Step 6 of 8)")]);
    $install->start_message_table($q, "/images/error_32.gif",
	gettext("Error"), gettext("Incorrect Path Information"),
	gettext("The following installation paths could not be reached:"));
    
    # Print unreachable nodes

    foreach my $error (@errors) {
	print $q->p({class => "message-command-text"}, $error);
    }

    $install->end_message_table($q);
    print $q->p({class => "action-window-normal-text"});
    print gettext("Make sure that the specified paths are correct and ".
	"that they can be reachable by all nodes of the cluster.  The paths ".
	"of CDROM images should contain the .cdtoc file.  Return to ".
	"the previous page and enter the correct path(s).");

    print $q->start_form();

    $install->start_button_table($q);

    print $q->button({ value   => gettext("  < Back  "),
		       onClick => "history.back()"
		       });
    $install->end_button_table($q, "cancel");

    print $q->end_form();
    
    # End the html document
    $install->footer($q);

    # Exit gracefully
    exit(0);
}

# If no patch directory was specified then there is no need to display the
# directory's contents.  So redirect to sccheck_step1.pl
if ( ! $q->param('patch_path')) {
    my $query_string = $q->get_cluster_query_string();
    print $q->redirect("/cgi-bin/installation/sccheck_step1.pl$query_string");
    exit;
}

# Start the html document
$install->header($q, [gettext("Sun Cluster Installation"),
    "/cgi-bin/installation/step0.pl",
    gettext("Confirm Patch Information (Step 6 of 8)")]);
print $q->p({class => "action-window-normal-text"});

# Refresh the page in case lots of patches. 
print gettext("Uncompressing and locating all patches...");

# Get global values
my $cluster_name = $q->param('cluster_name');
my $patch_path = $q->param('patch_path');
my $uncompressed_patch_path = $install->get_uncompresspatchdir();
my $error_file = "/var/cluster/spm/setup_patch_dir_errors";
my $node;
my @patches;
my @patch_errors;

# Copy the contents of $patch_path to the well-known directory
# $uncompressed_patch_path.  Then uncompress the patches, if necessary.  Do
# this is parallel for performance.
foreach $node ( @nodelist ) {
    $install->copy_uncompress_patches($node, $patch_path, $uncompressed_patch_path, $error_file, $q);
}

# Wait for all nodes to complete the copy and optional uncompression.
foreach my $node (@nodelist) {
    # redo comes here
    if ($install->get_install_state($node, "DONE_COPY_UNCOMPRESS_PATCHES") ne 'YES') {
        $install->syslog("Copying and uncompressing the patches has NOT ".
	"finished on $node");
        sleep(10);
	print ".";
        redo;
    } else {
        $install->syslog("Copying and uncompressing the patches has finished ".
	"on $node");
    }
}

# Print newlines
print "<br><br>\n";

# Get the patch contents to display to the user.  Since all the patches should
# be identical, just grab the contents of the first node.
foreach $node ( @nodelist ) {
    @patches = $install->get_patch_list($node, $uncompressed_patch_path, $q);
    last;
}
# last come here

# Now check to see if there were any errors in copying or uncompressing
foreach $node ( @nodelist ) {
    push (@patch_errors, $install->check_patch_errors($node, $error_file));
}
if (@patch_errors) {
    # There were errors when copying or uncompressing certain patches!
    $install->start_message_table($q, "/images/error_32.gif", gettext("Error"), gettext("Patch Installation"), gettext("The following error(s) occurred during the copying of the directory <b>$patch_path</b> to the directory <b>$uncompressed_patch_path</b> or during the uncompression of patches:"));
    foreach my $p_e ( @patch_errors ) {
         # Print message for the user
         print $q->p({CLASS => "message-command-text"});
         print $q->sprintfn(gettext("%1"), $p_e);
         $install->syslog("$p_e");
         $q->br();
    }
    $install->end_message_table($q);
    print $q->br();
    print $q->p({CLASS => "action-window-normal-text"});
    print gettext("Resolve the above error(s) and return to the previous page to try again.");
    print $q->br();
    print $q->br();
    print gettext("If you do not wish to add patches through the SunPlex Manager, then return to the previous page and clear the patch directory path.");
    print $q->br();

    print $q->start_form({ action => '/cgi-bin/installation/step5.pl' });
    # Print the hidden cluster fields
    $q->print_hidden_cluster_fields();
    print $q->hidden({ name    => "novalidate", value => 1});

    $install->start_button_table($q);

    print $q->button({ value   => gettext("  < Back  "),
		       onClick => "history.back()"
		       });


    # Put spaces between the buttons
    print "&nbsp;" ;

    $install->end_button_table($q, "cancel");

    print $q->end_form();
    print $q->end_form();
    print $q->end_td();
    print $q->end_Tr();
    print $q->end_table();
    
    # End the html document
    $install->footer($q);

    # Exit gracefully
    exit(0);
}
# If no patches were returned, then print a warning.
if ( ! @patches ) {
    $install->start_message_table($q, "/images/warning_32.gif",
	gettext("Warning"),
	sprintf(gettext("No valid patches found in %s."), $patch_path));

    print $q->br();

    $install->end_message_table($q);
    print $q->p({class => "action-window-normal-text"});

    print gettext("Return to the previous page and specify another patch directory.");
    print $q->br();
    print gettext("If you do not wish to add patches through the SunPlex Manager, then return to the previous page and clear the patch directory path.");
    print $q->br();

    print $q->start_form({ action => 
			   '/cgi-bin/installation/sccheck_step1.pl' });

    # Print the hidden cluster fields
    $q->print_hidden_cluster_fields();
    print $q->hidden({ name    => "novalidate", value => 1});

    $install->start_button_table($q);

    print $q->button({ value   => gettext("  < Back  "),
		       onClick => "history.back()"
		       });


    # Put spaces between the buttons
    print "&nbsp;" ;

    $install->end_button_table($q, "cancel");

    print $q->end_form();
    print $q->end_form();
    print $q->end_td();
    print $q->end_Tr();
    print $q->end_table();
    
    # End the html document
    $install->footer($q);

    # Exit gracefully
    exit(0);
} else {
    # At least one patch was found in the specified directory
    # Print general info
    $install->start_message_table($q, "/images/info_32.gif", gettext("Informational"), gettext("Patches Discovered"), gettext("The following patches were discovered in the patch directory $patch_path:"));
    print $q->br();
    print $q->start_table({	BORDER		=> 0,
				CELLSPACING	=> 0,
				CELLPADDING	=> 0,
				ALIGN		=> "Left"
			});
    foreach my $p (@patches) {
	print $q->start_Tr();
	print $q->start_td();
	$q->start_table_text("breadcrumb-text");
        print sprintf(gettext("%s"), $p);
        $q->end_table_text();
        print $q->end_td();
        print $q->end_Tr();
    } 
    print $q->end_table();
    $install->end_message_table($q);
    
    # Print advisory to user
    print $q->p({CLASS => "action-window-normal-text"});
    print gettext("If this is the correct set of patches to install on all nodes, click 'Next'.  Otherwise if this is not the correct set, then click 'Back' and either modify the contents of the patch directory or select another patch directory location.");
 
    print $q->start_form({ action =>
			   '/cgi-bin/installation/sccheck_step1.pl' });

    # Print the hidden cluster fields
    $q->print_hidden_cluster_fields();

    $install->start_button_table($q);
    
    print $q->button({ value   => gettext("  < Back  "),
    		   onClick => "history.back()"
    		   });

    # Put spaces between the buttons
    print '&nbsp;';

    print $q->submit({ name  => 'submit_step_5',
		   value => gettext("  Next >  ")
		   });
    $install->end_button_table($q, "cancel");


    # End the form
    print $q->end_form();
    
    # End the html document
    $install->footer($q);
    # Exit gracefully
    exit(0);
}
