#pragma ident  "@(#)alntest.c 1.1     04/05/07 SMI"

/*
 *	Copyright (c) 2004, Sun Microsystems, Inc.
 *	All Rights Reserved.
 */

/*
 * This program demonstrates async link notification (ALN) for the
 * following SONET link events:
 *
 *	LOS	loss of signal
 *	LOF	loss of frame
 *	LRDI	link remote defect indication
 *	PRDI	path remote defect indication
 *
 * After processing the argument list, a A_SONET_LINKNOTIF_REQ ioctl
 * is sent to the ba driver which enables the selected interrupts.
 *
 * N.B. This ioctl is only supported in the ba driver (on ATM cards
 * built around the PMC-Sierra PM5346 or S/UNI-155-LITE chip.
 *
 * The program then waits for STREAMS messages from the driver.
 *
 * The ba driver will send M_PROTO messages containing the current
 * state of the link on each state change (ie. if you pull the fibre,
 * you will get one message; when you reconnect the fibre, you will
 * get another message)
 *
 * These message only go to streams that have registered interest in
 * these events.  (De-registration will occur automatically when the
 * stream is closed.)
 *
 * You could, for example, have 4 of these programs running.  One for
 * each event type.
 *
 * The hardware interrupt enable/disable logic is a bitwise ORing of
 * all the registered streams.  ie. If they all registered, but used a
 * zero filled request structure then the hardware interrupts would be
 * disabled.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <stropts.h>
#include <poll.h>
#include <string.h>
#include <errno.h>
#include <atmlib.h>
#include <atm/atm.h>
#include <atm/atmioctl.h>


static char *progname;

void
usage(void)
{
	fprintf(stderr,
		"Usage: %s [dev [ 0|1 0|1 0|1 0|1 ]]\n\n"
		"  dev is ba0 or ba1\n\n"
		"  0 turns off and 1 turns on the\n"
		"  following event notifications:\n\n"
		"  LOS - loss of signal\n"
		"  LOF - loss of frame\n"
		"  LRDI - link remote defect indication\n"
		"  PRDI - path remote defect indication\n\n"
		"  With no arguments, the default is to enable"
		" all event indications on ba0.\n\n"
		"Examples:\n\n"
		"  %s             - enable all on ba0\n"
		"  %s ba1         - enable all on ba1\n"
		"  %s ba0 0 0 0 0 - enable nothing on ba0\n"
		"  %s ba1 1 0 1 0 - enable LOS, LRDI on ba1\n",
		progname, progname, progname, progname, progname);

	exit(1);
}

int
main(int argc, char **argv)
{
	int fd;
	int ppa;
	struct strbuf ret;
	struct strioctl ioc;
	atm_sonet_link_event_t link, rlink;
	struct pollfd fds[1];
	int flags;
	int rv;
	char *dev;

	/* Cheap basename */
	if ((progname = strrchr(argv[0], '/') + 1) == (char *)1)
		progname = argv[0];

	/* Unused  - only of use on received events */
	link.primitive = 0;

	/* Parse the arguments and build the link event structure */
	switch (argc) {
	case 1:
		dev = "ba0";
		link.lof = 1;
		link.los = 1;
		link.lrdi = 1;
		link.prdi = 1;
		break;
	case 2:
		if ((argv[1][0] == '-' && argv[1][1] == 'h') ||
		    !strcasecmp(argv[1], "--help"))
			usage();

		dev = argv[1];
		link.lof = 1;
		link.los = 1;
		link.lrdi = 1;
		link.prdi = 1;
		break;
	case 6:
		dev = argv[1];
		link.lof = atoi(argv[2]);
		link.los = atoi(argv[3]);
		link.lrdi = atoi(argv[4]);
		link.prdi = atoi(argv[5]);
		break;
	default:
		fprintf(stderr, "%s: incorrect number of arguments\n", progname);
		usage();
	}

	/* Open the ATM device */
	fd = atm_open(dev);
	if (fd < 0) {
		fprintf(stderr, "%s: atm_open of %s failed: %s\n",
			progname, dev, strerror(errno));
		exit(1);
	}

	/* attach */
	ppa = atm_get_ppa(dev);
	atm_attach(fd, ppa, 0);

	/* build the ioctl message and send it */
	ioc.ic_cmd = A_SONET_LINKNOTIF_REQ;
	ioc.ic_timout = 0;
	ioc.ic_len = sizeof (atm_sonet_link_event_t);
	ioc.ic_dp = (char *)&link;

	if (ioctl(fd, I_STR, &ioc) < 0) {
		fprintf(stderr, "%s: ioctl failed: %s\n",
			progname, strerror(errno));
		exit(2);
	}

	/* setup for poll */
	fds[0].fd = fd;
	fds[0].events = POLLIN;
	fds[0].revents = 0;

	for (;;) {
		rv = poll(fds, 1, -1);
		if (rv < 0) {
			fprintf(stderr, "%s: poll failed: %s\n",
				progname, strerror(errno));
			exit(3);
		}
		if (rv == 0)
			continue;

		/*
		 * we got something (we're only expecting a
		 * ATM_SONET_LINKNOTIF_IND, so we decode it as such.
		 */
		ret.maxlen = sizeof(atm_sonet_link_event_t);
		ret.len = 0;
		ret.buf = (char *)&rlink;

		if (getmsg(fd, &ret, NULL, &flags) < 0) {
			fprintf(stderr, "%s: getmsg failed: %s\n",
				progname, strerror(errno));
			exit(4);
		}

		printf("flags: %d\n", flags);
		printf("ret.len: %d\n"
		       "ret.buf: %p\n",
		       ret.len, ret.buf);

		printf("rlink.primitive: 0x%x\n"
		       "rlink.los:            %d\n"
		       "rlink.lof:            %d\n"
		       "rlink.prdi:           %d\n"
		       "rlink.lrdi:           %d\n"
		       "---------------------------\n",
		       rlink.primitive,
		       rlink.los,
		       rlink.lof,
		       rlink.prdi,
		       rlink.lrdi);

		/* once more with feeling */
	}
}
