/*

        Parametric stepped shaft design program

        A sample ADS application.

        Designed and implemented by John Walker in September of 1989.



        Minor Additions:
          Windows 3.0 DDE hot link support, Phil Ford, Nov, 1991.
          Windows 3.1 Registration Database Search for Excel path, Dec, 1992

     ________________________________________________________________________

      (C) Copyright 1990-1994 by Autodesk, Inc.

      Permission to use, copy, modify, and distribute this software and its
      documentation for any purpose and without fee is hereby granted.  

      THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY. 
      ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF 
      MERCHANTABILITY ARE HEREBY DISCLAIMED.                                
     ________________________________________________________________________



        Commands:

           IMPORT_SHAFT Import shaft model from spreadsheet.
           EXPORT_SHAFT Send current model from AutoCAD to the
                        spreadsheet.
           UPDATE_SHAFT Update the model in the AutoCAD database
                        from the spreadsheet.
           V3D          Show oblique 3D view without dimensions.
           VDIAM        Show end-on view with diameter dimensions.
           VLENGTH      Show side view with length dimensions.
           SHAFTDIALOG  Run DDE dialog to set application, workfile,
                        and automatic update flag ("warm link").


    This application works in conjunction with a Windows DDE  link  to
    an Excel spreadsheet named  SHAFT.XLS.   This  spreadsheet  allows
    design  of  the  shaft based on its mechanical properties.  Once a
    shaft is designed, it can be imported into an AutoCAD  model  with
    the IMPORT_SHAFT command.

    If  the  spreadsheet model is modified, the AutoCAD drawing can be
    updated by entering the UPDATE_SHAFT command.

    You can modify the model with AutoCAD by using the STRETCH command
    to adjust the length of segments of the shaft, then send the model
    back to the spreadsheet with the EXPORT_SHAFT command.  The  spreadsheet
    will  recalculate the mechanical properties of the shaft, then the
    AutoCAD model will be automatically updated.

    The shaft is represented as a 3D model, dimensioned  in  both  the
    length  and  diameter directions.  You can view the model in three
    different ways with the viewing commands.  V3D shows an oblique 3D
    view without the dimensions; VLENGTH shows a dimensioned side view
    suitable for application of the STRETCH command;  VDIAM  shows  an
    edge-on view with the diameters dimensioned.

*/

/* #define DEBUGDUMP */               /* Dump data sent and received ? */

#include "options.h"
#include "adslib.h"

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "winutil.h"
#include "ddewin.h"
#include "windde.h"
#include "spreadsh.h"


#define V  (void)

/*  Standard drawing object names  */

#define FrozenLayer     "FROZEN-SOLID" /* Frozen layer for information */
#define PartLayer       "SHAFT"        /* Shaft geometry layer */
#define LDimensionLayer "LDIMEN"       /* Length dimension layer */
#define DDimensionLayer "DDIMEN"       /* Diameter dimension layer */
#define TempUCS         "P$SHAFT"      /* Temporary saved UCS name */

/*  Interface definitions  */

static char Application[MAXNAME+1] = "Excel"; /* Default application name */
static char SpreadSheet[MAXNAME+1] = "shaft.xls"; /* Default spreadsheet name */
static char AppStart[MAXPATH+120] = "";
                                       /* Auto start up command line */ 
static char cells[32];                 /* Description of range of spreadsheet
                                          cells of interest */  
static ddeDATA *pDdeData;              /* DDE global data, dialog data */       
static PDDE pdde;                      /* DDE channel data */   

#define ModelRow        5              /* Model start row in spreadsheet */
#define ModelColumn     2              /* Model start column in spreadsheet */

/* Definitions  to  wrap  around  submission  of  AutoCAD commands to
   prevent their being echoed.  */

#define Cmdecho  FALSE             /* Make TRUE for debug command output */

#define CommandB()  { struct resbuf rBc, rBb, rBu, rBh; \
    ads_getvar("CMDECHO", &rBc); ads_getvar("BLIPMODE", &rBb); \
    ads_getvar("HIGHLIGHT", &rBh); \
    rBu.restype = RTSHORT; rBu.resval.rint = (int) Cmdecho; \
    ads_setvar("CMDECHO", &rBu); rBu.resval.rint = (int) FALSE; \
    ads_setvar("BLIPMODE", &rBu); ads_setvar("HIGHLIGHT", &rBu)

#define CommandE()  ads_setvar("CMDECHO", &rBc); \
    ads_setvar("BLIPMODE", &rBb); \
    ads_setvar("HIGHLIGHT", &rBh); }

/*  Definitions  that permit you to push and pop system variables with
    minimal complexity.  These don't work  (and  will  cause  horrible
    crashes  if  used)  with  string  variables,  but since all string
    variables are read-only, they cannot be saved and restored in  any
    case.  */

#define PushVar(var, cell, newval, newtype) { struct resbuf cell, cNeW; \
    ads_getvar(var, &cell); cNeW.restype = cell.restype; \
    cNeW.resval.newtype = newval; ads_setvar(var, &cNeW)

#define PopVar(var, cell) ads_setvar(var, &cell); }

/* Set point variable from three co-ordinates */

#define Spoint(pt, x, y, z)  pt[X] = (x);  pt[Y] = (y);  pt[Z] = (z)

/* Copy point  */

#define Cpoint(d, s)   d[X] = s[X];  d[Y] = s[Y];  d[Z] = s[Z]

/* Utility definition to get an  array's  element  count  (at  compile
   time).   For  example:  

   int  arr[] = {1,2,3,4,5};
   ... 
   printf("%d", ELEMENTS(arr));

   would print a five.  ELEMENTS("abc") can also be used to  tell  how
   many  bytes are in a string constant INCLUDING THE TRAILING NULL. */
   
#define ELEMENTS(array) (sizeof(array)/sizeof((array)[0]))

/* Utility definitions */

#ifdef max
#undef  max
#endif
#define max(a,b)    ((a)>(b) ? (a) : (b))

/*  Shaft definition item  */

#define MAXSTEPS 10                /* Maximum steps in shaft */

typedef struct {
    ads_real shleng;               /* Length of this step */
    ads_real idiam;                /* Inside diameter */
    ads_real odiam;                /* Outside diameter */
} shafti;
static int shaftn = 0;             /* Number of steps in current shaft */
static shafti shaft[MAXSTEPS];     /* Definition of current shaft */
static ads_point updpoint;         /* Save position for update */
static BOOL updpos = FALSE;        /* Force point on import() ? */
static BOOL imported = FALSE;      /* Has anything been imported ? */
static BOOL antirecurso = FALSE;   /* Import() anti-recursion disable flag */



/* All Function Prototypes for shaft.c */

VOID main(int argc, char **argv);
static BOOL funcload(void);
static BOOL initacad(BOOL reset);
static BOOL importedp(void);
static void import(void);
static void update(void);
static void vlength(void);
static void vdiam(void);
static void v3d(void);
static void export(void);
static void listfunc(void);
static PDDE chk_dde(int pokeflag);
static int AdviseFunc(PDDE pdde);
static void shaftdialog(void);
static struct resbuf *assoc(struct resbuf *ent, int type);

/*  Table  of  AutoCAD function (or command) names and corresponding C
    function name */

CMDTAB cmdtab[] = {
    /*   Name           Function      */
    {"C:IMPORT_SHAFT",  import},     /* Import model from spreadsheet */
    {"C:EXPORT_SHAFT",  export},     /* Export model to spreadsheet */
    {"C:VLENGTH",       vlength},    /* View length dimensions */
    {"C:VDIAM",         vdiam},      /* View diameter dimensions */
    {"C:V3D",           v3d},        /* View 3D view */
    {"C:UPDATE_SHAFT",  update},     /* Modify drawing */
    {"C:SHAFT",         listfunc},   /* List all the function names 
                                   in this module */
    {"C:SHAFTDIALOG", shaftdialog} /* Run DDE dialog */
};






/********************************************************************

                          Main Program

********************************************************************/
        
VOID main(int argc, char **argv)
{
    short scode = RSRSLT;
    short cindex;
    short ads_req; 

    /* Initialize DDE Manager with the name, instance handle, and
       DDEML initialization handle. */
    pDdeData = DdeInit(adsw_AppName, adsw_hInstance, adsw_idInst);
    /* In case there is no info in windows\shaft.ini yet, use these
       defaults.  AppStart will be set to something like
       "d:\acadwin\acad.exe shaft.xls" based on Windows registration
       database. */
    DdeDefaults(Application, SpreadSheet, AppStart);
    /* This is the range of spreadsheet cells in which we're 
       interested */
    CellRange(cells, ModelRow, ModelColumn, MAXSTEPS, 3);

    /* Initialize ADS system */
    ads_init(argc, argv); 

#ifdef SHOW_AS_ICON
    /* Display ADS window as icon */
    ShowWindow(adsw_hWnd, SW_SHOWMINNOACTIVE);
    UpdateWindow(adsw_hWnd);
#endif

    /* Read info (app, topic, etc) from windows\shaft.ini */
    DdeGetProfile((char *)NULL);

    for ( ;; ) {

        /* This process will block (stop) here until AutoLISP has a 
           request for us */

        ads_req = ads_link(scode);        

        if (ads_req < 0)
            break;

        scode = RSRSLT;               /* Default return code */

        switch (ads_req) { 

        case RQXLOAD:                 /* Load request. Send function names. */
            scode = -(funcload() ? RSRSLT : RSERR);
            break;

        /* Execute a "loaded" function that was defined via RQXLOAD. */

        case RQSUBR:
            cindex = ads_getfuncode();
            V initacad(FALSE);
            if (cindex >= 0 && cindex < ELEMENTS(cmdtab))
                (*cmdtab[cindex].cmdfunc)();
            ads_retvoid();            /* Return void result */
            break;

        /* (xunload "shaft") entered or called in AutoCAD */ 

        case RQXUNLD:
            /* "xunload" or AutoCAD terminating */
            DdeQuit();                /* Terminate DDE channels, free up 
                                         memory, etc */
            break;

        default:
            break;
        }
    }

}

/* FUNCLOAD -- Register our functions with the ADS system.  */

static BOOL funcload(void)
{
    int idx;
    char *cname;

    for (idx = 0; idx < ELEMENTS(cmdtab); ++idx) {
        cname = cmdtab[idx].cmdname;
        ads_defun(cname, (short)idx);
    }
    imported = FALSE;                 /* No model loaded in this drawing */
    V initacad(TRUE);
    return TRUE;
}


/*  INITACAD  --  Initialise the required modes in the AutoCAD
                  drawing.  */

static BOOL initacad(BOOL reset)
{
    static BOOL initdone = FALSE, initok = TRUE;

    if (reset) {
        initdone = FALSE;
        initok = TRUE;
    } else {
        if (!initdone) {
            struct resbuf rb;
            struct resbuf *ep;

            initok = FALSE;

              /* Enable  handles  if  they aren't  already on.  We use
                 handles to link  from  our  in-memory  table  to  the
                 masses  in  the  AutoCAD  database,  and we also keep
                 track of the reference frame entity  by  its  handle. */

            ads_getvar("HANDLES", &rb);
            if (!rb.resval.rint) {
                CommandB();
                ads_command(RTSTR, "handles", RTSTR, "on", RTNONE);
                CommandE();
                ads_getvar("HANDLES", &rb);
                if (!rb.resval.rint) {
                    ads_printf("Cannot enable handles.\n");
                    initdone = TRUE;
                    return (initok = FALSE);
                }
            }

           /* Create required drawing objects. */

           /* If there isn't already a "frozen solid" layer, create one. */

            if ((ep = ads_tblsearch("LAYER", FrozenLayer, FALSE)) == NULL) {
                CommandB();
                ads_command(RTSTR, "_LAYER", RTSTR, "_NEW", RTSTR, FrozenLayer,
                    RTSTR, "_FREEZE", RTSTR, FrozenLayer, RTSTR, "", RTNONE);
                ads_command(RTSTR, "_LAYER", RTSTR, "_NEW",
                    RTSTR, PartLayer, RTSTR, "", RTNONE);
                ads_command(RTSTR, "_LAYER", RTSTR, "_NEW",
                    RTSTR, LDimensionLayer, RTSTR, "", RTNONE);
                ads_command(RTSTR, "_LAYER", RTSTR, "_NEW",
                    RTSTR, DDimensionLayer, RTSTR, "", RTNONE);
                ads_command(RTSTR, "_ORTHO", RTSTR, "_ON", RTNONE);
                ads_command(RTSTR, "_SNAP", RTREAL, 0.1, RTNONE);
                ads_command(RTSTR, "_UNITS", RTSTR, "2", RTSTR, "2",
                    RTSTR, "", RTSTR, "", RTSTR, "", RTSTR, "", RTNONE); 
                CommandE();
            } else {
                ads_relrb(ep);
            }

            initdone = initok = TRUE;
        }
    }
    return initok;
}

/*  IMPORTEDP  --  Test if a valid model has been imported into this
                   drawing.  */

static BOOL importedp(void)
{
    if (!imported) {
        ads_printf("No model imported from %s|%s.\n", Application,
                                SpreadSheet);
            return FALSE;
    }
    return TRUE;
}

/*  IMPORT  --  Import a shaft design from the spreadsheet.  */

static void import(void)
{
    int i, j, k;
    ads_point pos, pos1, pos2, pos3;
    ads_real celev, maxrad = -1.0;
    struct resbuf clayer, dimtxt;
    ads_real diadim[MAXSTEPS * 2], diadih[MAXSTEPS * 2];

    /* Check if a channel to app and topic is already open.  If not, 
       try to start one. */
    if (!chk_dde(FALSE))
        return;

    /* We  only allow one model in the drawing at a time.  If a model
       is already imported, just update it.  */

    if (imported && !antirecurso) {
        update();
        return;
    }

    /* First we magically obtain the shaft design parameters from  the
       spreadsheet into the shaft array.  */

    for (i = 0; i < MAXSTEPS; i++)
        shaft[i].shleng = 0.0;

    if (RequestRealArray(pdde, (ads_real *)shaft, ModelRow, ModelColumn,
                         MAXSTEPS, 3)) {

        /* Determine the number of segments in the shaft by searching
           for the first zero-length segment.  */

        for (shaftn = 0; shaftn < MAXSTEPS; shaftn++) {
            if (shaft[shaftn].shleng == 0.0)
                break;
        }
#ifdef DEBUGDUMP
        ads_printf("Shaftn = %d\n", shaftn);
        for (i = 0; i < shaftn; i++) {
            ads_printf("%d:  %f %f %f\n", i, shaft[i].shleng,
                shaft[i].idiam, shaft[i].odiam);
        }
#endif
    } else {
        ads_printf("Error getting range from %s|%s.\n", Application,
                                SpreadSheet);
        return;
    }

    /* Determine the maximum radius of any step on the shaft. */

    for (i = 0; i < shaftn; i++) {
        maxrad = max(maxrad, shaft[i].odiam / 2);
    }

    /* Save the current layer */

    ads_getvar("CLAYER", &clayer);

    /* First, create the three dimensional model of the shaft in the
       current drawing.  */

    if (updpos) {
        Cpoint(pos, updpoint);
        updpos = FALSE;
    } else {
        ads_initget(1 + 8 + 16, NULL);
        if (ads_getpoint(NULL, "\nPosition: ", pos) != RTNORM)
            return;
    }

    /* Draw the shaft.  Solid cylinder steps are drawn as extruded
       circles, hollow steps as extruded doughnuts.  */

    celev = 0.0;
    CommandB();
    PushVar("EXPERT", sexp, 4, rint);
    PushVar("ELEVATION", selev, 0.0, rreal);
    PushVar("THICKNESS", sthick, 0.0, rreal);
    ads_command(RTSTR, "_UCS", RTSTR, "_SAVE", RTSTR, TempUCS, RTNONE);
    ads_command(RTSTR, "_UCS", RTSTR, "_ORIGIN", RT3DPOINT, pos, RTNONE);
    ads_command(RTSTR, "_UCS", RTSTR, "_Y", RTSTR, "<<90.0", RTNONE);
    ads_command(RTSTR, "_LAYER",
        RTSTR, "_THAW", RTSTR, LDimensionLayer,
        RTSTR, "_THAW", RTSTR, DDimensionLayer,
        RTSTR, "_SET", RTSTR, PartLayer,
        RTSTR, "", RTNONE);
    Spoint(pos1, 0.0, 0.0, 0.0);
    Spoint(pos2, 0.0, 0.0, 0.0);
    ads_command(RTSTR, "_POINT", RT3DPOINT, pos1, RTNONE);
    for (i = 0; i < shaftn; i++) {
        pos2[Z] = celev;
        ads_command(RTSTR, "_ELEV", RTREAL, celev, RTREAL,
            shaft[i].shleng, RTNONE);
        if (shaft[i].idiam > 0.0) {
            ads_command(RTSTR, "_DOUGHNUT", RTREAL, shaft[i].idiam,
                RTREAL, shaft[i].odiam, RT3DPOINT, pos1, RTSTR, "", RTNONE);
        } else {
            ads_command(RTSTR, "_CIRCLE", RT3DPOINT, pos2, RTSTR, "_D",
                RTREAL, shaft[i].odiam, RTNONE);
        }
        celev += shaft[i].shleng;
    }

    /* Now draw the dimensions for the  diameter  of  each  step.   We
       build  a table of the diameter dimensions used in the shaft and
       sort them in ascending order.  */

    ads_getvar("DIMTXT", &dimtxt);
    ads_command(RTSTR, "_LAYER", RTSTR, "_SET", RTSTR, DDimensionLayer,
        RTSTR, "", RTNONE);
    ads_command(RTSTR, "_UCS", RTSTR, "_Z", RTSTR, "<<90", RTNONE);

    celev = 0.0;
    for (i = k = 0; i < shaftn; i++) {
        diadih[k] = celev + shaft[i].shleng / 2;
        diadim[k++] = shaft[i].odiam;
        if (shaft[i].idiam > 0.0) {
            diadih[k] = diadih[k - 1];
            diadim[k++] = shaft[i].idiam;
        }
        celev += shaft[i].shleng;
    }
    /* Bubble (ugh, but the array isn't that big) sort the diameters
       in ascending order. */
    for (i = 0; i < k - 1; i++) {
        for (j = i + 1; j < k; j++) {
            if (diadim[i] > diadim[j]) {
                ads_real r = diadim[j];
                diadim[j] = diadim[i];
                diadim[i] = r;

                r = diadih[j];
                diadih[j] = diadih[i];
                diadih[i] = r;
            }
        }
    }

    /* Draw the diameter dimensions for each segment. */

    Spoint(pos1, 0.0, 0.0, celev);
    Spoint(pos2, 0.0, 0.0, celev);
    Spoint(pos3, maxrad * 1.2, 0.0, celev);
    for (i = 0; i < k; i++) {
        pos1[Z] = pos2[Z] = pos3[Z] = diadih[i];
        ads_command(RTSTR, "_ELEV", RTREAL, diadih[i],
            RTREAL, 0.0, RTNONE);
        pos1[Y] = -(pos2[Y] = diadim[i] / 2);
        ads_command(RTSTR, "_DIM1", RTSTR, "_VERT",
            RT3DPOINT, pos1, RT3DPOINT, pos2,
            RT3DPOINT, pos3, RTSTR, "", RTNONE);
        /* Offset dimension if not identical to last one drawn.  */
        if ((i >= (k - 1)) || (diadim[i + 1] != diadim[i]))
            pos3[X] += dimtxt.resval.rreal * 5;
    }

    PopVar("THICKNESS", sthick);
    PopVar("ELEVATION", selev);

    /* Affix the horizontal dimensions giving the step lengths. */

    ads_command(RTSTR, "_LAYER", RTSTR, "_SET", RTSTR, LDimensionLayer,
        RTSTR, "", RTNONE);
    ads_command(RTSTR, "_UCS", RTSTR, "_RESTORE", RTSTR, TempUCS, RTNONE);
    ads_command(RTSTR, "_UCS", RTSTR, "_ORIGIN", RT3DPOINT, pos, RTNONE);
    Spoint(pos, 0.0, 0.0, 0.0);
    Cpoint(pos1, pos);
    Cpoint(pos2, pos);
    pos1[X] += shaft[0].shleng;
    pos2[Y] += maxrad * 1.2;
    ads_command(RTSTR, "_DIM", RTSTR, "_HORIZ", RT3DPOINT, pos,
        RT3DPOINT, pos1, RT3DPOINT, pos2, RTSTR, "", RTNONE);
    for (i = 1; i < shaftn; i++) {
        pos1[X] += shaft[i].shleng;
        ads_command(RTSTR, "_CONT", RT3DPOINT, pos1, RTSTR, "", RTNONE);
    }
    ads_command(RTSTR, "_EXIT", RTNONE);

    /* Now draw a 3D Polyline for the centreline of the  shaft.   This
       gets  modified  as the shaft is stretched and is used to export
       the lengths.  */

    ads_command(RTSTR, "_3Dpoly", RTNONE);
    Spoint(pos, 0.0, 0.0, 0.0);
    celev = 0.0;
    for (i = 0; i < shaftn; i++) {
        pos[X] = celev;
        ads_command(RT3DPOINT, pos, RTNONE);
        celev += shaft[i].shleng;
    }
    /* Tack on end of shaft point to centreline. */
    pos[X] = celev;
    ads_command(RT3DPOINT, pos, RTNONE);
    ads_command(RTSTR, "", RTNONE);

    ads_command(RTSTR, "_UCS", RTSTR, "_RESTORE", RTSTR, TempUCS, RTNONE);
    ads_command(RTSTR, "_UCS", RTSTR, "_DELETE", RTSTR, TempUCS, RTNONE);
    ads_command(RTSTR, "_LAYER", RTSTR, "_SET",
        RTSTR, clayer.resval.rstring, RTSTR, "", RTNONE);
    free(clayer.resval.rstring);

    PopVar("EXPERT", sexp);
    CommandE();
    imported = TRUE;                  /* Mark model in this drawing */
}

/*  UPDATE  --  Update an existing definition in the drawing.  */

static void update(void)
{
    int i;
    struct resbuf rbet, rbl;
    ads_name vname;
    long l;
    static char *delayers[] = {
        PartLayer,
        LDimensionLayer,
        DDimensionLayer
    };

    if (!importedp())
        return;

    /* Build the SSGET entity buffer to filter entities on the named
       layer. */

    rbet.restype = 0;                 /* Entity type */
    rbet.resval.rstring = "POINT";
    rbet.rbnext = &rbl;
    rbl.restype = 8;                  /* Layer name */
    rbl.rbnext = NULL;
    rbl.resval.rstring = PartLayer;

    /* Obtain the origin point for the part and set it to be the
       origin of the updated version.  */

    if (ads_ssget("X", NULL, NULL, &rbet, vname) == RTNORM) {

        if (ads_sslength(vname, &l) < 0)
            l = 0;

        if (l > 0) {
            ads_name ename;

            if (ads_ssname(vname, 0L, ename) == RTNORM) {
                struct resbuf *eb = ads_entget(ename);

                if (eb != NULL) {
                    ads_real *rp = assoc(eb, 10)->resval.rpoint;
                    Cpoint(updpoint, rp);
                    updpos = TRUE;
                }
            }
        }
        ads_ssfree(vname);            /* Release selection set */
    }

    for (i = 0; i < ELEMENTS(delayers); i++) {
        rbl.resval.rstring = delayers[i];
        if (ads_ssget("X", NULL, NULL, &rbl, vname) == RTNORM) {

            /* Delete all definitions found. */

            if (ads_sslength(vname, &l) < 0)
                l = 0;

            if (l > 0) {
                CommandB();
                ads_command(RTSTR, "_ERASE", RTPICKS, vname,
                    RTSTR, "", RTNONE);
                CommandE();
            }
            ads_ssfree(vname);        /* Release selection set */
        }
    }

    /* O.K.  Now that we've gotten rid of the old model, import the
       new one from the spreadsheet.  */

    antirecurso = TRUE;
    import();
    antirecurso = FALSE;
}

/*  VLENGTH  --  View length dimensions.  */

static void vlength(void)
{
    if (!importedp())
        return;

    CommandB();
    ads_command(RTSTR, "_LAYER", RTSTR, "_FREEZE", RTSTR, DDimensionLayer,
        RTSTR, "_THAW", RTSTR, LDimensionLayer, RTSTR, "", RTNONE);
    ads_command(RTSTR, "_VPOINT", RTSTR, "0,0,1", RTNONE);
    CommandE();
}

/*  VDIAM  --  View diameter dimensions.  */

static void vdiam(void)
{
    if (!importedp())
        return;

    CommandB();
    ads_command(RTSTR, "_LAYER", RTSTR, "_FREEZE", RTSTR, LDimensionLayer,
        RTSTR, "_THAW", RTSTR, DDimensionLayer, RTSTR, "", RTNONE);
    ads_command(RTSTR, "_VPOINT", RTSTR, "1,0,0", RTNONE);
    CommandE();
}

/*  V3D  --  View three dimensional part.  */

static void v3d(void)
{
    if (!importedp())
        return;

    CommandB();
    ads_command(RTSTR, "_LAYER", RTSTR, "_FREEZE", RTSTR, LDimensionLayer,
        RTSTR, "_FREEZE", RTSTR, DDimensionLayer, RTSTR, "", RTNONE);
    ads_command(RTSTR, "_VPOINT", RTSTR, "1,-1,1", RTNONE);
    CommandE();
}

/*  EXPORT  --  Export the current model in the drawing (possibly
                modified as a result of the STRETCH command) to the
                spreadsheet.  */

static void export(void)
{
    int i;
    struct resbuf rbet, rbl;
    ads_name vname;
    long l;

    if (!importedp())
        return;

    /* Check for open channel. Try to open one if none open. */
    if (!chk_dde(TRUE))
        return;

    /* Build the SSGET entity buffer to filter entities on the named
       layer.  */

    rbet.restype = 0;                 /* Entity type */
    rbet.resval.rstring = "POLYLINE";
    rbet.rbnext = &rbl;
    rbl.restype = 8;                  /* Layer name */
    rbl.rbnext = NULL;
    rbl.resval.rstring = LDimensionLayer;

    /* Find the centreline Polyline for the shaft. */

    if (ads_ssget("X", NULL, NULL, &rbet, vname) == RTNORM) {

        if (ads_sslength(vname, &l) < 0) 
            l = 0;

        if (l > 0) {
            ads_name ename;
            ads_point tp, lp;

            if (ads_ssname(vname, 0L, ename) == RTNORM) {
                for (i = 0; ads_entnext(ename, ename) == RTNORM; i++) {
                    struct resbuf *eb = ads_entget(ename);

                    if ((eb == NULL) || (strcmp(assoc(eb, 0)->resval.rstring, 
                                        "VERTEX") != 0))
                        break;
                    Cpoint(tp, assoc(eb, 10)->resval.rpoint);
                    if (i > 0)
                        shaft[i - 1].shleng = ads_distance(lp, tp);
                    Cpoint(lp, tp);
                }
            }
        }
        ads_ssfree(vname);         /* Release selection set */

#ifdef DEBUGDUMP
        ads_printf("Shaftn = %d\n", shaftn);
        for (i = 0; i < shaftn; i++) {
            ads_printf("%d:  %f %f %f\n", i, shaft[i].shleng,
                shaft[i].idiam, shaft[i].odiam);
        }
#endif

        /* Transmit the values in the current model table back
           to the corresponding spreadsheet cells.  */

        PokeRealArray(pdde, (ads_real *)shaft, ModelRow, ModelColumn,
                      shaftn,  3);

        /*      This is another way to do it, but it's slower.

        for (i = 0; i < shaftn; i++) {
            PokeReal(pdde, shaft[i].shleng, ModelRow + i, ModelColumn);
            PokeReal(pdde, shaft[i].idiam, ModelRow + i, ModelColumn + 1);
            PokeReal(pdde, shaft[i].odiam, ModelRow + i, ModelColumn + 2);
        }
        */

        /* Now that we've exported the values, we might as well update
           the drawing.  If the advise hot link is in effect, the update
           will occur "automatically", but for some reason we lose 
           the Excel link, so re-establish it. */

        if (pDdeData->flags.advise)
            DdeAdviseCur(pdde, TRUE, WARMLINK);
        else
            update();
    } else {
        ads_printf("Unable to locate centreline of shaft.\n");
    }
}


/* Equivalent of AutoLISP assoc function. 
   Find the sub-entity with a certain group code and return 
   a pointer to it. 
*/
static struct resbuf *assoc(struct resbuf *ent, int type)
{
    struct resbuf *pent;

    for (pent = ent; pent != NULL; pent = pent->rbnext) {
        if (pent->restype == type) {
            return pent;
        }
    }
    return NULL;
}



/* LISTFUNC  --  List function names in this module.  */

static void listfunc(void)
{
    int idx;
    char *cname;

    for (idx = 0; idx < ELEMENTS(cmdtab); ++idx) {
        cname = cmdtab[idx].cmdname;
        ads_printf(cname);
        ads_printf("\n");
    }
    ads_retint(idx);
}


/* SHAFTDIALOG -- Run DDE dialog.  Allows turning on/off hot link. */ 

static void shaftdialog(void)
{
    DdeDialog(adsw_hWnd);
}


/* Check for channel, attempt to start spreadsheet and initiate 
   DDE channel, put up dialog if fail and start over */

static PDDE chk_dde(int pokeflag)
{
    /* See if we have a channel with default app, topic */
    pdde = DdeFindChnl(Application, SpreadSheet);
    if (pdde == NULL) {
        /* Try to create link, put up dialog if unable */
        pdde = DdeDlgStart(adsw_hWnd, FALSE);
        if (!pdde)
            return NULL;
        /* Set up hot link item, command to AutoCAD, and function 
           to send the command. */
        DdeSetAdvise(pdde, cells, "update_shaft ", AdviseFunc);
        if (!pokeflag) {
            if (pDdeData->flags.advise)
                DdeAdviseCur(pdde, TRUE, WARMLINK);
        } else {
            if (pdde->ChannelState.advise_on)
                /* Turn off advise if we're about to send new data */
                DdeAdviseCur(pdde, FALSE, WARMLINK);
        }
    }
    return pdde;
}


/* Send command such as "update_shaft " to interrupt AutoCAD 
   to perform command (warm link function). */
static int AdviseFunc(PDDE pdde)
{
    /* Interrupt AutoCAD to execute our function */
    if (pdde && pdde->acadadvise) {
        DdeSendAcadCmd(pdde->acadadvise);
    }
    return 0;
}

