/*
        DdeConv (Entity Conversion to DDE/Clipboard Text Format)

        AutoCAD Drawing Entity DDE Conversion and Transfer Functions

          by Phil Ford

     ________________________________________________________________________

      (C) Copyright 1990-1994 by Autodesk, Inc.

      Permission to use, copy, modify, and distribute this software and its
      documentation for any purpose and without fee is hereby granted.  

      THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY. 
      ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF 
      MERCHANTABILITY ARE HEREBY DISCLAIMED.                                
     ________________________________________________________________________

*/


/*
See DDE.DOC.
*/

#include "options.h"

#include <stdio.h>
#include <math.h>

#include "adslib.h"

#include <process.h>
#include <string.h>
#include <memory.h>
#include <dos.h>
#include <ctype.h>

#include "winutil.h"
#include "ddewin.h"
#include "ddeconv.h"
#include "spreadsh.h"
#include "acaddefs.h"


extern ddeDATA DdeData;
extern char RealFmt[];

#define DEFSSWIDTH 64
#define CONST_MARK "Constant"         /* Used instead of handle for
                                         inserts of constant attr data */       
#define LIFE_FREQ 200                 /* Show sign of life, check ^C every
                                         so many rows of spreadsheet */ 

/* Put recognizable mark at end of DDE transfer to make it 
   easy to detect when importing back into AutoCAD */
#define ENDMARK -99                   /* numeric value of eot */        
static char EndString[] =  "-99\tEnd of DDE\t\t\t\t";
static char EndStr[] = "-99";
static char BlankRowStr[] = "\t\t\t\t\t";
static char TableString[] = "\tEnd of AutoCAD Tables\t\t\t\t";

#define STOPSCAN 10                   /* Number of blank lines to 
                                         signal stop inputting data */
#define COLWIDTH 4
#define BLANKWIDTH 6

/* AutoCAD table and block names, for tbl... functions.  NOTE:
   change defines VPORT, etc if table changes */
char *TableList[] = { "VPORT", "LTYPE", "LAYER", "STYLE",
                      "VIEW", "UCS", /* "DWGMGR", */ "BLOCK", 
                      "DIMSTYLE", NULL };

/* Set by "setddeformat" function in ddeapp.c.   This global
   controls in which format the drawing entities will be layed out.
   Handles, full list, are the default. */
static short Frmt = HANDLE_MODE;


/* Bits in "flags" below */
#define USE_NAME     0x0001
#define SINGLE_ROW   0x0002
#define DHANDLE      0x0008           /* handle converted to decimal */
#define VARWIDTH     0x0010           /* variable column width */

typedef struct tagFRMTTABLE {
    short flags;                        
    short entwidth;                   /* width in cell rows of this format. */
    short subent[FRMTMAX];            /* group codes of each sub-entity 
                                         desired to be converted, ending 
                                         the list with ENTERM */
} FRMTTABLE;


/* DDE/Clipboard Text Format Table
   Format index:
        ENAME_MODE  0
        HANDLE_MODE 1
        ATTR_MODE   2
        ATTR_MODE2  3
*/

FRMTTABLE FrmtTable[] = {

    { USE_NAME,     4, ALL_ENTS },       /* full entity conversion
                                         using entity names instead of 
                                         handles. */
    { 0,            4, ALL_ENTS },       /* full entity, but with handles 
                                         and not names */
    { SINGLE_ROW | DHANDLE | VARWIDTH, COLMAX,
                                G_HANDLE, G_TAGNAME, G_ATTRNAM, ENTTERM },
                                         /* 1: handle, and attribute 
                                         text only */
    { SINGLE_ROW | DHANDLE | VARWIDTH, COLMAX,
                                G_HANDLE, G_ATTRNAM, ENTTERM },
};


char AcFldSep[10] = " . ";
char AcRecSep[10] = "\n";
ads_real DdeTolerance = 0.001;        /* for EntCompare() and CmpReals()--
                                         used to decide whether entity has
                                         changed when being read in from
                                         text format. */


/* cell item to ads_real (double) conversions */
char PointScan[] = "%lf\t%lf\t%lf";   /* DDE text format of point:
                                         real<TAB>real<TAB>real */
char LongScan[] = "%lX";
char EnameScan[] = "%ul";
char HexFmt[] = "x%.8lx";



/* The global linked list of constant block attribute data.
   Gets freed on drawing quit, end, or save, which will cause
   the next request for attribute export to create a new
   one.  This avoids creating it on every request since a large
   symbol table can take awhile to scan for these. */   
BlkList *ConstBlkList = NULL;

static short locflatland = 0;
static short pointtype = RT3DPOINT;
static int DdeBrkFlg = 0;

#define SEQEND "SEQEND"
#define ATTRIB "ATTRIB"
#define INSERT "INSERT"
#define POLYLINE "POLYLINE"


/* Static Function Prototypes for ddeconv.c */

static UINT ModTextBlock(CONVDATA *ConvData);
static short ModTextEnt(CONVDATA *ConvData);
static short EntBuild(CONVDATA *ConvData);
static void SubEntBuild(RESBUF *pEnt, short GrpCode, char *Line, short etype);
static short EntRowBuild(CONVDATA *ConvData);
static int EntCompare(RESBUF *pDrawEnt, RESBUF *pDdeEnt);
static int CmpReals(ads_real RealAry1[], ads_real RealAry2[], int AryLen);
static char *GetNextLine(PHDATA pData);
static short ChkNextLine(PHDATA pData);
static int InFrmtTable(short GrpCode);
static void PokeEndString(PDDE pdde, UINT row, UINT col, UINT nrow);
static void SaveItem(PDDE pdde, UINT row, UINT col, UINT nrow);
static void SaveItemConv(CONVDATA *ConvData);
static int PokeBlankRowConv(CONVDATA *ConvData);
static int UpdateCols(CONVDATA *ConvData);
static int IsAttrFormat(void);
static void NewConv(CONVDATA *ConvData);




/* Set the DDE/AutoCAD format:
        ENAME_MODE  0
        HANDLE_MODE 1
        ATTR_MODE   2
        ATTR_MODE2  3
*/
void SetDdeFormat(int format)
{
    if (format < 0 || format >= ELEMENTS(FrmtTable))
        format = HANDLE_MODE;
    Frmt = format;   
}


/* Get the current DDE/AutoCAD format */

int GetDdeFormat(void)
{
    return Frmt;
}




/*


                AutoCAD Drawing Entity DDE Transfers



   Send all drawing tables and entities to spreadsheet, starting
   at row 
*/
UINT PokeDrawing(PDDE pdde, UINT sRow, UINT sCol)
{
    ads_name ename;
    UINT rows = 0, row = sRow, prows;
    short entstat;
    int attrflag = FALSE;

    if (Frmt != 0)
        SetHandles();
    ShowLife(0);                      /* Initialize sign of life */     

    /* default maximum cols used by any row for DXF type data */
    pdde->maxcols = FrmtTable[Frmt].entwidth; 

    /* Don't send table if we're sending attributes only */
    if (!IsAttrFormat()) {
        prows = PokeAllTables(pdde, row, sCol);
        row += prows;
        rows += prows;
    } else
        attrflag = TRUE;

    if (attrflag) {
        /* Get selection set of inserts with data following */
        entstat = GetInserts(ename, FALSE);
        if (entstat != RTNORM)
            return 0;
        prows = PokeSet(pdde, ename, NULL, row, sCol);
        ads_ssfree(ename);
        rows += prows;
        row += prows;
    } else {
        entstat = ads_entnext(NULL, ename);
        if (entstat != RTNORM)
            return rows;
        prows = PokeEntList(pdde, row, sCol, ename, ET_NORM);
        rows += prows;
        row += prows;
    }
    PokeEndString(pdde, sRow, sCol, rows);
    if (attrflag)
        return rows;
    else
        return rows + TABLEGAP;
}


/* Request data from remote DDE app and modify drawing
   with it, starting at row, limit of NumRows.   Returns 
   number of entities modified. 
*/
UINT ModDrawing(PDDE pdde, UINT row, UINT col, UINT NumRows)
{
    UINT RowCnt = 0;
    UINT RowTot = 0;
    UINT modcnt = 0;
    char Item[30];
    int entwidth;
    int more = TRUE;

    entwidth = ((FrmtTable[Frmt].flags & VARWIDTH) ? pdde->maxcols :
                                FrmtTable[Frmt].entwidth);
    if (entwidth <= 0)
        /* User could be reading in a spreadsheet from a previous
           session.  We should probably store the total rows and
           columns in the first row of the spreadsheet, but for 
           now, just read in a wide swath. */
        pdde->maxcols = entwidth = DEFSSWIDTH;
    ShowLife(0);                      /* Initialize sign of life */     
    while (more) {
        /* start at row, col, REQROWS rows, 4 columns wide */
        CellRange(Item, row, col, REQROWS, entwidth);
        /* If an entity straddles the buffer boundary, we'll pick 
           it up on the next loop since the row counter wasn't
           incremented past the beginning of the entity.  That is,
           the next DDE request will start with the beginning of
           the last entity in the current request buffer. */
        modcnt += ReqMod(pdde, Item, &RowCnt, &more);
        if (!RowCnt)
            break;
        RowTot += RowCnt;
        row += RowCnt;
        if (ShowLife(row))
            break;
    }
    return modcnt;
}



/*

            DDE and AutoCAD Symbol Table Functions

*/


/* Send all the table data to remote DDE spreadsheet, starting
   at sRow, sCol.   Return the number of rows taken.  
*/
UINT PokeAllTables(PDDE pdde, UINT sRow, UINT sCol)
{
    int rows = 0, row = sRow, idx, prows;
    char *table;
    char Cell[25];

    ShowLife(0);                      /* Initialize sign of life */     
    for (idx = 0; (table = TableList[idx]) != NULL; ++idx) {
        prows = PokeTable(pdde, idx, NULL, row, sCol, FALSE);
        rows += prows;
        row += prows;
    }
    PokeBlankRow(pdde, row, sCol);
    CellRange(Cell, row+1, sCol, 1, COLWIDTH);
    DdePokeString(pdde, Cell, TableString);
    PokeBlankRow(pdde, row+2, sCol);
    PokeBlankRow(pdde, row+3, sCol);
    rows += TABLEGAP;                 /* leave space between
                                         tables and drawing ents */
    return rows;
}



/* Send data from one table to DDE app. 
   TableNum is VPORT, VIEW, etc. 
*/
UINT PokeTable(PDDE pdde, int TableNum, char *item, UINT sRow,
                 UINT sCol, int WriteEnd)
{
    UINT prows, row, rows;
    int rewind = TRUE;
    RESBUF *pEnt, *pTab;
    char *table;
    int etype;
    int nRows, nCols;

    if ((sRow == 0 || sCol == 0) && item != NULL)
        GetCellRange(item, &sRow, &sCol, &nRows, &nCols);

    row = sRow;
    rows = 0;

    if (WriteEnd) {
        if (!IsAttrFormat())
            pdde->maxcols = 4;
        if (Frmt != ENAME_MODE)
            SetHandles();
        ShowLife(0);
    }

    etype = ET_TBL + TableNum;
    table = TableList[TableNum];
    for (rewind = TRUE; pEnt = ads_tblnext(table, rewind); rewind = FALSE) {
        prows = EntPokeCell(pdde, row, sCol, pEnt, etype);
        rows += prows;
        row += prows;
        if (TableNum == BLOCK) {
            for (pTab = pEnt; pTab; pTab = pTab->rbnext) {
                if (pTab->restype != SEQMARK)
                    continue;
                /* For -2 in the BLOCK table, the ename points to the
                   start of the block entities */       
                prows = PokeEntList(pdde, row, sCol, pTab->resval.rlname, 
                                                etype);
                row += prows;
                rows += prows;
                break;
            }
        }           
        ads_relrb(pEnt);
        if (DdeBrkFlg)
            break;
    }
    if (WriteEnd)
        PokeEndString(pdde, sRow, sCol, rows);
    return rows;
}



/*

            DDE and AutoCAD Selection Set Functions

*/




/* Send selection set data to DDE app, according to
   format, Frmt.  Caller must still call ads_ssfree(). 
*/
UINT PokeSet(PDDE pdde, ads_name ename, char *item, UINT sRow, UINT sCol)
{
    CONVDATA ConvData;
    UINT rows = 0, row = sRow, prows;
    int cont, entstat;
    RESBUF *pEnt;

    if (Frmt != 0)
        SetHandles();

    ShowLife(0);                      /* Initialize sign of life */     
    pdde->maxcols = (FrmtTable[Frmt].flags & VARWIDTH) ?  0 :
                                FrmtTable[Frmt].entwidth;
    NewConv(&ConvData);
    ads_name_set(ename, ConvData.sset);
    ConvData.etype = ET_SSET;
    ConvData.pdde = pdde;
    if (!item) {
        ConvData.row = ConvData.row1 = sRow;
        ConvData.col1 = sCol;
    }
    if (IsAttrFormat()) {
        prows = PokeAttrSet(&ConvData);
        return prows;
    }

    /* Poke unfiltered sset */
    for (cont = TRUE; cont; ) {
        pEnt = NULL;
        if (ConvData.status.SEQ) {
            /* normal entity or in sset sequence (attribute, polyline) */
            if (!ConvData.status.FIRST) {
                entstat = ads_entnext(ConvData.ename, ConvData.ename);
                if (entstat != RTNORM)
                    break;
            }
            pEnt = ads_entget(ConvData.ename);  /* get start node of linked
                                              list of sub-entities */
            if (pEnt == NULL)
                break;
            if (IsEntType(pEnt, SEQEND)) 
                /* Found end of sequence of polylines or attributes */
                ConvData.status.SEQ = FALSE;
        } else {
            /* Selection Set */ 
            entstat = ads_ssname(ConvData.sset, ConvData.ssidx++, 
                                                        ConvData.ename);
            if (entstat != RTNORM)
                break;
            /* get start node of linked list of sub-entities */
            pEnt = ads_entget(ConvData.ename); 
            if (!pEnt)
                break;
            /* See if we need to follow a chain of polylines or 
               block attributes */
            ConvData.status.SEQ = (IsSequence(pEnt) != 0);
        } 
        if (pEnt == NULL)
            break;
        prows = EntPokeCell(pdde, row, sCol, pEnt, ET_SSET);
        ads_relrb(pEnt);
        row += prows;
        rows += prows;
        ConvData.status.FIRST = FALSE;
        if (ShowLife(row))
            break;
    }
    if (rows)
        PokeEndString(pdde, sRow, sCol, rows);
    return rows;
}




/* Poke selection set of attribute data */

int PokeAttrSet(CONVDATA *ConvData) 
{
    int entstat;
    int rows = 0;

    if (ConstBlkList == NULL)
        CreateConstList(&ConstBlkList);
    PokeConstList(ConvData->pdde, ConstBlkList);
    ConvData->status.ATTR_FILTER = TRUE ;
    for (ConvData->ssidx = 0; ; ++ConvData->ssidx) {
        entstat = ads_ssname(ConvData->sset, ConvData->ssidx, 
                                                        ConvData->ename);
        if (entstat != RTNORM)
            break;
        PokeAttr(ConvData);
        if (DdeBrkFlg)
            break;
    }
    PokeEndStringConv(ConvData);

    /* You could speed things up by not freeing this until asked 
       by some dialog button or whatever.  */
    FreeConstList(&ConstBlkList);
    return ConvData->TotRows;
}

/* Create ConvData->textlist and poke the list. */

int PokeAttr(CONVDATA *ConvData)
{
    int stat;
    RESBUF *pEnt;
    int idx, ary_cnt;
    
    ConvData->textlist = NULL;
    pEnt = ads_entget(ConvData->ename);
    if (pEnt == NULL)
        return 0;
    /* See if we have an array from a MINSERT */
    ary_cnt = InsertArrayCnt(pEnt);
    ads_relrb(pEnt);
    stat = CreateAttrList(ConvData);
    if (stat && (ConvData->textlist != NULL)) {
        for (idx = 0; idx < ary_cnt; ++idx)
            stat = PokeConvList(ConvData);
    }
        /* Free Text List */
    RemoveText((void **)&ConvData->textlist, NULL);
    ShowLife(ConvData->row);
    return stat;
}


/* Create linked list of strings until SEQEND found.  Returns TRUE 
   if ename is valid.  "ename" cannot be NULL.  Check ConvData->textlist
   to see if an attribute was added to list. */

int CreateAttrList(CONVDATA *ConvData)
{
    RESBUF *pEnt;
    int found = FALSE, ary_len = 0;
    Tlink **ptlist;
    int idx, grpcode, seqflag;

    if ((pEnt = ads_entget(ConvData->ename)) == NULL)
        return 0;
    seqflag = (IsSequence(pEnt) == SEQ_ATTR);
    ary_len = ConstList(&ConvData->textlist, pEnt);
    if (seqflag == 0) {
        /* No attributes follow or constant attribs */
        ads_relrb(pEnt);
        return ary_len;
    }
    ptlist = &ConvData->textlist;
    ConvData->cols = 0;

    /* Loop until SEQEND is found, or an attribute is found for
       non sequence */
    for ( ; ; ) {       
        if (IsEntType(pEnt, SEQEND))
            break;      
        else if (IsEntType(pEnt,  ATTRIB)) {
            /* We've found an attribute */
            found = TRUE;
            ary_len = 1;
            for (idx = 0; idx < FRMTMAX; ++idx) {
                grpcode = FrmtTable[Frmt].subent[idx];
                if (grpcode == ENTTERM)
                    break;
                AddEntList(ConvData, pEnt, grpcode);
            }
            
        }
        ads_relrb(pEnt);
        pEnt = NULL;    
        if (!seqflag && found)
            break;
        if (ads_entnext(ConvData->ename, ConvData->ename) != RTNORM)
            break;        
        pEnt = ads_entget(ConvData->ename);
                                      /* get start node of linked       
                                         list of sub-entities */            
        if (!pEnt)
            break;  
    }    
    if (pEnt)
        /* SEQEND found */
        ads_relrb(pEnt);            
    UpdateCols(ConvData);
    return 1;
}

/* Look up constant attribute associated with this INSERT, if any, 
   and add text to the linked list of text items.  Returns array 
   length (MINSERT) or 1 for normal INSERT
*/
int ConstList(Tlink **ptlist, RESBUF *pEnt)
{
    BlkList *blkLink;
    BlkAttr *attrLink;
    RESBUF *pIns;
    char *block;
    int stat = FALSE;
    int ary_len = 0;

    if (!IsEntType(pEnt, INSERT))
        return 0;
    pIns = assoc(pEnt, G_BLNAME);
    if (pIns == NULL)
        return 0;
    block = pIns->resval.rstring;

    blkLink = InTextList(ConstBlkList, block);
    if (blkLink == NULL)
        return 0;

    stat = TRUE;            
    ary_len = max(blkLink->ary_cnt, 1);     
    for (attrLink = blkLink->attrlist; attrLink != NULL;            
                                attrLink = attrLink->next) {                
        char cell[25];              
        UINT col = attrLink->col;                   

        /* Insert the cell reference to the constant data                   
           decimal HANDLE in the first row, rather than the constant            
           attribute value itself.  Eg, "=AF1".  This is                    
           the purpose of spreadsheets. */                  
        CellRef(cell, 1, col);              
        AddText((void **)ptlist, cell);             
        ++col;              

        /* Show it's a constant attribute.  For variable attributes,            
           this slot would hold the handle (eg "x45E"). */                  
        AddText((void **)ptlist, CONST_MARK);               
        ++col;              

        /* Insert the cell reference to the constant data                   
           TAG in the first row if tags are desired */              
        if (Frmt == ATTR_MODE && attrLink->tag != NULL) {                   
            /* Put the tag there too */                     
            CellRef(cell, 1, col);                  
            AddText((void **)ptlist, cell);         
            ++col;                  
        }                   

        /* Insert the cell reference to the constant data                   
           VALUE in the first row */                
        CellRef(cell, 1, col);              
        AddText((void **)ptlist, cell);             
    } 
    return ary_len;
}





/* Create a list of block names and the constant attribute value
   in the symbol table.  Pass in address of pointer to BlkList that was 
   initialized to NULL.  blkList will be filled in with the
   block name and constant value list if found.  Call FreeConstList
   to free the whole list.

   BLOCK1 - CONSTANT VALUE 1 -- VALUE 2 -- VALUE 3
     |
     |
   BLOCK2 - CONSTANT VALUE 1 -- VALUE 2
     |
     |
   BLOCK3 - CONSTANT VALUE 1 -- VALUE 2  -- VALUE 3
     |

   Eg,
   BlkList *blkList = NULL;
   CreateConstList(&blkList);
*/
void CreateConstList(BlkList **blkList)
{
    RESBUF *pEnt, *pBlkName;
    int first = TRUE;
    int first2 = TRUE;
    int etype, rewind;
    BlkList *blkLink;
    BlkAttr *attrList, *attrLink;
    char *blk;
    int col = 1;

    etype = ET_TBL + BLOCK;
    for (rewind = TRUE; pEnt = ads_tblnext("BLOCK", rewind); rewind = FALSE) {
        pBlkName = assoc(pEnt, G_BLNAME);       
        if (pBlkName == NULL) {
            ads_relrb(pEnt);
            continue;
        }
        blk = pBlkName->resval.rstring;
        attrList = NULL;
        attrLink = CreateConstSubList(&attrList, NULL, pEnt, &col);
        if (attrLink != NULL) {
            /* Find or create a block link to attach this list to */
            blkLink = FindOrCreateBlock(blkList, blk);
            if (blkLink != NULL)
                blkLink->attrlist = attrList;
        }
        ads_relrb(pEnt);
    }
}




/* Create a list of constant attribute data for a certain block.
   Pass in address of pointer to AttrList that was 
   initialized to NULL.  attrList will be filled in with the
   constant value found.  "blkname" can be NULL if the first entity
   is already known.  Returns last link added, NULL means no constant
   data found.

   [BLOCK] - CONSTANT VALUE 1 -- VALUE 2 -- VALUE 3

   Eg,
   BlkAttr *attrList = NULL;
   if (CreateConstSubList(&attrList, "MYBLOCK", NULL, 1) != NULL) ...
*/
BlkAttr *CreateConstSubList(BlkAttr **attrList, char *blkname, RESBUF *pBlk, 
                int *col)
{
    RESBUF *pEnt, *pBlkData, *pAttr, *pConst;
    ads_name en;
    int first = TRUE;
    int stat;
    BlkAttr *attrLink = NULL;
    int ary_cnt = 1;

    if (blkname == NULL)
        pEnt = pBlk;
    else 
        pEnt = ads_tblsearch("BLOCK", blkname, FALSE);
    if (pEnt == NULL)
        return NULL;

    /* For -2 in the BLOCK table, the ename points to the
       start of the block entities in an entget chain that
       is terminated by a non-RTNORM from entget */     
    pBlkData = assoc(pEnt, SEQMARK);
    if (pBlkData == NULL) {
        if (blkname != NULL)
            ads_relrb(pEnt);    
        return NULL;
    }   
    ads_name_set(pBlkData->resval.rlname, en);  
    first = TRUE;       
    while (TRUE) {      
        /* We're at the start of the block data */              
        if (!first) {   
            stat = ads_entnext(en, en); 
            if (stat != RTNORM) 
                break;  
        } else  
            first = FALSE;      
        pAttr = ads_entget(en); 
        if (pAttr == NULL)      
            break;      
        if (!IsEntType(pAttr, "ATTDEF")) {      
            ads_relrb(pAttr);   
            continue;   
        }       
        pConst = assoc(pAttr, G_INT);           
        if (!(pConst->resval.rint & 0x02)) {    
            ads_relrb(pAttr);   
            continue;   
        }       
        /* We have a constant attdef */ 
        pConst = assoc(pAttr, G_PNAME);         
        if (pConst == NULL)     {       
            ads_relrb(pAttr);   
            continue;   
        }       
        /* Add the attribute data to the attrlist in the block  
           link */      
        attrLink = AddLink((void **)attrList, sizeof(BlkAttr)); 
        attrLink->value = _strdup(pConst->resval.rstring);       
        pConst = assoc(pAttr, G_HANDLE);                
        if (pConst != NULL)     
            attrLink->handle = _strdup(pConst->resval.rstring);  
        pConst = assoc(pAttr, G_TAGNAME);               
        if (pConst != NULL)     
            attrLink->tag = _strdup(pConst->resval.rstring);     
        attrLink->col = *col;   
        /* Include tag field for ATTR_MODE.     
           DECIMAL HANDLE - xHANDLE - TAG - ATTR VALUE or       
           DECIMAL HANDLE - xHANDLE - ATTR VALUE */     
        *col += (Frmt == ATTR_MODE ? 4 : 3);    
        ads_relrb(pAttr);       
    }   
    if (blkname != NULL)
        ads_relrb(pEnt);        
    return attrLink;
}        


/* Find or create a block link 
*/
BlkList *FindOrCreateBlock(BlkList **blkList, char *blk)
{
    BlkList *blkLink = NULL;

    /* We don't already have a link for this block.  Create     
       the block link first. */ 
    if (*blkList == NULL ||     
                        (blkLink = InTextList((void *)*blkList, blk)) == NULL) {        
        blkLink = AddLink((void **)blkList, sizeof(BlkList));   
        blkLink->block = _strdup(blk);   
    }   
    return blkLink;
}


/* Free Constant Attribute list.  Pass NULL for the global list.
*/
void FreeConstList(BlkList **blkList)
{
    BlkList *blkLink, *blkNext;

    if (blkList == NULL)
        blkList = &ConstBlkList;

    for (blkLink = *blkList; blkLink != NULL; blkLink = blkNext) {
        blkNext = blkLink->next;
        if (blkLink->block != NULL) {
            /* Free up the attribute list */
            FreeConstSubList(&blkLink->attrlist);
            free(blkLink->block);               
        }    
        free(blkLink);
    }
    *blkList = NULL;
}

/* Free Constant Attribute list 
*/
void FreeConstSubList(BlkAttr **attrList)
{
    BlkAttr *attrLink, *attrNext;

    /* Free up the attribute list */
    for (attrLink = *attrList; attrLink != NULL; attrLink = attrNext) {
        attrNext = attrLink->next;      
        if (attrLink->tag != NULL)      
            free(attrLink->tag);                        
        if (attrLink->value != NULL)    
            free(attrLink->value);                      
        if (attrLink->handle != NULL)   
            free(attrLink->handle);                     
        free(attrLink); 
    }           
    *attrList = NULL;
}


/* Poke the constant definitions from a block list into the first row */

void PokeConstList(PDDE pdde, BlkList *blkList)
{
    BlkList *blkLink;

    /* Walk the block list */
    for (blkLink = blkList; blkLink != NULL; blkLink = blkLink->next) {
        /* Walk the attribute list poking certain items into the 
           first row of the spreadsheet */
        if (blkLink->attrlist != NULL)
            PokeConstAttrList(pdde, blkLink->attrlist);
    }
}


/* Poke the constant definitions from an attr list into the first row */

void PokeConstAttrList(PDDE pdde, BlkAttr *attrList)
{
    UINT col;
    BlkAttr *attrLink;
    char cell[25];
    char handle_str[25];

    for (attrLink = attrList; attrLink != NULL; attrLink = attrLink->next) {
        col = attrLink->col;
        DecimalHandle(attrLink->handle, handle_str);
        CellItemName(cell, 1, col);
        DdePokeString(pdde, cell, handle_str);
        ++col;
        handle_str[0] = HPREFIX;
        strcpy(handle_str+1, attrLink->handle);
        CellItemName(cell, 1, col);
        DdePokeString(pdde, cell, handle_str);
        ++col;
        if (Frmt == ATTR_MODE && attrLink->tag) {
            CellItemName(cell, 1, col);
            DdePokeString(pdde, cell, attrLink->tag);
            ++col;
        }
        if (attrLink->value) {
            CellItemName(cell, 1, col);
            DdePokeString(pdde, cell, attrLink->value);
        }
    }
}


/* Add filtered entity string to linked list */

int AddEntList(CONVDATA *ConvData, RESBUF *pEnt, int grpcode)
{                   
#define HANDLEN 32
    RESBUF *pSubEnt;
    char *str;
    Tlink **ptlist = &ConvData->textlist;

    pSubEnt = assoc(pEnt, grpcode);                 
    if (pSubEnt == NULL)
        str = "?";
    else
        str = pSubEnt->resval.rstring;
    if (grpcode == G_HANDLE && pSubEnt) {       
        char buf[HANDLEN+1];    
        /* Format handle to 10 for database ID */               
        DecimalHandle(str, buf);                                    
        AddText((void **)ptlist, buf);                              
        ++ConvData->cols;                               
        /* Hex rep of handle as presented                           
           by AutoCAD, with preceeding "x" */                       
        buf[0] = HPREFIX;                               
        strzcpy(buf+1, str, HANDLEN-1);                             
        AddText((void **)ptlist, buf);                              
        ++ConvData->cols;                               
    } else {                    
        AddText((void **)ptlist, str);                  
        ++ConvData->cols;                       
    }                   
    return TRUE;
}                   
                    



/* Poke a horizontal row of strings from the list based on ConvData
   values: pdde, textlist, row, & col1. */

int PokeConvList(CONVDATA *ConvData) 
{
    int stat;
    PDDE pdde = ConvData->pdde;

    int listlen = GetListLen(ConvData->textlist);
    stat = PokeList(pdde, ConvData->textlist, ConvData->row, 
                ConvData->col1, 1, listlen);
    if (stat) {
        IncRow(ConvData, 1);
        if (listlen > (int)pdde->maxcols)
            pdde->maxcols = listlen;
    }
    return stat;
}



 

/* Put a list of entities into spreadsheet cell at row and col.
   Returns number of rows (length of list). 
*/
UINT PokeEntList(PDDE pdde, UINT Row, UINT Col, ads_name ename, int etype)
{
    ads_name en;
    int first = TRUE, status;
    int row = Row, rows = 0, prows;
    RESBUF *pEnt;

    ads_name_set(ename, en);
    while (TRUE) {
        if (!first) {
            status = ads_entnext(en, en);
            if (status != RTNORM)
                break;
        } else
            first = FALSE;
        pEnt = ads_entget(en);
        if (pEnt == NULL)
            break;
        prows = EntPokeCell(pdde, row, Col, pEnt, etype);
        ads_relrb(pEnt);
        rows += prows;
        row += prows;
        if (ShowLife(row))
            return rows;
    }
    return rows;
}


/* Put an entity into spreadsheet cell at row and col.
   Returns number of rows (length of list) 
*/
UINT EntPokeCell(PDDE pdde, UINT Row, UINT Col, RESBUF *pEnt, int etype)
{
    UINT nRows;
    char Cell[25];
    short DdeResult;

    nRows = GetEntLen(pEnt);
    CellRange(Cell, Row, Col, nRows, ENT_WIDTH);
    DdeResult = EntPoke(pdde, Cell, pEnt, etype);
    if (DdeResult)
        return nRows;
    else
        return 0;
}



/* Put an entity into remote app using DDE item of ItemStr.
   Find number of rows with GetEntLen. 
   "etype" is ET_RB, ET_NORM, ET_SSET, or ET_TBL. 
*/
UINT EntPoke(PDDE pdde, char *ItemStr, RESBUF *pEnt, int etype)
{
    char *DispBuff;
    UINT DdeResult;

    DispBuff = EntListFormat(pEnt, etype);
    if (!DispBuff)
        return 0;
    /* get formatted string of entities */
    DdeResult = DdePokeString(pdde, ItemStr, DispBuff);
    free(DispBuff);
    return DdeResult;
}


/*
         DDE Receive (request) Entity Functions

   Some functions return the number of entities modified.
   Others the number of rows received, so the next request
   can start from the end of the previous request.  
*/


/* Request data from spreadsheet Item, and update
   entities with it that have changed.  Returns the
   number entities modified.  RowCnt gets the number
   of rows (sub-entities) used.   The row counter can
   be bumped by this amount for next call.
*/
UINT ReqMod(PDDE pdde, char *Item, UINT *RowCnt, int *more)
{
    UINT mod;
    DWORD result;
    CONVDATA ConvData;
    RESBUF *pEnt = NULL;

    NewConv(&ConvData);
    ConvData.etype = ET_NORM;
    ConvData.ppEnt = &pEnt;

    *more = 0;
    *RowCnt = 0;
    if (IsAttrFormat()) 
        ConvData.status.ATTR_FILTER = TRUE;
    result = DdeReqString(pdde, Item);
    if (!pdde->DataSize)
        return 0;
    /* End of spreadsheet data: <CR><LF><EOS> */
    if (!strcmp(pdde->pData, CrLf)) {
        DdeFreeData(pdde);
        return 0;
    }
    ConvData.pBuff = pdde->pData;
    mod = ModTextBlock(&ConvData);
    *RowCnt = ConvData.TotRows;
    DdeFreeData(pdde);
    if (!ConvData.status.END_ALL)
        *more = 1;
    return mod;
}


/* Using a text stream, pData, modify entities in
   the drawing.  Return number modified, row count in RowCnt. 
*/
UINT ModTextStream(PHDATA pData, UINT *RowCnt)
{
    UINT mod;
    CONVDATA ConvData;
    RESBUF *pEnt = NULL;

    NewConv(&ConvData);
    ConvData.etype = ET_NORM;
    ConvData.ppEnt = &pEnt;
    ConvData.pBuff = pData;

    *RowCnt = 0;
    if (IsAttrFormat()) 
        ConvData.status.ATTR_FILTER = TRUE;
    mod = ModTextBlock(&ConvData);
    *RowCnt = ConvData.TotRows;
    return mod;
}


/* Using a block of text data, modify a drawing.  Loops 
   until the end of the current block of data.  Could be
   more data still from next request.
*/
static UINT ModTextBlock(CONVDATA *ConvData)
{
    while (!ConvData->status.END_BLK)
        ModTextEnt(ConvData);
    return ConvData->ModCnt;
}


/* From text data (string<TAB>string<TAB>string<CR><LF>...)
   build list of resbufs, and modify any entities that have
   changed.  Return number acutally modified.  If first link
   is -1, it uses that as the name, else it looks for a 
   handle (5). 
*/
static short ModTextEnt(CONVDATA *ConvData)
{
    short modstat, cmpstat;
    RESBUF *pEnt = NULL;
    RESBUF *pEntDde = NULL;
    RESBUF **pEntAddr;
    UINT NumRows;
    ads_name ename;

    /* build list of entities from text data representation.  Free pEntDde
       with EntFree. */
    NumRows = EntBuild(ConvData);
    if (ConvData->status.END_BLK && !ConvData->status.END_ALL)
        /* Could be incomplete entity.  Return so we can
           back up a little and get the whole thing on next request */
        return NumRows;
    pEntAddr = ConvData->ppEnt;
    pEntDde = *pEntAddr;
    if (pEntDde == NULL) 
        return 0;

    /* If no name, or format is 2 (handle format), 
       get handle and then name. */
    if (pEntDde->restype != ENTMARK) {
        pEnt = GetEntFromHandle(pEntDde);
        if (!pEnt) {
            EntFree(pEntAddr);
            return 0;
        }

        /* Have to have name to entmod it. */
        if (pEnt->restype != ENTMARK) {
            EntFree(pEntAddr);
            ads_relrb(pEnt);
            return 0;
        }
        ads_name_set(pEnt->resval.rlname, ename);
    } else {
        RESBUF *pSEnt;

        if (pSEnt = assoc(pEntDde, G_HANDLE)) {
            char *handle;

            /* Verify leading "x" */
            handle = pSEnt->resval.rstring;
            if (*handle != HPREFIX) {
                EntFree(pEntAddr);
                return 0;
            }
            /* Remove leading "x" */
            str_delete(handle, 1);
        }
        ads_name_set(pEntDde->resval.rlname, ename);
        pEnt = ads_entget(ename);     /* get start node of linked
                                         list of sub-entities */
    }

    if (!pEnt) {
        EntFree(pEntAddr);
        return 0;
    }

    cmpstat = EntCompare(pEnt, pEntDde);
    if (cmpstat > 0) {                /* name same, something 
                                         else different */
        modstat = ads_entmod(pEnt);
        if (modstat == RTNORM) {
            ++ConvData->ModCnt;
            ads_entupd(ename);        /* modify polyline, etc.,
                                         and show it */
        }
    }
    if (pEntDde)
        EntFree(pEntAddr);
    if (pEnt)
        ads_relrb(pEnt);
    *(ConvData->ppEnt) = NULL;
    return ConvData->ModCnt;
}


/* Note: If the section of a spreadsheet read in by DDE has no data,
   the data will look like:  <0D><0A><00><00>.  This pattern could be
   interpreted to mean end-of-data. */



/* Build an entity from DDE text format.
   Return number of rows (LF's).
   Call until END_ALL is TRUE.
   pEnt must be set to NULL if it's a new list.  
   ConvData.status will contain result after the call
   Call EntFree to free the entity list in ConvData.

        CONVDATA ConvData;
        RESBUF *pEnt = NULL;

        ConvData.ppEnt = &pEnt;
        ConvData.pBuff = pdde->pData;
        EntBuild(&ConvData);

        ads_entmod(pEnt);
        EntFree(&pEnt);     to free list. 
*/
static short EntBuild(CONVDATA *ConvData)
{
    short GrpCode, EntStart;
    short Sep;
    UINT NumRows = 0;
    UINT Blanks = 0;
    RESBUF *pEnt;
    RESBUF **ppEnt;
    PHDATA pLine;
    char *pWord;
    UINT nEnt = 0;
    short blank;

    ppEnt = ConvData->ppEnt;
    pLine = ConvData->pBuff;
    ConvData->EntRows = 0;
    if (FrmtTable[Frmt].flags & USE_NAME)
        EntStart = ENTMARK;           /* -1 if using names */
    else
        EntStart = G_START;           /* use group code 0 as start of
                                         entity for handle format (not using
                                         ent name). */
    ConvData->status.END_BLK = FALSE;
    do  {
        blank = FALSE;
        if (*pLine == EOS) { 
            /* Normal end of block read from spreadsheet */     
            ConvData->status.END_BLK = TRUE;
            break;
        }
        if (!ChkNextLine(pLine)) {    /* blank line? */
            blank = TRUE;
            ++Blanks;
            if (Blanks > STOPSCAN) {
                ConvData->status.END_ALL = TRUE;
                ConvData->status.END_BLK = TRUE;
                /* blank lines mean end of spreadsheet section */
                break;
            }
        }

        /* Get group code: every non-blank line starts with
           an integer group code, or <= -4 for user comment */
        if (!blank)
            GrpCode = atoi(pLine);
        else
            GrpCode = -9;             /* treat comments (<-4) and 
                                         blank lines the same */

        if (ConvData->status.ATTR_FILTER) {
            ConvData->pBuff = pLine;
            return EntRowBuild(ConvData);
        }

        if (GrpCode == ENDMARK) {     /* end mark from previous transmission */
            ConvData->status.END_ALL = TRUE;
            ConvData->status.END_BLK = TRUE;
            break;
        } else if (GrpCode == SEQMARK) { /* -2: signals end of 
                                         entity, but don't include it */
            if (nEnt)
                /* Normal end of entity */
                break;
        }

        /* first sub-entity
           -1: entity name
            0: handle */
        else if (GrpCode == EntStart) {
            if (!nEnt) {
                /* Normal start of entity */
                ++ConvData->EntCnt;
                ++nEnt;
            } else
                /* Normal end of entity */
                break;
        }
        if (nEnt > 0) {               /* we've started an entity */
            ++ConvData->EntRows;      /* we'll back up if this
                                         block of data ends mid-
                                         entity--we'll subtract
                                         ->EntRows from total. */
            if ((GrpCode >= EntStart) && *pLine >= SP) {
                /* not empty line or comment*/
                pEnt = (RESBUF *)AddLink((void **)ppEnt, RBSIZE);
                pWord = GetNextFld(pLine, &Sep);
                if (!Sep) {           /* end of this block */
                    ConvData->status.END_BLK = TRUE;
                    break;
                }
                SubEntBuild(pEnt, GrpCode, pWord, ConvData->etype);
            }
        }
        ++NumRows;
    } while (pLine = GetNextLine(pLine));

    if (ConvData->status.END_BLK && !ConvData->status.END_ALL) {
        /* Start next read at the start of this entity */
        if (*ppEnt != NULL)
            EntFree(ppEnt);
        return 0;
    }
    if (NumRows) {                    /* Got complete entity. */
        ConvData->pBuff = pLine;      /* next call can continue 
                                         where we left off */
        ConvData->BlkRows += NumRows;
        ConvData->TotRows += NumRows;
    }
    return NumRows;
}





/*
                DXF/Text Conversions
*/

/* Format a list of entities into DDE or AutoCAD 
   text display, including all links.  
   FldSepStr and RecSepStr and used to separate items.
     -1 <FldSepStr> x60000048 <FldSepStr> 160000345 <FldSepStr> 160023456 <RecSepStr>
      8 <FldSepStr> "LAYER" <RecSepStr>
     etc.
   Use free() to free the pointer returned.   
*/
char *EntListFormat(RESBUF *pEnt, int etype)
{
    char *EntBuff = NULL;
    ULONG Len;
    UINT LinkCnt;

    Len = EntFormat(pEnt, NULL, &LinkCnt, etype);
    if (Len)
        EntBuff = malloc((int)Len + 1);
    if (!EntBuff)
        return NULL;
    EntFormat(pEnt, EntBuff, &LinkCnt, etype);
    return EntBuff;
}



/* Format a list of sub-entities into DDE or AutoCAD 
   text display, including all links. 
   Set PHDATA to NULL to just find length of
   formatted buffer, for malloc.  Returns length. 
*/
ULONG EntFormat(RESBUF *pEnt, char *pData, UINT *LinkCnt, int etype)
{
    RESBUF *pEnt1;
    char *ptr;
    UINT Len;
    ULONG TotalLen = 0;
    short cLink = 0;
    short NameFlag = TRUE;
    char LineBuff[2*ENTSTRMAX+12];

    if (!pEnt)
        return 0;

    if (FrmtTable[Frmt].flags & SINGLE_ROW) {   
                                         /* format is all on one line 
                                         (e.g. handle, tab, attr) */
        Len = EntRowFormat(pEnt, LinkCnt, LineBuff);
        if (pData && Len)
            strcpy(pData, LineBuff);
        return Len;
    }
    if (pData) {
        pData[0] = EOS;
        ptr = pData;
    }
    NameFlag = FrmtTable[Frmt].flags & USE_NAME;
    for (pEnt1 = pEnt; pEnt1 != NULL; pEnt1 = pEnt1->rbnext) {
        if (!NameFlag && pEnt1->restype == ENTMARK)
            /* skip name if using handle */
            continue;
        Len = SubEntFormat(pEnt1, LineBuff, etype);
        TotalLen += Len;
        if (pData) {
            strcpy(ptr, LineBuff);
            ptr += Len;
        }
        ++cLink;
    }
    *LinkCnt = cLink;
    return TotalLen;                  /* add 1 if you want the EOS at end */
}


/* Format entity all on one row.
        e.g., handle<TAB>tag<TAB>attribute text<CR><LF> 
*/
UINT EntRowFormat(RESBUF *pEnt, UINT *LinkCnt, char *EntTextStrm)
{
    short type;
    char *EntStr[FRMTMAX];
    RESBUF *ent;
    BOOL attrflag = FALSE;
    short pos, idx;
    BOOL found = FALSE;
    char dec_handle[22];
    int dformat = FrmtTable[Frmt].flags & DHANDLE;

    *LinkCnt = 0;
    if (pEnt == NULL)
        return 0;

    for (idx = 0; idx < FRMTMAX; ++idx)
        EntStr[idx] = NULL;

    dec_handle[0] = EOS;
    for (ent = pEnt; ent != NULL; ent = ent->rbnext) {
        type = ent->restype;
        if (type == 0) {
            if (!_stricmp(ent->resval.rstring, ATTRIB))
                attrflag = TRUE;
            else if (!_stricmp(ent->resval.rstring, "ATTDEF"))
                attrflag = TRUE;
            else
                break;                /* Not an attribute */
        } else if ((pos = InFrmtTable(type)) >= 0) {
            EntStr[pos] = ent->resval.rstring;
            if (type == G_HANDLE && dformat)
                DecimalHandle(ent->resval.rstring, dec_handle);
            found = TRUE;
        }
    }

    if (!attrflag)
        return 0;
    if (dformat)
        strcpy(EntTextStrm, dec_handle);
    else
        EntTextStrm[0] = EOS;
    for (idx = 0; ; ++idx) {
        type = FrmtTable[Frmt].subent[idx];
        if (type == ENTTERM)
            break;
        if (EntTextStrm[0] != EOS)
            strcat(EntTextStrm, FldSepStr);
        if (EntStr[idx]) {
            if (type == G_HANDLE)
                strcat(EntTextStrm, "x");
            strcat(EntTextStrm, EntStr[idx]);
        } else
            strcat(EntTextStrm, "No Handle");
        if (idx >= FRMTMAX)
            break;
    }

    if (found) {
        strcat(EntTextStrm, RecSepStr);
        *LinkCnt = 1;
        return strlen(EntTextStrm);
    } else
        return 0;
}



/* Format the value from an RESBUF or RESBUF into your string, ValStr.
   Return the length of ValStr.  Uses globals FldSepStr & RecSepStr to 
   separate each item.  For example:
   "<GroupCode><TAB>X<TAB>Y<TAB>Z<CR><LF>", or
   "10<TAB>1.23456<TAB>4.32434<TAB>-45.5555<CR><LF>". 
   Returns the length of the formatted string. 
   "etype" is either ET_NORM, ET_SSET, ET_TBL, or ET_RB. 
*/
UINT SubEntFormat(RESBUF *pEnt, char *ValStr, int etype)
{
    short GrpCode;
    short typeval;
    char DataStr[140];
    char NumStr[FPSTR+1];
    ULONG ename0;

    if (pEnt == NULL)
        return 0;

    GrpCode = pEnt->restype;
    if (etype == ET_RB)
        typeval = GrpCode;
    else
        typeval = dxftype(GrpCode, etype);

    _itoa(GrpCode, ValStr, 10);
    strcat(ValStr, FldSepStr);

    switch (typeval) {
    case RTSHORT:
        _itoa(pEnt->resval.rint, DataStr, 10);
        break;

    case RTANG:
    case RTREAL:
        RealToStr(pEnt->resval.rreal, DataStr);
        break;

    case RTSTR:                       /* Handles are hex, and the spreadsheet takes
                                         2E8 and thinks it's 2E+8, or 2 times 10
                                         to the eight power.  Put x in front of it. 
                                      */
        if (GrpCode == G_HANDLE) {
            DataStr[0] = HPREFIX;
            strzcpy(DataStr+1, pEnt->resval.rstring, ENTSTRMAX);
        }
        else
            strzcpy(DataStr, pEnt->resval.rstring, ENTSTRMAX);
        break;

    case RTPOINT:
    case RT3DPOINT:
        RealToStr(pEnt->resval.rpoint[0], DataStr);

        strcat(DataStr, FldSepStr);
        RealToStr(pEnt->resval.rpoint[1], NumStr);

        strcat(DataStr, NumStr);
        if (typeval == RT3DPOINT)
            RealToStr(pEnt->resval.rpoint[2], NumStr);
        else
            /* strcpy(NumStr, "0.0"); */
            break;
        strcat(DataStr, FldSepStr);
        strcat(DataStr, NumStr);
        break;

    case RTPICKS:                     /* selection set--treat same as 
                                         entity name */
    case RTENAME:                     /* Field 1: hex display of ename[0], 
                                              to match AutoCAD command line
                                              representation
                                         Field 2: long ename[0]
                                         Field 3: long ename[1] */

        if (!(FrmtTable[Frmt].flags & USE_NAME) && GrpCode == SEQMARK) {
            /* handle mode--forget sequence end ent name */
            DataStr[0] = EOS;
            break;
        }

        ename0 = pEnt->resval.rlname[0];
        sprintf(DataStr, HexFmt, ename0);
        strcat(DataStr, FldSepStr);

        _ultoa(ename0, NumStr, 10);
        strcat(DataStr, NumStr);

        strcat(DataStr, FldSepStr);

        /* sprintf(NumStr, HexFmt, pEnt->resval.rlname[1]); */
        _ultoa(pEnt->resval.rlname[1], NumStr, 10);
        strcat(DataStr, NumStr);
        break;

    case RTLONG:
        _ltoa(pEnt->resval.rlong, DataStr, 10);
        break;

    default:
        DataStr[0] = 0;
        // strcpy(DataStr, "Unknown type");    
        break;
    }

    strcat(ValStr, DataStr);
    strcat(ValStr, RecSepStr);
    return strlen(ValStr);
}

/* fill in a sub-resbuf according to GrpCode and a line
   of chars to be converted to numbers, separated by TABs. 
   Use EntFree to free it. 
*/
static void SubEntBuild(RESBUF *pEnt, short GrpCode, char *Line, short etype)
{
    short ctype;
    short Sep;
    short blank;
    char chr;

    pEnt->restype = GrpCode;
    pEnt->rbnext = NULL;
    chr = *Line;
    if (!chr)
        return;

    blank = isspace( chr );

    ctype = dxftype(GrpCode, etype);
    switch (ctype) {
    case RTSTR:
        /*  *Line should be 'x' for G_HANDLE, or 5 group code */
        pEnt->resval.rstring = FieldSave(Line);
        break;

    case RTSHORT:
        if (blank)
            pEnt->resval.rint = 0;
        else
            pEnt->resval.rint = atoi(Line);
        break;

    case RTLONG:
        if (blank)
            pEnt->resval.rlong = 0L;
        else
            pEnt->resval.rlong = atol(Line);
        break;

    case RT3DPOINT:
    case RTPOINT:
        pEnt->resval.rpoint[0] = StrToReal(Line);
        Line = GetNextFld(Line, &Sep);
        if (Sep != FldSepChr)         /* new line or end of data */
            break;
        pEnt->resval.rpoint[1] = StrToReal(Line);
        if (ctype == RTPOINT)
            break;

        Line = GetNextFld(Line, &Sep);
        if (Sep != FldSepChr)         /* new line or end of data */
            break;
        pEnt->resval.rpoint[2] = StrToReal(Line);
        break;

    case RTREAL:
    case RTANG:
        pEnt->resval.rreal = StrToReal(Line);
        break;

        /* Excel couldn't handle longs in decimal format--too long.  
           Using hex to make shorter string. */
    case RTPICKS:                     /* selection set--treat same as 
                                         entity name */
    case RTENAME:
        Line = GetNextFld(Line, &Sep);/* skip hex display field */
        if (Sep != FldSepChr)         /* new line or end of data */
            break;
        if (*Line < SP)
            pEnt->resval.rlname[0] = 0L;
        else
            /* sscanf(Line, EnameScan, &LongVal);
            pEnt->resval.rlname[0] = LongVal; */
            pEnt->resval.rlname[0] = atol(Line);
        Line = GetNextFld(Line, &Sep);
        if (Sep != FldSepChr)         /* new line or end of data */
            break;
        if (*Line < SP)
            pEnt->resval.rlname[1] = 0L;
        /* sscanf(Line, EnameScan, &LongVal);
        pEnt->resval.rlname[1] = LongVal; */
        else
            pEnt->resval.rlname[1] = atol(Line);
        break;
    }
}


/* Build entity from spreadsheet data all on one row, ATTR_MODE2.  Eg, 
   decimal handle<TAB>handle<TAB>attribute<TAB>decimal handle...<CR><LF> 
   Returning 0 breaks block scan loop.  Sets ENT_MORE normally.
*/
static short EntRowBuild(CONVDATA *ConvData)
{
    RESBUF *pEnt;
    short idx, grpcode;
    short FoundOne = FALSE;
    char chr;
    short Sep;
    PHDATA pLine = ConvData->pBuff;
    int dhandle = FrmtTable[Frmt].flags & DHANDLE;
    int toss = FALSE;
    int const_attr = FALSE;

    *(ConvData->ppEnt) = NULL;
    ConvData->status.END_BLK = FALSE;

    /* Move past any empty cells at the end of the row */
    while (*pLine == FldSepChr) {
        pLine = GetNextLine(pLine);
        if (*pLine == EOS) {
            ConvData->status.END_BLK = TRUE;
            return 0;
        }
    }

    /* Check of end mark, -99, or any other -xx */
    if (!strncmp(pLine, EndStr, 3)) {
        ConvData->status.END_ALL = TRUE;  /* terminates req loop */
        ConvData->status.END_BLK = TRUE;
        return 0;
    }
        
    /* Move past decimal handle */
    if (dhandle) {
        pLine = GetNextFld(pLine, &Sep);
        if (Sep != FldSepChr) {                 /* new line or end of data */
            ConvData->pBuff = pLine;    
            return 0;
        }
    }
    for (idx = 0; (grpcode = FrmtTable[Frmt].subent[idx]) != ENTTERM; ++idx) {
        chr = *pLine;
        if (chr >= SP) {
            /* not empty line or comment */
            FoundOne = TRUE;
            if (grpcode == G_HANDLE) {
                /* Verify "x" prefix */
                if (*pLine != HPREFIX) {
                    /* No valid handle.  Could be "Constant" data.
                       Let scanner finish this one anyway to stay
                       in sync. */
                    const_attr = TRUE;
                }
            }
            if (!const_attr) {
                pEnt = (RESBUF *)AddLink((void **)ConvData->ppEnt, RBSIZE);
                pEnt->restype = grpcode;
                pEnt->resval.rstring = FieldSave(pLine);
            }
            pLine = GetNextFld(pLine, &Sep);
            if (Sep != FldSepChr)     /* new line or end of data */
                break;
        } else {
            /* We're out of sync with spreadsheet.  Bail out and move
               to next line. */
            toss = TRUE;
            break;
        }
    }
    if (toss) {
        EntFree(ConvData->ppEnt);
        if (chr == EOS)
            ConvData->status.END_BLK = TRUE;
        else {
            pLine = GetNextLine(pLine);
            if (*pLine == EOS)
                ConvData->status.END_BLK = TRUE;
        }
    }
    ConvData->pBuff = pLine;
    if (FoundOne) {
        ConvData->status.END_BLK = FALSE;
        ConvData->EntRows = 1;
        ++ConvData->EntCnt;
        /* We're not really uping the row, but upper level 
           routines use this to make sure we're not stuck in
           a loop */
        ++ConvData->TotRows;          
        if (const_attr) {
            EntFree(ConvData->ppEnt);
            return 0;
        } else  
            return 1;
    } else {
        ConvData->status.END_BLK = TRUE;
        return 0;
    }
}


#if 0
#define ET_RB   0                     /* result buffer */
#define ET_NORM 1                     /* normal entity */
#define ET_SSET 2                     /* selection set */
#define ET_TBL  3                     /* table--above this reserved for adding
                                      actual table number */
/* table numbers, after adding ET_TBL */
#define ET_VPORT  3
#define ET_LTYPE  4
#define ET_LAYER  5
#define ET_STYLE  6
#define ET_VIEW   7
#define ET_UCS    8
#define ET_BLOCK  9
#define ET_DIMSTYLE  10

#endif

/* Get basic C language type from AutoCAD dxf group code (RTREAL, 
   RTANG are doubles, RTPOINT double[2], RT3DPOINT double[3], 
   RTENAME long[2].  The "etype" is one of the ET_ definitions. 
   Returns RTNONE if grpcode isn't one of the known group codes. 
*/
int dxftype(int grpcode, int etype)
{
    short rbtype = RTNONE;

    if (grpcode <= 49) {
        if (grpcode >= 20)            /* 20 to 49 */
            rbtype = RTREAL;
        else if (grpcode >= 10) {     /* 10 to 19 */
            if (etype == ET_VIEW)     /* special table cases */
                rbtype = RTPOINT;
            else if (etype == ET_VPORT && grpcode <= 15)
                rbtype = RTPOINT;
            else                      /* normal point */
                rbtype = pointtype;   /* 10: start point, 11: end point */
        }
        else if (grpcode >= 0)        /* 0 to 9 */
            rbtype = RTSTR;
        else if (grpcode >= -2)       /* -1 = start of normal entity
                                         -2 = sequence end, etc. */
            rbtype = RTENAME;
        else if (grpcode == -3)
            rbtype = RTEXTDS;         /* extended data sentinals */
    }
    else {
        if (grpcode <= 59)            /* 50 to 59 */
            rbtype = RTANG;           /* double */
        else if (grpcode <= 79)       /* 60 to 79 */
            rbtype = RTSHORT;
        else if (grpcode < 210)
            ;
        else if (grpcode <= 239)      /* 210 to 239 */
            rbtype = RT3DPOINT;
        else if (grpcode == 999)      /* comment */
            rbtype = RTSTR;
        else if (grpcode >= 1000)     /* extended data group codes */
            rbtype = RTEXTDG;
    }

    return rbtype;
}



/* Free a list of entities created with the upper level
   heap calls, instead of by ADS.  Pass address of first 
   node.  
*/
void EntFree(RESBUF **pEntDde)
{
    RESBUF *pEnt;
    RESBUF *next;
    short type;

    for (pEnt = *pEntDde; pEnt != NULL; pEnt = next) {
        next = pEnt->rbnext;
        type = dxftype(pEnt->restype, ET_NORM);
        if (type == RTSTR)
            free(pEnt->resval.rstring);
        free(pEnt);
    }
    *pEntDde = NULL;
}




/* Compare an entity from ads_entget (pEnt) with an entity built from 
   text data (pEntDde).  If any values are different, change the
   value in the pEnt list and return 1.
   Returns  0 if no differences or name doesn't match.
   Returns -1 if incomplete data (end of block of
   DDE data?).  
   To test for entity that has been changed:
        if (EntCompare(pEnt, pEntDde) > 0)
            ads_entmod(pEnt);
*/
static int EntCompare(RESBUF *pDrawEnt, RESBUF *pDdeEnt)
{
    RESBUF *pDdeSub, *pDrawSub;
    int ctype;
    int nLink = 0;
    int ret = 0;
    int AryLen, GrpCode;
    char *str;

    if (!pDrawEnt || !pDdeEnt)
        return 0;

    /* Loop through subentities from the spreadsheet and find associated
       subentities in the drawing.  Compare and update drawin 
       subentities if different. */
    for (pDdeSub = pDdeEnt; pDdeSub != NULL; pDdeSub = pDdeSub->rbnext) {
        ++nLink;
        GrpCode = pDdeSub->restype;
        pDrawSub = assoc(pDrawEnt, GrpCode);
        if (pDrawSub == NULL)
            return 0;
        ctype = dxftype(GrpCode, ET_NORM);

        switch (ctype) {
        case RTSHORT:
            if (pDrawSub->resval.rint != pDdeSub->resval.rint) {
                pDrawSub->resval.rint = pDdeSub->resval.rint;
                ret = 1;
            }
            break;

        case RTREAL:
        case RTANG:
            if (!CmpReals(&pDrawSub->resval.rreal, &pDdeSub->resval.rreal, 1))
                ret = 1;
            break;

        case RTSTR:                   /* case sensitive compare */
            if ((str = pDdeSub->resval.rstring) == NULL)
                break;
            if (strcmp(pDrawSub->resval.rstring, str) != 0) {
                if (strlen(str) > strlen(pDrawSub->resval.rstring)) {
                    free(pDrawSub->resval.rstring); /* free old one */
                    pDrawSub->resval.rstring = _strdup(str);
                } else
                    strcpy(pDrawSub->resval.rstring, str);
                ret = 1;
            }
            break;

        case RT3DPOINT:
        case RTPOINT:
            if (ctype == RT3DPOINT)
                AryLen = 3;
            else
                AryLen = 2;
            if (!CmpReals(pDrawSub->resval.rpoint,
                          pDdeSub->resval.rpoint, AryLen))
                ret = 1;
            break;

        case RTPICKS:                 /* selection set--treat same as 
                                      entity name */
        case RTENAME:
            if ((pDrawSub->resval.rlname[0] != pDdeSub->resval.rlname[0]) &&
                        FrmtTable[Frmt].flags & USE_NAME)
                /* name flag */
                return 0;
            break;

        case RTLONG:
            if (pDrawSub->resval.rlong != pDdeSub->resval.rlong) {
                pDrawSub->resval.rlong = pDdeSub->resval.rlong;
                ret = 1;
            }
            break;

        default:
            break;
        }
    }
    return ret;
}


/* Returns TRUE if both arrays of reals are
   the same, within DdeTolerance.  If not, the
   first is set to the 2nd. 
*/
static int CmpReals(ads_real RealAry1[], ads_real RealAry2[], int AryLen)
{
    static ads_real rdiff;
    short Idx;
    ads_real diff;
    short ret = TRUE;

    for (Idx = 0; Idx < AryLen; ++Idx) {
        #if USEFABS
        rdiff = RealAry1[Idx] - RealAry2[Idx];
        diff = fabs(rdiff);

        #else

        diff = RealAry1[Idx] - RealAry2[Idx];
        if (diff < 0.0)
            diff = 0.0 - diff;
        #endif

        if (diff >= DdeTolerance) {
            RealAry1[Idx] = RealAry2[Idx];
            ret = FALSE;
        }
    }
    return ret;
}


/* 

                Low Level Utility Functions

*/


/* Modify conversion strings for converting entities to
   text streams and back.  Set any args to NULL to accept
   default.  E.g., for 6 digit accuracy,  .001 tolerance (defaults):
        ConvertInit(6, 0.001, "\t", "\r\n"); 
   When entities are converted from strings back to reals,
   if the change is greater than Tolerance, EntCompare says
   they're different (so entmod can be called to change 
   them in AutoCAD).   Depends on digits of accuracy.  
*/
void ConvPrecision(int precision, ads_real Tolerance)
{
    char NumStr[25];

    if (precision) {
        FpPrecision = precision;
        /* char RealFmt[10] = "%+.6lf"; */
        strcpy(RealFmt, "%+.");
        _itoa(precision, NumStr, 10);
        strcat(RealFmt, NumStr);
        strcat(RealFmt, "lf");
    }
    DdeTolerance = Tolerance;
}


/* Set field record separators for text formatted 
   AutoCAD drawing entities.  Default is TAB field separator, 
   CRLF record separator. 
*/
void ConvSeparators(char *FldSeparator, char *RecSeparator)
{
    if (FldSeparator)
        strcpy(FldSepStr, FldSeparator);
    if (RecSeparator)
        strcpy(RecSepStr, RecSeparator);
}





/* Call after a drawing has been opened (ads RQXLOAD request)
   to set the local flatland variable, so dxf point data
   (group code 10, 11,  etc.) will be interpreted correctly. 
*/
short SetFlatLand(void)
{
    RESBUF rb;

    pointtype = RT3DPOINT;
    ads_getvar("flatland", &rb);
    if (rb.restype != RTSHORT)
        return -1;
    locflatland = rb.resval.rint;
    pointtype = (locflatland ? RTPOINT : RT3DPOINT);
    return locflatland;
}


/* Turn on handle mode for AutoCAD drawing.  Since handles
   are valid between AutoCAD sessions, and names aren't (they're
   memory load dependent), data can be saved and used later
   with a drawing.  Necessary with Frmt = 2, although the 
   drawing could already have handles on.  Returns previous
   state, TRUE if on. 
*/
BOOL SetHandles(void)
{
    RESBUF rb;

    ads_getvar("HANDLES", &rb);

    if (!rb.resval.rint) {
        ads_command(RTSTR, "handles", RTSTR, "on", RTNONE);
        return FALSE;
    }
    else
        return TRUE;
}



/* Get number of rows taken by all the tables and blocks 
*/
ULONG LenAllTable(void)
{
    short idx;
    ULONG rows = 0L;
    char *tblname;

    for (idx = 0; (tblname = TableList[idx]) != 0; ++idx)
        rows += LenTable(idx);
    return rows;
}


/* Get number of rows taken by a table or all the blocks 
*/
UINT LenTable(short TableNum)
{
    RESBUF *pEnt;
    UINT Len = 0;
    UINT first = TRUE;

    while (TRUE) {
        pEnt = ads_tblnext(TableList[TableNum], first);
        first = FALSE;
        if (pEnt) {
            Len += GetListLen((PLINK)pEnt);
            ads_relrb(pEnt);
        }
        else
            break;
    }
    return Len;
}





/*

                AutoCAD Text Screen Messages

*/


/* Use AutoCAD to prompt for an application and topic name,
   initiate a DDE channel with the names, and return the 
   channel number if successful, 0 if not.  "app" and "topic"
   need to be able to store MAXNAME+1 characters.  Use DdeDlgStart
   if you'd rather use a Windows dialog.
*/
PDDE AdsChnl(char *app, char *topic)
{
    PDDE pdde;
    int stat;

    stat = AcPrompt("DDE application", app, app);
    if (stat < 0)
        return NULL;
    stat = AcPrompt("DDE work file", topic, topic);
    if (stat < 0)
        return NULL;
    /* Get channel on app and topic. */
    pdde = DdeChannel(app, topic);
    return pdde;
}


/* Display a list of DDE channels in AutoCAD, such as:
                1 Excel Sheet1
                2 AutoCAD Columbia
*/
short AcPrintChnlList(void)
{
    PLINK list = NULL;
    short NumChnl;

    NumChnl = BuildChnlList(&list, " ");
    if (list) {
        AcPrintList(list, "\n");
        RemoveText((void **)&list, NULL);
    }
    return NumChnl;
}



/* Build a list of DDE channels for display
   for dialogs or whatever. 
                1 Excel Sheet1
                2 AutoCAD Columbia
*/
short BuildChnlList(PLINK *list, char *fldsep)
{
    char line[120];
    char numstr[20];
    PDDE pdde;
    short NumChnl = 0;

    for (pdde = DdeData.ChnlList; pdde != NULL; pdde = pdde->next) {
        ++NumChnl;
        _itoa(pdde->channel, numstr, 10);
        strcpy(line, numstr);
        strcat(line, fldsep);
        strcat(line, pdde->app);
        strcat(line, fldsep);
        strcat(line, pdde->topic);
        AddText((void **)list, line);
    }
    return NumChnl;
}


/* Use AutoCAD to input a string.  Show prompt, with 
   <default>:, get user input in result.  Set default to 
   NULL if none.  Negative return means failure. 
   E.g.  
        AcPrompt("DDE Application", "Excel", TestApp); 

                     would look like:

           DDE Application <Excel>: 
*/
int AcPrompt(char *prompt, char *DefString, char *result)
{
    char adsprompt[120];
    char adsresult[120];
    int stat;

    strcpy(adsprompt, prompt);
    if (DefString && DefString[0] && DefString[0] != LF && 
                        DefString[0] != CR) {
        strcat(adsprompt, " <");
        strcat(adsprompt, DefString);
        strcat(adsprompt, ">: ");
    }
    else
        strcat(adsprompt, ": ");
    stat = ads_getstring(0, adsprompt, adsresult);
    if (stat < 0)
        return stat;
    if (strlen(adsresult)) {
        strzcpy(result, adsresult, MAXNAME);
    } else {
        if (DefString)
            strzcpy(result, DefString, MAXNAME);
        else
            result[0] = 0;
    }
    return stat;
}


/* Display any text linked list, using the Separator string
   to separate one item from the next. 
*/
void AcPrintList(PLINK TextList, char *Separator)
{
    PLINK TextLink;
    if (!TextList)
        ads_printf("Empty List\n");

    for (TextLink = TextList; TextLink != NULL; TextLink = TextLink->next) {
        ads_printf(TextLink->text);
        if (Separator)
            ads_printf(Separator);
    }
}


/* Display string conversion value of resbuf in AutoCAD.  If 
   it's an entity type (RESBUF), set EntityFlag = TRUE. 
*/
void AcPrintRb(RESBUF *pEnt, short etype)
{
    char MsgLine[120];
    char *FldSepSave;
    char *RecSepSave;

    FldSepSave = FldSepStr;           /* swap in " . " for TAB, 
                                      "\n" for "\n\r", for correct
                                      display in AutoCAD. */
    RecSepSave = RecSepStr;
    FldSepStr = AcFldSep;
    RecSepStr = AcRecSep;
    if (!pEnt)
        ads_printf("Empty\n");
    else {
        MsgLine[0] = 0;
        SubEntFormat(pEnt, MsgLine, etype); /* 0 = just rb, not entity */
        ads_printf(MsgLine);
    }
    FldSepStr = FldSepSave;
    RecSepStr = RecSepSave;
}





/*
               Text Parsing Routines
*/


/* Get pointer to the start of the next line
   in a cell data list
   Check *pData==EOS for end-of-data. 
*/
static char *GetNextLine(PHDATA pData)
{
    PHDATA ptr;
    char chr;

    for (ptr = pData; chr = *ptr; ++ptr) {
        if (chr == RecSepChr2) {
            ++ptr;
            break;
        }
    }
    return ptr;
}



/* Check to see if next line has data--could just be
  <TAB><TAB><CR><LF> or such. 
*/
static short ChkNextLine(PHDATA pData)
{
    char chr;
    char *ptr;
    short FoundData = FALSE;

    for (ptr = pData; chr = *ptr; ++ptr) {
        if (chr == RecSepChr2)
            break;
        if (chr >= SP) {
            FoundData = TRUE;
            break;
        }
    }
    return FoundData;
}





#if INCL_ALL

/* copy the word at src into dest, until a TAB, CR, or LF is
   encountered (anything less than SP). 
*/
short FldCopy(char *dest, char *src)
{
    char chr;
    while (chr = *src) {
        if (chr < SP) {
            *dest = NULL;
            break;
        }
        *dest++ = *src++;
    }
    return (short)chr;
}


#endif  /* INCL_ALL */





/* Is the group code in the current table list? 
   Return table index (0 based), or ENTTERM if not in list. 
*/
static int InFrmtTable(short GrpCode)
{
    short idx, tablenum;

    for (idx = 0; idx < FRMTMAX; ++idx) {
        tablenum = FrmtTable[Frmt].subent[idx];
        if (tablenum == ENTTERM)
            break;
        if (tablenum == GrpCode)
            return idx;
    }
    return -1;

}





/* Return the number of rows taken by an entity in DDE text format 
*/
UINT GetEntLen(RESBUF *pEnt)
{
    UINT nRow;

    if (!pEnt)
        return 0;

    if (FrmtTable[Frmt].flags & SINGLE_ROW)  /* whole entity on one row */
        return 1;

    nRow = GetListLen((PLINK) pEnt);
    if (nRow && pEnt->restype == ENTMARK && 
                        !(FrmtTable[Frmt].flags & USE_NAME))
        /* named entity with handle mode--skip name */
        --nRow;
    return nRow;
}

/* Write a blank line, a "-99\tEnd of DDE", and another blank line 
  to the spreadsheet to mark end of data. */

static void PokeEndString(PDDE pdde, UINT row, UINT col, UINT nrow)
{
    char Cell[20];
    UINT erow = row + nrow;

    PokeBlankRow(pdde, erow, col);
    CellRange(Cell, erow+1, col, 1, BLANKWIDTH);
    DdePokeString(pdde, Cell, EndString);
    PokeBlankRow(pdde, erow+2, col);
    SaveItem(pdde, row, col, nrow);
}

/* Write a blank line, a "-99\tEnd of DDE", and another blank line 
  to the spreadsheet to mark end of data. */

void PokeEndStringConv(CONVDATA *ConvData)
{
    char Cell[20];
    PDDE pdde = ConvData->pdde;

    PokeBlankRowConv(ConvData);
    CellRange(Cell, ConvData->row+1, ConvData->col1, 1, BLANKWIDTH);
    DdePokeString(pdde, Cell, EndString);
    PokeBlankRow(pdde, ConvData->row+2, ConvData->col1);
    SaveItemConv(ConvData);
}

/* Save item for DDE "warm link" requests */
static void SaveItem(PDDE pdde, UINT row, UINT col, UINT nrow)
{
    char Cell[20];
    int maxcols;

    maxcols = max(pdde->maxcols, 4);
    CellRange(Cell, row, col, nrow, maxcols);
    store_string(&pdde->item, Cell);
}

/* Save item for DDE "warm link" requests */
static void SaveItemConv(CONVDATA *ConvData)
{
    PDDE pdde = ConvData->pdde;
    int cols;
    char Cell[20];

    /* If variable width format, use computed maxcols for width */
    cols = (FrmtTable[Frmt].flags & VARWIDTH) ? ConvData->pdde->maxcols : 
                        FrmtTable[Frmt].entwidth;
    cols = max(cols, 4);
    CellRange(Cell, ConvData->row1, ConvData->col1, ConvData->TotRows, 
                        cols);
    store_string(&pdde->item, Cell);
}

/* Poke a blank row into spreadsheet of a certain width */

static int PokeBlankRowConv(CONVDATA *ConvData)
{
    char Cell[20];

    CellRange(Cell, ConvData->row, ConvData->col1, 1, BLANKWIDTH);
    return DdePokeString(ConvData->pdde, Cell, BlankRowStr);
}

/* Poke a blank row into spreadsheet at row and col of a certain width */

int PokeBlankRow(PDDE pdde, int row, int col)
{
    char Cell[20];

    CellRange(Cell, row, col, 1, BLANKWIDTH);
    return DdePokeString(pdde, Cell, BlankRowStr);
}

/* Increment all row counters in a CONVDATA structure */

void IncRow(CONVDATA *ConvData, int inc)
{
    ConvData->row += inc;
    ConvData->TotRows += inc;
    ConvData->BlkRows += inc;
}


static int UpdateCols(CONVDATA *ConvData)
{
    if (ConvData->cols > (int)ConvData->pdde->maxcols) {
        ConvData->pdde->maxcols = ConvData->cols;
        return TRUE;
    } else
        return FALSE;
}

static int IsAttrFormat(void)
{
    if (Frmt >= ATTR_MODE)
        return TRUE;
    else
        return FALSE;
}



/* 

                General ADS Routines

*/


/* Equivalent of AutoLISP assoc function. 
   Find the sub-entity with a certain group code, return that
   pointer to sub-entity. 
*/
RESBUF *assoc(RESBUF *Ent, int type)
{
    RESBUF *pEnt;

    for (pEnt = Ent; pEnt != NULL; pEnt = pEnt->rbnext) {
        if (pEnt->restype == type) {
            return pEnt;
        }
    }
    return NULL;
}


/* Is this entity an "ATTRIB", "BLOCK", etc ("entstr") 
        Eg:
        IsEntType(pEnt, "ATTRIB");
*/

int IsEntType(RESBUF *pEnt, char *entstr)
{
    RESBUF *pSubEnt;

    pSubEnt = assoc(pEnt, G_START);     
    if (pSubEnt != NULL)
        if (!_stricmp(pSubEnt->resval.rstring, entstr))
            return TRUE;
    return FALSE;
}

/* Search a list of entity data, find handle, and use it
   to get the entity from AutoCAD.  Call ads_relrb  later
   if return is non NULL.
*/
RESBUF *GetEntFromHandle(RESBUF *pEntDde)
{
    RESBUF *ent, *enth;
    short stat;
    ads_name ename;
    char *handle;

    ent = assoc(pEntDde, G_HANDLE);
    if (ent == NULL)
        return NULL;
    handle = ent->resval.rstring;
    if (*handle != HPREFIX)
        /* Verify "x" for handle */
        return NULL;
    /* Remove "x" for entmod */
    str_delete(handle, 1);
    stat = ads_handent(handle, ename);
    if (stat != RTNORM)
        return NULL;
    enth = ads_entget(ename);
    return enth;
}


/* Format Hex handle to decimal for database ID.  Put decimal string
   in result, return value. */

long DecimalHandle(char *hexstr, char *result)
{
    long lhandle;

    sscanf(hexstr, "%lX", &lhandle);
    /* Get decimal string for hex handle, for database ID */
    _ltoa(lhandle, result, 10);
    return lhandle;
}


/* Is this selection set entity the start of a sequence of attributes
   or polylines?  Returns ATTRS_FOLLOW, SEQ_ATTR, SEQ_PLINE, or 0. */

int IsSequence(RESBUF *pEnt)
{
    RESBUF *pEnt2;

    if (IsEntType(pEnt, INSERT)) {
        pEnt2 = assoc(pEnt, G_ATFLAG);
        if (pEnt2 != NULL && pEnt2->resval.rint)
            return SEQ_ATTR;
    } else if (IsEntType(pEnt, POLYLINE)) {
#ifdef CHK_ATFLAG       
        /* Manual says "always 1", AutoCAD Reference Manual, 
           rel 11, Appendix C, Pg. 543. */
        pEnt2 = assoc(pEnt, G_ATFLAG);
        if (pEnt2 != NULL && pEnt2->resval.rint)
#endif  /* CHK_ATFLAG */        
            return SEQ_PLINE;
    }
    return 0;
}



/* Use ads_ssget to get all INSERTS.  Set skip_const to select only those
   with attributes following (<66 . 1>). */

int GetInserts(ads_name ss, int skip_const)
{
    RESBUF rbmask, rbmask2;

    /* Get all INSERT entities with following attributes */
    rbmask.restype = 0;
    rbmask.resval.rstring = "INSERT";
    if (!skip_const)
        rbmask.rbnext = NULL;
    else {
        rbmask.rbnext = &rbmask2;
        rbmask2.restype = G_ATFLAG;
        rbmask2.resval.rint = 1;
        rbmask2.rbnext = NULL;
    }
    return ads_ssget("X", NULL, NULL, &rbmask, ss);
}


/* Get AutoCAD function arguments.  Send array of 
   ARGLIST structures preloaded with
   expected types, such as [RTSHORT, RTSTR, RTENAME],
   and number of args.  Loads pointers to arguments
   in array.  Returns true if all argument types are correct.     
   Handles error messages to AutoCAD otherwise.
*/
int GetArgList(ARGLIST *arglist, char *funcname)
{
    int idx;
    int ActualType, ExpectedType;
    short cnt = 0;
    RESBUF *rb;
    RESBUF *rb1;

    if ((rb1 = ads_getargs()) == NULL)
        return FALSE;
    for (idx = 0, rb = rb1; rb != NULL; rb = rb->rbnext, ++idx) {
        ExpectedType = arglist[idx].argtype;
        if (ExpectedType == RTNONE) {
            ads_printf(funcname);
            ads_printf(": too many arguments [%d].\n", idx+1);
            return FALSE;
        }
        ActualType = rb->restype;
        if (ActualType == RTPICKS)
            ActualType = RTENAME;     /* so functions can use either */
        if (ActualType != ExpectedType) {
            ads_printf(funcname);
            ads_printf(": incorrect type, argument %d.\n", idx+1);
            return FALSE;
        }
        /* Fill in caller's structure */
        arglist[idx].rb = rb;
        if (idx >= MAXARGS) {
            ads_printf(funcname);
            ads_printf(": too many arguments [%d].\n", idx+1);
            return FALSE;
        }
    }
    if (arglist[idx].argtype != RTNONE) {
        ads_printf(funcname);
        ads_printf(": insufficient number of arguments.\n");
        return FALSE;
    }
    return TRUE;
}

/* See if this insert was a MINSERT, return cols * rows 
   Return 1 if not an array. */

int InsertArrayCnt(RESBUF *pEnt)
{
    RESBUF *psEnt;
    int col_cnt = 1, row_cnt = 1;

    /* See if we have an array from a MINSERT */
    psEnt = assoc(pEnt, 70);
    if (psEnt != NULL)
        col_cnt = max(psEnt->resval.rint, 1);
    psEnt = assoc(psEnt, 71);
    if (psEnt != NULL)
        row_cnt = max(psEnt->resval.rint, 1);
    return row_cnt * col_cnt;    
}




/* Initialize a CONVDATA structure */
static void NewConv(CONVDATA *ConvData)
{
    memset(ConvData, 0, sizeof(CONVDATA));
    ConvData->status.FIRST = TRUE;
}


/* Show current spreadsheet row in AutoCAD prompt area.  Check
   for ^C */
int ShowLife(UINT row)
{
    static UINT lastrow = 0;
    static int lastlen = 0;
    static UINT init = 0;
    char rstr[25];
    char lstr[50];
    int bk, len;
    int check;

    if (row == 0) {
        lastrow = 0;
        lastlen = 0;
        if (!init) {
            ads_printf("\nScanning drawing data...");
            ads_printf("\nSpreadsheet row: ");
            init = TRUE;
        }
        return FALSE;
    } else
        init = FALSE;

    if (IsAttrFormat())
        check = 5;
    else
        check = LIFE_FREQ;
    if ((int)(row - lastrow) < check)
        return FALSE;
    lastrow = row;
    _itoa(row, rstr, 10);
    len = strlen(rstr);
    /* Back up over the last status output */
    for (bk = 0; bk < lastlen; ++bk)
        lstr[bk] = '\b';
    lstr[bk] = EOS;
    lastlen = len;
    strcat(lstr, rstr);
    ads_printf(lstr);
    DdeBrkFlg = ads_usrbrk();
    return DdeBrkFlg;
}



/* end of file */

