/* Next available MSG number is    11 */

/*****************************************************************************
      CALSTDF.C
      (C) Copyright 1988-1994 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1988-1993 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

  Description: Standard calculator functions.

*****************************************************************************/


/****************************************************************************/
/*  INCLUDES                                                                */
/****************************************************************************/

#define MODULE_ID CALSTDF_C_

#include "cal.h"

#include "xmf.h"
#include "ads_ix.h"


/****************************************************************************/
/*  STATIC VARIABLES AND FUNCTIONS                                          */
/****************************************************************************/

static int pick_end_end   _((ads_point p1, ads_point p2, char fcn_name[]));
static void set_lastpoint _((ads_point p));
static void cal_snap      _((char snap_mode[]));


/****************************************************************************/
/*.doc sets_lastpoint(internal)*/
/*+
  Sets the value of the LASTPOINT AutoCAD variable.
-*/
/****************************************************************************/


static void
/*FCN*/set_lastpoint(p)

  ads_point p;
{
    struct resbuf rb;

    rb.restype = RT3DPOINT;
    CPY_PNT(rb.resval.rpoint, p);
    ads_setvar(/*MSG0*/"LASTPOINT", &rb);
} /*set_lastpoint*/


/****************************************************************************/
/*.doc cal_rXof_func(internal)*/
/*+
  Calculator 'rXof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_rXof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"rXof");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"rXof");
        return;
    }

    result.type = real_type;
    result.r    = params[0].v[X];
} /*cal_rXof_func*/


/****************************************************************************/
/*.doc cal_rYof_func(internal)*/
/*+
  Calculator 'rYof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_rYof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"rYof");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"rYof");
        return;
    }

    result.type = real_type;
    result.r    = params[0].v[Y];
} /*cal_rYof_func*/


/****************************************************************************/
/*.doc cal_rZof_func(internal)*/
/*+
  Calculator 'rZof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_rZof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"rZof");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"rZof");
        return;
    }

    result.type = real_type;
    result.r    = params[0].v[Z];
} /*cal_rZof_func*/


/****************************************************************************/
/*.doc cal_Xof_func(internal)*/
/*+
  Calculator 'Xof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_Xof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"Xof");
        return;
    }

    result.type = vector_type;
    result.v[X] = result.v[Y] = result.v[Z] = 0.0;

    if (params[0].type == vector_type) {
        result.v[X] = params[0].v[X];
    } else {
        result.v[X] = params[0].r;
    }
} /*cal_Xof_func*/


/****************************************************************************/
/*.doc cal_Yof_func(internal)*/
/*+
  Calculator 'Yof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_Yof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"Yof");
        return;
    }

    result.type = vector_type;
    result.v[X] = result.v[Y] = result.v[Z] = 0.0;

    if (params[0].type == vector_type) {
        result.v[Y] = params[0].v[Y];
    } else {
        result.v[Y] = params[0].r;
    }
} /*cal_Yof_func*/


/****************************************************************************/
/*.doc cal_Zof_func(internal)*/
/*+
  Calculator 'Zof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_Zof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"Zof");
        return;
    }

    result.type = vector_type;
    result.v[X] = result.v[Y] = result.v[Z] = 0.0;

    if (params[0].type == vector_type) {
        result.v[Z] = params[0].v[Z];
    } else {
        result.v[Z] = params[0].r;
    }
} /*cal_Zof_func*/


/****************************************************************************/
/*.doc cal_XYof_func(internal)*/
/*+
  Calculator 'XYof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_XYof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"XYof");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"XYof");
        return;
    }

    result = params[0];
    result.v[Z] = 0.0;
} /*cal_XYof_func*/


/****************************************************************************/
/*.doc cal_YZof_func(internal)*/
/*+
  Calculator 'YZof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_YZof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"YZof");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"YZof");
        return;
    }

    result = params[0];
    result.v[X] = 0.0;
} /*cal_YZof_func*/


/****************************************************************************/
/*.doc cal_XZof_func(internal)*/
/*+
  Calculator 'XZof()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_XZof_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"XZof");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"XZof");
        return;
    }

    result = params[0];
    result.v[Y] = 0.0;
} /*cal_XZof_func*/


/****************************************************************************/
/*.doc cal_ABS_func(internal)*/
/*+
  Calculator 'ABS()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ABS_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"ABS");
        return;
    }

    result.type = real_type;
    if (params[0].type == vector_type) {
        result.r = LENGTH(params[0].v);
    } else {
        result.r = fabs(params[0].r);
    }
} /*cal_ABS_func*/


/****************************************************************************/
/*.doc cal_ASIN_func(internal)*/
/*+
  Calculator 'ASIN()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ASIN_func()
{
    double a;

    if (no_of_params != 1) {
        error(4, /*MSG0*/"ASIN");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"ASIN");
        return;
    }

    a = params[0].r;
    result.type = real_type;

    if (fabs(a) > 1.1) {
        error(13, /*MSG0*/"ASIN");
        return;
    }

    if (a >  1.0) {
        a =  1.0;
    } else if (a < -1.0) {
        a = -1.0;
    }

    result.r = asin(a) / DEGRAD;
} /*cal_ASIN_func*/


/****************************************************************************/
/*.doc cal_ACOS_func(internal)*/
/*+
  Calculator 'ACOS()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ACOS_func()
{
    double a;

    if (no_of_params != 1) {
        error(4, /*MSG0*/"ACOS");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"ACOS");
        return;
    }

    a = params[0].r;
    result.type = real_type;

    if (fabs(a) > 1.1) {
        error(13, /*MSG0*/"ACOS");
        return;
    }

    if (a >  1.0) {
        a =  1.0;
    } else if (a < -1.0) {
        a = -1.0;
    }

    result.r = acos(a) / DEGRAD;
} /*cal_ACOS_func*/


/****************************************************************************/
/*.doc cal_ATAN_func(internal)*/
/*+
  Calculator 'ATAN()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ATAN_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"ATAN");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"ATAN");
        return;
    }

    result.type = real_type;
    result.r    = atan(params[0].r) / DEGRAD;
} /*cal_ATAN_func*/


/****************************************************************************/
/*.doc cal_SIN_func(internal)*/
/*+
  Calculator 'SIN()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_SIN_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"SIN");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"SIN");
        return;
    }

    result.type = real_type;
    result.r    = sin(DEGRAD * params[0].r);
} /*cal_SIN_func*/


/****************************************************************************/
/*.doc cal_COS_func(internal)*/
/*+
  Calculator 'COS()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_COS_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"COS");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"COS");
        return;
    }

    result.type = real_type;
    result.r    = cos(DEGRAD * params[0].r);
} /*cal_COS_func*/


/****************************************************************************/
/*.doc cal_TANG_func(internal)*/
/*+
  Calculator 'TANG()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_TANG_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"TANG");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"TANG");
        return;
    }
    if (fabs(fmod(fabs(params[0].r), 180.0) - 90.0) < EPS) {
        error(24, NULL);
        return;
    }

    result.type = real_type;
    result.r    = tan(DEGRAD * params[0].r);
} /*cal_TANG_func*/


/****************************************************************************/
/*.doc cal_ROUND_func(internal)*/
/*+
  Calculator 'ROUND()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ROUND_func()
{
    double r;

    if (no_of_params != 1) {
        error(4, /*MSG0*/"ROUND");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"ROUND");
        return;
    }

    r = floor(params[0].r + 0.5);

    if ((r > 32767) || (r < -32768)) {
        error(14, NULL);
        return;
    }

    result.type = int_type;
    result.r    = r;
} /*cal_ROUND_func*/


/****************************************************************************/
/*.doc cal_TRUNC_func(internal)*/
/*+
  Calculator 'TRUNC()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_TRUNC_func()
{
    double r;

    if (no_of_params != 1) {
        error(4, /*MSG0*/"TRUNC");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"TRUNC");
        return;
    }

    r = params[0].r;

    if ((r > 32767) || (r < -32768)) {
        error(14, NULL);
        return;
    }

    result.type = int_type;
    result.r    = (int)(r);
} /*cal_TRUNC_func*/


/****************************************************************************/
/*.doc cal_SQR_func(internal)*/
/*+
  Calculator 'SQR()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_SQR_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"SQR");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"SQR");
        return;
    }

    result.type = real_type;
    result.r    = SQR(params[0].r);
} /*cal_SQR_func*/


/****************************************************************************/
/*.doc cal_SQRT_func(internal)*/
/*+
  Calculator 'SQRT()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_SQRT_func()
{
    double r;

    if (no_of_params != 1) {
        error(4, /*MSG0*/"SQRT");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"SQRT");
        return;
    }

    r = params[0].r;

    if (r < -EPS) {
        error(15, NULL);
        return;
    }
    if (r < 0.0) {
        r = 0.0;
    }

    result.type = real_type;
    result.r    = sqrt(r);
} /*cal_SQRT_func*/


/****************************************************************************/
/*.doc cal_R2Dfunc(internal)*/
/*+
  Calculator 'R2D()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_R2D_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"R2D");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"R2D");
        return;
    }

    result.type = real_type;
    result.r    = params[0].r / DEGRAD;
} /*cal_R2D_func*/


/****************************************************************************/
/*.doc cal_D2R_func(internal)*/
/*+
  Calculator 'D2R()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_D2R_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"D2R");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"D2R");
        return;
    }

    result.type = real_type;
    result.r    = params[0].r * DEGRAD;
} /*cal_D2R_func*/


/****************************************************************************/
/*.doc cal_LN_func(internal)*/
/*+
  Calculator 'LN()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_LN_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"LN");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"LN");
        return;
    }
    if (params[0].r <= 0.0) {
        error(38, /*MSG0*/"LN");
        return;
    }
    
    result.type = real_type;
    result.r    = log(params[0].r);
} /*cal_LN_func*/


/****************************************************************************/
/*.doc cal_LOG_func(internal)*/
/*+
  Calculator 'LOG()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_LOG_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"LOG");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"LOG");
        return;
    }
    if (params[0].r <= 0.0) {
        error(38, /*MSG0*/"LOG");
        return;
    }

    result.type = real_type;
    result.r    = log10(params[0].r);
} /*cal_LOG_func*/


/****************************************************************************/
/*.doc cal_EXP_func(internal)*/
/*+
  Calculator 'EXP()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_EXP_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"EXP");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"EXP");
        return;
    }

    result.type = real_type;
    result.r    = exp(params[0].r);
} /*cal_EXP_func*/


/****************************************************************************/
/*.doc cal_EXP10_func(internal)*/
/*+
  Calculator 'EXP10()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_EXP10_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"EXP10");
        return;
    }
    if (!IS_REAL(0)) {
        error(12, /*MSG0*/"EXP10");
        return;
    }

    result.type = real_type;
    result.r    = exp(log(10.0) * params[0].r);
} /*cal_EXP10_func*/


/****************************************************************************/
/*.doc cal_U2W_func(internal)*/
/*+
  Calculator 'U2W()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_U2W_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"U2W");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"U2W");
        return;
    }

    result.type = vector_type;
    sa_u2w(params[0].v, result.v);
} /*cal_U2W_func*/


/****************************************************************************/
/*.doc cal_W2U_func(internal)*/
/*+
  Calculator 'W2U()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_W2U_func()
{
    if (no_of_params != 1) {
        error(4, /*MSG0*/"W2U");
        return;
    }
    if (!IS_VECTOR(0)) {
        error(11, /*MSG0*/"W2U");
        return;
    }

    result.type = vector_type;
    sa_w2u(params[0].v, result.v);
} /*cal_W2U_func*/


/****************************************************************************/
/*.doc cal_ANG_func(internal)*/
/*+
  Calculator 'ANG()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ANG_func()
{
    double    ang;
    ads_point p0, p1, p2, p3, origin;
    int       i, success;
    int       all_vectors = TRUE;

    for (i = 0; i < no_of_params; i++) {
        if (!IS_VECTOR(i)) {
            all_vectors = FALSE;
            break;
        }
    }

    if (!all_vectors || (no_of_params < 1) || (no_of_params > 4)) {
        error(26, /*MSG0*/"ANG(v), ANG(p1,p2), ANG(apex,p1,p2), ANG(apex,p1,p2,p)");
        return;
    }

    CPY_PNT(p0, params[0].v);
    CPY_PNT(p1, params[1].v);
    CPY_PNT(p2, params[2].v);
    CPY_PNT(p3, params[3].v);
    origin[X] = origin[Y] = origin[Z] = 0.0;

    if (no_of_params == 1) {
        p0[Z] = 0.0;
        if (LENGTH(p0) < EPS) {
            error(56, NULL);
            return;
        }
        ang = ads_angle(origin, p0);

    } else if (no_of_params == 2) {
        p0[Z] = p1[Z] = 0.0;
        if (DISTANCE(p0, p1) < EPS) {
            error(39, NULL);
            return;
        }
        ang = ads_angle(p0, p1);

    } else if (no_of_params == 3) {
        p0[Z] = p1[Z] = p2[Z] = 0.0;

        if ((DISTANCE(p0, p1) < EPS) ||
            (DISTANCE(p0, p2) < EPS) ||
            (DISTANCE(p1, p2) < EPS)) {
            error(57, NULL);
            return;
        }
        ang = ads_angle(p0, p2) - ads_angle(p0, p1);

    } else {                          /*no_of_params == 4*/

        ads_point normal, v1, v2;
        double    d, dist;

        if ((DISTANCE(p0, p1) < EPS) || (DISTANCE(p0, p2) < EPS) ||
            (DISTANCE(p0, p3) < EPS) || (DISTANCE(p1, p2) < EPS) ||
            (DISTANCE(p1, p3) < EPS) || (DISTANCE(p2, p3) < EPS))  {
            error(37, NULL);
            return;
        }

        success = sa_plane_equation(TRUE, p0, p1, p2, normal, &d);
        if (success == RTNORM) {
            dist = DOTPROD(normal, p3) + d;
            if (fabs(dist) < EPS) {
                error(36, NULL);
                return;
            }
            if (dist > 0.0) {
                ADD_PNT(p3, p0, normal);
            } else {
                SUB_PNT(p3, p0, normal);
            }
            ang = sa_angle_around_axis(p0, p3, p1, p2);
        } else {
            SUB_PNT(v1, p1, p0);
            SUB_PNT(v2, p2, p0);
            if (DOTPROD(v1, v2) > 0.0) {
                ang = 0.0;
            } else {
                ang = PI;
            }
        }
    }

    sa_normalize_angle(&ang);
    result.type = real_type;
    result.r    = ang / DEGRAD;
} /*cal_ANG_func*/


/****************************************************************************/
/*.doc cal_RAD_func(internal)*/
/*+
  Calcualtor 'RAD' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_RAD_func()
{
    int    success;
    double rad;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"RAD");
        return;
    }

    success = sa_get_radius_of_picked_entity(
XMSG(">> Select circle, arc or polyline segment for RAD function:\n", 1),
                                             &rad);
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    result.type = real_type;
    result.r    = rad;
} /*cal_RAD_func*/


/****************************************************************************/
/*.doc cal_NOR_func(internal)*/
/*+
  Claculator 'NOR()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_NOR_func()
{
    int        i, success;
    ads_point  nor;
    double     len;
    int        all_vectors = TRUE;

    for (i = 0; i < no_of_params; i++) {
        if (!IS_VECTOR(i)) {
            all_vectors = FALSE;
            break;
        }
    }

    if (!all_vectors || (no_of_params > 3)) {
        error(26, /*MSG0*/"NOR, NOR(v), NOR(p1,p2), NOR(p1,p2,p3)");
        return;
    }

    if (no_of_params == 0) {          /*3D normal to picked entity*/
        ads_point origin, xaxis, yaxis, zaxis;

        success = sa_get_cs_of_picked_entity(
XMSG(">> Select circle, arc or polyline for NOR function:\n", 2),
FALSE, origin,
xaxis, yaxis, zaxis);
        if (success != RTNORM) {
            error(0, NULL);
            return;
        }

        SUB_PNT(nor, zaxis, origin);
        sa_normalize(nor);

    } else if (no_of_params == 1) {   /*2D normal to vector*/

        nor[X] = -params[0].v[Y];
        nor[Y] =  params[0].v[X];
        nor[Z] = 0.0;

        len = LENGTH(nor);
        if (len < EPS) {
            error(30, NULL);
            return;
        }

        nor[X] /= len;
        nor[Y] /= len;

    } else if (no_of_params == 2) {   /*2D normal to line*/

        nor[X] = params[0].v[Y] - params[1].v[Y];
        nor[Y] = params[1].v[X] - params[0].v[X];
        nor[Z] = 0.0;

        len = LENGTH(nor);
        if (len < EPS) {
            error(61, /*MSG0*/"NOR(p1,p2)");
            return;
        }

        nor[X] /= len;
        nor[Y] /= len;

    } else {                          /*3D normal to plane by 3 points*/
        double d;

        if (!IS_VECTOR(0) || !IS_VECTOR(1) || !IS_VECTOR(2)) {
            error(26, /*MSG0*/"NOR, NOR(v), NOR(p1,p2), NOR(p1,p2,p3)");
            return;
        }
        success = sa_plane_equation(TRUE, params[0].v, params[1].v, 
                                    params[2].v,
                                    nor, &d);
        if (success != RTNORM) {
            error(51, NULL);
            return;
        }
    }

    result.type = vector_type;
    CPY_PNT(result.v, nor);
} /*cal_NOR_func*/


/****************************************************************************/
/*.doc cal_ILL_func(internal)*/
/*+
    Calculator 'ILL()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ILL_func()
{
    int success;

    if ((no_of_params != 4) ||
        !IS_VECTOR(0) || !IS_VECTOR(1) || !IS_VECTOR(2) || !IS_VECTOR(3)) {
        error(26, /*MSG0*/"ILL(p1,p2,p3,p4)");
        return;
    }

    if ((DISTANCE(params[0].v, params[1].v) < EPS) ||
        (DISTANCE(params[2].v, params[3].v) < EPS)) {
        error(40, /*MSG0*/"ILL");
        return;
    }

    result.type = vector_type;
    success = ads_inters(params[0].v, params[1].v,
                         params[2].v, params[3].v, 0, result.v);

    if (success != RTNORM) {
        error(54, /*MSG0*/"ILL");
        return;
    }
} /*cal_ILL_func*/


/****************************************************************************/
/*.doc cal_ILP_func(internal)*/
/*+
    Calculator 'ILP()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ILP_func()
{
    ads_point p, v, n;
    double    d, t, denom;
    int       success;

    if ((no_of_params != 5)            || 
        !IS_VECTOR(0) || !IS_VECTOR(1) || 
        !IS_VECTOR(2) || !IS_VECTOR(3) || !IS_VECTOR(4)) {
        
        error(26, /*MSG0*/"ILP(p1,p2,p3,p4,p5)");
        return;
    }
    
    CPY_PNT(p, params[0].v);
    SUB_PNT(v, params[1].v, params[0].v);

    if (LENGTH(v) < EPS) {
        error(40, /*MSG0*/"ILP");
        return;
    }

    success = sa_plane_equation(TRUE, params[2].v, params[3].v,
                                params[4].v, 
                                n, &d);
    if (success != RTNORM) {
        error(59, NULL);
        return;
    }

    denom = DOTPROD(n, v);
    if (fabs(denom) < EPS) {
        if (fabs(DOTPROD(p, n) + d) < EPS)
            error(62, NULL);
        else
            error(55, NULL);
        return;
    }
    t = -(d + DOTPROD(n, p)) / denom;

    result.type = vector_type;
    result.v[X] = p[X] + t * v[X];
    result.v[Y] = p[Y] + t * v[Y];
    result.v[Z] = p[Z] + t * v[Z];

} /*cal_ILP_func*/


/****************************************************************************/
/*.doc cal_DPL_func(internal)*/
/*+
    Calculator 'DPL()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_DPL_func()
{
    ads_point q, p, v, qp, a;
    double    len, t;

    if ((no_of_params != 3) ||
        !IS_VECTOR(0) || !IS_VECTOR(1) || !IS_VECTOR(2)) {
        
        error(26, /*MSG0*/"DPL(p,p1,p2)");
        return;
    }

    CPY_PNT(q, params[0].v);
    CPY_PNT(p, params[1].v);
    SUB_PNT(v, params[2].v, params[1].v);

    len = LENGTH(v);
    if (len < EPS) {
        error(40, /*MSG0*/"DPL");
        return;
    }

    v[X] /= len;
    v[Y] /= len;
    v[Z] /= len;

    SUB_PNT(qp, q, p);
    t = DOTPROD(v, qp);
    a[X] = p[X] + t * v[X];
    a[Y] = p[Y] + t * v[Y];
    a[Z] = p[Z] + t * v[Z];

    result.type = real_type;
    result.r    = DISTANCE(q, a);

} /*cal_DPL_func*/


/****************************************************************************/
/*.doc cal_DPP_func(internal)*/
/*+
    Calculator 'DPP()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_DPP_func()
{
    ads_point n;
    double    d;
    int       success;

    if ((no_of_params != 4) ||
        !IS_VECTOR(0) || !IS_VECTOR(1) || !IS_VECTOR(2) || !IS_VECTOR(3)) {
        
        error(26, /*MSG0*/"DPP(p,p1,p2,p3)");
        return;
    }
    
    success = sa_plane_equation(TRUE, params[1].v, params[2].v,
                                params[3].v, 
                                n,&d);
    if (success != RTNORM) {
        error(59, NULL);
        return;
    }

    result.type = real_type;
    result.r    = fabs(DOTPROD(n, params[0].v)+d);
} /*cal_DPP_func*/


/****************************************************************************/
/*.doc cal_PLT_func(internal)*/
/*+
    Calculator 'PLT()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_PLT_func()
{
    double t1, t2;

    if ((no_of_params != 3) ||
        !IS_VECTOR(0) || !IS_VECTOR(1) || !IS_REAL(2)) {
        error(26, /*MSG0*/"PLT(p1,p2,t)");
        return;
    }

    t2 = params[2].r;
    t1 = 1.0 - t2;

    result.type = vector_type;
    result.v[X] = t1 * params[0].v[X] + t2 * params[1].v[X];
    result.v[Y] = t1 * params[0].v[Y] + t2 * params[1].v[Y];
    result.v[Z] = t1 * params[0].v[Z] + t2 * params[1].v[Z];

} /*cal_PLT_func*/


/****************************************************************************/
/*.doc cal_PLD_func(internal)*/
/*+
    Calculator 'PLD()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_PLD_func()
{
    double d, d12, t1, t2;

    if ((no_of_params != 3) ||
        !IS_VECTOR(0) || !IS_VECTOR(1) || !IS_REAL(2)) {
        error(26, /*MSG0*/"PLD(p1,p2,dist)");
        return;
    }

    d = params[2].r;
    d12 = DISTANCE(params[0].v, params[1].v);

    if (d12 < EPS) {
        error(58, NULL);
        return;
    }

    t2 = d / d12;
    t1 = 1.0 - t2;

    result.type = vector_type;
    result.v[X] = t1 * params[0].v[X] + t2 * params[1].v[X];
    result.v[Y] = t1 * params[0].v[Y] + t2 * params[1].v[Y];
    result.v[Z] = t1 * params[0].v[Z] + t2 * params[1].v[Z];

} /*cal_PLD_func*/


/****************************************************************************/
/*.doc pick_end_end(internal) */
/*+
    Prompts the user to pick two entities by cursor and returns coordinates 
    of their endpoints in 'p1' and 'p2'.

    Function returns one of the standard ADS result codes.
-*/
/****************************************************************************/


static int
/*FCN*/pick_end_end(p1, p2, fcn_name)

  ads_point p1, p2;                   /* Returned endpoints */
  char      fcn_name[];               /* Prompt             */
{
    int  success;
    char prompt[128];
    int  ok = TRUE;

    do {
        strcpy(prompt, XMSG(">> Select one endpoint for ", 3));
        strcat(prompt, fcn_name);
        strcat(prompt, ":\n");

        success = sa_snap(prompt, /*MSG0*/"_END", p1);
        if (success != RTNORM)
            return(success);

        strcpy(prompt, XMSG(">> Select another endpoint for ", 4));
        strcat(prompt, fcn_name);
        strcat(prompt, ":\n");

        success = sa_snap(prompt, /*MSG0*/"_END", p2);
        if (success != RTNORM)
            return(success);

        if (DISTANCE(p1, p2) < EPS) {
            ads_printf(XMSG("\nThe two picked endpoints must be different.\n\n", 5));
            ok = FALSE;
        } else {
            ok = TRUE;
        }
    } while (!ok);

    return(RTNORM);
} /*pick_end_end*/


/****************************************************************************/
/*.doc cal_MEE_func(internal)*/
/*+
    Calculator 'MEE' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_MEE_func()
{
    ads_point p1, p2;
    int       success;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"MEE");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"MEE");
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    result.type = vector_type;
    result.v[X] = (p1[X] + p2[X]) / 2;
    result.v[Y] = (p1[Y] + p2[Y]) / 2;
    result.v[Z] = (p1[Z] + p2[Z]) / 2;

} /*cal_MEE_func*/


/****************************************************************************/
/*.doc cal_DEE_func(internal)*/
/*+
    calculator 'DEE' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_DEE_func()
{
    ads_point p1, p2;
    int       success;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"DEE");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"DEE");
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    result.type = real_type;
    result.r    = DISTANCE(p1, p2);

} /*cal_DEE_func*/


/****************************************************************************/
/*.doc cal_NEE_func(internal)*/
/*+
    Calculator 'NEE' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_NEE_func()
{
    ads_point p1, p2;
    ads_point n;
    double    len;
    int       success;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"NEE");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"NEE");
    if (success != RTNORM)  {
        error(61, /*MSG0*/"NEE");
        return;
    }

    n[X] = p1[Y] - p2[Y];
    n[Y] = p2[X] - p1[X];
    n[Z] = 0.0;

    len = LENGTH(n);
    if (len < EPS) {
        error(61, /*MSG0*/"NEE");
        return;
    }

    result.type = vector_type;
    result.v[X] = n[X] / len;
    result.v[Y] = n[Y] / len;
    result.v[Z] = 0.0;
} /*cal_NEE_func*/


/****************************************************************************/
/*.doc cal_VEE_func(internal)*/
/*+
    Calculator 'VEE' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_VEE_func()
{
    ads_point p1, p2;
    int       success;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"VEE");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"VEE");
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    result.type = vector_type;
    SUB_PNT(result.v, p2, p1);
} /*cal_VEE_func*/


/****************************************************************************/
/*.doc cal_VEE1_func(internal)*/
/*+
    Calculator 'VEE1' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_VEE1_func()
{
    ads_point p1, p2;
    int       success;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"VEE1");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"VEE1");
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    result.type = vector_type;
    SUB_PNT(result.v, p2, p1);
    sa_normalize(result.v);
} /*cal_VEE1_func*/


/****************************************************************************/
/*.doc cal_PLTEE_func(internal)*/
/*+
    Calculator 'PLTEE()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_PLTEE_func()
{
    double    t1, t2;
    int       success;
    ads_point p1, p2;

    if ((no_of_params != 1) || !IS_REAL(0)) {
        error(26, /*MSG0*/"PLTEE(t)");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"PLTEE");
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    t2 = params[0].r;
    t1 = 1.0 - t2;

    result.type = vector_type;
    result.v[X] = t1 * p1[X] + t2 * p2[X];
    result.v[Y] = t1 * p1[Y] + t2 * p2[Y];
    result.v[Z] = t1 * p1[Z] + t2 * p2[Z];

} /*cal_PLTEE_func*/


/****************************************************************************/
/*.doc cal_PLDEE_func(internal)*/
/*+
    Calculator 'PLDEE()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_PLDEE_func()
{
    double    d, d12, t1, t2;
    ads_point p1, p2;
    int       success;

    if ((no_of_params != 1) || !IS_REAL(0)) {
        error(26, /*MSG0*/"PLDEE(dist)");
        return;
    }

    success = pick_end_end(p1, p2, /*MSG0*/"PLDEE");
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    d = params[0].r;
    d12 = DISTANCE(p1, p2);

    t2 = d / d12;
    t1 = 1.0 - t2;

    result.type = vector_type;
    result.v[X] = t1 * p1[X] + t2 * p2[X];
    result.v[Y] = t1 * p1[Y] + t2 * p2[Y];
    result.v[Z] = t1 * p1[Z] + t2 * p2[Z];

} /*cal_PLDEE_func*/


/****************************************************************************/
/*.doc cal_ILLE_func(internal)*/
/*+
    Calculator 'ILLE()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ILLE_func()
{
    int       success;
    ads_point p1, p2, p3, p4;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"ILLE");
        return;
    }

    success = pick_end_end(p1, p2, XMSG("ILLE:First line", 6));
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    success = pick_end_end(p3, p4, XMSG("ILLE:Second line", 7));
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }

    result.type = vector_type;
    success = ads_inters(p1, p2, p3, p4, 0, result.v);
    if (success != RTNORM) {
        error(54, /*MSG0*/"ILLE");
        return;
    }

} /*cal_ILLE_func*/


/****************************************************************************/
/*.doc cal_DISP_func(internal)*/
/*+
    Calculator 'DIST()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_DIST_func()
{
    if ((no_of_params != 2) || !IS_VECTOR(0) || !IS_VECTOR(1)) {
        error(26, /*MSG0*/"DIST(p1,p2)");
        return;
    }

    result.type = real_type;
    result.r    = DISTANCE(params[0].v, params[1].v);
} /*cal_DIST_func*/


/****************************************************************************/
/*.doc cal_VEC_func(internal)*/
/*+
    Calculator 'VEC()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_VEC_func()
{
    if ((no_of_params != 2) || !IS_VECTOR(0) || !IS_VECTOR(1)) {
        error(26, /*MSG0*/"VEC(FromPt,ToPt)");
        return;
    }

    result.type = vector_type;
    SUB_PNT(result.v, params[1].v, params[0].v);
} /*cal_VEC_func*/


/****************************************************************************/
/*.doc cal_VEC1_func(internal)*/
/*+
    Calculator 'VEC1()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_VEC1_func()
{
    if ((no_of_params != 2) || !IS_VECTOR(0) || !IS_VECTOR(1)) {
        error(26, /*MSG0*/"VEC1(FromPt,ToPt)");
        return;
    }

    result.type = vector_type;
    SUB_PNT(result.v, params[1].v, params[0].v);
    sa_normalize(result.v);
} /*cal_VEC1_func*/


/****************************************************************************/
/*.doc cal_ROT_func(internal)*/
/*+
    Calculator 'ROT()' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_ROT_func()
{
    ads_point p, p1, p2;
    double    angle;
    int       success, given_3_points, given_4_points;

    given_3_points = ((no_of_params == 3) &&
                      IS_VECTOR(0) && IS_VECTOR(1) && IS_REAL(2));
    given_4_points = ((no_of_params == 4) &&
                      IS_VECTOR(0) && IS_VECTOR(1) &&
                      IS_VECTOR(2) && IS_REAL(3));

    if (!given_3_points && !given_4_points) {
        error(26, /*MSG0*/"ROT(p,origin,angle) or ROT(p,p1ax,p2ax,angle)");
        return;
    }

    CPY_PNT(p,  params[0].v);
    CPY_PNT(p1, params[1].v);

    if (given_3_points) {
        CPY_PNT(p2, p1);
        p2[Z] += 1.0;
        angle = params[2].r;
    } else {
        CPY_PNT(p2, params[2].v);
        angle = params[3].r;
    }

    if (DISTANCE(p1, p2) < EPS) {
        error(53, NULL);
        return;
    }

    success = sa_rotate_point_around_axis(p, (angle * DEGRAD), p1, p2);
    if (success != RTNORM) {
        error(48, NULL);
        return;
    }

    result.type = vector_type;
    CPY_PNT(result.v, p);
} /*cal_ROT_func*/


/****************************************************************************/
/*.doc cal_CUR_func(internal)*/
/*+
    Calculator 'CUR' function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_CUR_func()
{
    int success;

    if (no_of_params != 0) {
        error(25, /*MSG0*/"CUR");
        return;
    }
    
    result.type = vector_type;
    ads_initget(1+8, NULL);
    
    success = ads_getpoint(NULL, XMSG(">> Enter a point:\n", 8), result.v);
    
    if (success != RTNORM) {
        error(0, NULL);
        return;
    }
    set_lastpoint(result.v);
} /*cal_CUR_func*/


/****************************************************************************/
/*.doc cal_snap(internal)*/
/*+
   Local snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_snap(snap_mode)

  char snap_mode[];                   /* String "END", "MID", "CEN", etc. */
{
    char prompt[128];
    int  success;

    if (no_of_params != 0) {
        error(25, snap_mode);
        return;
    }

    strcpy(prompt, XMSG(">> Select entity for ", 9));

    /* Skip the underscore prefix of the snap mode */

    if (snap_mode[0] != '_') {
        strcat(prompt, snap_mode);
    } else {
        strcat(prompt, snap_mode + 1);
    }
    strcat(prompt, XMSG(" snap:\n", 10));

    result.type = vector_type;
    success = sa_snap(prompt, snap_mode, result.v);

    if (success != RTNORM) {
        error(0, NULL);
    }
    set_lastpoint(result.v);
} /*cal_snap*/


/****************************************************************************/
/*.doc cal_END_func(internal)*/
/*+
    Calculator 'END' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_END_func()
{
    cal_snap(/*MSG0*/"_END");
} /*cal_END_func*/


/****************************************************************************/
/*.doc cal_MID_func(internal)*/
/*+
    Calculator 'MID' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_MID_func()
{
    cal_snap(/*MSG0*/"_MID");
} /*cal_MID_func*/


/****************************************************************************/
/*.doc cal_CEN_func(internal)*/
/*+
    Calculator 'CEN' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_CEN_func()
{
    cal_snap(/*MSG0*/"_CEN");
} /*cal_CEN_func*/


/****************************************************************************/
/*.doc cal_INT_func(internal)*/
/*+
    Calculator 'INT' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_INT_func()
{
    cal_snap(/*MSG0*/"_INT");
} /*cal_INT_func*/


/****************************************************************************/
/*.doc cal_NEA_func(internal)*/
/*+
    Calculator 'NEA' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_NEA_func()
{
    cal_snap(/*MSG0*/"_NEA");
} /*cal_NEA_func*/


/****************************************************************************/
/*.doc cal_NOD_func(internal)*/
/*+
    Calculator 'NOD' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_NOD_func()
{
    cal_snap(/*MSG0*/"_NOD");
} /*cal_NOD_func*/


/****************************************************************************/
/*.doc cal_QUA_func(internal)*/
/*+
    Calculator 'QUA' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_QUA_func()
{
    cal_snap(/*MSG0*/"_QUA");
} /*cal_QUA_func*/


/****************************************************************************/
/*.doc cal_PER_func(internal)*/
/*+
    Calculator 'PER' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_PER_func()
{
    cal_snap(/*MSG0*/"_PER");
} /*cal_PER_func*/


/****************************************************************************/
/*.doc cal_TAN_func(internal)*/
/*+
    Calculator 'TAN' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_TAN_func()
{
    cal_snap(/*MSG0*/"_TAN");
} /*cal_TAN_func*/


/****************************************************************************/
/*.doc cal_INS_func(internal)*/
/*+
    Calculator 'INS' snap function.
-*/
/****************************************************************************/


static void
/*FCN*/cal_INS_func()
{
    cal_snap(/*MSG0*/"_INS");
} /*cal_INS_func*/


/****************************************************************************/
/*.doc cal_register_standard_functions(external)*/
/*+
  Registers the standard calculator functions in this file. This function
  is called only once during the phase of loading the calculator.
-*/
/****************************************************************************/


void
/*FCN*/cal_register_standard_functions()
{
    cal_register_function(/*MSG0*/"END",     cal_END_func  );
    cal_register_function(/*MSG0*/"MID",     cal_MID_func  );
    cal_register_function(/*MSG0*/"CEN",     cal_CEN_func  );
    cal_register_function(/*MSG0*/"INT",     cal_INT_func  );
    cal_register_function(/*MSG0*/"NEA",     cal_NEA_func  );
    cal_register_function(/*MSG0*/"NOD",     cal_NOD_func  );
    cal_register_function(/*MSG0*/"QUA",     cal_QUA_func  );
    cal_register_function(/*MSG0*/"PER",     cal_PER_func  );
    cal_register_function(/*MSG0*/"TAN",     cal_TAN_func  );
    cal_register_function(/*MSG0*/"INS",     cal_INS_func  );

    cal_register_function(/*MSG0*/"ANG",     cal_ANG_func  );
    cal_register_function(/*MSG0*/"RAD",     cal_RAD_func  );
    cal_register_function(/*MSG0*/"NOR",     cal_NOR_func  );

    cal_register_function(/*MSG0*/"SIN",     cal_SIN_func  );
    cal_register_function(/*MSG0*/"COS",     cal_COS_func  );
    cal_register_function(/*MSG0*/"TANG",    cal_TANG_func );
    cal_register_function(/*MSG0*/"ASIN",    cal_ASIN_func );
    cal_register_function(/*MSG0*/"ACOS",    cal_ACOS_func );
    cal_register_function(/*MSG0*/"ATAN",    cal_ATAN_func );

    cal_register_function(/*MSG0*/"ABS",     cal_ABS_func  );
    cal_register_function(/*MSG0*/"LN",      cal_LN_func   );
    cal_register_function(/*MSG0*/"LOG",     cal_LOG_func  );
    cal_register_function(/*MSG0*/"EXP",     cal_EXP_func  );
    cal_register_function(/*MSG0*/"EXP10",   cal_EXP10_func);
    cal_register_function(/*MSG0*/"SQR",     cal_SQR_func  );
    cal_register_function(/*MSG0*/"SQRT",    cal_SQRT_func );

    cal_register_function(/*MSG0*/"XOF",     cal_Xof_func  );
    cal_register_function(/*MSG0*/"YOF",     cal_Yof_func  );
    cal_register_function(/*MSG0*/"ZOF",     cal_Zof_func  );
    cal_register_function(/*MSG0*/"RXOF",    cal_rXof_func );
    cal_register_function(/*MSG0*/"RYOF",    cal_rYof_func );
    cal_register_function(/*MSG0*/"RZOF",    cal_rZof_func );
    cal_register_function(/*MSG0*/"XYOF",    cal_XYof_func );
    cal_register_function(/*MSG0*/"YZOF",    cal_YZof_func );
    cal_register_function(/*MSG0*/"XZOF",    cal_XZof_func );

    cal_register_function(/*MSG0*/"R2D",     cal_R2D_func  );
    cal_register_function(/*MSG0*/"D2R",     cal_D2R_func  );
    cal_register_function(/*MSG0*/"ROUND",   cal_ROUND_func);
    cal_register_function(/*MSG0*/"TRUNC",   cal_TRUNC_func);

    cal_register_function(/*MSG0*/"U2W",     cal_U2W_func  );
    cal_register_function(/*MSG0*/"W2U",     cal_W2U_func  );

    cal_register_function(/*MSG0*/"ILL",     cal_ILL_func  );
    cal_register_function(/*MSG0*/"ILP",     cal_ILP_func  );
    cal_register_function(/*MSG0*/"DPL",     cal_DPL_func  );
    cal_register_function(/*MSG0*/"DPP",     cal_DPP_func  );
    cal_register_function(/*MSG0*/"PLT",     cal_PLT_func  );
    cal_register_function(/*MSG0*/"PLD",     cal_PLD_func  );

    cal_register_function(/*MSG0*/"MEE",     cal_MEE_func  );
    cal_register_function(/*MSG0*/"DEE",     cal_DEE_func  );
    cal_register_function(/*MSG0*/"NEE",     cal_NEE_func  );
    cal_register_function(/*MSG0*/"VEE",     cal_VEE_func  );
    cal_register_function(/*MSG0*/"VEE1",    cal_VEE1_func );
    cal_register_function(/*MSG0*/"PLTEE",   cal_PLTEE_func);
    cal_register_function(/*MSG0*/"PLDEE",   cal_PLDEE_func);
    cal_register_function(/*MSG0*/"ILLE",    cal_ILLE_func );

    cal_register_function(/*MSG0*/"DIST",    cal_DIST_func );
    cal_register_function(/*MSG0*/"VEC",     cal_VEC_func  );
    cal_register_function(/*MSG0*/"VEC1",    cal_VEC1_func );
    cal_register_function(/*MSG0*/"ROT",     cal_ROT_func  );
    cal_register_function(/*MSG0*/"CUR",     cal_CUR_func  );

} /*cal_register_standard_functions*/


