{$G-,R-,S-,Q-}
{{$DEFINE QUANTIZETIMER}
{If defined, results from the timer method are adjusted by discarding the
least-significant bits.  The faster a machine you have, the less this is
necessary; if you have 12MHz or higher, don't use it at all.}
{{$DEFINE PARANOID} {extra wrapping around supposedly atomic events}

{
Unit to read joystick values from a db15 analog joystick.
20060225, trixter@oldskool.org.  Some information by Lou Duchez; assembler
implementations inspired by James P. McAdams and Bret Johnson
20090309: altered to use constants that make sense from TInterrupts;
adding quantization for timer-based method

Background:

The basic approach to reading a joystick is to monitor port 201h.  The eight
bits at that port correspond to:

01h - Joystick A, "X" position
02h - Joystick A, "Y" position
04h - Joystick B, "X" position
08h - Joystick B, "Y" position
10h - Joystick A, button 1
20h - Joystick A, button 2
40h - Joystick B, button 1
80h - Joystick B, button 2

The buttons are easy: a bit of "0" means "pressed" and "1" means "not
pressed".  But how do you get a variable X and Y axis from a single bit?!
Here's what you do:

1. Write a value -- any value -- to port 201h.  The four lowest bits will then
all assume a value of "1".

2. Start a counter, and see how many iterations it takes for your desired bit
to go to zero.  The number of iterations = the joystick position, with lower
values corresponding to "left" or "up" and higher values corresponding to
"right" or "down".

Now, what method to use to get the values?  You can do one of three things:

1. For maximum speed, LOOP with CX.  The CX register, coupled with LOOPNZ,
is the fastest method of doing this and results in the highest resolution.

2. On a machine with a BIOS date after November 8th, 1982, call the
Int 15h,84h BIOS routine.  On a real PC/XT, this employs method #1 for
all 4 axis and buttons (other implementations may vary).

3. For maximum compatibility, use the 8259 timer.  It's speed is constant
across all PC implementations.

Which method should you use?  That depends on whether or not you want
PRECISION or ACCURACY.  High PRECISION means you will get the most number of
values returned per axis.  High ACCURACY means you will get consistent values
for that joystick no matter how fast a machine you plug it into.  Here's a
handy chart:

System timer method:
PROS: High accuracy
CONS: Low precision on slow machines.  4.77MHz machine produces less than 50
positions per axis.

CX/LOOPNZ method:
PROS: High precision on any machine
CONS: Inconsistent accuracy (faster machines produce larger numbers)

If you're writing a game, the system timer method is best.  If you're using
the joystick as an all-points-addressable device (such as a drawing
program), the CX/LOOPNZ method is best (assuming the speed of the machine is
not variable).

Misc. notes:

- There is a bug in the implementation of some joysticks/adapters in that,
after waiting for one axis to go to 0, you need to wait for the other one to
go to 0 as well.  If you don't, and you fire the one-shots to read the stick a
second time before the other axis bit has settled, your results are
unpredictable (ie. wrong, stuck, always clear, etc.).

- You may be tempted to write a single routine that monitors all four bits and
spits out all four values, does the waiting, etc. -- but since that takes
time, on a slow machine all that processing can actually affect
the results!  I've personally measured this on a 4.77MHz 8088; extended
processing of the bits takes so long that the numbers returned by the
timer-based routine start to become very coarse.  This unit intentionally
treats each axis in a seperate block of code to ensure the least amount of
processing per axis, resulting in the most granular results possible.

- If you want to use the BIOS, your machine must be made after November
8th, 1982, because that's when IBM started putting the code in there.
Here is how:

                     INT 15,84 - Joy-Stick Support

        AH = 84h
        DX = 0 to read the current switch settings
           = 1 to read the joystick position (resistive inputs)

        on return (DX=0, read switch setting):
        CF = 0 if successful
           = 1 if error
        AH = 80h error code if CF set on PC
           = 86h error code if CF set on XT before 11/8/82
        AL = switch settings in bits 7-4 (if read switch function)

        on return (DX=1, read joystick position):
        AX = A(X)
        BX = A(Y)
        CX = B(X)
        DX = B(Y)
}

unit joystick;

interface

type
  polltype=(p_timer,p_loop,usebios);

const
  JoyPortAddr:word=$201;
  {bits for joystick_pos and buttons}
  jax=$01; {joystick a, x axis}
  jay=$02; {joystick a, y axis}
  jbx=$04; {joystick b, x axis}
  jby=$08; {joystick b, y axis}
  ja1=$10; {joystick a, button 1}
  ja2=$20; {joystick a, button 2}
  jb1=$40; {joystick b, button 1}
  jb2=$80; {joystick b, button 2}
  {unit variables}
  GotA:boolean=false;
  GotB:boolean=false;
  smoothing:boolean=false;
  pollmethod:polltype=p_timer; {This *must* be p_timer for this units's
  start-up code to work on all machines.}

function joystick_position(which_bit:byte):word;
function joystick_button(which_bit:byte):boolean;

implementation

uses
  TInterrupts;

const
  J_maxtimer=9999; {Maximum number of 0.8381 usec ticks}
  j_maxpolls=$4ff; {Maximum number of times we're willing to loop polling the
  joystick before we give up (cpu too fast, joystick not working, etc.)
  This number was derived from the IBM PC/XT BIOS disassembly; it limits the
  range from 0-1279.  The maximum number obtained from a 1.8 GHz Athlon was
  less than 1100 so it seems this number is okay.  In fact, since BIOS code
  runs faster than code in main memory, it is especially safe for us to use
  this number.}

function joystick_button;assembler;
{readport high-level:  result:=(port[$201] and mask)=0}
asm
  mov    bl,pollmethod
  cmp    bl,usebios     {compare; BIOS method requested?}
  jne    @readport      {no?  then jmp to port method, otherwise fall through}

@askbios:
  mov    ax,8400h       {AH=function call; set AL=0 for later}
  xor    dx,dx          {0 = read switch settings}
  int    15h
  jc     @buttonexit    {carry set?  if so, something went wrong}
  jmp    @returnbutton  {if not, process our bits}

@readport:
  MOV    DX,JoyPortADDR {PorT ADDR of JOYSTICKS}
  MOV    ah,which_bit   {MASK For DESIRED 1-SHOT}
  in     al,dx          {get joystick port bits}

@returnbutton:
  xor    al,$FF         {flip all bits}
  and    al,which_bit   {mask off our bit... al is our function result return}
@buttonexit:
end;

function joystick_position;assembler;
{
Returns:
  ax=number of ticks as joystick position (each tick is 0.8381 usec.)
}
asm
  mov    dx,JoyPortADDR {used in both methods' inner loops}
  mov    bl,pollmethod
  cmp    bl,p_loop      {compare; loop method requested?}
  je     @loop_method
  cmp    bl,usebios     {BIOS method requested?}
  je     @bios_method
  {otherwise, fall through to timer method}

@timer_method:
  mov    bl,which_bit   {mask for desired bit}
                        {Channel 0, Latch Counter, Rate Generator, Binary}
  mov    bh,iMC_Chan0+iMC_LatchCounter+iMC_OpMode3+iMC_BinaryMode
  mov    cx,j_maxtimer  {maximum compare value for inner loop below}
  MOV    AL,bh          {Begin building timer count}
  {$IFDEF PARANOID}
  PUSHF                 {Save interrupt state}
  CLI                   {Disable interrupts so our operationg is atomic}
  {$ENDIF}
  OUT    43h,AL         {Tell timer about it}
  IN     AL,40h         {Get LSB of timer counter}
  xchg   al,ah          {Save it in ah (xchg accum,reg is 3c 1b}
  IN     AL,40h         {Get MSB of timer counter}
  {$IFDEF PARANOID}
  POPF                  {Restore interrupt state}
  {$ENDIF}
  xchg   al,ah          {Put things in the right order; AX:=starting timer}
  mov    di,ax          {okay, now DI:=original timer value too}
  out    dx,al          {write random data to start the one-shots}
@read:
  MOV    AL,bh          {Use same Mode/Command as before (latch counter, etc.)}
  {$IFDEF PARANOID}
  PUSHF                 {Save interrupt state}
  CLI                   {Disable interrupts so our operation is atomic}
  {$ENDIF}
  OUT    43h,AL         {Tell timer about it}
  IN     AL,40h         {Get LSB of timer counter}
  xchg   al,ah          {Save it in ah for a second}
  IN     AL,40h         {Get MSB of timer counter}
  {$IFDEF PARANOID}
  POPF                  {Restore interrupt state}
  {$ENDIF}
  xchg   al,ah          {AX:=new timer value}
  mov    si,di          {copy original value to scratch}
  sub    si,ax          {subtract new value from old value}
  cmp    si,cx          {compare si to maximum time allowed}
  ja     @nostick       {if above, then we've waited too long -- blow doors}
  in     al,dx          {if we're still in time, read all eight bits}
  test   al,bl          {check axis bit we care about}
  jnz    @read          {loop while the bit tested isn't zero}
  {$IFDEF QUANTIZETIMER}{see top of file for explanation}
  shr    si,1
  shr    si,1
  shr    si,1
  {$ENDIF}
  jmp    @joy_exit      {si holds number of ticks}

@loop_method:
  mov    cx,j_maxpolls  {number of ticks to count down}
  mov    si,cx          {used later}
  mov    ah,which_bit   {mask for desired bit}
  PUSHF                 {save interrupt flag state, in case we are being called from an interrupt-driven procedure ourselves}
  CLI                   {turn off interrupts so our timing loop isn't affected}
  out    dx,al          {write random data to start the one-shots}
@readit:
  in     al,dx          {read all eight bits}
  test   al,ah          {check desired bit}
  loopnz @readit        {loop while the bit tested isn't zero (all hail LOOPNZ!)}
  POPF                  {turn interrupts back on}
  jcxz   @nostick       {if cx is 0 then we timed out and should abort}
  sub    si,cx          {si:=j_maxpolls - ticks counted}
  jmp    @joy_exit

@bios_method:
  mov    ax,8400h       {joystick function}
  mov    dx,1           {read sticks}
  xor    si,si          {assume the worst}
  int    15h            {do it}
  jc     @returnticks   {error?  si already 0, jmp to done}
  {process int15 -- this isn't efficient if you ONLY want to use BIOS method}
  mov    si,ax          {assume we wanted jax}
  test   which_bit,jax  {were we right?}
  jnz    @returnticks   {jump if desired AND jax <> 0}
  mov    si,bx          {assume jay}
  test   which_bit,jay
  jnz    @returnticks
  mov    si,cx          {assume jbx}
  test   which_bit,jbx
  jnz    @returnticks
  mov    si,dx          {oh well, must have been jby}
  jmp    @returnticks   {don't need to wait for bits to settle if using BIOS routine}

@nostick:
  xor    si,si          {no stick?  return 0}

@joy_exit:
  mov    ah,00001111b   {mask all four axis bits}
  mov    cx,j_maxpolls
@clearstick:            {This took me days to find.  You have to wait}
  in     al,dx          {for all AXIS bits to go to 0 before you can}
  test   al,ah          {reliably read the stick again!  See Misc. notes above}
  loopnz @clearstick

@returnticks:
  mov    ax,si          {return # of ticks as the value}
end;

begin
  {Set "GotA and "GotB" booleans by checking for "0" from the X axis
  of both sticks.  This is "safe" because it uses the most compatible
  method, timer-based polling.}
  GotA:=(joystick_position(jax)<>0);
  GotB:=(joystick_position(jbx)<>0);
end.
