{
JOYCALIB
Calibrates joysticks connected to a standard 15-pin joystick
port on any IBM PC or compatible.
20090315 trixter@oldskool.org

Version history:

20090309: First release, all features implemented.
}

{{$DEFINE DEBUG}

program joycalib;

uses
  strings,cmdlin,totfast,totsys,joystick,support;

const
  {global vars init'd to most common settings}
  jport:byte=0; {0=porta 1=portb}
  {joyportaddr and pollmethod are defined in the joystick unit}
  flakystick:boolean=false; {set TRUE if we encounter a bad joystick}

Procedure jInit;
{determine which port we're going to test, what address it is at,
what polling method to use}
const
  helplines=15;
  helptext:array [0..helplines-1] of pchar=(
  'Command-line usage: (parameters are case-sensitive)',
  '',
  '  /a        Tests joystick port A (default)',
  '  /b        Tests joystick port B',
  '  /mxxxx    Selects the polling method.  Choices are:',
  '              8253 - Uses the 8253 timer (default)',
  '              loop - Counts in a tight loop',
  '              bios - Uses interrupt 15,84 provided by the BIOS',
  '  /pxxx     Joystick hardware address in hex (default is 201)',
  '  /h or /?  Prints this help text',
  '',
  'Notes:',
  '',
  '- /p is provided for custom hardware projects only.  You should never have',
  '  a use for /p if you are calibrating standard joystick hardware.'
  );

  prelines=3;
  pretext:array [0..prelines-1] of pchar=(
  'Before starting, make sure that your joystick''s "springs" are enabled, so',
  'that the stick returns to the center position when let go.  Calibration only',
  'applies to self-centering sticks.'
  );
var
  c:char;
  bool:boolean;
  b:byte;
begin
  with Screen do begin
    {$IFDEF DEBUG}
    SnowProne:=0;
    {$ENDIF}
    clear($07,#32);
    writeln('JOYCALIB - Calibrates joysticks on IBM PCs and compatibles'#13#10'trixter@oldskool.org');
    if is_param('?') or is_param('h') then begin
      for b:=0 to helplines-1 do writeln(strpas(helptext[b]));
      halt(1);
    end else begin
      writeln('Use /? for command-line options.'#13#10);
    end;
    {check for different polling method - defaults to timer-based}
    if Param_Text('m')='loop' then pollMethod:=p_loop;
    if Param_Text('m')='bios' then pollMethod:=usebios;
    {check for which port to test}
    if is_param('b') then jport:=1;
    {check for different joystick address}
    if is_param('p') then JoyPortAddr:=HexStrToLong(Param_Text('p'));
    {finally, print everything out and wait before starting}
    writeln('Ready to start with the following settings: ');
    write('  '); if jport=0 then write('Port A') else write('Port B'); writeln(' will be tested.');
    write('  '); case pollMethod of
      p_timer:write('8253 Timer-based');
      p_loop:write('Tight loop-based');
      usebios:write('BIOS INT 15,84');
    end;
    writeln(' polling will be used to sample axis values.');
    writeln('  Joystick hardware is at port address $'+hex(joyportaddr)+' ('+inttostr(joyportaddr)+' dec).'#13#10);
    for b:=0 to prelines-1 do writeln(strpas(pretext[b]));
    writeln(#13#10'Press a key to begin, or ESC to exit.');
    c:=readkeychar;
    if c=#27 then begin
      writeln(#13#10'Exited by user...');
      halt(3);
    end;
    {is the stick even detected?  gota and gotb are acquired using the
    timer-based routine, which works on every compatible}
    if jport=0 then bool:=gota else bool:=gotb;
    if not bool then begin
      if jport=0 then c:='A' else c:='B';
      writeln(#13#10'Joystick not detected at port '+c+'!  Try other port?');
      halt(2);
    end;
  end; {with screen}
end;

Procedure jCalibrate;
{
Sampling PC analog joysticks can be tricky because poorly-made sticks
can have different ranges left/up and right/down of center.  However,
*calibrating* a bad stick is not the same as *sampling* a bad stick.
The same procedure works for calibrating bad sticks as well as good
ones, so we won't try to handle wacky hardware ranges.

Sadly, noisy sticks are more common (with age) and we do need to handle
that.  So we will take an average of samples for the boundaries instead
of a single one.  The user can see how crappy their stick is while
they're adjusting it; we need stable values for calibration.

Basic procedure is:
- Sample upper right, then lower left
- User lets stick return to center
- User adjusts trims so that spring-centered stick returns values in the
  middle of the sampled range

There are many different variations on the above, such as procedures that
attempt to compensate for uneven ranges, but Occam's Razor usually applies
in a situation this simple.
}

type
  sticksample=record
    x,y:word;
    but1,but2:boolean;
  end;
  srect=record
    x1,y1,x2,y2:byte;
  end;
const
  numsamples=32; {number of samples to take and average at stick boundaries; necessary for noisy/dirty sticks}
  diagrect:srect=(
    x1:1;y1:1;
    x2:18;y2:10
  );
  calibrect:srect=(
    x1:28;y1:2;
    x2:80;y2:24 {leave room at left and top sides for x/y axis, and bottom for status display}
  );
  {display labels}
  statusline:byte=25; coordsline=12;

var
  w,wx,wy:word;
  l,lx,ly:longint;
  minsample,maxsample,cursample,prevsample:sticksample;
  joybut1,joybut2,joyxval,joyyval:byte; {masks for joystick buttons and joystick axis}
  b:byte; c:char;
  s,st1,st2:string;
  calibcx,calibcy:byte; {center of calibration area}
  sampcounter,oldtick:longint;
  prevcalib,curcalib:sticksample; {used for repainting the screen}
  xr,yr:real;

  procedure updatestatus(s:string);
  begin
    with Screen do begin
      partclear(1,statusline,80,statusline,$07,#32);
      writeHI(1,statusline,$0f,$07,s);
    end;
  end;

  procedure samplestick(var samp:sticksample);
  begin
    samp.x:=joystick_position(joyxval);
    samp.y:=joystick_position(joyyval);
    samp.but1:=joystick_button(joybut1);
    samp.but2:=joystick_button(joybut2);
  end;

begin
  {assume port a; fill to port b if requested}
  joybut1:=ja1; joybut2:=ja2; joyxval:=jax; joyyval:=jay;
  if jport=1 then begin joybut1:=jb1; joybut2:=jb2; joyxval:=jbx; joyyval:=jby; end;
  with screen do begin
    Monitor^.Set25;
    clear($07,#32);
    {draw joystick diag to guide user; get stick boundaries}
    with diagrect do begin
      titledbox(x1,y1,x2,y2,$07,$0F,$07,2,'Stick Position');

      {get upper right boundary}
      writeat(x1+1,y1+1,$07,'');
      updatestatus('Move stick to the ~upper right~ position and press any joystick button.');
      repeat until joystick_button(joybut1) or joystick_button(joybut2);
      {we have to handle debounce ourselves}
      writeat(x1+1,y1+1,$0F,'');
      updatestatus('(release the joystick button)');
      repeat until not joystick_button(joybut1) and not joystick_button(joybut2);
      writeat(x1+1,y1+1,$07,'');
      updatestatus('Continue to ~hold~ the stick in this position while the stick is sampled');
      lx:=0; ly:=0;
      for b:=0 to numsamples-1 do begin
        {get an average of samples in this position}
        inc(lx,joystick_position(joyxval));
        inc(ly,joystick_position(joyyval));
      end;
      minsample.x:=lx div numsamples;
      minsample.y:=ly div numsamples;
      writeat(x1+1,y1+1,$07,'  ');

      {get lower left boundary}
      writeat(x2-2,y2-1,$07,'');
      updatestatus('Move stick to the ~lower right~ position and press any joystick button.');
      repeat until joystick_button(joybut1) or joystick_button(joybut2);
      {we have to handle debounce ourselves}
      writeat(x2-2,y2-1,$0F,'');
      updatestatus('(release the joystick button)');
      repeat until not joystick_button(joybut1) and not joystick_button(joybut2);
      writeat(x2-2,y2-1,$07,'');
      updatestatus('Continue to ~hold~ the stick in this position while the stick is sampled');
      lx:=0; ly:=0;
      for b:=0 to numsamples-1 do begin
        {get an average of samples in this position}
        inc(lx,joystick_position(joyxval));
        inc(ly,joystick_position(joyyval));
      end;
      maxsample.x:=lx div numsamples;
      maxsample.y:=ly div numsamples;
      {in case user was trigger happy and we get bogus reading, prevent
      division by zero by existing early}
      if (maxsample.x = minsample.x) or (maxsample.y = minsample.y) then with Screen do begin
        clear($07,#32);
        CursReset;
        writeln('Error: Both stick ranges were identical; did you move the stick as instructed?');
        writeln('It is also possible that your joystick buttons don''t debounce properly.');
        writeln('Run this program again, but hold the buttons down instead of tapping them.');
        halt(4);
      end;
      writeat(x2-2,y2-1,$0F,'  ');
      {show "stick" in the middle}
      writeat(((x1+x2) div 2) + 2,((y1+y2) div 2) + 1,$0F,'');
      box(x1+5,y1+2,x2-1,y2-1,$07,1);
    end; {with diagrect}

    {paint static unchanging areas}
    writeat(1,coordsline+0,$07,'Min. coords.: ('+intpadded(minsample.x,4)+','+intpadded(minsample.y,4)+')');
    writeat(1,coordsline+1,$07,'Max. coords.: ('+intpadded(maxsample.x,4)+','+intpadded(maxsample.y,4)+')');
    writeHI(1,coordsline+3,$0f,$07,'Ideal center: ~('
      +intpadded((minsample.x+maxsample.x) div 2,4)+','
      +intpadded((minsample.y+maxsample.y) div 2,4)+')~');
    with calibrect do begin
      titledbox(x1,y1,x2,y2,$07,$07,$07,2,'Calibration Area');
      {calc center of area for use later}
      calibcx:=x1+((x2-x1) div 2);
      calibcy:=y1+((y2-y1) div 2);
    end; {with calibrect}
    {centering box}
    box(calibcx-1,calibcy-1,calibcx+1,calibcy+1,$0f,1);

    {redraw dynamic areas until user presses escape}
    sampcounter:=0;
    updatestatus('Adjust your stick''s ~trims~ until the marker is centered.  Press ~Esc~ when done.');
    oldtick:=TicksSinceMidnight;
    repeat
      {update counter fun about once a second}
      {$IFDEF DEBUG}
      writeat(1,coordsline+11,$07,inttostr(tickssincemidnight)+' '+inttostr(oldtick+18));
      {$ENDIF}
      if tickssincemidnight>=oldtick+18 then begin
        oldtick:=tickssincemidnight;
        s:='~Samples/second: '+intpadded(sampcounter,6);
        writeat(1,coordsline+10,$07,s);
        sampcounter:=0;
      end;
      samplestick(cursample);
      {Clip stick values -- This is helpful in detecting noisy/dirty sticks
      but also helps keep the program sane.  First, let's determine if the
      stick is seriously going bad by seeing if any value is over twice
      the maximum (ie. takes twice as long to return)}
      if (cursample.x > maxsample.x*2)
      or (cursample.y > maxsample.y*2)
        then flakystick:=true;
      {Now we clip:}
      if cursample.x<minsample.x then cursample.x:=minsample.x;
      if cursample.y<minsample.y then cursample.y:=minsample.y;
      if cursample.x>maxsample.x then cursample.x:=maxsample.x;
      if cursample.y>maxsample.y then cursample.y:=maxsample.y;
      inc(sampcounter); {keep track of how many samples we've taken}
      {do we have a new sample to do something with?}
      {if so, update the realtime displays}
      if (cursample.x <> prevsample.x) or (cursample.y <> prevsample.y)
      or (cursample.but1 <> prevsample.but1) or (cursample.but2 <> prevsample.but2) then begin
        prevsample:=cursample; {update holding tank for previous sample}
        writeat(1,coordsline+2,$07,'Cur. coords.: ('+intpadded(cursample.x,4)+','+intpadded(cursample.y,4)+')');
        s:='Button 1: '; if cursample.but1 then s:=s+'ON ' else s:=s+'OFF';
        writeat(1,coordsline+4,$07,s);
        s:='Button 2: '; if cursample.but2 then s:=s+'ON ' else s:=s+'OFF';
        writeat(1,coordsline+5,$07,s);
        {update the little stick diagram area}
        with diagrect do begin
          {show little button displays}
          if cursample.but1 then writeat(x1+1,y1+2,$0f,'') else writeat(x1+1,y1+2,$07,'');
          if cursample.but2 then writeat(x1+3,y1+1,$0f,'') else writeat(x1+3,y1+1,$07,'');
          {and it couldn't hurt to show some little directional arrows with 50% thresholding}
          if cursample.y < round(maxsample.y * 0.25)
            then writeat(x1+10,y1+3,$07,#24#24)
            else writeat(x1+10,y1+3,$07,'  ');
          if cursample.y > round(maxsample.y * 0.75)
            then writeat(x1+10,y1+7,$07,#25#25)
            else writeat(x1+10,y1+7,$07,'  ');
          if cursample.x < round(maxsample.x * 0.25)
            then writeat(x1+6,y1+5,$07,#27#27)
            else writeat(x1+6,y1+5,$07,'  ');
          if cursample.x > round(maxsample.x * 0.75)
            then writeat(x1+14,y1+5,$07,#26#26)
            else writeat(x1+14,y1+5,$07,'  ');
        end; { with diagrect}
        {now the calibration area}
        with calibrect do begin
          WriteHScrollBar(X1,X2,Y1-1,$07,cursample.x-minsample.x,maxsample.x-minsample.x);
          WriteVScrollBar(x1-1,y1,y2,$07,cursample.y-minsample.y,maxsample.y-minsample.y);
        end; {with calibrect}
        {and finally the hovering centerhighlight}
        prevcalib:=curcalib;
        with prevcalib do attrib(x-1,y-1,x+1,y+1,$07);
        curcalib.x:=calibrect.x1 + round(((cursample.x-minsample.x)/(maxsample.x-minsample.x))*(calibrect.x2-calibrect.x1));
        curcalib.y:=calibrect.y1 + round(((cursample.y-minsample.y)/(maxsample.y-minsample.y))*(calibrect.y2-calibrect.y1));
        with curcalib do attrib(x-1,y-1,x+1,y+1,$70);
      end;
      if keypressed then c:=readkeychar;
    until c=#27;
  end;
end;

Procedure jDone;
begin
  {$IFNDEF DEBUG}
  with screen do begin
    Clear($07,#32);
    CursReset;
    Writeln('Finished.  Re-run this program a second time if necessary to refine results.'#13#10);
    if flakystick then begin
      writeln('During calibration, your stick sometimes took more than twice as much time');
      writeln('as usual to be sampled.  This usually indicates a worn-out or "dirty" stick.');
      writeln('If you were moving the stick normally during the test (ie. you didn''t wildly');
      writeln('thrash the stick around), consider repairing or replacing your joystick.');
    end;
  end;
  {$ENDIF}
end;

begin
  jInit;
  jCalibrate;
  jDone;
end.
